/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.json.osgi;

import java.util.concurrent.ExecutorService;
import javax.servlet.ServletException;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceReference;
import org.osgi.service.http.HttpService;
import org.osgi.service.http.NamespaceException;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import com.openexchange.groupware.settings.PreferencesItemService;
import com.openexchange.groupware.settings.tree.modules.interfaces.USM_JSON;
import com.openexchange.groupware.settings.tree.modules.interfaces.USM_JSONActive;
import com.openexchange.startup.ThreadControlService;
import com.openexchange.usm.api.configuration.ConfigurationManager;
import com.openexchange.usm.api.configuration.ConfigurationProperties;
import com.openexchange.usm.api.contenttypes.common.ContentTypeManager;
import com.openexchange.usm.api.exceptions.USMInvalidConfigurationException;
import com.openexchange.usm.api.exceptions.USMStartupException;
import com.openexchange.usm.api.session.SessionManager;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.USMJSONVersion;
import com.openexchange.usm.util.AbstractUSMActivator;

/**
 * {@link USMJSONServletActivator}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class USMJSONServletActivator extends AbstractUSMActivator {

    private volatile String servletAlias;
    private volatile ServiceTracker<ExecutorService, ExecutorService> executorTracker;
    private volatile ServiceTracker<ThreadControlService, ThreadControlService> threadControlTracker;

    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { HttpService.class, SessionManager.class, ContentTypeManager.class, ConfigurationManager.class };
    }

    @Override
    protected void register() {
        HttpService httpService = (HttpService) context.getService(presentServices.get(HttpService.class));
        SessionManager sessionManager = (SessionManager) context.getService(presentServices.get(SessionManager.class));
        ContentTypeManager ctManager = (ContentTypeManager) context.getService(presentServices.get(ContentTypeManager.class));
        ConfigurationManager cfManager = (ConfigurationManager) context.getService(presentServices.get(ConfigurationManager.class));

        String servletAlias;
        try {
            servletAlias = cfManager.getProperty(ConfigurationProperties.JSON_SERVLET_ALIAS_PROPERTY, ConfigurationProperties.JSON_SERVLET_ALIAS_DEFAULT, true);
            this.servletAlias = servletAlias;
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(ConnectorBundleErrorCodes.INVALID_ALIAS_FOR_SERVLET, e.getMessage(), e);
        }

        USMJSONServlet servlet = new USMJSONServlet(sessionManager, ctManager, cfManager);

        try {
            httpService.registerServlet(servletAlias, servlet, null, httpService.createDefaultHttpContext());
            LOG.info("Connector servlet registered under alias " + servletAlias);
            LOG.info("Open-Xchange USM/JSON Version " + USMJSONVersion.VERSION + " (Build " + USMJSONVersion.BUILD_NUMBER + ") successfully started");
        } catch (ServletException e) {
            throw new USMStartupException(ConnectorBundleErrorCodes.SERVLET_REGISTRATION_FAILED, "Servlet registration failed", e);
        } catch (NamespaceException e) {
            throw new USMStartupException(ConnectorBundleErrorCodes.SERVLET_ALIAS_IN_USE, "Servlet alias already in use:" + servletAlias, e);
        }

        registerService(USMJSONServlet.class, servlet);
        LOG.info("USM-JSON Servlet registered.");

        USM_JSON usmJSON = new USM_JSON();
        context.registerService(PreferencesItemService.class, usmJSON, null);
        LOG.info("USM_JSON Preferences Item Service registered");

        USM_JSONActive usmJSONActive = new USM_JSONActive();
        context.registerService(PreferencesItemService.class, usmJSONActive, null);
        LOG.info("USM_JSON active Preferences Item Service registered");

        //TODO provide tracking functionality via the AbstractUSMActivator
        final BundleContext context = this.context;
        ServiceTracker<ExecutorService, ExecutorService> executorTracker = new ServiceTracker<ExecutorService, ExecutorService>(context, ExecutorService.class, new ServiceTrackerCustomizer<ExecutorService, ExecutorService>() {
            @Override
            public ExecutorService addingService(ServiceReference<ExecutorService> reference) {
                Object vendor = reference.getProperty(Constants.SERVICE_VENDOR);
                if (null != vendor && "OX Software GmbH".equals(vendor.toString())) {
                    ExecutorService executor = context.getService(reference);
                    USMJSONServlet.setExecutorService(executor);
                    return executor;
                }

                return null;
            }

            @Override
            public void modifiedService(ServiceReference<ExecutorService> reference, ExecutorService executor) {
                // nothing
            }

            @Override
            public void removedService(ServiceReference<ExecutorService> reference, ExecutorService executor) {
                if (null != executor) {
                    Object vendor = reference.getProperty(Constants.SERVICE_VENDOR);
                    if (null != vendor && "OX Software GmbH".equals(vendor.toString())) {
                        USMJSONServlet.setExecutorService(null);
                    }
                    context.ungetService(reference);
                }
            }
        });
        this.executorTracker = executorTracker;
        executorTracker.open();

        ServiceTracker<ThreadControlService, ThreadControlService> threadControlTracker = new ServiceTracker<ThreadControlService, ThreadControlService>(context, ThreadControlService.class, new ThreadControlTracker(context));
        this.threadControlTracker = threadControlTracker;
        threadControlTracker.open();
    }

    @Override
    protected void unregister() {
        ServiceTracker<ExecutorService, ExecutorService> executorTracker = this.executorTracker;
        if (null != executorTracker) {
            executorTracker.close();
            this.executorTracker = null;
        }

        ServiceTracker<ThreadControlService, ThreadControlService> threadControlTracker = this.threadControlTracker;
        if (null != threadControlTracker) {
            this.threadControlTracker = null;
            threadControlTracker.close();
        }

        HttpService httpService = (HttpService) getService(HttpService.class);
        String servletAlias = this.servletAlias;
        if (servletAlias != null && httpService != null) {
            httpService.unregister(servletAlias);
            LOG.info("Connector servlet unregistered from alias " + servletAlias);
            this.servletAlias = null;
        } else {
            LOG.info("Connector servlet unregistered");
        }
        USMJSONServlet.setExecutorService(null);
        LOG.info("USM-JSON unregistered.");
    }

}
