/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.connector.sync;

import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.assets.ChangeState;

public class EmailSizeFilter implements DataObjectFilter {

    private static final int DEFAULT_MAIL_SIZE = 1 << 20;

    private final long _originalSyncID;

    private final int _oneSyncSizeLimit;

    private final int _maxEmailSize;

    private int _size;

    private boolean _hadTooBigEmails;

    public EmailSizeFilter(long originalSyncID, int onesyncSizeLimit, int maxEmailSize) {
        _originalSyncID = originalSyncID;
        _oneSyncSizeLimit = onesyncSizeLimit;
        _maxEmailSize = maxEmailSize;
    }

	@Override
    public void initialize() {
        _size = 0;
        _hadTooBigEmails = false;
    }

	@Override
    public boolean accept(DataObject object) {
	    if(_maxEmailSize <= 0)
	        return true;
        boolean isTooBig = getMailSize(object) > _maxEmailSize;
        _hadTooBigEmails |= isTooBig;
        return !isTooBig;
    }

	@Override
    public boolean addOperationToResult(ChangeState action, DataObject object) {
        if (object == null)
            return false;
        if (!DefaultContentTypes.MAIL_ID.equals(object.getContentType().getID()))
            return false;
        switch (action) {
        case CREATED:
        case MODIFIED:
            int mailSize = getMailSize(object);
            if (_size > 0 && _size + mailSize >= _oneSyncSizeLimit)
                return false;
            _size += mailSize;
            return true;
        default:
            return true;
        }
    }

    private int getMailSize(DataObject object) {
        Object o = object.getFieldContent("size");
        return (o instanceof Number) ? ((Number) o).intValue() : DEFAULT_MAIL_SIZE;
    }

	@Override
    public long getOriginalSyncID() {
        return _originalSyncID;
    }

    public boolean hadTooBigEmails() {
        return _hadTooBigEmails;
    }
}
