/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.connector.commands;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.HttpSessionUtil;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.USMJSONVersion;
import com.openexchange.usm.json.USMSessionCredentials;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.json.streaming.SimpleJSONReader;
import com.openexchange.usm.util.TempFile;
import com.openexchange.usm.util.TempFileStorage;
import com.openexchange.usm.util.Toolkit;

/**
 * {@link UploadDataHandler} Stores large data elements sent from the client locally in the server. These objects can be referenced by their
 * tempids on later sync-requests to avoid loading them completely into memory.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class UploadDataHandler implements CommandHandler {

    private static final String _ORIGINAL_SIZE = "original_size";

    private static final int _BUFFER_SIZE = 8192; // multiple of 4 for correct Base64-decoding of smaller parts

    private static boolean isValidBase64Character(int c) {
        return (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c >= '0' && c <= '9') || c == '+' || c == '/' || c == '-' || c == '_' || c == '=';
    }

    private final USMJSONServlet _servlet;

    private final HttpServletRequest _request;

    private Session _session;

    private TempFile _tempFile;

    public UploadDataHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        _servlet = servlet;
        _request = request;
    }

    @Override
    public Session getSession() {
        return _session;
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        SimpleJSONReader reader = createReader();
        try {
            if (reader.readNonWhitespace() != '[')
                throw new JSONException("Request is not a JSON array");
            String sessionID = reader.readString("sessionid", reader.readNonWhitespace());
            reader.read(',');
            USMSessionCredentials creds = HttpSessionUtil.getSession(_request.getSession(), sessionID);
            _session = BaseCommandHandler.retrieveUSMSession(_servlet, _request, creds);
            USMJSONVersion.verifyProtocolVersion(_session, USMJSONVersion.SUPPORTS_ATTACHMENT_STREAMING);
            BaseCommandHandler.initSessionRequestData(_servlet, _session);
            String tempid = reader.readString("tempid", reader.readNonWhitespace());
            reader.read(',');
            long offset = 0L;
            int c = reader.readNonWhitespace();
            if (c >= '0' && c <= '9') {
                offset = reader.readLong("offset", c, ',');
                c = reader.readNonWhitespace();
            }
            if (!SimpleJSONReader.isQuotationMark(c))
                throw new JSONException("data not provided as string");
            _tempFile = TempFileStorage.getTempFileForWrite(_session, tempid, offset);
            long originalSize = _tempFile.getSize();
            writeToFile(reader, c);
            reader.read(']');
            reader.read(-1);
            JSONObject data = new JSONObject();
            data.put(_ORIGINAL_SIZE, originalSize);
            return new ResponseObject(ResponseStatusCode.SUCCESS, data);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_UPLOAD_DATA_INVALID_REQUEST_FORMAT,
                ResponseStatusCode.BAD_REQUEST,
                "Invalid request data: " + e.getMessage());
        } catch (IOException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_UPLOAD_DATA_IOEXCEPTION_ON_LOCAL_FILE,
                ResponseStatusCode.INTERNAL_ERROR,
                "Error storing data locally: " + e.getMessage());
        } finally {
            Toolkit.close(_tempFile);
            Toolkit.close(reader);
        }
    }

    private SimpleJSONReader createReader() throws USMJSONAPIException {
        try {
            return new SimpleJSONReader(new BufferedReader(new InputStreamReader(_request.getInputStream(), CommandConstants.UTF_8)));
        } catch (UnsupportedEncodingException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_UPLOAD_DATA_UTF_8_NOT_AVAILABLE,
                ResponseStatusCode.INTERNAL_ERROR,
                CommandConstants.UTF_8 + " not available");
        } catch (IOException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_UPLOAD_DATA_CANNOT_OPEN_INPUT_STREAM,
                ResponseStatusCode.BAD_REQUEST,
                "Can not access InputStream of HttpRequest");
        }
    }

    private void writeToFile(SimpleJSONReader reader, int expectedQuotationMark) throws JSONException, IOException {
        StringBuilder sb = new StringBuilder(_BUFFER_SIZE);
        for (;;) {
            int c = reader.read();
            if (!isValidBase64Character(c)) {
                if (sb.length() > 0)
                    writeBufferToFile(sb);
                if (c == expectedQuotationMark)
                    return;
                throw new JSONException("Unexpected character in encoded content: '" + ((char) c) + '\'');
            }
            sb.append((char) c);
            if (sb.length() == _BUFFER_SIZE) {
                writeBufferToFile(sb);
                sb.delete(0, _BUFFER_SIZE);
            }
        }
    }

    private void writeBufferToFile(StringBuilder sb) throws IOException {
        _tempFile.write(Toolkit.decodeBase64(sb.toString()));
    }

    @Override
    public void disposeResources() {
        // do nothing
    }
}
