/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.connector.commands;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.UUID;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.ObjectChanges;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.session.dataobject.DataObjectSet;

/**
 * This class implements the PushNotificationType interface. It realized the listener for 'udp' art of the push notification. The objects of
 * this class stored the specified host and port of the client, that must be received information, if the change event occurs in the server.
 * 
 * @author ibr
 */
public class PushNotificationTypeUDP implements PushNotificationType {

    private String _host;

    private int _port;

    private Session _session;

    private static final String TYPE = "udp";

    public PushNotificationTypeUDP(String host, int port, Session session) {
        _host = host;
        _port = port;
        _session = session;
    }

    /**
     * this method realizes the reaction to the changes by the server content If the changes occurs, the udp data packet must be send to the
     * specified host and port
     */
    @Override
    public void changesOccurred(ObjectChanges changes) {
        if (changes.hasFolderStructureChanged()) {

            String[] parents = changes.getContentChangeParentIDs();
            StringBuilder packageData = new StringBuilder(256);
            for (String parentId : parents) {
                try {
                    if (parentId != null) {
                        UUID uuid = null;
                        Folder[] folders = _session.getCachedFolders();
                        DataObjectSet set = new DataObjectSet(folders);
                        DataObject folder = set.get(parentId);
                        if (folder != null) {
                            uuid = folder.getUUID();
                        } else {
                            SyncResult result = _session.syncWithServer(null, 0, null, false, null);
                            DataObject[] objects = result.getNewState();
                            DataObjectSet set2 = new DataObjectSet(objects);
                            DataObject folder2 = set2.get(parentId);
                            if (folder2 != null)
                                uuid = folder2.getUUID();

                        }
                        if (uuid != null) {
                            packageData.append(uuid);
                            packageData.append('\1');
                        }
                    }
                } catch (IllegalArgumentException e) {
                    continue;
                } catch (USMException e) {
                    // DEBUG
                    e.printStackTrace();
                    continue;
                }

            }
            try {
                InetAddress inetAdr = InetAddress.getByName(_host);
                DatagramSocket socket = new DatagramSocket();
                byte[] buf = packageData.toString().getBytes();
                DatagramPacket pack = new DatagramPacket(buf, buf.length, inetAdr, _port);
                pack.setLength(buf.length);
                socket.send(pack);
            } catch (UnknownHostException ignored) {
                // ignored
            } catch (IOException e1) {
                // DEBUG
                e1.printStackTrace();
            }
        }
    }

    @Override
    public String getType() {
        return TYPE;
    }

    public void setHost(String host) {
        _host = host;
    }

    public void setPort(int port) {
        _port = port;
    }

}
