/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.database.ox;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.context.ContextService;
import com.openexchange.database.DatabaseService;
import com.openexchange.exception.OXException;
import com.openexchange.usm.api.database.DatabaseAccessWithNoTimeoutSupport;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.util.SQLToolkit;
import com.openexchange.usm.util.Toolkit;

public class OXDBService implements DatabaseAccessWithNoTimeoutSupport {

    private static final String UNHEX = "UNHEX";

    private static final String HEX = "HEX";

    private final DatabaseService _delegate;

    private static final Log LOG = LogFactory.getLog(OXDBService.class);

    private final ContextService _contextService;

    /**
     * Initializes a new {@link OXDBService}.
     *
     * @param contextService
     * @param databaseService
     */
    public OXDBService(ContextService contextService, DatabaseService databaseService) {
        _contextService = contextService;
        _delegate = databaseService;
        LOG.info("USM Connection to OX DB service activated");
    }

    @Override
    public void backReadOnly(int contextID, Connection connection) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Releasing read only JDBC Connection for " + contextID);
        }
        _delegate.backReadOnly(contextID, connection);
    }

    @Override
    public void backWritable(int contextID, Connection connection) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Releasing read/write JDBC Connection for " + contextID);
        }
        _delegate.backWritable(contextID, connection);
    }

    @Override
    public void backNoTimeout(int contextID, Connection connection) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Releasing no-timeout JDBC Connection for " + contextID);
        }
        _delegate.backForUpdateTask(contextID, connection);
    }

    @Override
    public EncapsulatedConnection getReadOnly(int contextID) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Retrieving read only JDBC Connection for " + contextID);
        }
        try {
            return new EncapsulatedConnection(this, _delegate.getReadOnly(contextID), contextID, true);
        } catch (OXException e) {
            String errorMessage = "Couldn't get read only connection for " + contextID;
            LOG.error(errorMessage, e);
            throw new StorageAccessException(USMDatabaseOXErrorCodes.GET_READ_ONLY_ERROR, errorMessage, e);
        }
    }

    @Override
    public EncapsulatedConnection getWritable(int contextID) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Retrieving read/write JDBC Connection for " + contextID);
        }
        try {
            return new EncapsulatedConnection(this, _delegate.getWritable(contextID), contextID, false);
        } catch (OXException e) {
            String errorMessage = "Couldn't get read/write connection for " + contextID;
            LOG.error(errorMessage, e);
            throw new StorageAccessException(USMDatabaseOXErrorCodes.GET_READ_WRITE_ERROR, errorMessage, e);
        }
    }

    @Override
    public EncapsulatedConnection getNoTimeout(int contextID) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Retrieving no-timeout JDBC Connection for " + contextID);
        }
        try {
            return new EncapsulatedConnection(this, _delegate.getForUpdateTask(contextID), contextID, EncapsulatedConnection.Nature.NO_TIMEOUT);
        } catch (OXException e) {
            String errorMessage = "Couldn't get no-timeout connection for " + contextID;
            LOG.error(errorMessage, e);
            throw new StorageAccessException(USMDatabaseOXErrorCodes.GET_READ_WRITE_ERROR, errorMessage, e);
        }
    }

    @Override
    public void generateSequence(int contextID, String sequenceName) throws USMStorageException, StorageAccessException {
        EncapsulatedConnection con = getWritable(contextID);
        Statement stmt = null;
        try {
            try {
                stmt = con.createStatement();
                con.setAutoCommit(false);
                stmt.execute(SQLToolkit.createSequenceStatement(sequenceName));
                stmt.execute(SQLToolkit.createSequenceProcedure(sequenceName));
                con.commit();
                LOG.info("Creation of sequence " + sequenceName + " successfully performed");
            } catch (SQLException e) {
                con.rollback();
                throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR, e.getMessage(), e);
            } finally {
                con.setAutoCommit(true);
                Toolkit.close(stmt);
                Toolkit.close(con);
            }
        } catch (SQLException e) {
            throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR_AUTOCOMMIT_SETTINGS, e.getMessage(), e);
        }

    }

    @Override
    public int getNextIdFromSequence(int contextID, String sequenceName) throws USMStorageException, StorageAccessException {
        EncapsulatedConnection con = getWritable(contextID);
        try {
            return SQLToolkit.getNextIdFromSequence(con, sequenceName, contextID);
        } catch (SQLException e) {
            throw new USMStorageException(USMDatabaseOXErrorCodes.GET_NEXT_FROM_SEQUENCE_ERROR, e.getMessage(), e);
        } finally {
            Toolkit.close(con);
        }
    }

    @Override
    public String retrieveHexFunctionName() {
        return HEX;
    }

    @Override
    public String retrieveUnHexFunctionName() {
        return UNHEX;
    }

    @Override
    public List<Integer> getAllContextIds() throws StorageAccessException {
        try {
            return _contextService.getAllContextIds();
        } catch (OXException e) {
            throw new StorageAccessException(USMDatabaseOXErrorCodes.GET_ALL_CONTEXT_IDS_ERROR, "Error while retrieving all context ids", e);
        }
    }

    @Override
    public int[] getContextsInSameSchema(int contextId) throws StorageAccessException {
        try {
            return _delegate.getContextsInSameSchema(contextId);
        } catch (OXException e) {
            throw new StorageAccessException(
                USMDatabaseOXErrorCodes.GET_CONTEXTS_IN_SAME_SCHEMA_ERROR,
                "Error while retrieving all context ids in same schema as " + contextId,
                e);
        }
    }

    @Override
    public boolean supportsOnDuplicateKeyUpdateStatement() {
        return true;
    }

    @Override
    public boolean supportsInsertIgnoreStatement() {
        return true;
    }
}
