/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.datatypes.folder;

import java.io.Serializable;
import org.json.JSONException;
import org.json.JSONObject;

public class FolderPermission implements Serializable {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	// For non-mail folders, a number as described:
	/*
	Bits 	 Value
	0-6 	Folder permissions:
	    0 	No permissions.
	    1 	See the folder.
	    2 	Create objects in the folder.
	    4 	Create subfolders.
	    64 	Admin.
	7-13 	Read permissions for objects in the folder:
	    0 	No permissions.
	    1 	Read only own objects.
	    2 	Read all objects.
	    64 	Admin.
	14-20 	Write permissions for objects in the folder:
	    0 	No permissions.
	    1 	Modify only own objects.
	    2 	Modify all objects.
	    64 	Admin.
	21-27 	Delete permissions for objects in the folder:
	    0 	No permissions.
	    1 	Delete only own objects.
	    2 	Delete all objects.
	    64 	Admin.
	28 	Admin flag:
	    0 	No permissions.
	    1 	Allowed to modify permission
	 * 
	 */
	private final int _bits;
	public static final String BITS = "bits";
	// For mail folders, the rights string as defined in RFC 2086. 
	private final String _rights;
	public static final String RIGHTS = "rights";
	// User ID of the user or group to which this permission applies. 
	private final int _entity;
	public static final String ENTITY = "entity";
	// true if entity refers to a group, false if it refers to a user. 
	private final boolean _group;
	public static final String GROUP = "group";

	/**
	 * Constructor
	 * 
	 * @param bits
	 * @param rights
	 * @param entity
	 * @param group
	 */
	public FolderPermission(int bits, String rights, int entity, boolean group) {
		_bits = bits;
		_rights = rights;
		_entity = entity;
		_group = group;
	}

	public FolderPermission() {
		_bits = 0;
		_rights = "";
		_entity = 0;
		_group = false;
	}

	public FolderPermission(JSONObject jSONObject) throws JSONException {
		_bits = jSONObject.has(BITS) ? jSONObject.getInt(BITS) : 0;
		_rights = jSONObject.has(RIGHTS) ? jSONObject.getString(RIGHTS) : "";
		_entity = jSONObject.has(ENTITY) ? jSONObject.getInt(ENTITY) : 0;
		_group = jSONObject.has(GROUP) ? jSONObject.getBoolean(GROUP) : true;
	}

	/**
	 * 
	 * @return
	 */
	public int getBits() {
		return _bits;
	}

	/**
	 * 
	 * @return
	 */
	public String getRights() {
		return _rights;
	}

	public int getEntity() {
		return _entity;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isGroup() {
		return _group;
	}

	@Override
	public int hashCode() {
		final int prime = 37;
		int result = 1;
		result = prime * result + (_group ? 1 : 0);
		result = prime * result + _entity;
		result = prime * result + _rights.hashCode();
		result = prime * result + _bits;
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		FolderPermission other = (FolderPermission) obj;
		if (_group != other._group)
			return false;
		if (!_rights.equals(other._rights))
			return false;
		if (_entity != other._entity)
			return false;
		if (_bits != other._bits)
			return false;
		return true;
	}

	@Override
	public String toString() {
		return '{' + GROUP + ':' + _group + ';' + RIGHTS + ':' + _rights + ';' + ENTITY + ':' + _entity + ';' + BITS
				+ ':' + _bits + '}';
	}

	public JSONObject toJSONObject() throws JSONException {
		JSONObject js = new JSONObject();
		js.put(BITS, _bits); // XXX Not always present (only non-mail folders), do not throw Exception if not present
		js.put(RIGHTS, _rights); // XXX Not always present (only mail folders), do not throw Exception if not present
		js.put(ENTITY, _entity);
		js.put(GROUP, _group);
		return js;
	}
}
