/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.clt.eas12;

import org.apache.commons.codec.binary.Base64;

/**
 * {@link EASBase64RequestParser} copied from com.openexchange.usm.eas.servlet
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class EASBase64RequestParser {

    private final static String _HEX_CHARS = "0123456789abcdef";

    private static byte[] decodeBase64(String encodedString) {
        try {
            return Base64.decodeBase64(encodedString.getBytes("UTF-8"));
        } catch (Exception e) {
            throw new IllegalArgumentException("queryString not properly Base64 encoded", e);
        }
    }

    private final byte[] _requestData;

    private int _cursor;

    public EASBase64RequestParser(String queryString) {
        if (queryString == null)
            throw new IllegalArgumentException("No query String");
        _requestData = decodeBase64(queryString);
    }

    public EASBase64RequestParser(byte[] requestData) {
        _requestData = requestData;
    }

    public boolean hasMoreData() {
        return _cursor < _requestData.length;
    }

    public int readByte() {
        if (!hasMoreData())
            throw new IllegalArgumentException("Invalid encoded query String");
        return _requestData[_cursor++] & 0xFF;
    }

    public int readShort() {
        return readByte() * 0x100 + readByte();
    }

    public long readUnsignedInt() {
        return readShort() * 0x10000L + readShort();
    }

    public int[] readByteArray() {
        int length = readByte();
        int[] data = new int[length];
        for (int i = 0; i < data.length; i++) {
            data[i] = readByte();
        }
        return data;
    }

    public String readString() {
        int[] data = readByteArray();
        boolean hasInvalidCharacters = false;
        for (int i = 0; i < data.length; i++) {
            if (data[i] < 0x20 || data[i] > 0x7f) {
                hasInvalidCharacters = true;
                break;
            }
        }
        StringBuilder sb = new StringBuilder(hasInvalidCharacters ? data.length * 2 + 2 : data.length);
        if (hasInvalidCharacters) {
            // sb.append('0').append('x');
            for (int c : data)
                sb.append(_HEX_CHARS.charAt(c >> 4)).append(_HEX_CHARS.charAt(c & 0xF));
        } else {
            for (int c : data)
                sb.append((char) c);
        }
        return sb.toString();
    }
}
