/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.session.assets;

import java.io.Serializable;
import java.util.UUID;

/**
 * Holds the fields that identify an USM Object. 
 * @author ldo
 *
 */
public class OXObjectID implements Serializable, Comparable<OXObjectID> {

	public static OXObjectID fromSimplifiedUUID(UUID uuid) {
		if (uuid == null)
			return null;
		return new OXObjectID((int) (uuid.getMostSignificantBits() & 0xFFFFFFFFL), (int) (uuid
				.getLeastSignificantBits() >> 32), (int) (uuid.getLeastSignificantBits() & 0xFFFFFFFFL));
	}

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private final int _cid;
	private final int _contentType;
	private final int _objectId;

	public OXObjectID(int cid, int contentType, int objectId) {
		_cid = cid;
		_contentType = contentType;
		_objectId = objectId;
	}

	public int getCid() {
		return _cid;
	}

	public int getObjectId() {
		return _objectId;
	}

	public int getContentType() {
		return _contentType;
	}

	@Override
    public int compareTo(OXObjectID o) {
		int result = _cid == o._cid ? 0 : 1;
		if (result == 0) {
			result = _objectId == o._objectId ? 0 : 1;
			if (result == 0) {
				result = _contentType == o._contentType ? 0 : 1;
			}
		}
		return result;
	}

	@Override
	public String toString() {
		return "(cid: " + _cid + ", type: " + _contentType + ", id: " + _objectId + ')';
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + _cid;
		result = prime * result + _objectId;
		result = prime * result + _contentType;
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof OXObjectID))
			return false;
		OXObjectID other = (OXObjectID) obj;

		if (_cid != other._cid)
			return false;

		if (_objectId != other._objectId)
			return false;

		if (_contentType != other._contentType)
			return false;

		return true;
	}

	public UUID convertToSimplifiedUUID() {
		return new UUID(_cid, (((long) _contentType) << 32L) | (_objectId));
	}
}
