/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.session;

import java.io.Serializable;
import java.util.BitSet;
import java.util.UUID;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.exceptions.DeserializationFailedException;
import com.openexchange.usm.api.session.assets.ChangeState;

/**
 * Generic structure that holds an object of a specific ContentType and
 * can also be used to track or initiate changes on that object. The type
 * of change (creation, deletion, update) is stored in the ChangeState of
 * the object.
 * The DataObject is both used to notify clients of changes and by clients
 * to initiate changes to the system
 * 
 * @author afe
 *
 */
public interface DataObject extends Serializable {
	/**
	 * @return Session with which this DataObject is associated
	 */
	Session getSession();

	/**
	 * @return ContentType of this DataObject
	 */
	ContentType getContentType();

	/**
	 * 
	 * @return Type of change that either was performed or will be performed if submitted by the client
	 */
	ChangeState getChangeState();

	/**
	 * This method is used to mark a DataObject as CREATED or DELETED. Any other
	 * values are not allowed. A DataObject will be automatically marked as UNMODIFIED
	 * when calling commitChanges(), and an unmodified DataObject will be automatically
	 * marked MODIFIED as soon as a field is set to a different value.<br>
	 * Note that a DataObject is only marked as MODIFIED as long as at least one field
	 * differs, so if all modified fields are set to their original values, it will
	 * automatically change its state back to UNMODIFIED.
	 * 
	 * @param newState type of change this DataObject will be set to
	 * @throws IllegalArgumentException if newState is not CREATED or DELETED
	 */
	void setChangeState(ChangeState newState);

	/**
	 * @return for updates the timestamp of the DataObject before the change
	 */
	long getOriginalTimestamp();

	/**
	 * @return for updates the timestamp to use for the DataObject if the change is committed
	 */
	long getTimestamp();

	/**
	 * Sets the timestamp to use for the modified DataObject
	 * 
	 * @param timestamp milliseconds since midnight, January 1, 1970 UTC.
	 */
	void setTimestamp(long timestamp);

	/**
	 * @return original parent Folder ID (before changes)
	 */
	String getOriginalParentFolderID();

	/**
	 * @return currently set parent Folder ID
	 */
	String getParentFolderID();

	/**
	 * Changes the current parent Folder, i.e. sets the parent folder ID to the
	 * ID of the given Folder. As long as the parent folder ID or the parent folder
	 * are not changed, the method getParentFolder() will return the object that was
	 * used in this method. Note that setFieldContent() may update the parent folder id
	 * WITHOUT resetting the stored parent folder
	 * 
	 * @param folder
	 */
	void setParentFolder(Folder folder);

	/**
	 * Changes the current parent Folder ID
	 * 
	 * @param folderID
	 */
	void setParentFolderID(String folderID);

	/**
	 * 
	 * @return Folder that was previously used in setParentFolder, or null if the ID has been explicitly set
	 */
	Folder getParentFolder();

	/**
	 * @return original ID (before changes)
	 */
	String getOriginalID();

	/**
	 * @return currently set ID
	 */
	String getID();

	/**
	 * Sets the current ID to the given value
	 * 
	 * @param id
	 */
	void setID(String id);

	/**
	 * Retrieve current (new/modified) content of a field. For indexed fields this is
	 * always an array.
	 * 
	 * @param fieldName name of the field to retrieve
	 * @return field data
	 */
	Object getFieldContent(String fieldName);

	/**
	 * Retrieve current (new/modified) content of a field. For indexed fields this is
	 * always an array.
	 * 
	 * @param fieldIndex index of the field
	 * @return field data
	 */
	Object getFieldContent(int fieldIndex);

	/**
	 * @param fieldName
	 * @return true if the original and the current field content differ
	 */
	boolean isFieldModified(String fieldName);

	/**
	 * @param fieldIndex
	 * @return true if the original and the current field content differ
	 */
	boolean isFieldModified(int fieldIndex);

	/**
	 * Retrieve original (before modification) content of a field. For indexed fields this is
	 * always an array.
	 * 
	 * @param fieldName name of the field to retrieve
	 * @return field data
	 */
	Object getOriginalFieldContent(String fieldName);

	/**
	 * Retrieve original (before modification) content of a field. For indexed fields this is
	 * always an array.
	 * 
	 * @param fieldIndex
	 * @return field data
	 */
	Object getOriginalFieldContent(int fieldIndex);

	/**
	 * Change the current content of the specified field
	 * 
	 * @param fieldName
	 * @param data
	 */
	void setFieldContent(String fieldName, Object data);

	/**
	 * Change the current content of the specified field
	 * 
	 * @param fieldIndex
	 * @param data
	 */
	void setFieldContent(int fieldIndex, Object data);

	/**
	 * 
	 * @return true if at least one field is modified
	 */
	boolean isModified();

	/**
	 * Returns the current value of the failed flag (as previously set by setFailed or reset by commitChanges() or rollbackChanges())
	 * 
	 * @return
	 */
	boolean isFailed();

	/**
	 * Marks the current operation as failed.
	 * 
	 * @param failed true if current operation should be marked as failed
	 */
	void setFailed(boolean failed);

	/**
	 * Moves all current values to the original values and marks the DataObject
	 * as UNMODIFIED. This method only updates the internal fields of this DataObject,
	 * it does not perform any synchronization with the server/clients.<br>
	 * This method also resets the failed-flag to false.
	 */
	void commitChanges();

	/**
	 * Resets all current values to the original values and marks the DataObject
	 * as UNMODIFIED. This method only updates the internal fields of this DataObject,
	 * it does not perform any synchronization with the server/clients.<br>
	 * This method also resets the failed-flag to false.
	 * 
	 */
	void rollbackChanges();

	/**
	 * Create an exact copy of the current state of this DataObject.
	 * 
	 * @param linkUUIDs if true, modifications to a UUID will be performed in both the original and the copy
	 * @return
	 */
	DataObject createCopy(boolean linkUUIDs);

	/**
	 * performs as toString(), but only includes the requested fields (if the field index is set in the BitField) in the output
	 * 
	 * @param includedFields BitSet containing the indexes of fields to include in the output, if not specified (null), behaves as toString()
	 * @return
	 */
	String toString(BitSet includedFields);

	/**
	 * performs as toString(requestedFields), but only includes those fields that are modified or do not contain the default value
	 * 
	 * @param requestedFields
	 * @param skipUnsetFields
	 * @return
	 */
	String toString(BitSet requestedFields, boolean skipUnsetFields);

	/**
	 * Maps a given fieldName to a field index, throws an IllegalArgumentException for an unknown field name
	 * 
	 * @param fieldName
	 * @return index of the field with the given name
	 */
	int getFieldIndex(String fieldName);

	/**
	 * This method behaves as equals(), but it does not use the DataObjects ID for checking equality, i.e.
	 * equalsWithoutID() returns true if the objects are of the same type, have the same ChangeState, have the
	 * same session and all fields except the one for the ID have the same original and current content.<br>
	 * Note that equalsWithoutID() and equals() both ignore the failed flag and the original and current
	 * timestamps and last_modified fields for checking equality.
	 * 
	 * @param other
	 * @return true if both DataObjects are considered equal, ignoring the ID and last_modified field
	 */
	boolean equalsWithoutID(DataObject other);

    /**
     * This method behaves as equals(), but it also takes the last_modified-field into account. This field is normally not used by
     * protocols, but required to minimize local communication between USM and OX and therefore its current value must be stored in all
     * SyncStates.
     * 
     * @param other
     * @return true if both DataObjects are considered equal, ignoring no field
     */
    boolean equalsWithLastModified(DataObject other);

	/**
	 * This method stores all current individual field information into an array of Serializable objects.
	 * Only the current field information will be serialized, no extra information like ContentType, Session,
	 * ChangeState or Timestamp.
	 * 
	 * @return
	 */
	Serializable[] serialize();

	/**
	 * This method reads in previously serialized information. It also sets the timestamp to the provided
	 * value. After deserialization the DataObject will have the ChangeState UNMODIFIED.
	 * 
	 * @param timestamp
	 * @param data
	 * @throws DeserializationFailedException if the serialized information is not compatible with the ContentType of this DataObject, i.e. if the field count mismatches or at least one field is not of the correct type
	 */
	void deserialize(long timestamp, Serializable[] data) throws DeserializationFailedException;

	/**
	 * This method sets an optional UUID that can be used to uniquely identify this object. It will be
	 * stored in the sync states so that later calls can reference the DataObject by this UUID.
	 * Note that a UUID is shared between all copies of a DataObject, if you call createCopy() and later
	 * modify the UUID of either the original or the copy, the change is reflected in the other
	 * 
	 * @param uuid
	 */
	void setUUID(UUID uuid);

	/**
	 * Retrieves a previously set UUID.
	 * 
	 * @return
	 */
	UUID getUUID();

	/**
	 * Links the UUID of this DataObject to that of the source
	 * 
	 * @param source
	 */
	void linkUUIDTo(DataObject source);
	
	/**
	 * 
	 * @param info
	 */
	void setProtocolInformation(ProtocolInformation info);
	
	/**
	 * retrieves previously set protocol information. 
	 * @return
	 */
	ProtocolInformation getProtocolInformation();
	
	/**
	 * retrieves the user id of the parent folder owner.
	 * @return
	 */
	int getParentFolderOwnerID();
	
	/**
	 * sets the parent folder owner id field. 
	 * @param id
	 */
	void setParentFolderOwnerID(int id);
}
