/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.contenttypes.common;

import java.io.Serializable;
import com.openexchange.usm.api.contenttypes.transferhandlers.ContentTypeTransferHandler;
import com.openexchange.usm.api.exceptions.DeserializationFailedException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Session;

/**
 * This interface defines all relevant information for content types, i.e. type of objects that can be synchronized.<br>
 * Providers of ContentTypes have to make sure that their ContentTypes correctly implement hashCode() and equals() since ContentTypes will
 * be stored in Maps.<br>
 * By convention, the first field of a ContentType has to contain the ID for objects of that ContentType. In the second field, the parent
 * folder ID has to be stored. Using this convention, the USM sync system can make sure that the required fields are always retrieved
 * regardless of the field filter set by protocols.
 * 
 * @author afe
 */
public interface ContentType extends Serializable {

    /**
     * @return unique identifier for this content type, e.g. "task", "contacts"
     */
    String getID();

    /**
     * unique code for the content type, e.g. "tasks" == 4;
     * 
     * @return
     */
    int getCode();

    /**
     * @return Array of fields provided by this content type
     */
    ContentTypeField[] getFields();

    /**
     * This method is used to create a DataObject for this ContentType. A ContentType can either use a generic implementation that is
     * normally available through a separated bundle, or it can provide a more specific implementation that allows for better access in Java
     * through explicit getter- and setter-methods for its fields.
     * 
     * @param session Session for which the DataObject should be created
     * @return DataObject for this ContentType
     */
    DataObject newDataObject(Session session);

    /**
     * @return ContentTypeTransferHandler that is used by this ContentType for communication with an OX server
     */
    ContentTypeTransferHandler getTransferHandler();

    /**
     * This method is used by the USM on slow synchronizations to determine possible matches between client and server objects that have
     * different (or in the case of the client, possibly no) IDs. This method should examine the current content of both DataObjects and
     * return a value that indicates the likeliness of a match. A value < 0 means that the objects do not match, a value >= 0 indicates a
     * possible match. The higher the returned value is, the more likely is a match between the two DataObjects. Since only DataObjects of
     * the same ContentType can be compared, the ContentType can use any value range ( >= 0) and is not required to limit the result to a
     * specific range.<br>
     * The USM sync mechanism compares all client candidates with all server candidates, joins the two with the highest rating and
     * eliminates them from the candidate list, and repeats this process until no pair with a rating >= 0 is found.<br>
     * Note that normally only DataObjects within the same Folder are compared, only Folder elements are usually compared also if they have
     * different parent folders.
     * 
     * @param client DataObject of this ContentType as reported from a client
     * @param server DataObject of this ContentType as reported from the server
     * @return < 0 if no match, >= 0 if a match is likely, the higher, the likelier
     */
    int getMatchRating(DataObject client, DataObject server);

    /**
     * This method is called by the persistence subsystem of the sync system to restore the field content of a previously serialized
     * DataObject. Normally this method can simply copy the data into the field content array, but ContentTypes may change their fields and
     * old serialized information may be persisted. To enable the USM to handle "old" serialized data, each ContentType can introduce
     * additional code to perform a mapping from older versions to the current version.
     * 
     * @param data data that was deserialized, may contain extra information before the actual fields
     * @param i starting index to use for field extraction, all following indexes contain field data
     * @param fieldContent target field array of a DataObject in which the deserialized data should be stored
     * @throws DeserializationFailedException if for some reason the desrialization can not be performed
     */
    void deserializeData(Serializable[] data, int i, Object[] fieldContent) throws DeserializationFailedException;

    /**
     * Returns true if the content type supports attachments;
     * 
     * @return
     */
    boolean supportsPIMAttachments();

    /**
     * Returns the object id or the id of the "parent" object. For ex. when the object is Exception from Series Appointment it returns the
     * id of the Series App.
     * 
     * @param object
     * @return
     */
    Object getObjectGroupOwner(DataObject object);

    /**
     * Returns true if the content type can be folder elements content type. ContenTypes like folder or groups return false because objects
     * from this content type can not be elements in a folder.
     * 
     * @return
     */
    boolean canBeFolderElementsContentType();
}
