/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.configuration;

import com.openexchange.usm.api.exceptions.USMInvalidConfigurationException;

/**
 * Service that provides access to OX configuration or alternative configuration
 * 
 * <br>TODO Define which methods are provided, for now simple interface with no listeners, maybe adjust this to use OX config bundle ?
 * 
 * @author afe
 *
 */
public interface ConfigurationManager {

	/**
	 * Retrieves the given String property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. If warningIfNotPresent is set to true, a warning
	 * will be logged if the property is not set in the config file (the default value will be returned).
	 * 
	 * @param key
	 * @param defaultValue
	 * @param warningIfNotPresent 
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	String getProperty(String key, String defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given boolean property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Boolean parsing is performed as in Boolean.parseBoolean().
	 * If warningIfNotPresent is set to true, a warning will be logged if the property is not set in the
	 * config file (the default value will be returned).
	 * 
	 * @param key
	 * @param defaultValue
	 * @param warningIfNotPresent 
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	boolean getProperty(String key, boolean defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given integer property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Also, if the value in the property can not be parsed
	 * as an integer, this is logged as an error and USMInvalidConfigurationException is thrown. If warningIfNotPresent
	 * is set to true, a warning will be logged if the property is not set in the config file (the
	 * default value will be returned).
	 * 
	 * @param key
	 * @param defaultValue
	 * @param warningIfNotPresent 
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	int getProperty(String key, int defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given double property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Also, if the value in the property can not be parsed
	 * as a double, this is logged as an error and USMInvalidConfigurationException is thrown. If warningIfNotPresent
	 * is set to true, a warning will be logged if the property is not set in the config file (the default
	 * value will be returned).
	 * 
	 * @param key
	 * @param defaultValue
	 * @param warningIfNotPresent
	 * @throws USMInvalidConfigurationException 
	 * @return
	 */
	double getProperty(String key, double defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException;
	
	
	/**
	 * Retrieves the given String property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. If warningIfNotPresent is set to true, a warning
	 * will be logged if the property is not set in the config file (the default value will be returned).
	 * 
	 * @param key
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	String getPropertyAsString(String key) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given boolean property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Boolean parsing is performed as in Boolean.parseBoolean().
	 * If warningIfNotPresent is set to true, a warning will be logged if the property is not set in the
	 * config file (the default value will be returned).
	 * 
	 * @param key
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	boolean getPropertyAsBoolean (String key) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given integer property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Also, if the value in the property can not be parsed
	 * as an integer, this is logged as an error and the default value is returned. If warningIfNotPresent
	 * is set to true, a warning will be logged if the property is not set in the config file (the
	 * default value will be returned).
	 * 
	 * @param key
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	int getPropertyAsInt (String key) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given double property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Also, if the value in the property can not be parsed
	 * as a double, this is logged as an error and the default value is returned. If warningIfNotPresent
	 * is set to true, a warning will be logged if the property is not set in the config file (the default
	 * value will be returned).
	 * 
	 * @param key
	 * @throws USMInvalidConfigurationException 
	 * @return
	 */
	double getPropertyAsDouble (String key) throws USMInvalidConfigurationException;
}
