package com.openexchange.office.rt2.core.jms;

import java.text.NumberFormat;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import org.apache.activemq.pool.PooledConnectionFactory;
import org.eclipse.microprofile.health.HealthCheck;
import org.eclipse.microprofile.health.HealthCheckResponse;
import org.eclipse.microprofile.health.HealthCheckResponse.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jms.JmsException;
import org.springframework.stereotype.Service;
import com.openexchange.office.tools.annotation.RegisteredService;
import com.openexchange.office.tools.common.TimeStampUtils;
import com.openexchange.office.tools.jms.EnhActiveMQSSLConnectionFactory;
import com.openexchange.office.tools.jms.PooledConnectionFactoryProxy;

@Service
@RegisteredService(registeredClass=HealthCheck.class)
public class RT2JmsClientHealthCheck implements HealthCheck {

    private static final Logger log = LoggerFactory.getLogger(RT2JmsClientHealthCheck.class);
    private static final long MAX_HEALTHCHECK_TIMEOUT = 3000; // ms
    private final String JMS_CLIENT_HEALTH_CHECK = "RT2JmsClientHealthCheck";
    private final String JMS_MSG_ROUNDTRIP_TIME = "jmsMsgRoundtripTime";
    private final String JMS_BROKER_URL = "jmsBrokerUrl";
    private final String JMS_BROKER_URL_NOT_AVAILABLE = "n/a";
    private final String SERVICE_AVAILABLE = "serviceAvailable";

    @Autowired
    RT2AdminJmsConsumer jmsAdminConsumer;

    @Autowired
    PooledConnectionFactoryProxy pooledConnectionFactoryProxy;

    @Override
    public HealthCheckResponse call() {
        final Map<String, Object> healthData = new HashMap<>();
        long duration = -1;
        boolean serviceAvailable = false;

        if (jmsAdminConsumer != null) {
            try {
                long started = System.nanoTime();
                jmsAdminConsumer.sendAdminHealthCheckRequest();

                boolean healthCheckResponseReceived = false;
                long elapsed;
                do {
                    long timeReceived = jmsAdminConsumer.getLastHealthCheckResponseReceived();
                    healthCheckResponseReceived = (timeReceived >= started);
                    if (healthCheckResponseReceived) {
                        duration = TimeStampUtils.timeDiff(started, timeReceived);
                        serviceAvailable = true;
                    } else {
                        try {
                            Thread.sleep(MAX_HEALTHCHECK_TIMEOUT / 100);
                        } catch (@SuppressWarnings("unused") InterruptedException e) {
                            Thread.currentThread().interrupt();
                            break;
                        }
                    }
                    elapsed = TimeStampUtils.nanoToMillis(TimeStampUtils.timeDiff(started, System.nanoTime()));
                } while (!healthCheckResponseReceived && (elapsed < MAX_HEALTHCHECK_TIMEOUT));
            } catch (JmsException e) {
                log.error("RT2JmsClientHealthCheck caught exception trying to send/receive health check admin msg", e);
            }
        }

        // set JMS message roundtrip time
        healthData.put(JMS_MSG_ROUNDTRIP_TIME, formatDuration(duration) + "ms");

        // set currently used JMS broker Url
        final PooledConnectionFactory pooledConnFact = (null != pooledConnectionFactoryProxy) ?
            pooledConnectionFactoryProxy.getPooledConnectionFactory() :
                null;
        final EnhActiveMQSSLConnectionFactory amqConnFact = (EnhActiveMQSSLConnectionFactory) ((null != pooledConnFact) ?
            pooledConnFact.getConnectionFactory() :
                null);
        final String jmsBrokerUrl = (null != amqConnFact) ? amqConnFact.getBrokerURL() : null;

        healthData.put(JMS_BROKER_URL, (null != jmsBrokerUrl) ? jmsBrokerUrl : JMS_BROKER_URL_NOT_AVAILABLE);
        healthData.put(SERVICE_AVAILABLE, Boolean.valueOf(serviceAvailable));

        return new HealthCheckResponse(JMS_CLIENT_HEALTH_CHECK, Status.UP, Optional.of(healthData));
    }

    private String formatDuration(long duration) {
        NumberFormat nf = NumberFormat.getNumberInstance(Locale.US);
        nf.setMaximumFractionDigits(3);
        nf.setMinimumFractionDigits(3);
        return nf.format(duration / 1000000F);
    }
}
