/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.office.rest;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Collections;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.BoundedInputStream;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.validator.routines.UrlValidator;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RestController;

import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.upload.UploadFile;
import com.openexchange.imagetransformation.ImageInformation;
import com.openexchange.imagetransformation.ImageMetadata;
import com.openexchange.imagetransformation.ImageMetadataOptions;
import com.openexchange.imagetransformation.ImageMetadataService;
import com.openexchange.imagetransformation.Utility;
import com.openexchange.java.BoolReference;
import com.openexchange.net.HostList;
import com.openexchange.office.document.api.DocFileServiceFactory;
import com.openexchange.office.document.tools.DocFileHelper;
import com.openexchange.office.imagemgr.IResourceManager;
import com.openexchange.office.imagemgr.IResourceManagerFactory;
import com.openexchange.office.imagemgr.Resource;
import com.openexchange.office.rest.tools.RESTException;
import com.openexchange.office.rest.tools.RequestDataHelper;
import com.openexchange.office.tools.common.DigestHelper;
import com.openexchange.office.tools.common.IOHelper;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.common.memory.MemoryObserver;
import com.openexchange.office.tools.doc.DocumentFormat;
import com.openexchange.office.tools.doc.StreamInfo;
import com.openexchange.tools.encoding.Base64;
import com.openexchange.tools.images.ImageTransformationUtility;
import com.openexchange.tools.net.URITools;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link AddFileAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
@SuppressWarnings("rawtypes")
@RestController
public class AddFileAction extends DocumentRESTAction implements InitializingBean {

	public static final String HOST_BLACKLIST_DEFAULT = "127.0.0.1-127.255.255.255,localhost";

    private static final Set<String> SUPPORTED_IMAGE_TYPES;
    static {
        final Set<String> supportedImageTypes = new HashSet<>();
        supportedImageTypes.add("png");
        supportedImageTypes.add("jpg");
        supportedImageTypes.add("jpeg");
        supportedImageTypes.add("gif");
        supportedImageTypes.add("tif");
        supportedImageTypes.add("tiff");
        supportedImageTypes.add("bmp");
        SUPPORTED_IMAGE_TYPES = Collections.unmodifiableSet(supportedImageTypes);
    }

    @SuppressWarnings("deprecation")
	private static final Log LOG = com.openexchange.log.Log.loggerFor(AddFileAction.class);
    private static final int MAX_IMAGE_SIZE_IN_MB = 10;
    private static final int MAX_URL_LENGTH = 255;

    static final protected String RESOURCE_ID_KEY = "added_fileid";
    static final protected String RESOURCE_NAME_KEY = "added_filename";
    static final protected String IMAGE_TRANSFORMATION_KEY = "image_transformation";
    static final protected String EXIF_ORIENTATION_KEY = "exif_orientation";

    private long maxImageSizeSupported = MAX_IMAGE_SIZE_IN_MB * 1024L * 1024L;

    // - Members ---------------------------------------------------------------

    @Autowired
    private IResourceManagerFactory resourceManagerFactory = null;

    @Autowired
    private ConfigurationService configurationService;

    @Autowired
    private ImageMetadataService imageMetadataService;

    @Autowired
    private DocFileServiceFactory docFileServiceFactory;

    private IResourceManager resourceManager;

    private HostList blackImageUrlHostlist;
    private String blackImageUrlHostlistAsStr;
    private HostList whiteImageUrlHostlist;
    private String whiteImageUrlHostlistAsStr;
    private int maxUrlLength;

    @Override
	public void afterPropertiesSet() throws Exception {
          maxImageSizeSupported = configurationService.getIntProperty("com.openexchange.office.maxImageSize", MAX_IMAGE_SIZE_IN_MB) * 1024L * 1024L;
          maxUrlLength = configurationService.getIntProperty("com.openexchange.office.maxUrlLength", MAX_URL_LENGTH);
	      blackImageUrlHostlistAsStr = configurationService.getProperty("com.openexchange.office.upload.blacklist", HOST_BLACKLIST_DEFAULT);
	      blackImageUrlHostlist = HostList.valueOf(blackImageUrlHostlistAsStr);
	      whiteImageUrlHostlistAsStr = configurationService.getProperty("com.openexchange.office.upload.whitelist", "");
	      whiteImageUrlHostlist = HostList.valueOf(whiteImageUrlHostlistAsStr);
	      resourceManager = resourceManagerFactory.createInstance(AddFileAction.class.getCanonicalName());
	}

	/*
     * (non-Javadoc)
     * @see com.openexchange.ajax.requesthandler.AJAXActionService#perform(com.openexchange.ajax.requesthandler.AJAXRequestData,
     * com.openexchange.tools.session.ServerSession)
     */
    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) throws OXException {
        final RequestDataHelper requestDataHelper = new RequestDataHelper(request);
        final JSONObject resultData = new JSONObject();
        AJAXRequestResult ajaxResult = null;
        String sha256 = null;

        final String p = requestDataHelper.getParameter("add_hash");
        if (StringUtils.isNotEmpty(p)) {
            if (DigestHelper.isSHA256HexString(p)) {
                sha256 = p;
            } else {
                LOG.error("Hash parameter is not a valid SHA-256 hex string: " + p);
                final ErrorCode errorCode = ErrorCode.GENERAL_ARGUMENTS_ERROR;
                errorCode.setValue(p);
                final JSONObject requestResult = errorCode.getAsJSONResultObject();
                return getAjaxRequestResult(requestResult);
            }
        }

        final String extension = requestDataHelper.getParameter("add_ext");
        final String data64 = requestDataHelper.getParameter("add_filedata");
        final String fileId = requestDataHelper.getParameter("add_fileid");
        final String folderId = requestDataHelper.getParameter("add_folderid");
        final String imageUrlAsStr = requestDataHelper.getParameter("add_imageurl");

        byte[] data = null;

        LOG.debug("AddFileAction called with ext=" + extension + ", fileId=" + fileId + ", folderId=" + folderId + ", sha256=" + sha256 + ", data-length=" + ((data != null) ? data.length : "0"));

        if ((null != fileId) && (null != folderId)) {
            final StreamInfo streamInfo = docFileServiceFactory.createInstance().getDocumentStream(session, folderId, fileId, "");
            final InputStream inputStream = streamInfo.getDocumentStream();
            if (null != inputStream) {
                try {
                    data = IOUtils.toByteArray(inputStream);
                } catch (IOException e) {
                    LOG.error("Could not read file stream", e);
                } finally {
                    IOHelper.closeQuietly(inputStream);
                    try {
                        streamInfo.close();
                    } catch (IOException e) {
                        LOG.warn("Could not correctly close IDBasedFileAccess instance", e);
                    }
                }
            } else {
                try {
                    streamInfo.close();
                } catch (IOException e) {
                    LOG.warn("Could not correctly close IDBasedFileAccess instance", e);
                }
            }
        } else if (StringUtils.isNotEmpty(imageUrlAsStr)) {
            try {
                data = getImageDataFromUrl(imageUrlAsStr);
            } catch (RESTException e) {
                final ErrorCode errorCode = e.getErrorCode();
                final JSONObject requestResult = errorCode.getAsJSONResultObject();
                ajaxResult = getAjaxRequestResult(requestResult);
            }
        } else {
            if (null != data64) {

                final String searchPattern = "base64,";
                int pos = data64.indexOf(searchPattern);

                if ((-1 != pos) && ((pos += searchPattern.length()) < (data64.length() - 1))) {
                    data = Base64.decode(data64.substring(pos));
                }
            } else { // in IE9 data64 is not readable from request
                InputStream uploadStm = null;

                try {
                    if (request.hasUploads()) {
                        final UploadFile uploadFile = request.getFiles().get(0);
                        uploadStm = new FileInputStream(uploadFile.getTmpFile());
                    }
                } catch (final Exception e) {
                    LOG.error("Could not create file stream", e);
                }

                if (null != uploadStm) {
                    try {
                        data = IOUtils.toByteArray(uploadStm);
                    } catch (IOException e) {
                        LOG.error("Could not read file stream", e);
                    } finally {
                    	IOHelper.closeQuietly(uploadStm);
                    }
                }
            }
        }

        if ((data != null) && (data.length > 0)) {
            if (data.length > maxImageSizeSupported) {
            	throwMaxImageSizeExceeding(maxImageSizeSupported);
            }


            try {
                if (StringUtils.isEmpty(sha256)) {
                    sha256 = Resource.getSHA256(data);
                }

                String uid = resourceManager.addResource(sha256, data);

                String resname = null;
                final DocumentFormat docFormat = DocFileHelper.getDocumentFormat(request.getParameter("filename"));
                switch (docFormat) {
                    case DOCX: resname = "word/media/"; break;
                    case PPTX: resname = "ppt/media/"; break;
                    case ODT:
                    case ODS:
                    case ODP: resname = "Pictures/"; break;
                    case XLSX: resname = "xl/media/"; break;
                    case ODG:
                    case NONE:
                    default:
                        LOG.warn("Unsupported document format used " + docFormat);
                        break;
                }

                if (resname != null) {
                    resname = resname + Resource.RESOURCE_UID_PREFIX + uid + "." + extension;
                    final Resource resource = resourceManager.getResource(uid);
                    String managedResourceId = null;

                    // add the managed file id to the result object
                    if ((null != resource) && (null != (managedResourceId = resource.getManagedId()))) {
                        resultData.put(RESOURCE_ID_KEY, managedResourceId);
                    }

                    // add the filename to the result object
                    resultData.put(RESOURCE_NAME_KEY, resname);
                }

                if (data.length > 2 && ((data[0] & 0x00ff) == 0x00ff) && ((data[1] & 0x00ff) == 0x00d8)) {
                    final ImageInformation information = ImageTransformationUtility.readImageInformation(new BufferedInputStream(new ByteArrayInputStream(data)), new BoolReference(false));
                    if (information.orientation > 1) {
                        final JSONObject imageTransformation = new JSONObject(1);
                        imageTransformation.put(EXIF_ORIENTATION_KEY, information.orientation);
                        resultData.put(IMAGE_TRANSFORMATION_KEY, imageTransformation);
                    }
                }

            } catch (Exception e1) {
                LOG.error("Could not add to resource manager", e1);
            }
        }

        // set resource id at connection object; return the
        // object with fragment name and id in case of success
        if (resultData.has(RESOURCE_ID_KEY)) {
            try {
                ajaxResult = getAjaxRequestResult(resultData);

                LOG.debug("AddFileAction with sha256=" + sha256 + " results in " + RESOURCE_ID_KEY + "=" + resultData.getString(RESOURCE_ID_KEY));
            } catch (JSONException e) {
                LOG.error("Could not send resource to realtime session", e);
            }
        }

        return ajaxResult;
    }

    /**
     * Validates the given URL according to whitelisted prtocols ans blacklisted hosts.
     *
     * @param url The URL to validate
     * @return An optional OXException
     */
    private Function<URL, Optional<OXException>> validator = (url) -> {

    	String urlasStr = url.toString();
    	if (urlasStr.length() > maxUrlLength) {
    		return Optional.of(new OXException(new RESTException("The length " + blackImageUrlHostlistAsStr.length() + " is to long!", ErrorCode.GENERAL_ARGUMENTS_ERROR)));
    	}

        final UrlValidator defaultValidator = new UrlValidator();
        if (!defaultValidator.isValid(urlasStr)) {
            return Optional.of(new OXException(new RESTException(ErrorCode.GENERAL_ARGUMENTS_ERROR)));
        }
		String host = url.getHost();
		if (StringUtils.isBlank(whiteImageUrlHostlistAsStr)) {
			if (blackImageUrlHostlist.contains(host)) {
				return Optional.of(new OXException(new RESTException("Host " + host + " is on blacklist!", ErrorCode.GENERAL_ARGUMENTS_ERROR)));
			}
		} else {
			if (!whiteImageUrlHostlist.contains(host)) {
				if (blackImageUrlHostlist.contains(host)) {
					return Optional.of(new OXException(new RESTException("Host " + host + " is not on whithlist and is on blacklist!", ErrorCode.GENERAL_ARGUMENTS_ERROR)));
				}
			}
		}
        return Optional.empty();
    };

    private static void throwMaxImageSizeExceeding(long maxSize) throws RESTException {
        final ErrorCode errorCode = ErrorCode.GENERAL_IMAGE_SIZE_EXCEEDS_LIMIT_ERROR;
        errorCode.setValue(new Long(maxSize / (1024L * 1024L)).toString());
        throw new RESTException("Image size exceeds limit " + maxSize, errorCode);
    }

    private byte[] getImageDataFromUrl(String fileURL) throws RESTException {
    	byte[] result = null;
    	try {
        	HttpURLConnection httpConn;
        	try {
        		httpConn = (HttpURLConnection) URITools.getTerminalConnection(fileURL, Optional.of(validator), Optional.empty());
            } catch (Exception e) {
                throw new RESTException(ErrorCode.GENERAL_MISSING_SERVICE_ERROR, e);
            }
    		try {
		        int responseCode = httpConn.getResponseCode();

		        // always check HTTP response code first
		        if (responseCode == HttpURLConnection.HTTP_OK) {
		            final int contentLength = httpConn.getContentLength(); // contentLenght is -1 if the length is not yet available
		            if (contentLength > maxImageSizeSupported) {
		                throwMaxImageSizeExceeding(maxImageSizeSupported);
		            }
		            final boolean notEnoughMemory = MemoryObserver.get().willHeapLimitBeReached(contentLength == -1 ? maxImageSizeSupported : contentLength); // assuming maxImageSizeSupported if contentLength is not available
		            if (notEnoughMemory) {
		                throw new RESTException("Memory threshold will be exceeded due to reading image data in memory", ErrorCode.GENERAL_MEMORY_TOO_LOW_ERROR);
		            }

		            // opens input stream from the HTTP connection
                    InputStream inputStream = httpConn.getInputStream();
		            if(contentLength == -1) {
	                    result = IOUtils.toByteArray(new BoundedInputStream(inputStream, maxImageSizeSupported + 1));
	                    if(result.length > maxImageSizeSupported) {
	                        throwMaxImageSizeExceeding(maxImageSizeSupported);
	                    }
		            }
		            else {
	                    result = IOUtils.toByteArray(inputStream, contentLength);
		            }
		            final String imageType = imageTypeDetection(new ByteArrayInputStream(result));
	                if (StringUtils.isEmpty(imageType) || !SUPPORTED_IMAGE_TYPES.contains(imageType)) {
	                    throw new RESTException(ErrorCode.GENERAL_IMAGE_TYPE_NOT_SUPPORTED_ERROR);
	                }
                } else {
                    throw new RESTException("Error: " + responseCode + ", " + httpConn.getResponseMessage(), ErrorCode.GENERAL_ARGUMENTS_ERROR);
                }
	        }
    		finally {
	        	httpConn.disconnect();
			}
	    } catch (RESTException e) {
	        LOG.debug("RESTException caught while trying to get image data from provided url", e);
	        throw e;
	    } catch (Exception e) {
	        LOG.error("Error while processing the retrieved information: " + e.getMessage(), e);
	        throw new RESTException(ErrorCode.GENERAL_UNKNOWN_ERROR);
	    }
    	return result;
    }

    private String imageTypeDetection(InputStream imageInputStm) throws OXException {

        String imageFormatName = null;
        if (null != imageInputStm) {
            try {
                final ImageMetadata imageMetadata = imageMetadataService.getMetadataFor(imageInputStm, null, null, ImageMetadataOptions.builder().withDimension().withFormatName().build());
                imageFormatName = imageMetadata.getFormatName();
            } catch (IOException e) {
                LOG.trace(e.getMessage());
            }
        }

        return imageFormatName;
    }
}
