/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.odp.dom.components;

import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.core.component.ComponentType;
import com.openexchange.office.filter.core.component.IComponent;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.components.OdfComponent;
import com.openexchange.office.filter.odf.draw.ConnectorShape;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.DrawImage;
import com.openexchange.office.filter.odf.draw.DrawTextBox;
import com.openexchange.office.filter.odf.draw.DrawingType;
import com.openexchange.office.filter.odf.draw.GroupShape;
import com.openexchange.office.filter.odf.draw.IDrawingType;
import com.openexchange.office.filter.odf.draw.LineShape;
import com.openexchange.office.filter.odf.draw.Shape;
import com.openexchange.office.filter.odf.table.Table;

public class ShapeGroupComponent extends OdfComponent implements IDrawingType {

    private final GroupShape groupShape;

    public ShapeGroupComponent(ComponentContext<OdfOperationDoc> parentContext, DLNode<Object> shapeNode, int componentNumber) {
		super(parentContext, shapeNode, componentNumber);
		this.groupShape = (GroupShape)getObject();
	}

    @Override
    public String simpleName() {
        return "Group";
    }

    @Override
    public DrawingType getType() {
		return ((Shape)getObject()).getType();
	}

	@Override
	public IComponent<OdfOperationDoc> getNextChildComponent(ComponentContext<OdfOperationDoc> previousChildContext, IComponent<OdfOperationDoc> previousChildComponent) {
        final int nextComponentNumber = previousChildComponent != null ? previousChildComponent.getNextComponentNumber() : 0;
        DLNode<Object> nextNode = previousChildContext != null ? previousChildContext.getNode().getNext() : ((Shape)getNode().getData()).getContent().getFirstNode();
		while(nextNode!=null) {
			final Object child = nextNode.getData();
			if(child instanceof GroupShape) {
			    return new ShapeGroupComponent(this, nextNode, nextComponentNumber);
			}
            else if(child instanceof DrawFrame) {
                return new FrameComponent(this, nextNode, nextComponentNumber);
            }
			else if(child instanceof ConnectorShape) {
			    return new ShapeConnectorComponent(this, nextNode, nextComponentNumber);
			}
			else if(child instanceof Shape) {
				return new ShapeComponent(this, nextNode, nextComponentNumber);
			}
			nextNode = nextNode.getNext();
		}
		return null;
	}

    @Override
    public IComponent<OdfOperationDoc> insertChildComponent(ComponentContext<OdfOperationDoc> parentContext, DLNode<Object> contextNode, int number, IComponent<OdfOperationDoc> child, ComponentType type, JSONObject attrs) {
		final Shape shape = (Shape)getObject();
    	DLList<Object> DLList = shape.getContent();
        DLNode<Object> referenceNode = child != null && child.getComponentNumber()== number ? child.getNode() : null;

        switch(type) {
            case AC_CONNECTOR:
            case AC_SHAPE: {
                final DLNode<Object> newShapeNode = new DLNode<Object>(Shape.createShape(operationDocument, attrs, groupShape, false, isContentAutoStyle()));
                DLList.addNode(referenceNode, newShapeNode, true);
                return new ShapeComponent(parentContext, newShapeNode, number);
            }
            case AC_GROUP: {
            	final DLNode<Object> newGroupNode = new DLNode<Object>(new GroupShape(operationDocument, groupShape, false, isContentAutoStyle()));
                DLList.addNode(referenceNode, newGroupNode, true);
                return new ShapeGroupComponent(parentContext, newGroupNode, number);
            }
            case AC_IMAGE: {
                final DrawFrame drawFrame = new DrawFrame(operationDocument, groupShape, false, isContentAutoStyle());
                final DLNode<Object> drawFrameNode = new DLNode<Object>(drawFrame);
                DLList.addNode(referenceNode, new DLNode<Object>(drawFrame), true);
                final DrawImage drawImage = new DrawImage(drawFrame);
                drawFrame.addContent(drawImage);
                return new FrameComponent(parentContext, drawFrameNode, number);
            }
            case AC_FRAME: {
                final DrawFrame drawFrame = new DrawFrame(operationDocument, groupShape, false, isContentAutoStyle());
                final DLNode<Object> drawFrameNode = new DLNode<Object>(drawFrame);
                DLList.addNode(referenceNode, new DLNode<Object>(drawFrame), true);
                final DrawTextBox drawTextBox = new DrawTextBox(drawFrame);
                drawFrame.addContent(drawTextBox);
                return new FrameComponent(parentContext, drawFrameNode, number);
            }
            case TABLE: {
                final DrawFrame drawFrame = new DrawFrame(operationDocument, groupShape, false, isContentAutoStyle());
                final DLNode<Object> drawFrameNode = new DLNode<Object>(drawFrame);
                DLList.addNode(referenceNode, new DLNode<Object>(drawFrame), true);
                final Table table = new Table("presentation");
                drawFrame.addContent(table);
                return new FrameComponent(parentContext, drawFrameNode, number);
            }
            default : {
                throw new UnsupportedOperationException();
            }
        }
    }

	@Override
	public void applyAttrsFromJSON(JSONObject attrs) {
	    try {
            groupShape.applyAttrsFromJSON(operationDocument, attrs, isContentAutoStyle());
        } catch (JSONException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
	}

	@Override
	public void createJSONAttrs(OpAttrs attrs) {

	    groupShape.createAttrs(operationDocument, attrs, isContentAutoStyle());
	}
}
