/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.fileitem.impl;

import com.openexchange.annotation.NonNull;
import com.openexchange.imageconverter.api.IFileItem;

/**
 * {@link FileItem}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
/**
 * {@link FileItem}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileItem implements IFileItem {

    /**
     * serialVersionUID
     */
    private static final long serialVersionUID = 1203862060096934396L;

    /**
     * Initializes a new {@link FileItem}.
     */
    @SuppressWarnings("unused")
    private FileItem() {
        // no empty Ctor
    }

    /**
     * Initializes a new {@link FileItem}.
     * @param groupId
     * @param subGroupId
     * @param fileId
     */
    public FileItem(@NonNull String groupId, @NonNull String subGroupId, @NonNull String fileId) {
        super();

        m_groupId = FileItemUtils.isValid(groupId) ?  groupId : FileItemService.GROUP_ID_DEFAULT;
        m_subGroupId = FileItemUtils.isValid(subGroupId) ? subGroupId : FileItemService.SUBGROUP_ID_DEFAULT;
        m_fileId = fileId;

        if (FileItemUtils.isInvalid(m_fileId) && FileItemUtils.isLogError()) {
            FileItemUtils.logError("Invalid fileId is not allowed! Set fileId to unique and not empty value.");
        }
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return new StringBuilder(256).
            append("FileItem").
            append(" [").
            append((null != m_fileStoreData) ? m_fileStoreData.toString() : "FileStoreData [null]").
            append(" , groupId: ").append(m_groupId).
            append(" , subGroupId: ").append(m_subGroupId).
            append(" , fileId: ").append(m_fileId).
            append(']').toString();
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItem#getGroupId()
     */
    @Override
    public String getGroupId() {
        return m_groupId;
    }

    /**
     * Gets the subGroupId
     *
     * @return The m_subGroupId
     */
    @Override
    public String getSubGroupId() {
        return m_subGroupId;
    }

    /**
     * Gets the fileId
     *
     * @return The m_fileId
     */
    @Override
    public String getFileId() {
        return m_fileId;
    }

    // - Package internal API  -------------------------------------------------

    /**
     * @return
     */
    protected String getKey() {
        return new StringBuilder(m_groupId.length() + m_subGroupId.length() + m_fileId.length() + 2).
            append(m_groupId).append('_').
            append(m_subGroupId).append('_').
            append(m_fileId).toString();
    }

    /**
     * @return
     */
    protected boolean isValid() {
        return FileItemUtils.isValid(m_groupId) && FileItemUtils.isValid(m_subGroupId) && FileItemUtils.isValid(m_fileId);
    }

    /**
     * @return
     */
    protected boolean isInvalid() {
        return FileItemUtils.isInvalid(m_groupId) || FileItemUtils.isInvalid(m_subGroupId) || FileItemUtils.isInvalid(m_fileId);
    }

    /**
     * Gets the storeId
     *
     * @return The m_storeId
     */
    protected FileStoreData getFileStoreData() {
        return m_fileStoreData;
    }

    /**
     * Gets the storeId
     *
     * @return The m_storeId
     */
    protected void setFileStoreData(final FileStoreData fileStoreData) {
        m_fileStoreData = fileStoreData;
    }

    // - Members ---------------------------------------------------------------

    protected String m_groupId = null;

    protected String m_subGroupId = null;

    protected String m_fileId = null;

    protected FileStoreData m_fileStoreData = null;
}
