/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.fileitem.impl;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.Date;
import java.util.NoSuchElementException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import com.openexchange.annotation.NonNull;
import com.openexchange.imageconverter.api.AccessOption;
import com.openexchange.imageconverter.api.ElementLocker;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItemReadAccess;

/**
 * {@link FileAccess}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileItemReadAccess implements IFileItemReadAccess {

    /**
     * Initializes a new {@link FileItemReadAccess}.
     * @param fileItem
     * @param fileItem
     * @throws FileItemException
     */
    protected FileItemReadAccess(@NonNull FileItemService fileItemService,
        @NonNull final FileItemDatabase database,
        @NonNull FileItem fileItem,
        final AccessOption... accessOptions) {

        super();

        m_fileItemService = fileItemService;
        m_database = database;
        m_fileItemKey = (m_fileItem = fileItem).getKey();
        m_accessOptions = AccessOption.getNormalizedOptions(accessOptions);
    }

    /* (non-Javadoc)
     * @see java.io.Closeable#close()
     */
    @Override
    public void close() throws IOException {
        try {
            // close InputStream and InputFile;
            FileItemUtils.close(m_fileContentInputStm);
            m_fileContentInputStm = null;

            FileUtils.deleteQuietly(m_fileContentInputFile);
            m_fileContentInputFile = null;
        } finally {
            // unlock access time lock (from open to close), that was
            // previously acquired in ReadAccess#open method
            ElementLocker.unlock(m_fileItemKey);
        }
    }

    // - IFileItemReadAccess -------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemReadAccess#getInputStream()
     */
    @Override
    public InputStream getInputStream() {
        return m_fileContentInputStm;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemReadAccess#getInputFile()
     */
    @Override
    public File getInputFile() throws IOException {
        if (null == m_fileContentInputFile) {
            if (null != m_fileContentInputStm) {
                if (null != (m_fileContentInputFile = FileItemUtils.createTempFile())) {
                    ((ByteArrayInputStream) m_fileContentInputStm).reset();
                    FileUtils.copyInputStreamToFile(m_fileContentInputStm, m_fileContentInputFile);

                    // the previously used output stream is closed and not valid anymore
                    FileItemUtils.close(m_fileContentInputStm);
                    m_fileContentInputStm = null;
                }
            } else {
                FileUtils.deleteQuietly(m_fileContentInputFile);
                m_fileContentInputFile = null;
            }
        }

        return m_fileContentInputFile;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemReadAccess#getLength()
     */
    @Override
    public long getLength() {
        return (null != m_fileItemProperties) ?
            m_fileItemProperties.getLength() :
                0;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemReadAccess#getCreateDate()
     */
    @Override
    public Date getCreateDate() {
        return (null != m_fileItemProperties) ?
            new Date(m_fileItemProperties.getCreateDateMillis()) :
                null;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemReadAccess#getModificationDate()
     */
    @Override
    public Date getModificationDate() {
        return (null != m_fileItemProperties) ?
            new Date(m_fileItemProperties.getModificationDateMillis()) :
                null;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemReadAccess#getKeyValue(java.lang.String)
     */
    @Override
    public String getKeyValue(String key) {
        return (null != m_fileItemProperties) ?
            m_fileItemProperties.getCustomKeyValue(key) :
                StringUtils.EMPTY;
    }

    // - Implementation --------------------------------------------------------

    /**
     * @return
     * @throws FileItemException
     * @throws NoSuchElementException
     * @throws IOException
     */
    protected void open() throws FileItemException, NoSuchElementException, IOException {
        // Global lock for the lifetime of this access object => will finally be unlocked in #close call
        // As such, an open call must be followed by a close call in order to ensure correct unlocking
        ElementLocker.lock(m_fileItemKey);

        if (!tryOpen()) {
            throw new NoSuchFileException("FileItem is not available: " + m_fileItem.toString());
        }

        m_fileItemProperties.reset();

        try (final InputStream inputStm = m_fileItemService.getFileStoreInputStream(m_fileItem)) {
            if (null != inputStm) {
                if (null != (m_fileContentInputFile = FileItemUtils.createTempFile())) {
                    FileUtils.copyInputStreamToFile(inputStm, m_fileContentInputFile);
                    m_fileContentInputStm = FileUtils.openInputStream(m_fileContentInputFile);
                }
            }

            if (null == m_fileContentInputStm) {
                throw new IOException("InputStream for FileStore file could not be opened although file item is valid: : " + m_fileItem.toString());
            }
        } catch (Exception e) {
            throw new IOException("Reading FileStore file not possible although file item is valid: " + m_fileItem.toString(), e);
        }
    }

    /**
     * @return
     */
    protected boolean tryOpen() {
        boolean ret = false;

        try {
            m_fileItemProperties = m_database.getFileItemProperties(m_fileItem);
            ret = (null != m_fileItemProperties) && FileItemUtils.isValid(m_fileItem.getFileStoreData());
        } catch (@SuppressWarnings("unused") FileItemException e) {
            // it's a try method!
        }

        return ret;
    }

    /**
     * @return
     */
    protected boolean isValid() {
        return (null != m_fileItemProperties);
    }

    /**
     * @return
     */
    protected FileItem getFileItem() {
        return m_fileItem;
    }

    /**
     * @return
     */
    protected AccessOption[] getAccessOptions() {
        return m_accessOptions;
    }

    /**
     * @return
     */
    protected FileItemProperties getFileItemProperties() {
        return m_fileItemProperties;
    }

    // - Members ---------------------------------------------------------------

    final protected FileItemService m_fileItemService;

    final protected FileItemDatabase m_database;

    protected ElementLocker m_fileItemLocker = null;

    final protected FileItem m_fileItem;

    protected String m_fileItemKey = null;

    protected AccessOption[] m_accessOptions = null;

    protected FileItemProperties m_fileItemProperties = null;

    protected InputStream m_fileContentInputStm = null;

    protected File m_fileContentInputFile = null;
}
