/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.fileitem.impl;

import static com.openexchange.fileitem.impl.FileItemUtils.LOG;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.WildcardNamePropertyFilter;
import com.openexchange.exception.OXException;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItemService;
import com.openexchange.imageconverter.api.ISubGroup;
import com.zaxxer.hikari.pool.HikariPool;
import liquibase.Liquibase;
import liquibase.changelog.ChangeSet;
import liquibase.changelog.ChangeSet.ExecType;
import liquibase.changelog.ChangeSet.RunStatus;
import liquibase.changelog.DatabaseChangeLog;
import liquibase.changelog.visitor.ChangeExecListener;
import liquibase.database.Database;
import liquibase.database.jvm.JdbcConnection;
import liquibase.exception.LiquibaseException;

/**
 * {@link FileItemDatabase}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileItemDatabase implements ChangeExecListener {

    // Predefined properties
    final public static String KEY_CREATEDATE = "CreateDate";
    final public static String KEY_MODIFICATIONDATE = "ModificationDate";
    final public static String KEY_LENGTH = "Length";

    final public static String[] DEFAULT_KEYS = { KEY_CREATEDATE, KEY_MODIFICATIONDATE, KEY_LENGTH };

    final public static String DB_VARCHAR_TYPE = "VARCHAR(190)";

    /**
     * Initializes a new {@link FileItemDatabase}.
     */
    public FileItemDatabase(@NonNull ConfigurationService configService) throws FileItemException {
        super();

        m_configService = configService;

        implCreateConnectionPools();
        implUpdateDatabase();
        implReadGroups();
    }

    /*
     * (non-Javadoc)
     *
     * @see liquibase.changelog.visitor.ChangeExecListener#willRun(liquibase.changelog.ChangeSet, liquibase.changelog.DatabaseChangeLog, liquibase.database.Database, liquibase.changelog.ChangeSet.RunStatus)
     */
    @Override
    public void willRun(ChangeSet changeSet, DatabaseChangeLog databaseChangeLog, Database database, RunStatus runStatus) {
        if (null != changeSet) {
            implLogChangeEvent(runStatus, changeSet.getDescription());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see liquibase.changelog.visitor.ChangeExecListener#ran(liquibase.changelog.ChangeSet, liquibase.changelog.DatabaseChangeLog, liquibase.database.Database, liquibase.changelog.ChangeSet.ExecType)
     */
    @Override
    public void ran(ChangeSet changeSet, DatabaseChangeLog databaseChangeLog, Database database, ExecType execType) {
        if (null != changeSet) {
            implLogChangeEvent(execType, changeSet.getDescription());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see liquibase.changelog.visitor.ChangeExecListener#rolledBack(liquibase.changelog.ChangeSet, liquibase.changelog.DatabaseChangeLog, liquibase.database.Database)
     */
    @Override
    public void rolledBack(ChangeSet changeSet, DatabaseChangeLog databaseChangeLog, Database database) {
        if (null != changeSet) {
            implLogChangeEvent(null, changeSet.getDescription());
        }
    }

    // - Database access methods -----------------------------------------------

    /**
     * @return The type of JDBC database read and write connections,
     *  allowing to optimize SQL queries
     */
    public IFileItemService.DatabaseType getDatabaseType() {
        return m_databaseType;
    }

    /**
     * @param groupId
     * @param customKeyN
     * @throws FileItemException
     */
    public void registerGroup(@NonNull final String groupId, @Nullable String... customKeyN) throws FileItemException {
        Set<String> keySet = null;

        synchronized (m_groupMap) {
            GroupConfiguration groupConfiguration = m_groupMap.get(groupId);
            final boolean groupExists = (null != groupConfiguration);

            if (groupExists) {
                // just get key set with all keys from existing group
                keySet = implGetKeySet(groupId, false);
            } else {
                // create new group and perform all necessary DB transactions, it it doesn't exist yet
                m_groupMap.put(groupId, groupConfiguration = new GroupConfiguration(groupId));

                // create Content and Property tables for new group
                implCreateGroupTables(groupId);

                 keySet = implGetKeySet(groupId, false);

                 // insert the default property keys, that are always available
                 try {
                     implInsertPropertyKey(groupId, KEY_CREATEDATE, "BIGINT(20)", "NOT NULL");
                 } catch (FileItemException e) {
                     LOG.error("FileItemService received exception when trying to register 'CreateData' default key for new group: {}", groupId);
                 }

                 try {
                     implInsertPropertyKey(groupId, KEY_MODIFICATIONDATE, "BIGINT(20)", "NOT NULL");
                 } catch (FileItemException e) {
                     LOG.error("FileItemService received exception when trying to register 'ModificationData' default key for new group: {}", groupId);
                 }

                 try {
                     implInsertPropertyKey(groupId, KEY_LENGTH, "BIGINT(20)", "NOT NULL");
                 } catch (FileItemException e) {
                     LOG.error("FileItemService received exception when trying to register 'Length' default key for new group: {}", groupId);
                 }
            }

            // add additional custom keys if they're not contained yet
            if (FileItemUtils.isValid(customKeyN)) {
                final Set<String> customPropertyKeySet = implGetKeySet(groupId, true);

                for (final String curKey : customKeyN) {
                    if (FileItemUtils.isValid(curKey) && !keySet.contains(curKey)) {
                        try {
                            implInsertPropertyKey(groupId, curKey, "TEXT", "NULL");

                            if (!implIsDefaultKey(curKey)) {
                                customPropertyKeySet.add(curKey);
                            }

                        } catch (FileItemException e) {
                            FileItemUtils.logError(STR_BUILDER("FileItemService received exception when trying to register customKey for groupId: ").
                                append(groupId).append(" => ").append(curKey).
                                append(" (").append(Throwables.getRootCause(e).getMessage()).append(')').toString());
                        }
                    }
                }
            }

            // finish group configuration, perform the synchronisation step only for new groups
            implFinishGroupConfiguration(groupConfiguration, !groupExists, null);
        }
    }

    /**
     * @param groupId
     * @return
     */
    public Set<String> getPropertyKeys(@NonNull final String groupId) {
        return implGetKeySet(groupId, false);
    }

    /**
     * @param groupId
     * @return
     */
    public Set<String> getCustomPropertyKeys(@NonNull final String groupId) {
        return implGetKeySet(groupId, true);
    }

    /**
     * @return
     */
    public boolean hasCustomPropertyKey(@NonNull final String groupId, @NonNull final String customKey) {
        return implGetKeySet(groupId, true).contains(customKey);
    }

    /**
     * getUserData
     *
     * @return
     */
    public JSONObject getUserData(@NonNull final String groupId) {
        final GroupConfiguration groupConfiguration = m_groupMap.get(groupId);

        return (null != groupConfiguration) ?
            groupConfiguration.getUserData() :
                new JSONObject();
    }

    /**
     * getUserData
     *
     * @return
     */
    public boolean setUserData(@NonNull final String groupId, @NonNull final JSONObject userData) throws FileItemException {
        final GroupConfiguration groupConfiguration = m_groupMap.get(groupId);
        String userDataText = userData.toString(false);

        if ((null != groupConfiguration) && (null != userDataText)) {
            final String updateGroupUserDataSql = STR_BUILDER("UPDATE ").append(TBL_GROUPADMIN("fa")).
                append(" SET fa.Data=?").
                append(" WHERE fa.GroupId=?").toString();

            Connection usedCon = null;

            for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS; sleepTimeoutMillis <<= 1) {
                try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper();
                     final PreparedStatement fgUserDataUpdateStmt = conWrapper.prepareStatement(updateGroupUserDataSql)) {

                    usedCon = conWrapper.getConnection();

                    // delete/insert groupId in Groups table
                    fgUserDataUpdateStmt.setString(1, userDataText);
                    fgUserDataUpdateStmt.setString(2, groupId);

                    // execute all statements and commit
                    fgUserDataUpdateStmt.executeUpdate();

                    usedCon.commit();
                    groupConfiguration.setUserData(userData);

                    return true;
                } catch (Exception e) {
                    final String cause = new StringBuilder(STR_BUILDER_LARGE_CAPACITY).
                        append("Exception occured when trying to update UserData column for GroupId: ").append(groupId).
                        append(" (").append(Throwables.getRootCause(e).getMessage()).append(')').toString();

                    FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, cause);
                }
            }
        }

        return false;
    }

    /**
     * @param fileStoreId
     * @param fileItem
     * @param fileItemProperties
     * @throws FileItemException
     */
    public boolean createEntry(@NonNull final FileItem fileItem, @NonNull final FileStoreData fileStoreData, @NonNull final FileItemProperties fileItemProperties) throws FileItemException {
        // check FileItem
        if (fileItem.isInvalid()) {
            throw new FileItemException("FileItem is not valid: " + fileItem.toString());
        }

        // check FileStoreData
        if (fileStoreData.isInvalid()) {
            throw new FileItemException("FileStoreData is not valid: " + fileStoreData.toString());
        }

        final String groupId = fileItem.getGroupId();
        final String subGroupId = fileItem.getSubGroupId();
        final String fileId = fileItem.getFileId();
        final Properties customProperties = fileItemProperties.getCustomProperties();
        final Set<Object> customKeys = customProperties.keySet();
        int index = 0;

        final String insertContentSql = STR_BUILDER("INSERT INTO ").append(TBL_CONTENT(groupId, null)).
            append(" (FileStoreNumber,FileStoreId,SubGroupId,FileId) VALUES (?,?,?,?)").
            append(" ON DUPLICATE KEY UPDATE").
            append(" FileStoreNumber=?, FileStoreId=?, SubGroupId=?, FileId=?").toString();
        final StringBuilder insertPropertiesBuilderStart = STR_BUILDER("INSERT INTO ").append(TBL_PROPERTIES(groupId, null)).
            append(" (FileStoreNumber,FileStoreId,CreateDate,ModificationDate,Length");
        final StringBuilder insertPropertiesBuilderEnd = STR_BUILDER(" VALUES (?,?,?,?,?");
        final StringBuilder updatePropertiesBuilder = STR_BUILDER(" ON DUPLICATE KEY UPDATE ").
            append(" FileStoreNumber=?,FileStoreId=?,CreateDate=?,ModificationDate=?,Length=?");

        if (customKeys.size() > 0) {
            for (final Object curCustomKey : customKeys) {
                insertPropertiesBuilderStart.append(',').append(curCustomKey);
                insertPropertiesBuilderEnd.append(",?");

                updatePropertiesBuilder.append(',').append(curCustomKey).append("=?");
            }
        }

        final String insertOrUpdatePropertiesSql = insertPropertiesBuilderStart.append(") ").append(insertPropertiesBuilderEnd).append(')').
            append(updatePropertiesBuilder).toString();

        for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= 1; sleepTimeoutMillis <<= 1) {
            Connection usedCon = null;

            // Insert new data into Content table if
            // fileElement and FileStoreData are valid
            try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper();
                 final PreparedStatement fcStmt = conWrapper.prepareStatement(insertContentSql);
                 final PreparedStatement fpStmt = conWrapper.prepareStatement(insertOrUpdatePropertiesSql)) {

                final int fileStoreNumber = fileStoreData.getFileStoreNumber();
                final String fileStoreId = fileStoreData.getFileStoreId();
                final long createDateMillis = fileItemProperties.getCreateDateMillis();
                final long modificationDateMillis = fileItemProperties.getModificationDateMillis();
                final long length = fileItemProperties.getLength();

                usedCon = conWrapper.getConnection();

                // create entry in Content table
                fcStmt.setInt(index = 1, fileStoreNumber);
                fcStmt.setString(++index, fileStoreId);
                fcStmt.setString(++index, subGroupId);
                fcStmt.setString(++index, fileId);

                fcStmt.setInt(++index, fileStoreNumber);
                fcStmt.setString(++index, fileStoreId);
                fcStmt.setString(++index, subGroupId);
                fcStmt.setString(++index, fileId);

                // create properties
                fpStmt.setInt(index = 1, fileStoreNumber);
                fpStmt.setString(++index, fileStoreId);
                fpStmt.setLong(++index, createDateMillis);
                fpStmt.setLong(++index, modificationDateMillis);
                fpStmt.setLong(++index, length);

                // set given insert key custom values
                if (customKeys.size() > 0) {
                    for (final Object curCustomKey : customKeys) {
                        fpStmt.setString(++index, fileItemProperties.getCustomKeyValue(curCustomKey.toString()));
                    }
                }

                fpStmt.setInt(++index, fileStoreNumber);
                fpStmt.setString(++index, fileStoreId);
                fpStmt.setLong(++index, createDateMillis);
                fpStmt.setLong(++index, modificationDateMillis);
                fpStmt.setLong(++index, length);

                // set given update custom values
                if (customKeys.size() > 0) {
                    for (final Object curCustomKey : customKeys) {
                        fpStmt.setString(++index, fileItemProperties.getCustomKeyValue(curCustomKey.toString()));
                    }
                }

                // execute both statements and commit
                fcStmt.executeUpdate();
                fpStmt.executeUpdate();

                usedCon.commit();

                // if inserting of new entries into both
                // tables went fine, set fileStoreData
                // at fileElement and set return value to true
                fileItem.setFileStoreData(fileStoreData);

                return true;
            } catch (Exception e) {
                FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, "Error while creating entry");
            }
        }

        return false;
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public FileItem[] getFileItems(@NonNull final String groupId, @NonNull final String subGroupId) throws FileItemException {
        final String querySql = STR_BUILDER("SELECT fc.FileStoreNumber, fc.FileStoreId, fc.SubGroupId, fc.FileId").
            append(" FROM ").append(TBL_CONTENT(groupId, "fc")).
            append(" WHERE fc.SubGroupId=?").toString();

        try (final ConnectionWrapper conWrapper = getReadConnectionWrapper();
             final PreparedStatement queryStmt = conWrapper.prepareStatement(querySql)) {

            queryStmt.setString(1, subGroupId);

            return implExecuteQuery(conWrapper.getConnection(), groupId, queryStmt);
        } catch (Exception e) {
            throw new FileItemException(e);
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public FileItem[] getFileItems(@NonNull final String groupId, @NonNull final Properties properties) throws FileItemException {
        final Properties customProperties = new Properties();

        for (final Object curKey : properties.keySet()) {
            customProperties.put("fp." + curKey, properties.getProperty(curKey.toString()));
        }

        final String querySql = FileItemUtils.createSqlWithPropertyVariables(
            STR_BUILDER("SELECT fc.FileStoreNumber, fc.FileStoreId, fc.SubGroupId, fc.FileId FROM ").append(TBL_CONTENT(groupId, "fc")).
            append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
            append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId ").
            append(" WHERE ").toString(),
            customProperties,
            " AND ");

        try (final ConnectionWrapper conWrapper = getReadConnectionWrapper();
             final PreparedStatement queryStmt = conWrapper.prepareStatement(querySql)) {

            FileItemUtils.setStatementValues(queryStmt, customProperties.values().toArray(), 1);

            return implExecuteQuery(conWrapper.getConnection(), groupId, queryStmt);
        } catch (Exception e) {
            throw new FileItemException(e);
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public long getGroupCount() {
        return m_groupMap.size();
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public long getSubGroupCount(@NonNull final String groupId) throws FileItemException {
        final GroupConfiguration groupConfiguration = m_groupMap.get(groupId);

        return (null != groupConfiguration) ?
            implExecuteGetSubGroupCount(groupId, groupConfiguration.isUseCountTable()) :
                0;
    }

    /**
     * @param groupId
     * @return
     */
    public boolean contains(@NonNull final String groupId) {
        return m_groupMap.containsKey(groupId);
    }

    /**
     * @param groupId
     * @return
     */
    public boolean contains(@NonNull final String groupId, @NonNull final String subGroupId) throws FileItemException {
        return implExecuteContains(groupId, STR_BUILDER("fc.SubGroupId='").append(subGroupId).append("'").toString());
    }

    /**
     * @param groupId
     * @return
     */
    public boolean contains(@NonNull final String groupId, @NonNull final String subGroupId,@NonNull final String fileId) throws FileItemException {
        return implExecuteContains(groupId,
            STR_BUILDER("fc.SubGroupId='").append(subGroupId).append("'").
                append(" AND fc.FileId='").append(fileId).append("'").toString());
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public String[] getSubGroupIds(@NonNull final String groupId) throws FileItemException {
        try {
            return implExecuteGetIds(STR_BUILDER("SELECT DISTINCT fc.SubGroupId FROM ").append(TBL_CONTENT(groupId, "fc")).toString());
        } catch (Exception e) {
            throw new FileItemException("Exception caught when getting SubGroupIds", e);
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public String[] getSubGroupIdsBy(@NonNull final String groupId, @Nullable final String whereClause, @Nullable String limitClause) throws FileItemException {
        final StringBuilder querySql = STR_BUILDER("SELECT fc.SubGroupId, SUM(fp.Length) AS SubGroupLength, MIN(fp.ModificationDate) AS SubGroupDate FROM ").
            append(TBL_CONTENT(groupId, "fc")).append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
            append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId ");

        if (isNotBlank(whereClause)) {
            querySql.append(" WHERE ").append(whereClause);
        }

        querySql.append(" GROUP BY fc.SubGroupId ORDER BY SubGroupDate ASC, SubGroupLength DESC");

        if (isNotBlank(limitClause)) {
            querySql.append(" LIMIT ").append(limitClause);
        }

        try {
            return implExecuteGetIds(querySql.toString());
        } catch (Exception e) {
            throw new FileItemException("Exception caught when getting SubGroupIds by WHERE and/or LIMIT clause", e);
        }
    }

    /**
     * @param groupId
     * @param whereClause
     * @param limitClause
     * @param subGroupConsumer
     * @return
     * @throws FileItemException
     */
    public long getSubGroupsBy(String groupId, String whereClause, String limitClause, Consumer<ISubGroup> subGroupConsumer) throws FileItemException {
        final StringBuilder querySql = STR_BUILDER("SELECT fc.SubGroupId,").
            append(" SUM(fp.Length) AS SubGroupLength,").
            append(" MIN(fp.CreateDate) AS SubGroupCreateDate,").
            append(" MAX(fp.ModificationDate) AS SubGroupModificationDate").
            append(" FROM ").append(TBL_CONTENT(groupId, "fc")).append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
            append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId ");

        if (isNotBlank(whereClause)) {
            querySql.append(" WHERE ").append(whereClause);
        }

        querySql.append(" GROUP BY fc.SubGroupId ORDER BY SubGroupModificationDate ASC, SubGroupLength DESC");

        if (isNotBlank(limitClause)) {
            querySql.append(" LIMIT ").append(limitClause);
        }

        try (final ConnectionWrapper conWrapper = getReadConnectionWrapper();
            final PreparedStatement stmt = conWrapper.prepareStatement(querySql.toString())) {

            long ret = 0;

            try (final ResultSet resultSet = stmt.executeQuery()) {
                if (null != resultSet) {
                    while (resultSet.next()) {
                        final String subGroupId = resultSet.getString(1);
                        final long subGroupLength = resultSet.getLong(2);
                        final long subGroupCreateDateMillis = resultSet.getLong(3);
                        final long subGroupModificationDateMillis = resultSet.getLong(4);

                        ++ret;

                        if (null != subGroupConsumer) {
                            subGroupConsumer.accept(new ISubGroup() {

                                @Override
                                public String getSubGroupId() {
                                    return subGroupId;
                                }

                                @Override
                                public long getLength() {
                                    return subGroupLength;
                                }

                                @Override
                                public Date getCreateDate() {
                                    return new Date(subGroupCreateDateMillis);
                                }

                                @Override
                                public Date getModificationDate() {
                                    return new Date(subGroupModificationDateMillis);
                                }
                            });
                        }
                    }
                }
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                conWrapper.commit();
            }

            return ret;
        } catch (Exception e) {
            throw new FileItemException("Exception caught when getting SubGroupIds by WHERE and/or LIMIT clause", e);
        }
    }

    /**
     * getTotalLength
     *
     * @param groupId
     * @return
     * @throws FileItemException
     */
    public long getGroupLength(@NonNull final String groupId) throws FileItemException {
        final GroupConfiguration groupConfiguration = m_groupMap.get(groupId);

        return (null != groupConfiguration) ?
            implExecuteGetGroupLength(groupId, groupConfiguration.isUseCountTable()) :
                0;
    }

    /**
     * @param groupId
     * @param subGroupId
     * @param properties
     * @return
     * @throws FileItemException
     */
    public long getGroupLength(@NonNull final String groupId, @Nullable final String subGroupId, @Nullable final Properties properties) throws FileItemException {
        final Properties whereProperties = new Properties();
        StringBuilder selectSumSql = STR_BUILDER("SELECT SUM(fp.Length) from ").append(TBL_PROPERTIES(groupId, "fp"));
        Object[] whereValues = null;

        if (FileItemUtils.isValid(subGroupId)) {
            whereProperties.put("fc.SubGroupId", subGroupId);
        }

        if (null != properties) {
            for (final Object curKey : properties.keySet()) {
                whereProperties.put("fp." + curKey, properties.getProperty(curKey.toString()));
            }
        }

        if (whereProperties.size() > 0) {
            selectSumSql = STR_BUILDER(FileItemUtils.createSqlWithPropertyVariables(selectSumSql.
                    append(" INNER JOIN ").append(TBL_CONTENT(groupId, "fc")).
                    append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId WHERE").toString(),
                whereProperties,
                " AND "));

            whereValues = whereProperties.values().toArray(new Object[whereProperties.size()]);
        }

        try {
            return implExecuteGetLong(selectSumSql.toString(), whereValues);
        } catch (FileItemException e) {
            throw new FileItemException("Exception caught when getting summed up length of FileItems", e);
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public String[] getGroupIds() {
        return m_groupMap.keySet().toArray(new String[m_groupMap.size()]);
    }

    /**
     * @param fileItem
     * @return
     * @throws FileItemException
     */
    public FileStoreData getFileStoreData(@NonNull final FileItem fileItem) throws FileItemException {
        final String selectSql = STR_BUILDER("SELECT fc.FileStoreNumber, fc.FileStoreId FROM ").append(TBL_CONTENT(fileItem.getGroupId(), "fc")).
            append(" WHERE fc.SubGroupId=? AND fc.FileId=?").toString();

        try (final ConnectionWrapper conWrapper = getReadConnectionWrapper();
             final PreparedStatement stmt = conWrapper.prepareStatement(selectSql)) {

            stmt.setString(1, fileItem.getSubGroupId());
            stmt.setString(2, fileItem.getFileId());

            try (final ResultSet resultSet = stmt.executeQuery()) {
                if ((null != resultSet) && resultSet.first()) {
                    return new FileStoreData(resultSet.getInt(1), resultSet.getString(2));
                }
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                conWrapper.commit();
            }
        } catch (Exception e) {
            throw new FileItemException(e);
        }

        return null;
    }

    /**
     * @param fileStoreId.get
     * @param fileProperties
     * @throws FileItemException
     */
    public FileItemProperties getFileItemProperties(@NonNull FileItem fileItem) throws FileItemException {
        final String groupId = fileItem.getGroupId();
        final Set<String> customKeys = getCustomPropertyKeys(groupId);
        final StringBuilder selectSqlBuilderStart = STR_BUILDER("SELECT fp.FileStoreNumber,fp.FileStoreId,fp.CreateDate,fp.ModificationDate,fp.Length");
        final String selectSqlEnd = STR_BUILDER(" FROM ").append(TBL_CONTENT(groupId, "fc")).
            append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, " fp ")).
            append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId").
            append(" WHERE fc.SubGroupId=? AND fc.FileId=?").toString();

        // append all property keys for given groupId to Sql statement
        for (final String curCustomKey : customKeys) {
            selectSqlBuilderStart.append(",fp." + curCustomKey);
        }

        final String selectSql = selectSqlBuilderStart.append(selectSqlEnd).toString();

        try (final ConnectionWrapper conWrapper = getReadConnectionWrapper();
             final PreparedStatement stmt = conWrapper.prepareStatement(selectSql)) {

            stmt.setString(1, fileItem.getSubGroupId());
            stmt.setString(2, fileItem.getFileId());

            try (final ResultSet resultSet = stmt.executeQuery()) {

                if ((null != resultSet) && resultSet.first()) {
                    int index = 0;

                    final int fileStoreNumber = resultSet.getInt(++index);
                    final String fileStoreId = resultSet.getString(++index);

                    fileItem.setFileStoreData(new FileStoreData(fileStoreNumber, fileStoreId));

                    final FileItemProperties ret = new FileItemProperties(customKeys);

                    ret.setCreateDateMillis(resultSet.getLong(++index));
                    ret.setModificationDateMillis(resultSet.getLong(++index));
                    ret.setLength(resultSet.getLong(++index));

                    // read all key property values for the
                    // given groupId and set at result FileItemProperties
                    for (final String curCustomKey : customKeys) {
                        ret.setCustomKeyValue(curCustomKey, resultSet.getString(++index));
                    }

                    return ret;
                }
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                conWrapper.commit();
            }
        } catch(Exception e) {
           throw new FileItemException(e);
        }

        return null;
    }

    /**
     * @param fileStoreData
     * @param fileProperties
     * @throws FileItemException
     */
    public void updateProperties(@NonNull final FileStoreData fileStoreData, @NonNull final String groupId, @NonNull final Properties properties) throws FileItemException {
        if (properties.size() > 0) {
            final Properties setProperties = new Properties();

            for (final Object curKey : properties.keySet()) {
                setProperties.put("fp." + curKey, properties.get(curKey));
            }

            final String sqlStmt = FileItemUtils.createSqlWithPropertyVariables(
                STR_BUILDER("UPDATE ").append(TBL_PROPERTIES(groupId, "fp")).append(" SET").toString(),
                    setProperties,
                    ",",
                    " WHERE fp.FileStoreNumber=? AND fp.FileStoreId=?");

            for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS; sleepTimeoutMillis <<= 1) {
                Connection usedCon = null;

                try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper();
                     final PreparedStatement stmt = conWrapper.prepareStatement(sqlStmt)) {

                    usedCon = conWrapper.getConnection();

                    int curIndex = FileItemUtils.setStatementValues(stmt, setProperties.values().toArray(), 1);

                    stmt.setInt(curIndex++, fileStoreData.getFileStoreNumber());
                    stmt.setString(curIndex++, fileStoreData.getFileStoreId());

                    stmt.executeUpdate();

                    usedCon.commit();
                    break;
                } catch (Exception e) {
                    FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, null);
                }
            }
        }
    }

    /**
     * @param fileStoreId
     * @throws FileItemException
     */
    public int deleteEntry(@NonNull final String groupId, @NonNull final String subGroupId, @NonNull final String fileId, @NonNull final List<FileStoreData> deletedFileStoreDataList) throws FileItemException {
        final String whereStr = STR_BUILDER("fc.SubGroupId=").append("'").append(subGroupId).append("'").
                append(" AND").
                append(" fc.FileId=").append("'").append(fileId).append("'").toString();

        return implExecuteDelete(groupId, whereStr, deletedFileStoreDataList);
    }

    /**
     * @param fileStoreId
     * @throws FileItemException
     */
    public int deleteByGroupSubgroup(@NonNull final String groupId, @Nullable final String subGroupId,  @NonNull final List<FileStoreData> deletedFileStoreDataList) throws FileItemException {
        final String whereStr = (null != subGroupId) ?
            STR_BUILDER("fc.SubGroupId=").append("'").append(subGroupId).append("'").toString() :
                null;

        return implExecuteDelete(groupId, whereStr, deletedFileStoreDataList);
    }

    /**
     * @param groupId
     * @param fileStoreData
     * @param deletedFileStoreDataList
     * @return
     * @throws FileItemException
     */
    public int deleteByFileStoreData(@NonNull final String groupId, @NonNull final FileStoreData fileStoreData, @NonNull final List<FileStoreData> deletedFileStoreDataList) throws FileItemException {
        final String whereStr = STR_BUILDER("fc.FileStoreNumber=").append("'").append(fileStoreData.getFileStoreNumber()).append("'").
                append(" AND").
                append(" fc.FileStoreId=").append("'").append(fileStoreData.getFileStoreId()).append("'").toString();

        return implExecuteDelete(groupId, whereStr, deletedFileStoreDataList);
    }

    /**
     * @param groupId
     * @param subGroupIds
     * @return
     * @throws FileItemException
     */
    public int deleteByGroupSubgroups(@NonNull final String groupId, @NonNull final String[] subGroupIds, @NonNull final List<FileStoreData> deletedFileStoreDataList) throws FileItemException {
        final int count = subGroupIds.length;
        int ret = 0;

        if (count > 0) {
            final int blockCount = 1024;
            int curPos = 0;

            // remove keys in blocks with a maximum of blockCount keys
            do {
                final StringBuilder whereStrBuilder = STR_BUILDER("fc.SubGroupId IN (");

                for (int i = 0, loopCount = Math.min(blockCount, count - curPos); i < loopCount; ++i, ++curPos) {
                    if (i > 0) {
                        whereStrBuilder.append(',');
                    }

                    whereStrBuilder.append("'").append(subGroupIds[curPos]).append("'");
                }

                whereStrBuilder.append(')');

                ret += implExecuteDelete(groupId, whereStrBuilder.toString(), deletedFileStoreDataList);
            } while (curPos < count);
        }

        return ret;
    }

    /**
     * @param fileStoreId
     * @throws FileItemException
     */
    public int deleteByProperties(@NonNull final String groupId, @NonNull final Properties properties, @NonNull final List<FileStoreData> deletedFileStoreDataList) throws FileItemException {
        final StringBuilder whereStrBuilder = STR_BUILDER(null);

        for (final Object curKey : properties.keySet()) {
            if (whereStrBuilder.length() > 0) {
                whereStrBuilder.append(" AND ");
            }

            whereStrBuilder.append("fp.").append(curKey).append("=").append("'").append(properties.get(curKey)).append("'");
        }

        return implExecuteDelete(groupId, whereStrBuilder.toString(), deletedFileStoreDataList);
    }

    //  - Public interface to retrieve connections -------------------------

    /**
     * getReadConnection
     */
    public ConnectionWrapper getReadConnectionWrapper() throws FileItemException {
        if (null == m_readConnectionPool) {
            throw new FileItemException("No read connection pool available. Please check logging output and setup!");
        }

        try {
            final ConnectionWrapper ret = new ConnectionWrapper(m_readConnectionPool.getConnection());
            final Connection readCon = ret.getConnection();

            // Throw exception in case of invalid connection
            if (null == readCon) {
                // Thrown exception is caught in outer catch block and rethrown there
                throw new SQLException("Could not get valid read connection from connection pool");
            }

            return ret;
        } catch (SQLException e) {
            throw new FileItemException("Exception caught when getting read connection", e);
        }
    }

    /**
     * getWriteConnection
     */
    public ConnectionWrapper getWriteConnectionWrapper() throws FileItemException {
        return implGetWriteConnectionWrapper(false);
    }

    /**
     * getWriteConnection
     */
    public ConnectionWrapper getExclusiveWriteConnectionWrapper() throws FileItemException {
        return implGetWriteConnectionWrapper(true);
    }

    // - Implementation --------------------------------------------------------

    private ConnectionWrapper implGetWriteConnectionWrapper(boolean exclusive) throws FileItemException {
        if (null == m_writeConnectionPool) {
            throw new FileItemException("No write connection pool available. Please check logging output and setup!");
        }

        // synchronize for exclusive connection gets and signal availability handling;
        // access to m_exclusiveWriteConnection needs to synchronized as well
        m_writeConnectionLock.lock();

        try {
            // if somebody is currently using an exclusive write connection =>
            // wait until this exclusive connection is closed,
            // m_exclusiveWriteConnectionreset is reset to null and/or we get signaled
            while (null != m_exclusiveWriteConnection) {
                try {
                    m_writeConnectionAvailableCondition.await();
                } catch (InterruptedException e) {
                    // ok
                }
            }

            final ConnectionWrapper ret = new ConnectionWrapper(m_writeConnectionPool.getConnection());
            final Connection writeCon = ret.getConnection();

            // Throw exception in case of invalid connection
            if (null == writeCon) {
                // Thrown exception is caught in outer catch block and rethrown there
                throw new SQLException("Could not get valid write connection from connection pool");
            }

            // set m_exclusiveWriteConnection to
            // just opened connection if requested
            if (exclusive) {
                m_exclusiveWriteConnection = writeCon;
            }

            // set handler interface so that we get notified when the
            // just retrieved connection is closed;
            // in case of an exclusive connection, all waiting threads
            // need to be signaled when the exclusive connection is closed
            ret.setConnectionClosedHandler((closedCon) -> {
                // handler is called from outside this method => synchronize!
                m_writeConnectionLock.lock();

                try {
                    // check if the closed connection is the one and only exclusive one
                    if (m_exclusiveWriteConnection == closedCon) {
                        // set m_exclusiveWriteConnection back to null
                        m_exclusiveWriteConnection = null;

                        // signal all waiting threads that no
                        // more exclusive connection is open
                        m_writeConnectionAvailableCondition.signalAll();
                    }
                } finally {
                    m_writeConnectionLock.unlock();
                }
            });

            return ret;
        } catch (SQLException e) {
            throw new FileItemException("Exception caught when getting write connection", e);
        } finally {
            m_writeConnectionLock.unlock();
        }

    }

    /**
     * @param selectSql
     * @param stmtValues
     * @return
     * @throws FileItemException
     * @throws SQLException
     */
    private String[] implExecuteGetIds(@NonNull final String selectSql, final Object... stmtValues) throws FileItemException, Exception {
        try (final ConnectionWrapper conWrapper = getReadConnectionWrapper();
             final PreparedStatement stmt = conWrapper.prepareStatement(selectSql)) {

            FileItemUtils.setStatementValues(stmt, stmtValues, 1);

            try (final ResultSet resultSet = stmt.executeQuery()) {
                final List<String> idList = new ArrayList<>();

                if (null != resultSet) {
                    while (resultSet.next()) {
                        idList.add(resultSet.getString(1));
                    }
                }

                return idList.toArray(new String[idList.size()]);
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                conWrapper.commit();
            }
        }
    }

    /**
     * @param selectSql
     * @param stmtValues
     * @return
     * @throws FileItemException
     * @throws SQLException
     */
    private long implExecuteGetLong(@NonNull final String selectLongReturnSql, final Object... stmtValues) throws FileItemException {
        try (final ConnectionWrapper conWrapper = getReadConnectionWrapper();
             final PreparedStatement stmt = conWrapper.prepareStatement(selectLongReturnSql)) {

            FileItemUtils.setStatementValues(stmt, stmtValues, 1);

            try (final ResultSet resultSet = stmt.executeQuery()) {
                if ((null != resultSet) && resultSet.first()) {
                    return resultSet.getLong(1);
                }
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                conWrapper.commit();
            }
        } catch (Exception e) {
            throw new FileItemException("Error while getting long value from database", e);
        }

        return 0;
    }

    /**
     * @param whereProperties
     * @return
     * @throws FileItemException
     */
    private boolean implExecuteContains(@NonNull final String groupId, @NonNull final String whereStr) throws FileItemException {
        if (m_groupMap.containsKey(groupId)) {
            try (final ConnectionWrapper conWrapper = getReadConnectionWrapper()) {
                final StringBuilder existsBuilder = STR_BUILDER("SELECT EXISTS (SELECT * FROM ").append(TBL_CONTENT(groupId, "fc")).
                    append(" WHERE ").append(whereStr).
                    append(" LIMIT 1)");

                try (final PreparedStatement existsStmt = conWrapper.prepareStatement(existsBuilder.toString());
                     final ResultSet resultSet = existsStmt.executeQuery()) {

                    return ((null != resultSet) && resultSet.first()) ?
                        resultSet.getBoolean(1) :
                            false;
                } finally {
                    // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                    conWrapper.commit();
                }
            } catch (Exception e) {
                throw new FileItemException("Error while deleting entry(ies))", e);
            }
        }

        return false;
    }

    /**
     * @param whereProperties
     * @return
     * @throws FileItemException
     */
    private int implExecuteDelete(@NonNull final String groupId, @Nullable final String whereStr, @NonNull final List<FileStoreData> deletedFileStoreDataList) throws FileItemException {
        final StringBuilder selectStrBuilder = STR_BUILDER("SELECT fc.FileStoreNumber, fc.FileStoreId FROM ").
            append(TBL_CONTENT(groupId, "fc")).append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
            append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId");

        final StringBuilder deleteStrBuilder = STR_BUILDER("DELETE fc, fp FROM ").
            append(TBL_CONTENT(groupId, "fc")).append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
            append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId");

        if (isNotBlank(whereStr)) {
            selectStrBuilder.append(" WHERE ").append(whereStr);
            deleteStrBuilder.append(" WHERE ").append(whereStr);
        }

        for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS; sleepTimeoutMillis <<= 1) {
            Connection usedCon = null;

            try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper();
                 final PreparedStatement selectStmt = conWrapper.prepareStatement(selectStrBuilder.toString());
                 final PreparedStatement deleteStmt = conWrapper.prepareStatement(deleteStrBuilder.toString())) {

                usedCon = conWrapper.getConnection();

                selectStmt.executeQuery();
                deleteStmt.executeUpdate();

                usedCon.commit();

                try (final ResultSet resultSet = selectStmt.getResultSet()) {
                   if (null != resultSet) {
                       while (resultSet.next()) {
                           deletedFileStoreDataList.add(new FileStoreData(resultSet.getInt(1), resultSet.getString(2)));
                       }
                   }
                }

                return deletedFileStoreDataList.size();
            } catch (Exception e) {
                FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, "Error while deleting entry(ies))");
            }
        }

        return 0;
    }

    /**
     * @param con
     * @param selectStmt
     * @return
     */
    private static FileItem[] implExecuteQuery(@NonNull final Connection con, @NonNull final String groupId, @NonNull final PreparedStatement queryStmt, Object... returnValues) throws Exception {
        try (final ResultSet resultSet = queryStmt.executeQuery()) {
            final ArrayList<FileItem> fileItemList = new ArrayList<>();

            if (null != resultSet) {
                while (resultSet.next()) {
                    final int fileStoreNumber = resultSet.getInt(1);
                    final String fileStoreId = resultSet.getString(2);
                    final String subGroupId = resultSet.getString(3);
                    final String fileId = resultSet.getString(4);

                    final FileItem fileItem = new FileItem(groupId, subGroupId, fileId);

                    fileItem.setFileStoreData(new FileStoreData(fileStoreNumber, fileStoreId));
                    fileItemList.add(fileItem);

                    if (ArrayUtils.isNotEmpty(returnValues)) {
                        for (int i = 0, count = returnValues.length; i < count; ++i) {
                            final Object curReturnObj = returnValues[i];

                            if (curReturnObj instanceof Collection<?>) {
                                @SuppressWarnings("unchecked") final Collection<Object> curReturnCollection = (Collection<Object>) curReturnObj;

                                switch (i) {
                                    case 0: curReturnCollection.add(Integer.valueOf(fileStoreNumber)); break;
                                    case 1: curReturnCollection.add(fileStoreId); break;
                                    case 2: curReturnCollection.add(groupId); break;
                                    case 3: curReturnCollection.add(subGroupId); break;
                                    case 4: curReturnCollection.add(fileId); break;

                                    default: {
                                        // GroupId is not part of the SQL query itself,
                                        // so that the index of the return paramater equals
                                        // the current return array position instead
                                        // of (position + 1)
                                        curReturnCollection.add(resultSet.getObject(i));
                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
            }

            return FileItemUtils.listToArray(fileItemList, FileItem.class);
        } catch (Exception e) {
            throw new FileItemException(e);
        } finally {
            // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
            con.commit();
        }
    }

    /**
     * implExecuteGetGroupLength
     *
     * @param groupId
     * @param usdeCountTable
     * @return
     * @throws FileItemException
     */
    private long implExecuteGetGroupLength(@NonNull final String groupId, final boolean useCountTable) throws FileItemException {
        try {
            final StringBuilder selectGroupLengthSql = useCountTable ?
                    STR_BUILDER("SELECT fa.TotalLength FROM ").append(TBL_GROUPADMIN("fa")).append(" WHERE fa.GroupId='").append(groupId).append("'") :
                        STR_BUILDER("SELECT SUM(fp.Length) from ").append(TBL_PROPERTIES(groupId, "fp"));

            return implExecuteGetLong(selectGroupLengthSql.toString());
        } catch (FileItemException e) {
            throw new FileItemException("Exception caught when getting summed up length of FileItems for group: " + groupId, e);
        }
    }

    /**
     * implExecuteGetSubGroupCount
     *
     * @param groupId
     * @param useCountTable
     * @return
     * @throws FileItemException
     */
    public long implExecuteGetSubGroupCount(@NonNull final String groupId, final boolean useCountTable) throws FileItemException {
        try {
            final StringBuilder subGroupCountSql = useCountTable ?
                    STR_BUILDER("SELECT fa.SubGroupCount FROM ").append(TBL_GROUPADMIN("fa")).append(" WHERE fa.GroupId='").append(groupId).append("'") :
                        STR_BUILDER("SELECT COUNT(DISTINCT fc.SubGroupId) FROM ").append(TBL_CONTENT(groupId, "fc"));

            return implExecuteGetLong(subGroupCountSql.toString());
        } catch (FileItemException e) {
            throw new FileItemException("Exception caught when getting SubGroupId count for group: " + groupId, e);
        }
    }

    /**
     * @throws FileItemException
     */
    private void implCreateConnectionPools() throws FileItemException {
        if (null != m_configService) {
            final String readDriverClass = m_configService.getProperty(CFG_KEY_1("readDriverClass"));
            final String writeDriverClass = m_configService.getProperty(CFG_KEY_1("writeDriverClass"));
            final String jdbcReadUrl = m_configService.getProperty(CFG_KEY_1("readUrl"));
            final String jdbcWriteUrl = m_configService.getProperty(CFG_KEY_1("writeUrl"));

            try {
                Class.forName(readDriverClass);
                Class.forName(writeDriverClass);
            } catch (ClassNotFoundException e) {
                throw new FileItemException(e);
            }

            if (isEmpty(jdbcReadUrl) || isEmpty(jdbcWriteUrl)) {
                throw new FileItemException("Property readUrl and writeUrl must not be empty (Example Url: jdbc:mysql://host:port/databasename)");
            }

            try {
                final int poolMaxSize = m_configService.getIntProperty(CFG_KEY_1("connectionpool.maxPoolSize"), HikariSetup.DEFAULT_POOL_MAXSIZE);
                final int poolConnectTimeoutMillis = m_configService.getIntProperty(CFG_KEY_1("connectionpool.connectTimeout"), (int) HikariSetup.DEFAULT_POOL_CONNECTTIMEOUTMILLIS);
                final int poolIdleTimeoutMillis = m_configService.getIntProperty(CFG_KEY_1("connectionpool.idleTimeout"), (int) HikariSetup.DEFAULT_POOL_IDLETIMEOUTMILLIS);
                final int poolMaxLifetimeMillis = m_configService.getIntProperty(CFG_KEY_1("connectionpool.maxLifetime"), (int) HikariSetup.DEFAULT_POOL_MAX_LIFETIMEMILLIS);
                Properties dbProperties = null;

                // DB read properties and DataSource
                implFillDBConnectionPropertiesFromConfigItems(
                    dbProperties = new Properties(),
                    m_configService.getProperties(new WildcardNamePropertyFilter(CFG_KEY_1("readProperty.*"))),
                    m_configService.getProperties(new WildcardNamePropertyFilter(CFG_KEY_2("readProperty.*"))));

                m_readConnectionPool = new HikariSetup(readDriverClass, jdbcReadUrl,
                    dbProperties.getProperty("user"), dbProperties.getProperty("password"),
                    dbProperties,
                    poolMaxSize, poolConnectTimeoutMillis, poolIdleTimeoutMillis, poolMaxLifetimeMillis).getConnectionPool(true);

                // DB write properties and DataSource
                implFillDBConnectionPropertiesFromConfigItems(
                    dbProperties = new Properties(),
                    m_configService.getProperties(new WildcardNamePropertyFilter(CFG_KEY_1("writeProperty.*"))),
                    m_configService.getProperties(new WildcardNamePropertyFilter(CFG_KEY_2("writeProperty.*"))));

                m_writeConnectionPool = new HikariSetup(writeDriverClass, jdbcWriteUrl,
                    dbProperties.getProperty("user"), dbProperties.getProperty("password"),
                    dbProperties,
                    poolMaxSize, poolConnectTimeoutMillis, poolIdleTimeoutMillis, poolMaxLifetimeMillis).getConnectionPool(false);
            } catch (OXException e) {
                throw new FileItemException(e);
            }

            // determine database type
            if ((StringUtils.containsIgnoreCase(jdbcReadUrl, "mysql") && StringUtils.containsIgnoreCase(jdbcWriteUrl, "mysql")) ||
                (StringUtils.containsIgnoreCase(jdbcReadUrl, "maria") && StringUtils.containsIgnoreCase(jdbcWriteUrl, "maria"))) {

                m_databaseType = IFileItemService.DatabaseType.MYSQL;
            }
        }
    }

    /**
     * @throws FileItemException
     * @throws LiquibaseException
     */
    private void implUpdateDatabase() throws FileItemException {
        try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper()) {
            final Liquibase liquibase = new Liquibase(
                "fileItemChangeLog.xml",
                new LiquibaseResourceAccessor(),
                new JdbcConnection(conWrapper.getConnection()));

            liquibase.setChangeExecListener(this);
            liquibase.update(null);
        } catch (Exception e) {
            throw new FileItemException(e);
        }
    }

    /**
     * @param fileItem
     * @return
     * @throws FileItemException
     */
    private void implReadGroups() throws FileItemException {
        final String selectSql = STR_BUILDER("SELECT fa.GroupId, fa.PropertyKeys, fa.Sync, fa.Data FROM ").append(TBL_GROUPADMIN("fa")).toString();

        synchronized (m_groupMap) {
            try (final ConnectionWrapper conWrapper = getReadConnectionWrapper();
                 final PreparedStatement stmt = conWrapper.prepareStatement(selectSql)) {

                try (final ResultSet resultSet = stmt.executeQuery()) {
                    if (null != resultSet) {
                        while (resultSet.next()) {
                            final String curGroupId = resultSet.getString(1);

                            if (FileItemUtils.isValid(curGroupId)) {
                                GroupConfiguration groupConfiguration = m_groupMap.get(curGroupId);

                                // create and insert GroupConfiguration for currently read group, if not available yet
                                if (null == groupConfiguration) {
                                    m_groupMap.put(curGroupId, groupConfiguration = new GroupConfiguration(curGroupId));
                                }

                                final Set<String> keySet = implGetKeySet(curGroupId, false);
                                final Set<String> customKeySet = implGetKeySet(curGroupId, true);
                                final String curPropertyKeys = resultSet.getString(2);

                                // add to set for given groupId, if not already present
                                if (isNotEmpty(curPropertyKeys)) {
                                    final String[] keys  = new String(curPropertyKeys).split(",", 0);

                                    if (null != keys) {
                                        for (final String curKey : keys) {
                                            if (isNotBlank(curKey)) {
                                                final String targetKey = curKey.trim();

                                                keySet.add(targetKey);

                                                if (!implIsDefaultKey(curKey)) {
                                                    customKeySet.add(targetKey);
                                                }
                                            }
                                        }
                                    }
                                }

                                // finish group configuration with e.g. trigger creation and syncing
                                implFinishGroupConfiguration(groupConfiguration, resultSet.getBoolean(3), resultSet.getString(4));
                            }
                        }
                    }
                } finally {
                    // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                    conWrapper.commit();
                }
            } catch (Exception e) {
                throw new FileItemException(e);
            }
        }
    }

    /**
     * implFinishGroupConfiguration
     *
     * @param groupId
     * @param sync
     */
    void implFinishGroupConfiguration(@NonNull final GroupConfiguration groupConfiguration, final boolean sync, @Nullable final String userData ) {
        // if count table is to be used for group, sync first if indicated =>
        // count table is used if indicated and in in sync
        groupConfiguration.setUseCountTable(
            implCreateTriggers(groupConfiguration) &&
            (!sync || implSyncGroup(groupConfiguration)));

        final String userDataToUse = StringUtils.isBlank(userData) ? "{}" : userData;

        try {
            groupConfiguration.setUserData(new JSONObject(userDataToUse));
        } catch (JSONException e) {
            LOG.error("FileItemService could not set user data for group {}: {}",groupConfiguration.getGroupId(), userData);
        }
    }


    /**
     * implSyncGroup
     *
     * @param groupId
     */
    private boolean implSyncGroup(@NonNull final GroupConfiguration groupConfiguration) {
        final String groupId = groupConfiguration.getGroupId();

        try {
            final String updateGroupCountsSql = STR_BUILDER("UPDATE ").append(TBL_GROUPADMIN("fa")).
                    append(" SET fa.SubGroupCount=?, fa.TotalLength=?, fa.Sync=?").
                    append(" WHERE fa.GroupId=?").toString();

            Connection usedCon = null;

            for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS; sleepTimeoutMillis <<= 1) {
                try (final ConnectionWrapper conWrapper = getExclusiveWriteConnectionWrapper();
                     final PreparedStatement fgGroupCountsUpdateStmt = conWrapper.prepareStatement(updateGroupCountsSql)) {

                    usedCon = conWrapper.getConnection();

                    final long subGroupCount = implExecuteGetSubGroupCount(groupId,false);
                    final long groupLength = implExecuteGetGroupLength(groupId, false);

                    // update 'SubGroupCount', 'TotalLength' and 'Sync' rows;
                    // Sync value  is set to 'false' if sync was successful
                    fgGroupCountsUpdateStmt.setLong(1, subGroupCount);
                    fgGroupCountsUpdateStmt.setLong(2, groupLength);
                    fgGroupCountsUpdateStmt.setBoolean(3, false);
                    fgGroupCountsUpdateStmt.setString(4, groupId);

                    // execute all statements and commit
                    fgGroupCountsUpdateStmt.executeUpdate();

                    usedCon.commit();

                    return true;
                } catch (Exception e) {
                    final String cause = new StringBuilder(STR_BUILDER_LARGE_CAPACITY).
                        append("Exception occured when trying to update/sync SubGroupCount and TotalLength columns for GroupId: ").append(groupId).
                        append(" (").append(Throwables.getRootCause(e).getMessage()).append(')').toString();

                    FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, cause);
                }
            }
        } catch (FileItemException e) {
            LOG.info("FileItemService database could not update count tables for group {}. Using database server SQL counting.", groupId);
        }

        return false;
    }

    /**
     * @param groupId
     */
    private boolean implCreateTriggers(@NonNull final GroupConfiguration groupConfiguration) {
        final String groupId = groupConfiguration.getGroupId();
        final boolean info = FileItemUtils.isLogInfo();

        try {
            implCreateInsertTrigger(groupId);
            implCreateDeleteTrigger(groupId);
            implCreateUpdateTrigger(groupId);

            if (info) {
                LOG.info("FileItemService successfully created triggers to use count table for group: {}", groupId);
            }

            return true;
        } catch (Exception e) {
            if (info) {
                FileItemUtils.logInfo(STR_BUILDER("FileItemService could not create triggers to use count table for group ").append(groupId).append('.').
                    append(" Falling back to triggerless operation. ").
                    append(" Please consider upgrading MySQL database server to support triggers and ensure that TRIGGER privilege is granted for current database user.").toString());
            }
        }

        return false;
    }

    /**
     * @param groupId
     */
    private void implCreateInsertTrigger(@NonNull final String groupId) throws FileItemException {
        final String contentTriggerName = STR_BUILDER("insert_").append(TBL_CONTENT(groupId, null)).append("_before").toString();
        final String dropInsertContentTriggerSql = STR_BUILDER("DROP TRIGGER IF EXISTS ").append(contentTriggerName).toString();
        final String createInsertContentTriggerSql = STR_BUILDER("CREATE TRIGGER ").append(contentTriggerName).append(" BEFORE INSERT ON ").
            append(TBL_CONTENT(groupId, null)).
            append(" FOR EACH ROW BEGIN").
            append("   UPDATE GroupAdmin fa SET fa.SubGroupCount=fa.SubGroupCount+1").
            append("   WHERE").
            append("     fa.GroupId='").append(groupId).append("'").
            append("     AND NOT EXISTS (SELECT * FROM ").append(TBL_CONTENT(groupId, "fc")).
            append(" WHERE fc.SubGroupId=NEW.SubGroupId);").
            append(" END").toString();

        final String propertiesTriggerName = STR_BUILDER("insert_").append(TBL_PROPERTIES(groupId, null)).append("_after").toString();
        final String dropInsertPropertiesTriggerSql = STR_BUILDER("DROP TRIGGER IF EXISTS ").append(propertiesTriggerName).toString();
        final String createInsertPropertiesTriggerSql = STR_BUILDER("CREATE TRIGGER ").append(propertiesTriggerName).append(" AFTER INSERT ON ").
            append(TBL_PROPERTIES(groupId, null)).
            append(" FOR EACH ROW BEGIN").
            append("   UPDATE GroupAdmin fa SET fa.TotalLength=fa.TotalLength+NEW.Length").
            append("   WHERE").
            append("     fa.GroupId='").append(groupId).
            append("';").
            append(" END").toString();

        Connection usedCon = null;

        for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS; sleepTimeoutMillis <<= 1) {
            try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper();
                 final Statement insertStmt = conWrapper.createStatement()) {

                usedCon = conWrapper.getConnection();

                insertStmt.executeUpdate(dropInsertContentTriggerSql);
                insertStmt.executeUpdate(createInsertContentTriggerSql);

                insertStmt.executeUpdate(dropInsertPropertiesTriggerSql);
                insertStmt.executeUpdate(createInsertPropertiesTriggerSql);

                conWrapper.commit();
                break;
            } catch (Exception e) {
                FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, "Error while creating insert trigger");
            }
        }
    }

    /**
     * @param groupId
     */
    private void implCreateDeleteTrigger(@NonNull final String groupId) throws FileItemException {
        final String contentTriggerName = STR_BUILDER("delete_").append(TBL_CONTENT(groupId, null)).append("_after").toString();
        final String dropDeleteContentTriggerSql = STR_BUILDER("DROP TRIGGER IF EXISTS ").append(contentTriggerName).toString();
        final String createDeleteContentTriggerSql = STR_BUILDER("CREATE TRIGGER ").append(contentTriggerName).append(" AFTER DELETE ON ").append(TBL_CONTENT(groupId, null)).
           append(" FOR EACH ROW BEGIN").
           append("   UPDATE GroupAdmin fa SET fa.SubGroupCount=fa.SubGroupCount-1").
           append("   WHERE").
           append("     fa.GroupId='").append(groupId).append("'").
           append("     AND NOT EXISTS (SELECT * FROM ").append(TBL_CONTENT(groupId, "fc")).append(" WHERE fc.SubGroupId=OLD.SubGroupId);").
           append(" END").toString();

        final String propertiesTriggerName = STR_BUILDER("delete_").append(TBL_PROPERTIES(groupId, null)).append("_after").toString();
        final String dropDeletePropertiesTriggerSql = STR_BUILDER("DROP TRIGGER IF EXISTS ").append(propertiesTriggerName).toString();
        final String createDeletePropertiesTriggerSql = STR_BUILDER("CREATE TRIGGER ").append(propertiesTriggerName).append(" AFTER DELETE ON ").append(TBL_PROPERTIES(groupId, null)).
           append(" FOR EACH ROW BEGIN").
           append("   UPDATE GroupAdmin fa SET fa.TotalLength=fa.TotalLength-OLD.Length").
           append("   WHERE").
           append("     fa.GroupId='").append(groupId).append("';").
           append(" END").toString();

        Connection usedCon = null;

        for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS; sleepTimeoutMillis <<= 1) {
            try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper();
                 final Statement deleteStmt = conWrapper.createStatement()) {

                usedCon = conWrapper.getConnection();

                deleteStmt.executeUpdate(dropDeleteContentTriggerSql);
                deleteStmt.executeUpdate(createDeleteContentTriggerSql);

                deleteStmt.executeUpdate(dropDeletePropertiesTriggerSql);
                deleteStmt.executeUpdate(createDeletePropertiesTriggerSql);

                conWrapper.commit();
                break;
            } catch (Exception e) {
                FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, "Error while creating delete trigger");
            }
        }
    }

    /**
     * @param groupId
     */
    private void implCreateUpdateTrigger(@NonNull final String groupId) throws FileItemException {
        final String propertiesTriggerName = STR_BUILDER("update_").append(TBL_PROPERTIES(groupId, null)).append("_after").toString();
        final String dropUpdatePropertiesTriggerSql = STR_BUILDER("DROP TRIGGER IF EXISTS ").append(propertiesTriggerName).toString();
        final String createUpdatePropertiesTriggerSql = STR_BUILDER("CREATE TRIGGER ").append(propertiesTriggerName).append(" AFTER UPDATE ON ").append(TBL_PROPERTIES(groupId, null)).
           append(" FOR EACH ROW BEGIN").
           append("   UPDATE GroupAdmin fa SET fa.TotalLength=fa.TotalLength+(NEW.length - OLD.Length)").
           append("   WHERE").
           append("     fa.GroupId='").append(groupId).append("';").
           append(" END").toString();

        Connection usedCon = null;

        for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS; sleepTimeoutMillis <<= 1) {
            try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper();
                 final Statement updateStmt = conWrapper.createStatement()) {

                usedCon = conWrapper.getConnection();

                updateStmt.executeUpdate(dropUpdatePropertiesTriggerSql);
                updateStmt.executeUpdate(createUpdatePropertiesTriggerSql);

                updateStmt.executeUpdate(dropUpdatePropertiesTriggerSql);
                updateStmt.executeUpdate(createUpdatePropertiesTriggerSql);

                conWrapper.commit();
                break;
            } catch (Exception e) {
                FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, "Error while creating update trigger");
            }
        }
    }

    /**
     * @param status
     * @param description
     */
    private static void implLogChangeEvent(final Object status, final String description) {
        if (FileItemUtils.isLogInfo()) {
            if (null != status) {
                LOG.info("FileItemService database update received status: {}", status.toString());
            }

            if (isNotEmpty(description)) {
                LOG.info("FileItemService database update description: {}", description);
            }
        }
    }

    /**
     * @param dbProperties
     * @param configItems
     */
    private static void implFillDBConnectionPropertiesFromConfigItems(@NonNull final Properties dbProperties, @NonNull Map<String, String>... configItems) {
        for (final Map<String, String> curConfigItems : configItems) {
            for (final String curKey : curConfigItems.keySet()) {
                final String dbConfigPair = curConfigItems.get(curKey);

                if (isNotEmpty(dbConfigPair)) {
                    final int assignPos = dbConfigPair.indexOf('=');

                    if ((assignPos > 0) && (assignPos < (dbConfigPair.length() - 1))) {
                        dbProperties.put(dbConfigPair.substring(0, assignPos), dbConfigPair.substring(assignPos + 1));
                    }
                }
            }
        }
    }

    /**
     * @param groupId
     * @return
     */
    private boolean implCreateGroupTables(@NonNull final String groupId) throws FileItemException {
        // create Content_GroupId table
        final String createContentTableSql = STR_BUILDER("CREATE TABLE ").append(TBL_CONTENT(groupId, null)).
            append(" (").
            append(" FileStoreNumber INT(10) NOT NULL,").
            append(" FileStoreId ").append(DB_VARCHAR_TYPE).append(" NOT NULL,").
            append(" SubGroupId ").append(DB_VARCHAR_TYPE).append(" NOT NULL,").
            append(" FileId ").append(DB_VARCHAR_TYPE).append(" NOT NULL").
            append(" )").toString();

        final String alterContentTableSql = STR_BUILDER("ALTER TABLE ").append(TBL_CONTENT(groupId, null)).
            append(" ADD PRIMARY KEY (SubGroupId, FileId)").toString();

        final String createSecondaryContentIndexSql = STR_BUILDER("CREATE INDEX ").append("idx_FileStore").
            append(" ON ").append(TBL_CONTENT(groupId, " ")).
            append(" (").
            append(" FileStoreNumber,").
            append(" FileStoreId (128)").
            append(" )").toString();

        // create Properties_GroupId table
        final String createPropertiesTableSql = STR_BUILDER("CREATE TABLE ").append(TBL_PROPERTIES(groupId, null)).
            append(" (").
            append(" FileStoreNumber INT(10) NOT NULL,").
            append(" FileStoreId ").append(DB_VARCHAR_TYPE).append(" NOT NULL").
            append(" )").toString();

        final String alterPropertiesTableSql = STR_BUILDER("ALTER TABLE ").append(TBL_PROPERTIES(groupId, null)).
            append(" ADD PRIMARY KEY (FileStoreNumber, FileStoreId)").toString();

        // insert new group into Group table
        final String insertGroupSql = STR_BUILDER("INSERT INTO ").append(TBL_GROUPADMIN(null)).
            append(" (GroupId,PropertyKeys,SubGroupCount,TotalLength,Sync,Data) VALUES (?,?,?,?,?,?)").toString();

        for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS; sleepTimeoutMillis <<= 1) {
            Connection usedCon = null;

            try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper()) {
                usedCon = conWrapper.getConnection();

                try (final PreparedStatement createStmt = usedCon.prepareStatement(createContentTableSql);
                     final PreparedStatement alterStmt = usedCon.prepareStatement(alterContentTableSql);
                     final PreparedStatement createSecondaryIndexStmt = usedCon.prepareStatement(createSecondaryContentIndexSql)) {

                    createStmt.executeUpdate();
                    alterStmt.executeUpdate();
                    createSecondaryIndexStmt.executeUpdate();
                } catch (SQLException e) {
                    // ignore, errors will be handled in outer block
                }

                try (final PreparedStatement createStmt = usedCon.prepareStatement(createPropertiesTableSql);
                     final PreparedStatement alterStmt = usedCon.prepareStatement(alterPropertiesTableSql)) {

                    createStmt.executeUpdate();
                    alterStmt.executeUpdate();
                } catch (SQLException e) {
                    // ignore, errors will be handled in outer block
                }

                try (final PreparedStatement insertStmt = usedCon.prepareStatement(insertGroupSql)) {
                    insertStmt.setString(1, groupId);
                    insertStmt.setString(2, "");
                    insertStmt.setLong(3, 0);
                    insertStmt.setLong(4, 0);
                    insertStmt.setBoolean(5, true);
                    insertStmt.setString(6, "{}");

                    insertStmt.executeUpdate();
                } catch (SQLException e) {
                    // ignore, errors will be handled in outer block
                }

                usedCon.commit();
                return true;
            } catch (Exception e) {
                FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, "Error while creating group tables");
            }
        }

        return false;
    }

    /**
     * @param groupId
     * @param key
     * @param dbType
     * @return
     * @throws FileItemException
     */
    private void implInsertPropertyKey(@NonNull final String groupId, @NonNull final String key, @NonNull final String dbType, @Nullable final String dbColumnConstraints) throws FileItemException {
        final Set<String> propertyKeySet = implGetKeySet(groupId, false);

        propertyKeySet.add(key);

        final String updateGroupPropertyKeysSql = STR_BUILDER("UPDATE ").append(TBL_GROUPADMIN("fa")).
                append(" SET fa.PropertyKeys=?").
                append(" WHERE fa.GroupId=?").toString();


        // create update string value for set of property keys
        final StringBuilder propertyKeysBuilder = STR_BUILDER("");

        for (final String curKey : propertyKeySet) {
            if (isNotEmpty(curKey)) {
                if (propertyKeysBuilder.length() > 0) {
                    propertyKeysBuilder.append(',');
                }

                propertyKeysBuilder.append(curKey);
            }
        }

        Connection usedCon = null;

        for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS; sleepTimeoutMillis <<= 1) {
            try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper();
                 final PreparedStatement fgPropertyKeysUpdateStmt = conWrapper.prepareStatement(updateGroupPropertyKeysSql)) {

                usedCon = conWrapper.getConnection();

                // delete/insert groupId in Groups table
                fgPropertyKeysUpdateStmt.setString(1, propertyKeysBuilder.toString());
                fgPropertyKeysUpdateStmt.setString(2, groupId);

                // execute all statements and commit
                fgPropertyKeysUpdateStmt.executeUpdate();

                usedCon.commit();
                break;
            } catch (Exception e) {
                final String cause = new StringBuilder(STR_BUILDER_LARGE_CAPACITY).
                    append("Exception occured when trying to update PropertyKeys column for GroupId: ").append(groupId).
                    append(" (").append(Throwables.getRootCause(e).getMessage()).append(')').toString();

                FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, cause);
            }
        }

        // Add new customKey column to Properties table
        final StringBuilder insertPropertiesColSql = STR_BUILDER("ALTER TABLE ").append(TBL_PROPERTIES(groupId, null)).
            append(" ADD COLUMN ").append(key).append(' ').append(dbType);

        if (null != dbColumnConstraints) {
            insertPropertiesColSql.append(' ').append(dbColumnConstraints);
        }

        for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS; sleepTimeoutMillis <<= 1) {
            try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper();
                 final PreparedStatement fpInsertColStmt = conWrapper.prepareStatement(insertPropertiesColSql.toString())) {

                usedCon = conWrapper.getConnection();
                fpInsertColStmt.executeUpdate();

                usedCon.commit();
                break;
            } catch (Exception e) {
                final String cause = new StringBuilder(STR_BUILDER_LARGE_CAPACITY).
                    append("Exception occured when trying to add new column for GroupId: ").append(groupId).
                    append(" (").append(Throwables.getRootCause(e).getMessage()).append(')').toString();

                FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, cause);
            }
        }

        // create index for new column
        final String indexName = "idx_" + key;
        final String createPropertiesColIndexSql = STR_BUILDER("CREATE INDEX ").append(indexName).
            append(" ON ").append(TBL_PROPERTIES(groupId, " ")).
            append(" (").
            append(key).append(dbType.equalsIgnoreCase("text") ? " (128)" : "").
            append(" )").toString();

        for (int sleepTimeoutMillis = 1; sleepTimeoutMillis <= MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS; sleepTimeoutMillis <<= 1) {
            try (final ConnectionWrapper conWrapper = getWriteConnectionWrapper();
                 final PreparedStatement fpCreateIndexStmt = conWrapper.prepareStatement(createPropertiesColIndexSql)) {

                usedCon = conWrapper.getConnection();

                fpCreateIndexStmt.executeUpdate();

                usedCon.commit();
                break;
            } catch (Exception e) {
                final String cause = new StringBuilder(STR_BUILDER_LARGE_CAPACITY).
                    append("Exception occured when trying to create new index for GroupId: ").append(groupId).
                    append(" (").append(Throwables.getRootCause(e).getMessage()).append(')').toString();

                FileItemUtils.handleException(usedCon, e, sleepTimeoutMillis, cause);
            }
        }
    }

    /**
     * @param groupId
     * @return
     */
    private @NonNull Set<String> implGetKeySet(@NonNull final String groupId, final boolean customKeysOnly) {
        final GroupConfiguration groupConfiguration = m_groupMap.get(groupId);

        return (null != groupConfiguration) ?
            (customKeysOnly ? groupConfiguration.getCustomKeys() : groupConfiguration.getKeys()) :
                new LinkedHashSet<>();
    }

    /**
     * implIsDefaultKey
     *
     * @param key
     * @return
     */
    final private static boolean implIsDefaultKey(@NonNull final Object key) {
        return ArrayUtils.contains(DEFAULT_KEYS, key);
    }

    /**
     * @param groupId
     * @param tableAlias
     * @return
     */
    final private static @NonNull StringBuilder TBL_GROUPADMIN(@Nullable String tableAlias) {
        final StringBuilder groupAdminTableNameBuilder = new StringBuilder(STR_BUILDER_SMALL_CAPACITY).append("GroupAdmin");

        return (null != tableAlias) ?
            groupAdminTableNameBuilder.append(' ').append(tableAlias) :
                groupAdminTableNameBuilder;
    }

    /**
     * @param groupId
     * @param tableAlias
     * @return
     */
    final private static @NonNull StringBuilder TBL_CONTENT(@NonNull final String groupId, @Nullable String tableAlias) {
        final StringBuilder contentTableNameBuilder = new StringBuilder(STR_BUILDER_SMALL_CAPACITY).append("Content").append('_').append(groupId);

        return (null != tableAlias) ?
                contentTableNameBuilder.append(' ').append(tableAlias) :
                    contentTableNameBuilder;
    }

    /**
     * @param groupId
     * @param tableAlias
     * @return
     */
    final private static @NonNull StringBuilder TBL_PROPERTIES(@NonNull final String groupId, @Nullable String tableAlias) {
        final StringBuilder propertiesTableNameBuilder = new StringBuilder(STR_BUILDER_SMALL_CAPACITY).append("Properties").append('_').append(groupId);

        return (null != tableAlias) ?
            propertiesTableNameBuilder.append(' ').append(tableAlias) :
                propertiesTableNameBuilder;
    }

    /**
     * CFG_VALUE_1 (with config prefix FILEITEM_CONFIG_PREFIX_1)
     *
     * @param key
     * @return
     */
    final private static @NonNull String CFG_KEY_1(final String unprefixedKey) {
        return STR_BUILDER(FILEITEM_CONFIG_PREFIX_1).append(unprefixedKey).toString();
    }

    /**
     * CFG_VALUE_2 (with config prefix FILEITEM_CONFIG_PREFIX_2)
     *
     * @param key
     * @return
     */
    final private static @NonNull String CFG_KEY_2(final String unprefixedKey) {
        return STR_BUILDER(FILEITEM_CONFIG_PREFIX_2).append(unprefixedKey).toString();
    }

    /**
     * @param initStr
     * @return
     */
    final private static @NonNull StringBuilder STR_BUILDER(@Nullable final String initStr) {
        final StringBuilder ret = new StringBuilder(STR_BUILDER_DEFAULT_CAPACITY);

        return (null != initStr) ?
                ret.append(initStr) :
                    ret;
    }

    // - Static members --------------------------------------------------------

    final private static int MAX_TRANSACTION_REPEAT_TIMEOUT_MILLIS = 1024;

    final private static int STR_BUILDER_SMALL_CAPACITY = 64;
    final private static int STR_BUILDER_DEFAULT_CAPACITY = 128;
    final private static int STR_BUILDER_LARGE_CAPACITY = 256;

    final private static String FILEITEM_CONFIG_PREFIX_1 = "com.openexchange.fileItem.";
    final private static String FILEITEM_CONFIG_PREFIX_2 = "com.openexchange.fileitem.";

    // - Members ---------------------------------------------------------------

    final private ReentrantLock m_writeConnectionLock = new ReentrantLock(true);

    final private Condition m_writeConnectionAvailableCondition = m_writeConnectionLock.newCondition();

    final private Map<String, GroupConfiguration> m_groupMap = Collections.synchronizedMap(new HashMap<>());

    final private ConfigurationService m_configService;

    private HikariPool m_readConnectionPool = null;

    private HikariPool m_writeConnectionPool = null;

    private IFileItemService.DatabaseType m_databaseType = IFileItemService.DatabaseType.STANDARD_SQL;

    private Connection m_exclusiveWriteConnection = null;
}
