/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.messaging.generic.internal;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import com.openexchange.context.ContextService;
import com.openexchange.crypto.CryptoService;
import com.openexchange.database.DatabaseService;
import com.openexchange.database.Databases;
import com.openexchange.datatypes.genericonf.storage.GenericConfigurationStorageService;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.contexts.Context;
import com.openexchange.messaging.MessagingAccount;
import com.openexchange.messaging.MessagingExceptionCodes;
import com.openexchange.messaging.MessagingService;
import com.openexchange.messaging.generic.DefaultMessagingAccount;
import com.openexchange.messaging.generic.services.MessagingGenericServiceRegistry;
import com.openexchange.messaging.registry.MessagingServiceRegistry;
import com.openexchange.secret.SecretEncryptionFactoryService;
import com.openexchange.secret.SecretEncryptionService;
import com.openexchange.secret.SecretEncryptionStrategy;
import com.openexchange.server.ServiceExceptionCode;
import com.openexchange.session.Session;
import com.openexchange.tools.session.ServerSession;
import gnu.trove.list.TIntList;
import gnu.trove.list.array.TIntArrayList;
import gnu.trove.map.TIntIntMap;
import gnu.trove.map.hash.TIntIntHashMap;

/**
 * {@link RdbMessagingAccountStorage} - The messaging account storage backed by database.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since Open-Xchange v6.16
 */
public class RdbMessagingAccountStorage implements MessagingAccountStorage, SecretEncryptionStrategy<GenericProperty> {

    /**
     * The {@link DatabaseService} class.
     */
    private static final Class<DatabaseService> CLAZZ_DB = DatabaseService.class;

    /**
     * The {@link GenericConfigurationStorageService} class.
     */
    private static final Class<GenericConfigurationStorageService> CLAZZ_GEN_CONF = GenericConfigurationStorageService.class;

    private static final RdbMessagingAccountStorage INSTANCE = new RdbMessagingAccountStorage();

    /**
     * Gets the database-backed instance of account storage manager.
     *
     * @return The database-backed instance of account storage manager
     */
    public static RdbMessagingAccountStorage getInstance() {
        return INSTANCE;
    }

    /**
     * Initializes a new {@link RdbMessagingAccountStorage}.
     */
    private RdbMessagingAccountStorage() {
        super();
    }

    private static final String SQL_SELECT = "SELECT confId, displayName FROM messagingAccount WHERE cid = ? AND user = ? AND serviceId = ? AND account = ?";

    private static final String SQL_SELECT_CONFIDS_FOR_USER = "SELECT confId, account FROM messagingAccount WHERE cid = ? AND user = ? AND serviceId = ?";

    @Override
    public MessagingAccount getAccount(final String serviceId, final int id, final Session session, final Modifier modifier) throws OXException {
        final DatabaseService databaseService = getService(CLAZZ_DB);
        /*
         * Readable connection
         */
        final int contextId = session.getContextId();
        final Connection rc = databaseService.getReadOnly(contextId);
        PreparedStatement stmt = null;
        ResultSet rs = null;
        try {
            stmt = rc.prepareStatement(SQL_SELECT);
            int pos = 1;
            stmt.setInt(pos++, contextId);
            stmt.setInt(pos++, session.getUserId());
            stmt.setString(pos++, serviceId);
            stmt.setInt(pos, id);
            rs = stmt.executeQuery();
            if (!rs.next()) {
                throw MessagingExceptionCodes.ACCOUNT_NOT_FOUND.create(Integer.valueOf(id), serviceId, Integer.valueOf(session.getUserId()), Integer.valueOf(contextId));
            }
            final MessagingServiceRegistry registry = getService(MessagingServiceRegistry.class);
            if (null == registry) {
                throw MessagingExceptionCodes.ACCOUNT_NOT_FOUND.create(Integer.valueOf(id), serviceId, Integer.valueOf(session.getUserId()), Integer.valueOf(contextId));
            }
            final MessagingService messagingService = registry.getMessagingService(serviceId, session.getUserId(), session.getContextId());
            final DefaultMessagingAccount account = new DefaultMessagingAccount();
            account.setId(id);
            account.setMessagingService(messagingService);
            account.setDisplayName(rs.getString(2));
            {
                final GenericConfigurationStorageService genericConfStorageService = getService(CLAZZ_GEN_CONF);
                final Map<String, Object> configuration = new HashMap<String, Object>();
                final int confId = rs.getInt(1);
                genericConfStorageService.fill(rc, getContext(session), confId, configuration);
                /*
                 * Decrypt password fields for clear-text representation in account's configuration
                 */
                final Set<String> secretPropNames = messagingService.getSecretProperties();
                if (!secretPropNames.isEmpty()) {
                    for (final String passwordElementName : secretPropNames) {
                        final String toDecrypt = (String) configuration.get(passwordElementName);
                        if (null != toDecrypt) {
                            try {
                                final String decrypted = decrypt(toDecrypt, serviceId, id, session, confId, passwordElementName);
                                configuration.put(passwordElementName, decrypted);
                            } catch (OXException x) {
                                // Must not be fatal
                                configuration.put(passwordElementName, "");
                                // Supply a (probably false) password anyway.
                            }
                        }
                    }
                }
                account.setConfiguration(configuration);
            }
            return modifier.modifyOutgoing(account);
        } catch (SQLException e) {
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            Databases.closeSQLStuff(rs, stmt);
            databaseService.backReadOnly(contextId, rc);
        }
    }

    @Override
    public List<MessagingAccount> getAccounts(final String serviceId, final Session session, final Modifier modifier) throws OXException {
        Context context = getContext(session);
        return getAccounts(serviceId, session.getUserId(), context, modifier);
    }

    /**
     * Gets all accounts belonging to specified user
     *
     * @param serviceId The service identifier
     * @param userId The user identifier
     * @param contextId The context identifier
     * @param modifier The optional modifier
     * @return The accounts
     * @throws OXException If accounts cannot be returned
     */
    public List<MessagingAccount> getAccounts(String serviceId, int userId, int contextId, Modifier modifier) throws OXException {
        Context context = getContext(contextId);
        return getAccounts(serviceId, userId, context, modifier);
    }

    private static final String SQL_SELECT_ACCOUNTS = "SELECT account, confId, displayName FROM messagingAccount WHERE cid = ? AND user = ? AND serviceId = ?";

    private List<MessagingAccount> getAccounts(String serviceId, int userId, Context context, Modifier modifier) throws OXException {
        final DatabaseService databaseService = getService(CLAZZ_DB);
        /*
         * Readable connection
         */
        int contextId = context.getContextId();
        final Connection rc = databaseService.getReadOnly(contextId);
        List<MessagingAccount> accounts;
        PreparedStatement stmt = null;
        ResultSet rs = null;
        try {
            stmt = rc.prepareStatement(SQL_SELECT_ACCOUNTS);
            int pos = 1;
            stmt.setInt(pos++, contextId);
            stmt.setInt(pos++, userId);
            stmt.setString(pos, serviceId);
            rs = stmt.executeQuery();
            if (rs.next()) {
                accounts = new ArrayList<MessagingAccount>(4);
                final GenericConfigurationStorageService genericConfStorageService = getService(CLAZZ_GEN_CONF);
                final MessagingService messagingService;
                {
                    final MessagingServiceRegistry registry = getService(MessagingServiceRegistry.class);
                    messagingService = registry.getMessagingService(serviceId, userId, contextId);
                }
                do {
                    final DefaultMessagingAccount account = new DefaultMessagingAccount();
                    account.setDisplayName(rs.getString(3));
                    final Map<String, Object> configuration = new HashMap<String, Object>();
                    genericConfStorageService.fill(rc, context, rs.getInt(2), configuration);
                    account.setConfiguration(configuration);
                    account.setId(rs.getInt(1));
                    account.setMessagingService(messagingService);
                    accounts.add(modifier.modifyOutgoing(account));
                } while (rs.next());
            } else {
                accounts = Collections.emptyList();
            }
            return accounts;
        } catch (SQLException e) {
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            Databases.closeSQLStuff(rs, stmt);
            databaseService.backReadOnly(contextId, rc);
        }
    }

    /**
     * Gets all accounts belonging to specified user.
     *
     * @param userId The user identifier
     * @param contextId The context identifier
     * @param modifier The optional modifier
     * @return All accounts for the user
     * @throws OXException If accounts cannot be returned
     */
    public List<MessagingAccount> getAccounts(int userId, int contextId, Modifier modifier) throws OXException {
        Context context = getContext(contextId);
        return getAccounts(userId, context, modifier);
    }

    private static final String SQL_SELECT_USER_ACCOUNTS = "SELECT account, confId, displayName, serviceId FROM messagingAccount WHERE cid = ? AND user = ?";

    private List<MessagingAccount> getAccounts(int userId, Context context, Modifier modifier) throws OXException {
        final DatabaseService databaseService = getService(CLAZZ_DB);
        /*
         * Readable connection
         */
        int contextId = context.getContextId();
        final Connection rc = databaseService.getReadOnly(contextId);
        List<MessagingAccount> accounts;
        PreparedStatement stmt = null;
        ResultSet rs = null;
        try {
            stmt = rc.prepareStatement(SQL_SELECT_USER_ACCOUNTS);
            int pos = 1;
            stmt.setInt(pos++, contextId);
            stmt.setInt(pos++, userId);
            rs = stmt.executeQuery();
            if (rs.next()) {
                accounts = new LinkedList<MessagingAccount>();
                final GenericConfigurationStorageService genericConfStorageService = getService(CLAZZ_GEN_CONF);
                do {
                    String serviceId = rs.getString(4);
                    final MessagingService messagingService;
                    {
                        final MessagingServiceRegistry registry = getService(MessagingServiceRegistry.class);
                        messagingService = registry.getMessagingService(serviceId, userId, contextId);
                    }
                    final DefaultMessagingAccount account = new DefaultMessagingAccount();
                    account.setDisplayName(rs.getString(3));
                    final Map<String, Object> configuration = new HashMap<String, Object>();
                    genericConfStorageService.fill(rc, context, rs.getInt(2), configuration);
                    account.setConfiguration(configuration);
                    account.setId(rs.getInt(1));
                    account.setMessagingService(messagingService);
                    accounts.add(null == modifier ? account : modifier.modifyOutgoing(account));
                } while (rs.next());
            } else {
                accounts = Collections.emptyList();
            }
            return accounts;
        } catch (SQLException e) {
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            Databases.closeSQLStuff(rs, stmt);
            databaseService.backReadOnly(contextId, rc);
        }
    }

    /**
     * Gets the identifiers of user-associated accounts of a certain service.
     *
     * @param serviceId The service identifier
     * @param session The session
     * @return The identifiers of user-associated accounts of a certain service
     * @throws OXException If identifiers cannot be returned
     */
    public TIntArrayList getAccountIDs(final String serviceId, final Session session) throws OXException {
        final DatabaseService databaseService = getService(CLAZZ_DB);
        if (null == databaseService) {
            throw ServiceExceptionCode.serviceUnavailable(CLAZZ_DB);
        }
        /*
         * Readable connection
         */
        final int contextId = session.getContextId();
        final Connection rc = databaseService.getReadOnly(contextId);
        TIntArrayList accounts;
        PreparedStatement stmt = null;
        ResultSet rs = null;
        try {
            stmt = rc.prepareStatement("SELECT account FROM messagingAccount WHERE cid = ? AND user = ? AND serviceId = ?");
            int pos = 1;
            stmt.setInt(pos++, contextId);
            stmt.setInt(pos++, session.getUserId());
            stmt.setString(pos, serviceId);
            rs = stmt.executeQuery();
            if (rs.next()) {
                accounts = new TIntArrayList(4);
                do {
                    accounts.add(rs.getInt(1));
                } while (rs.next());
            } else {
                accounts = new TIntArrayList(0);
            }
            return accounts;
        } catch (SQLException e) {
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            Databases.closeSQLStuff(rs, stmt);
            databaseService.backReadOnly(contextId, rc);
        }
    }

    private static final String SQL_INSERT = "INSERT INTO messagingAccount (cid, user, account, confId, serviceId, displayName) VALUES (?, ?, ?, ?, ?, ?)";

    @Override
    public int addAccount(final String serviceId, final MessagingAccount account, final Session session, final Modifier modifier) throws OXException {
        final DatabaseService databaseService = getService(CLAZZ_DB);
        /*
         * Writable connection
         */
        final int contextId = session.getContextId();
        final Connection wc;
        try {
            wc = databaseService.getWritable(contextId);
            wc.setAutoCommit(false); // BEGIN
        } catch (SQLException e) {
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        }
        PreparedStatement stmt = null;
        try {
            /*
             * Save account configuration using generic conf
             */
            modifier.modifyIncoming(account);
            final int genericConfId;
            {
                final GenericConfigurationStorageService genericConfStorageService = getService(CLAZZ_GEN_CONF);
                final Map<String, Object> configuration = new HashMap<String, Object>(account.getConfiguration());
                /*
                 * Encrypt password fields to not having clear-text representation in database
                 */
                final MessagingService messagingService = getService(MessagingServiceRegistry.class).getMessagingService(serviceId, session.getUserId(), session.getContextId());
                final Set<String> secretPropNames = messagingService.getSecretProperties();
                if (!secretPropNames.isEmpty()) {
                    for (final String passwordElementName : secretPropNames) {
                        final String toCrypt = (String) configuration.get(passwordElementName);
                        if (null != toCrypt) {
                            final String encrypted = encrypt(toCrypt, session);
                            configuration.put(passwordElementName, encrypted);
                        }
                    }
                }
                genericConfId = genericConfStorageService.save(wc, getContext(session), configuration);
            }
            /*
             * Insert account data
             */
            stmt = wc.prepareStatement(SQL_INSERT);
            int pos = 1;
            stmt.setInt(pos++, contextId);
            stmt.setInt(pos++, session.getUserId());
            stmt.setInt(pos++, genericConfId);
            stmt.setInt(pos++, genericConfId);
            stmt.setString(pos++, serviceId);
            stmt.setString(pos, account.getDisplayName());
            stmt.executeUpdate();
            wc.commit(); // COMMIT
            return genericConfId;
        } catch (OXException e) {
            Databases.rollback(wc); // ROLL-BACK
            throw e;
        } catch (SQLException e) {
            Databases.rollback(wc); // ROLL-BACK
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } catch (RuntimeException e) {
            Databases.rollback(wc); // ROLL-BACK
            throw MessagingExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            Databases.closeSQLStuff(stmt);
            Databases.autocommit(wc);
            databaseService.backWritable(contextId, wc);
        }
    }

    @Override
    public void update(final String recrypted, final GenericProperty prop) throws OXException {
        final HashMap<String, Object> update = new HashMap<String, Object>();
        update.put(prop.propertyName, recrypted);
        final GenericConfigurationStorageService genericConfStorageService = getService(CLAZZ_GEN_CONF);
        final Session session = prop.session;
        genericConfStorageService.update(getContext(session), prop.confId, update);
        // Invalidate account
        try {
            CachingMessagingAccountStorage.getInstance().invalidate(prop.serviceId, prop.id, session.getUserId(), session.getContextId());
        } catch (Exception e) {
            // Ignore
        }
    }

    private String encrypt(final String toCrypt, final Session session) throws OXException {
        final SecretEncryptionService<GenericProperty> encryptionService = getService(SecretEncryptionFactoryService.class).createService(this);
        return encryptionService.encrypt(session, toCrypt);
    }

    private String decrypt(final String toDecrypt, final String serviceId, final int id, final Session session, final int confId, final String propertyName) throws OXException {
        final SecretEncryptionService<GenericProperty> encryptionService = getService(SecretEncryptionFactoryService.class).createService(this);
        return encryptionService.decrypt(session, toDecrypt, new GenericProperty(confId, propertyName, serviceId, id, session));
    }

    @Override
    public void deleteAccount(final String serviceId, final MessagingAccount account, final Session session, final Modifier modifier) throws OXException {
        deleteAccounts(serviceId, new MessagingAccount[] { account }, new int[] { 0 }, session, modifier);
    }

    /**
     * Deletes specified accounts.
     *
     * @param serviceId The service identifier
     * @param accounts The accounts to delete
     * @param genericConfIds The associated identifiers of generic configuration
     * @param session The session
     * @param optModifier The modifier
     * @throws OXException If delete operation fails
     */
    public void deleteAccounts(final String serviceId, final MessagingAccount[] accounts, final int[] genericConfIds, final Session session, final Modifier optModifier) throws OXException {
        Context context = getContext(session);
        deleteAccounts(serviceId, accounts, genericConfIds, session.getUserId(), context, optModifier);
    }

    /**
     * Deletes specified accounts.
     *
     * @param serviceId The service identifier
     * @param accounts The accounts to delete
     * @param genericConfIds The associated identifiers of generic configuration
     * @param userId The user ID
     * @param contextId The context ID
     * @param optModifier The modifier
     * @param con The optional connection to use
     * @throws OXException If delete operation fails
     */
    public void deleteAccounts(final String serviceId, final MessagingAccount[] accounts, final int[] genericConfIds, int userId, int contextId, final Modifier optModifier, Connection con) throws OXException {
        Context context = getContext(contextId);
        if (null == con) {
            deleteAccounts(serviceId, accounts, genericConfIds, userId, context, optModifier);
        } else {
            deleteAccounts(serviceId, accounts, genericConfIds, userId, context, optModifier, con);
        }
    }

    private void deleteAccounts(String serviceId, MessagingAccount[] accounts, int[] genericConfIds, int userId, Context context, Modifier optModifier) throws OXException {
        DatabaseService databaseService = getService(CLAZZ_DB);
        int contextId = context.getContextId();
        Connection con = databaseService.getWritable(contextId);
        int rollback = 0;
        try {
            Databases.startTransaction(con);
            rollback = 1;

            deleteAccounts(serviceId, accounts, genericConfIds, userId, context, optModifier, con);

            con.commit();
            rollback = 2;
        } catch (SQLException e) {
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            if (rollback > 0) {
                if (rollback == 1) {
                    Databases.rollback(con);
                }
                Databases.autocommit(con);
            }
            databaseService.backWritable(contextId, con);
        }
    }

    private void deleteAccounts(String serviceId, MessagingAccount[] accounts, int[] genericConfIds, int userId, Context context, Modifier optModifier, Connection con) throws OXException {
        try {
            int contextId = context.getContextId();
            for (int i = 0; i < accounts.length; i++) {
                final MessagingAccount account = accounts[i];
                if (null != optModifier) {
                    optModifier.modifyIncoming(account);
                }
                final int accountId = account.getId();
                /*
                 * Delete account configuration using generic conf
                 */
                {
                    final GenericConfigurationStorageService genericConfStorageService = getService(CLAZZ_GEN_CONF);
                    int genericConfId = genericConfIds[i];
                    if (genericConfId <= 0) {
                        genericConfId = getGenericConfId(contextId, userId, serviceId, accountId, con);
                    }
                    genericConfStorageService.delete(con, context, genericConfId);
                }
                /*
                 * Delete account data
                 */
                PreparedStatement stmt = null;
                try {
                    stmt = con.prepareStatement("DELETE FROM messagingAccount WHERE cid = ? AND user = ? AND serviceId = ? AND account = ?");
                    int pos = 1;
                    stmt.setInt(pos++, contextId);
                    stmt.setInt(pos++, userId);
                    stmt.setString(pos++, serviceId);
                    stmt.setInt(pos, accountId);
                    stmt.executeUpdate();
                } catch (SQLException e) {
                    throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
                } finally {
                    Databases.closeSQLStuff(stmt);
                }
            }
        } catch (SQLException e) {
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } catch (RuntimeException e) {
            throw MessagingExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    private static final String SQL_UPDATE = "UPDATE messagingAccount SET displayName = ? WHERE cid = ? AND user = ? AND serviceId = ? AND account = ?";

    @Override
    public void updateAccount(final String serviceId, final MessagingAccount account, final Session session, final Modifier modifier) throws OXException {
        final DatabaseService databaseService = getService(CLAZZ_DB);
        /*
         * Writable connection
         */
        final int contextId = session.getContextId();
        final Connection wc;
        try {
            wc = databaseService.getWritable(contextId);
            wc.setAutoCommit(false); // BEGIN
        } catch (SQLException e) {
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        }
        PreparedStatement stmt = null;
        try {
            modifier.modifyIncoming(account);
            /*
             * Update account configuration using generic conf
             */
            {
                Map<String, Object> configuration = account.getConfiguration();
                if (null != configuration) {
                    configuration = new HashMap<String, Object>(configuration);
                    final GenericConfigurationStorageService genericConfStorageService = getService(CLAZZ_GEN_CONF);
                    final int genericConfId = getGenericConfId(contextId, session.getUserId(), serviceId, account.getId(), wc);
                    /*
                     * Encrypt password fields to not having clear-text representation in database
                     */
                    final MessagingService messagingService = getService(MessagingServiceRegistry.class).getMessagingService(serviceId, session.getUserId(), session.getContextId());
                    final Set<String> secretPropNames = messagingService.getSecretProperties();
                    if (!secretPropNames.isEmpty()) {
                        for (final String passwordElementName : secretPropNames) {
                            final String toCrypt = (String) configuration.get(passwordElementName);
                            if (null != toCrypt) {
                                final String encrypted = encrypt(toCrypt, session);
                                configuration.put(passwordElementName, encrypted);
                            }
                        }
                    }
                    genericConfStorageService.update(wc, getContext(session), genericConfId, configuration);
                }
            }
            /*
             * Update account data
             */
            final String displayName = account.getDisplayName();
            if (null != displayName) {
                stmt = wc.prepareStatement(SQL_UPDATE);
                int pos = 1;
                stmt.setString(pos++, displayName);
                stmt.setInt(pos++, contextId);
                stmt.setInt(pos++, session.getUserId());
                stmt.setString(pos++, serviceId);
                stmt.setInt(pos, account.getId());
                stmt.executeUpdate();
            }
            wc.commit(); // COMMIT
        } catch (OXException e) {
            Databases.rollback(wc); // ROLL-BACK
            throw e;
        } catch (SQLException e) {
            Databases.rollback(wc); // ROLL-BACK
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } catch (RuntimeException e) {
            Databases.rollback(wc); // ROLL-BACK
            throw MessagingExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            Databases.closeSQLStuff(stmt);
            Databases.autocommit(wc);
            databaseService.backWritable(contextId, wc);
        }
    }

    private static <S> S getService(final Class<? extends S> clazz) throws OXException {
        try {
            return MessagingGenericServiceRegistry.getService(clazz);
        } catch (RuntimeException e) {
            throw new OXException(e);
        }
    }

    private static Context getContext(final Session session) throws OXException {
        if (session instanceof ServerSession) {
            return ((ServerSession) session).getContext();
        }
        return getContext(session.getContextId());
    }

    private static Context getContext(int contextId) throws OXException {
        return getService(ContextService.class).getContext(contextId);
    }

    private static int getGenericConfId(final int contextId, final int userId, final String serviceId, final int accountId, final Connection con) throws OXException, SQLException {
        PreparedStatement stmt = null;
        ResultSet rs = null;
        try {
            stmt = con.prepareStatement(SQL_SELECT);
            int pos = 1;
            stmt.setInt(pos++, contextId);
            stmt.setInt(pos++, userId);
            stmt.setString(pos++, serviceId);
            stmt.setInt(pos, accountId);
            rs = stmt.executeQuery();
            if (!rs.next()) {
                throw MessagingExceptionCodes.ACCOUNT_NOT_FOUND.create(Integer.valueOf(accountId), serviceId, Integer.valueOf(userId), Integer.valueOf(contextId));
            }
            return rs.getInt(1);
        } finally {
            Databases.closeSQLStuff(rs, stmt);
        }
    }

    public String checkSecretCanDecryptStrings(final MessagingService parentService, final Session session, final String secret) throws OXException {
        final Set<String> secretProperties = parentService.getSecretProperties();
        if (secretProperties.isEmpty()) {
            return null;
        }
        final TIntList confIds = getConfIdsForUser(session.getContextId(), session.getUserId(), parentService.getId());
        final GenericConfigurationStorageService genericConfStorageService = getService(CLAZZ_GEN_CONF);
        final CryptoService cryptoService = getService(CryptoService.class);

        final Context ctx = getContext(session);
        final HashMap<String, Object> content = new HashMap<String, Object>();
        int confId = -1;
        try {
            for (int i = 0, size = confIds.size(); i < size; i++) {
                confId = confIds.get(i);
                content.clear();
                genericConfStorageService.fill(ctx, confId, content);

                for (final String field : secretProperties) {
                    final String encrypted = (String) content.get(field);
                    if (encrypted != null) {
                        cryptoService.decrypt(encrypted, secret);
                    }
                }
            }
        } catch (OXException e) {
            throw e;
        }
        return null;
    }

    private TIntIntMap getConfIdToAccountMappingForUser(final int contextId, final int userId, final String serviceId) throws OXException {
        final DatabaseService databaseService = getService(CLAZZ_DB);
        PreparedStatement stmt = null;
        ResultSet rs = null;
        Connection con = null;
        try {
            con = databaseService.getReadOnly(contextId);
            stmt = con.prepareStatement(SQL_SELECT_CONFIDS_FOR_USER);
            int pos = 1;
            stmt.setInt(pos++, contextId);
            stmt.setInt(pos++, userId);
            stmt.setString(pos++, serviceId);
            // Query
            rs = stmt.executeQuery();
            final TIntIntMap ret = new TIntIntHashMap(16);
            while (rs.next()) {
                ret.put(rs.getInt(1), rs.getInt(2));
            }
            return ret;
        } catch (SQLException e) {
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            Databases.closeSQLStuff(rs, stmt);
            if (con != null) {
                databaseService.backReadOnly(contextId, con);
            }
        }
    }

    private TIntList getConfIdsForUser(final int contextId, final int userId, final String serviceId) throws OXException {
        final DatabaseService databaseService = getService(CLAZZ_DB);
        PreparedStatement stmt = null;
        ResultSet rs = null;
        Connection con = null;
        try {
            con = databaseService.getReadOnly(contextId);
            stmt = con.prepareStatement(SQL_SELECT_CONFIDS_FOR_USER);
            int pos = 1;
            stmt.setInt(pos++, contextId);
            stmt.setInt(pos++, userId);
            stmt.setString(pos++, serviceId);
            // Query
            rs = stmt.executeQuery();
            final TIntList ret = new TIntArrayList(16);
            while (rs.next()) {
                ret.add(rs.getInt(1));
            }
            return ret;
        } catch (SQLException e) {
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            Databases.closeSQLStuff(rs, stmt);
            if (con != null) {
                databaseService.backReadOnly(contextId, con);
            }
        }
    }

    public void migrateToNewSecret(final MessagingService parentService, final String oldSecret, final String newSecret, final Session session) throws OXException {
        final Set<String> secretProperties = parentService.getSecretProperties();
        if (secretProperties.isEmpty()) {
            return;
        }
        final TIntList confIds = getConfIdsForUser(session.getContextId(), session.getUserId(), parentService.getId());
        final GenericConfigurationStorageService genericConfStorageService = getService(CLAZZ_GEN_CONF);
        final CryptoService cryptoService = getService(CryptoService.class);

        final Context ctx = getContext(session);
        final Map<String, Object> content = new HashMap<String, Object>();
        final Map<String, Object> update = new HashMap<String, Object>();
        for (int i = 0, size = confIds.size(); i < size; i++) {
            final int confId = confIds.get(i);
            content.clear();
            genericConfStorageService.fill(ctx, confId, content);
            update.clear();
            for (final String field : secretProperties) {
                final String encrypted = (String) content.get(field);
                if (!com.openexchange.java.Strings.isEmpty(encrypted)) {
                    try {
                        // Try using the new secret. Maybe this account doesn't need the migration
                        cryptoService.decrypt(encrypted, newSecret);
                    } catch (OXException x) {
                        // Needs migration
                        final String transcripted = cryptoService.encrypt(cryptoService.decrypt(encrypted, oldSecret), newSecret);
                        update.put(field, transcripted);
                    }
                }
            }
            if (!update.isEmpty()) {
                genericConfStorageService.update(ctx, confId, update);
            }
        }
    }

    private static final String ACCOUNT_EXISTS = "SELECT 1 FROM messagingAccount WHERE cid = ? AND user = ? LIMIT 1";

    public boolean hasAccount(final MessagingService parentService, final Session session) throws OXException {
        final Set<String> secretProperties = parentService.getSecretProperties();
        if (secretProperties.isEmpty()) {
            return false;
        }
        final DatabaseService databaseService = getService(CLAZZ_DB);
        PreparedStatement stmt = null;
        ResultSet rs = null;
        Connection con = null;
        try {
            con = databaseService.getReadOnly(session.getContextId());
            stmt = con.prepareStatement(ACCOUNT_EXISTS);
            int pos = 1;
            stmt.setInt(pos++, session.getContextId());
            stmt.setInt(pos++, session.getUserId());
            // Query
            rs = stmt.executeQuery();
            return rs.next();
        } catch (SQLException e) {
            throw MessagingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            Databases.closeSQLStuff(rs, stmt);
            if (con != null) {
                databaseService.backReadOnly(session.getContextId(), con);
            }
        }
    }

    public void cleanUp(final MessagingService parentService, final String secret, final Session session) throws OXException {
        final Set<String> secretProperties = parentService.getSecretProperties();
        if (secretProperties.isEmpty()) {
            return;
        }
        final String serviceId = parentService.getId();
        final TIntIntMap confId2AccountMap = getConfIdToAccountMappingForUser(session.getContextId(), session.getUserId(), serviceId);
        final GenericConfigurationStorageService genericConfStorageService = getService(CLAZZ_GEN_CONF);
        final CryptoService cryptoService = getService(CryptoService.class);
        // Proceed...
        final Context ctx = getContext(session);
        final Map<String, Object> content = new HashMap<String, Object>();
        final Map<String, Object> update = new HashMap<String, Object>();
        for (final int confId : confId2AccountMap.keys()) {
            content.clear();
            genericConfStorageService.fill(ctx, confId, content);
            update.clear();
            for (final Map.Entry<String, Object> entry : content.entrySet()) {
                final String field = entry.getKey();
                if (secretProperties.contains(field)) {
                    final String encrypted = entry.getValue().toString();
                    if (!com.openexchange.java.Strings.isEmpty(encrypted)) {
                        try {
                            // Check it
                            cryptoService.decrypt(encrypted, secret);
                        } catch (OXException x) {
                            // Discard
                            update.put(field, "");
                        }
                    }
                }
            }
            if (!update.isEmpty()) {
                genericConfStorageService.update(ctx, confId, update);
            }
        }
    }

    public void removeUnrecoverableItems(MessagingService parentService, String secret, Session session) throws OXException {
        final Set<String> secretProperties = parentService.getSecretProperties();
        if (secretProperties.isEmpty()) {
            return;
        }
        final String serviceId = parentService.getId();
        final TIntIntMap confId2AccountMap = getConfIdToAccountMappingForUser(session.getContextId(), session.getUserId(), serviceId);
        final GenericConfigurationStorageService genericConfStorageService = getService(CLAZZ_GEN_CONF);
        final CryptoService cryptoService = getService(CryptoService.class);
        // Proceed...
        final Context ctx = getContext(session);
        final Map<String, Object> content = new HashMap<String, Object>();

        List<MessagingAccount> accountsToDelete = new ArrayList<MessagingAccount>(confId2AccountMap.size());
        TIntArrayList confIdsToDelete = new TIntArrayList(confId2AccountMap.size());

        for (final int confId : confId2AccountMap.keys()) {
            content.clear();
            genericConfStorageService.fill(ctx, confId, content);
            for (final Map.Entry<String, Object> entry : content.entrySet()) {
                final String field = entry.getKey();
                if (secretProperties.contains(field)) {
                    final String encrypted = entry.getValue().toString();
                    if (!com.openexchange.java.Strings.isEmpty(encrypted)) {
                        try {
                            // Check it
                            cryptoService.decrypt(encrypted, secret);
                        } catch (OXException x) {
                            // Discard
                            if (!confIdsToDelete.contains(confId)) {
                                confIdsToDelete.add(confId);
                                DefaultMessagingAccount account = new DefaultMessagingAccount();
                                account.setId(confId2AccountMap.get(confId));
                                account.setServiceId(serviceId);
                                account.setMessagingService(parentService);
                                accountsToDelete.add(account);
                            }
                        }
                    }
                }
            }
        }

        deleteAccounts(serviceId, accountsToDelete.toArray(new MessagingAccount[accountsToDelete.size()]), confIdsToDelete.toArray(), session, null);
    }
}
