/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 Open-Xchange GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.ant.tasks;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.types.Path;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

/**
 * Parses Eclipse Launcher files for workspace bundles, outputs that to the named property
 *
 * @author <a href="mailto:felix.marx@open-xchange.com">Felix Marx</a>
 */
public class ParseLauncher extends Task {

    private Path launcher;
    private String property;
    private String delimiter;
    private HashSet<String> keys;

    public ParseLauncher() {
        super();
    }

    public final void setProperty(final String property) {
        this.property = property;
    }

    public final Path createLauncher() {
        launcher = new Path(getProject());
        return launcher;
    }

    public void setDelimiter(final String delimiter){
        if (delimiter.length() != 0){
            this.delimiter = delimiter;
        }
    }

    public void setKeys(final String keys){
        if (keys.length() != 0){
            this.keys = new HashSet<String>(Arrays.asList(keys.split(",", 0)));
        }
    }

    @Override
    public final void execute() throws BuildException {
        // Parse the PSF files.
        final String[] launcherFiles = launcher.list();
        if (0 == launcher.size()) {
            throw new BuildException("No launcher configuration found in \"" + launcher + "\".");
        }
//        if (1 != launcher.size()){
//            throw new BuildException("Should be called for only one launcher \"" + launcher + "\".");
//        }
        if (null == keys){
            throw new BuildException("At least one key must be definded");
        }

        if (delimiter == null) {
            log("No delimiter is set, will use \",\"", Project.MSG_ERR);
            delimiter = ",";
        }

        StringBuilder sb = new StringBuilder();

        for (final String launcherFile : launcherFiles) {
            log("Parsing " + launcherFile, Project.MSG_INFO);
            try {
                Document doc = new SAXBuilder().build(new File(launcherFile));

                Element launchConfiguration = doc.getRootElement();

                for (Object e : launchConfiguration.getChildren("stringAttribute")){
                    Element tmp = (Element) e;
                    for (String s : keys){
                        if (tmp.getAttribute("key").getValue().equalsIgnoreCase(s)){
                            String re1="((?:[a-z][a-z\\.\\d\\-]+)\\.(?:[a-z][a-z\\-]+))(?![\\w\\.])";   // Fully Qualified Domain Name 1
                            String re2="(@)";   // Any Single Character 1

                            Pattern p = Pattern.compile(re1+re2,Pattern.CASE_INSENSITIVE | Pattern.DOTALL);
                            Matcher m = p.matcher(tmp.getAttributeValue("value"));
                            while (m.find())
                            {
                                String fqdn1=m.group(1);
                                if (sb.length() != 0) {
                                    sb.append(delimiter);
                                }
                                sb.append(fqdn1.toString());
                            }
                        }
                    }
                }
            } catch (JDOMException jE) {
                log("JDOMException " + launcherFile + "\n" + jE, Project.MSG_ERR);
            } catch (IOException iOE) {
                log("IOException " + launcherFile + "\n" + iOE, Project.MSG_ERR);
            }
            log(property + "=" + sb.toString(), Project.MSG_INFO);
            getProject().setInheritedProperty(property , sb.toString());

        }

        // Log the project list and output the property for it.
//        final String projectNamesList = Tools.join(projectList, ",");
//        log(name + "=" + projectNamesList, Project.MSG_INFO);
//        getProject().setInheritedProperty(name, projectNamesList);
    }
}
