/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.pns.transport.fcm.internal;

import static com.openexchange.java.Autoboxing.I;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.ObjIntConsumer;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.util.tracker.ServiceTracker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.firebase.FirebaseApp;
import com.google.firebase.FirebaseOptions;
import com.google.firebase.messaging.AndroidConfig;
import com.google.firebase.messaging.AndroidConfig.Priority;
import com.google.firebase.messaging.AndroidNotification;
import com.google.firebase.messaging.BatchResponse;
import com.google.firebase.messaging.FirebaseMessaging;
import com.google.firebase.messaging.FirebaseMessagingException;
import com.google.firebase.messaging.MulticastMessage;
import com.google.firebase.messaging.SendResponse;
import com.openexchange.config.cascade.ComposedConfigProperty;
import com.openexchange.config.cascade.ConfigView;
import com.openexchange.config.cascade.ConfigViewFactory;
import com.openexchange.exception.OXException;
import com.openexchange.java.SortableConcurrentList;
import com.openexchange.osgi.util.RankedService;
import com.openexchange.pns.DefaultPushSubscription;
import com.openexchange.pns.EnabledKey;
import com.openexchange.pns.KnownTransport;
import com.openexchange.pns.PushExceptionCodes;
import com.openexchange.pns.PushMatch;
import com.openexchange.pns.PushMessageGenerator;
import com.openexchange.pns.PushMessageGeneratorRegistry;
import com.openexchange.pns.PushNotification;
import com.openexchange.pns.PushNotificationTransport;
import com.openexchange.pns.PushNotifications;
import com.openexchange.pns.PushSubscriptionRegistry;
import com.openexchange.pns.transport.fcm.FCMOptions;
import com.openexchange.pns.transport.fcm.FCMOptionsProvider;

/**
 * {@link FCMPushNotificationTransport}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class FCMPushNotificationTransport extends ServiceTracker<FCMOptionsProvider, FCMOptionsProvider> implements PushNotificationTransport {

    private static final Logger LOG = LoggerFactory.getLogger(FCMPushNotificationTransport.class);

    /**
     * Multicast size of registration tokens. Even though the online documentation states that are 1000 registration
     * ids allowed, the client API states that the limit is set to 500. We are going with the client API.
     * 
     * @see <a href="https://firebase.google.com/docs/cloud-messaging/js/topic-messaging">Topic Messaging</a>
     */
    private static final int MULTICAST_LIMIT = 500;
    private static final int MAX_PAYLOAD_SIZE = 4096;

    private static final String ID = KnownTransport.FCM.getTransportId();
    private static final String SCOPES = "https://www.googleapis.com/auth/firebase.messaging";

    private static final Cache<EnabledKey, Boolean> CACHE_AVAILABILITY = CacheBuilder.newBuilder().maximumSize(65536).expireAfterWrite(30, TimeUnit.MINUTES).build();

    private final ConfigViewFactory configViewFactory;
    private final PushSubscriptionRegistry pushSubscriptionRegistry;
    private final PushMessageGeneratorRegistry generatorRegistry;
    private final SortableConcurrentList<RankedService<FCMOptionsProvider>> trackedProviders;
    private ServiceRegistration<PushNotificationTransport> registration; // non-volatile, protected by synchronized blocks

    /**
     * Initialises a new {@link FCMPushNotificationTransport}.
     */
    public FCMPushNotificationTransport(PushSubscriptionRegistry subscriptionRegistry, PushMessageGeneratorRegistry generatorRegistry, ConfigViewFactory configViewFactory, BundleContext context) {
        super(context, FCMOptionsProvider.class, null);
        this.configViewFactory = configViewFactory;
        this.trackedProviders = new SortableConcurrentList<>();
        this.pushSubscriptionRegistry = subscriptionRegistry;
        this.generatorRegistry = generatorRegistry;
    }

    @Override
    public boolean isEnabled(String topic, String client, int userId, int contextId) throws OXException {
        EnabledKey key = new EnabledKey(topic, client, userId, contextId);
        Boolean result = CACHE_AVAILABILITY.getIfPresent(key);
        if (null == result) {
            result = Boolean.valueOf(doCheckEnabled(topic, client, userId, contextId));
            CACHE_AVAILABILITY.put(key, result);
        }
        return result.booleanValue();
    }

    @Override
    public void transport(PushNotification notification, Collection<PushMatch> matches) throws OXException {
        if (null == notification || null == matches) {
            return;
        }
        Map<String, List<PushMatch>> clientMatches = categorize(matches);
        for (Map.Entry<String, List<com.openexchange.pns.PushMatch>> entry : clientMatches.entrySet()) {
            transport(entry.getKey(), notification, entry.getValue());
        }
    }

    @Override
    public String getId() {
        return ID;
    }

    @Override
    public boolean servesClient(String client) throws OXException {
        try {
            return null != getHighestRankedFCMOptionsFor(client);
        } catch (OXException x) {
            return false;
        } catch (RuntimeException e) {
            throw PushExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    ////////////////////////////// SERVICE TRACKER ////////////////////////////

    @Override
    public synchronized FCMOptionsProvider addingService(ServiceReference<FCMOptionsProvider> reference) {
        int ranking = RankedService.getRanking(reference);
        FCMOptionsProvider provider = context.getService(reference);

        trackedProviders.addAndSort(new RankedService<>(provider, ranking));

        if (null == registration) {
            registration = context.registerService(PushNotificationTransport.class, this, null);
        }

        return provider;
    }

    @Override
    public synchronized void removedService(ServiceReference<FCMOptionsProvider> reference, FCMOptionsProvider provider) {
        trackedProviders.remove(new RankedService<FCMOptionsProvider>(provider, RankedService.getRanking(reference)));

        if (trackedProviders.isEmpty() && null != registration) {
            registration.unregister();
            registration = null;
        }

        context.ungetService(reference);
    }

    //////////////////////////// HELPERS //////////////////////////

    /**
     * Returns the highest ranked FCM options for the specified client
     * 
     * @param client The client for which to return the options
     * @return The options
     * @throws OXException if no options available for the specified client
     */
    private FCMOptions getHighestRankedFCMOptionsFor(String client) throws OXException {
        List<RankedService<FCMOptionsProvider>> list = trackedProviders.getSnapshot();
        for (RankedService<FCMOptionsProvider> rankedService : list) {
            FCMOptions options = rankedService.service.getOptions(client);
            if (null != options) {
                return options;
            }
        }
        throw PushExceptionCodes.UNEXPECTED_ERROR.create("No options found for client: " + client);
    }

    /**
     * Checks whether FCM is enabled for the specified client and topic
     * 
     * @param topic The topic
     * @param client The client
     * @param userId The user identifier
     * @param contextId The context identifier
     * @return <code>true</code> if enabled; <code>false</code> otherwise
     * @throws OXException if configuration cannot be retrieved
     */
    private boolean doCheckEnabled(String topic, String client, int userId, int contextId) throws OXException {
        ConfigView view = configViewFactory.getView(userId, contextId);

        String basePropertyName = "com.openexchange.pns.transport.fcm.enabled";

        ComposedConfigProperty<Boolean> property;
        property = null == topic || null == client ? null : view.property(basePropertyName + "." + client + "." + topic, boolean.class);
        if (null != property && property.isDefined()) {
            return property.get().booleanValue();
        }

        property = null == client ? null : view.property(basePropertyName + "." + client, boolean.class);
        if (null != property && property.isDefined()) {
            return property.get().booleanValue();
        }

        property = view.property(basePropertyName, boolean.class);
        if (null != property && property.isDefined()) {
            return property.get().booleanValue();
        }

        return false;
    }

    /**
     * Categorises the matches
     * 
     * @param matches The push matches to categorise
     * @return a map with all categorised pushed matches
     * @throws OXException if no push generator is found
     */
    private Map<String, List<PushMatch>> categorize(Collection<PushMatch> matches) throws OXException {
        Map<String, List<PushMatch>> clientMatches = new LinkedHashMap<>(matches.size());
        for (PushMatch match : matches) {
            String client = match.getClient();

            PushMessageGenerator generator = generatorRegistry.getGenerator(client);
            if (null == generator) {
                throw PushExceptionCodes.NO_SUCH_GENERATOR.create(client);
            }

            clientMatches.computeIfAbsent(client, c -> clientMatches.put(c, new LinkedList<>())).add(match);
        }
        return clientMatches;
    }

    /**
     * Sends the push notification to the specified devices
     * 
     * @param client The client
     * @param notification The notification to send
     * @param matches The devices to send it to
     * @throws OXException if an error is occurred
     */
    private void transport(String client, PushNotification notification, List<PushMatch> matches) throws OXException {
        if (null == notification || null == matches) {
            return;
        }

        LOG.debug("Going to send notification '{}' via transport '{}' for user {} in context {}", notification.getTopic(), ID, I(notification.getUserId()), I(notification.getContextId()));
        int size = matches.size();
        if (size <= 0) {
            return;
        }

        FirebaseMessaging sender = optClient(client);
        if (null == sender) {
            return;
        }
        List<String> registrationIDs = new ArrayList<>(size);
        for (int i = 0; i < size; i += MULTICAST_LIMIT) {
            registrationIDs.clear();

            int length = Math.min(size, i + MULTICAST_LIMIT) - i;
            //@formatter:off
            List<String> chunk = IntStream.range(i, i + length)
                    .mapToObj(j -> matches.get(j).getToken())
                    .collect(Collectors.toList());
            //@formatter:on
            registrationIDs.addAll(chunk);

            // Send chunk
            if (registrationIDs.isEmpty()) {
                continue;
            }
            try {
                BatchResponse batchResponse = sender.sendEachForMulticast(createMessage(client, notification, registrationIDs));
                // Might contain a partial failure
                processResponse(notification, registrationIDs, batchResponse);
            } catch (FirebaseMessagingException e) {
                // Total failure, no notification was sent at all
                LOG.warn("Error publishing push notification", e);
                handleException(notification, registrationIDs, e);
            }
        }
    }

    /**
     * (Optionally) Gets a FCM sender based on the configured API key for given client.
     *
     * @param client The client
     * @return The FCM sender or <code>null</code>
     */
    private FirebaseMessaging optClient(String client) {
        try {
            return getClient(client);
        } catch (Exception e) {
            LOG.error("Error getting FCM client", e);
        }
        return null;
    }

    /**
     * Gets a FCM sender based on the configured API key.
     *
     * @param client The client
     * @return The FCM sender
     * @throws OXException If FCM sender cannot be returned
     */
    private FirebaseMessaging getClient(String client) throws OXException {
        FCMOptions options = getHighestRankedFCMOptionsFor(client);
        return createOrGetApp(options.getClient(), options.getKeyPath());
    }

    /**
     * Creates or gets the app with the specified name and for the specified key path
     * 
     * @param name The app's name
     * @param keyPath The key path
     * @param userId the user identifier
     * @param contextId The context identifier
     * @return
     */
    private FirebaseMessaging createOrGetApp(String name, String keyPath) {
        try (FileInputStream keyStream = new FileInputStream(keyPath)) {
            GoogleCredentials googleCredentials = GoogleCredentials.fromStream(keyStream).createScoped(Arrays.asList(SCOPES));
            FirebaseOptions options = FirebaseOptions.builder().setCredentials(googleCredentials).build();
            return createOrGetApp(options, name);
        } catch (IOException e) {
            LOG.debug("Unable to load FCM key for app {} with path {}", name, keyPath, e);
            return null;
        }
    }

    /**
     * Creates the messaging app. If another app with the same name already exists,
     * it will be deleted and re-initialised
     * 
     * @param options The firebase options
     * @param name The app's name
     * @return The firebase messaging app
     */
    private FirebaseMessaging createOrGetApp(FirebaseOptions options, String name) {
        try {
            FirebaseApp firebaseApp = FirebaseApp.initializeApp(options, name);
            return FirebaseMessaging.getInstance(firebaseApp);
        } catch (IllegalStateException e) {
            LOG.debug("", e);
            FirebaseApp.getInstance(name).delete();
            FirebaseApp app = FirebaseApp.initializeApp(options, name);
            return FirebaseMessaging.getInstance(app);
        }
    }

    /**
     * Processes the response
     * 
     * @param notification the notification
     * @param registrationIDs The registration ids
     * @param batchResponse the batch response
     */
    private void processResponse(PushNotification notification, List<String> registrationIDs, BatchResponse batchResponse) {
        if (null == registrationIDs) {
            LOG.warn("Unable to process empty results");
            return;
        }
        String registrationIDsString = registrationIDs.stream().collect(Collectors.joining(","));
        if (batchResponse == null) {
            LOG.info("Failed to send notification '{}' via transport '{}' for user {} in context {} to registration ID(s): {}", notification.getTopic(), ID, I(notification.getUserId()), I(notification.getContextId()), registrationIDsString);
            return;
        }
        LOG.info("Sent notification '{}' via transport '{}' for user {} in context {} to registration ID(s): {}", notification.getTopic(), ID, I(notification.getUserId()), I(notification.getContextId()), registrationIDsString);
        LOG.debug("{}", batchResponse);

        if (batchResponse.getFailureCount() == 0) {
            return;
        }
        List<SendResponse> responses = batchResponse.getResponses();
        if (responses == null || responses.isEmpty()) {
            return;
        }

        int numOfResults = batchResponse.getSuccessCount() + batchResponse.getFailureCount();
        for (int i = 0; i < numOfResults; i++) {
            // Assuming results' order is the same as in the sent order...
            SendResponse response = responses.get(i);
            String registrationId = registrationIDs.get(i);
            if (response.getException() == null) {
                continue;
            }
            handleException(notification, Collections.singletonList(registrationId), response.getException());
        }
    }

    /**
     * Handles the specified {@link FirebaseMessagingException}
     * 
     * @param subscriptionStore The subscription store
     * @param contextID The context identifier
     * @param token The registration id, i.e. the token
     * @param e the exception
     */
    private void handleException(PushNotification notification, List<String> registrationIDs, FirebaseMessagingException e) {
        switch (e.getErrorCode()) {
            case ABORTED:
                LOG.warn("Push message could not be sent because the request was aborted.");
                return;
            case ALREADY_EXISTS:
                LOG.warn("Push message could not be sent because the resource the client tried to created already exists.");
                return;
            case CANCELLED:
                LOG.warn("Push message could not be sent because the request was cancelled.");
                return;
            case CONFLICT:
                LOG.warn("Push message could not be sent because there was a concurrency conflict.");
                return;
            case DATA_LOSS:
                LOG.warn("Push message could not be sent because there was an unrecoverable data loss or data corruption.");
                return;
            case DEADLINE_EXCEEDED:
                LOG.warn("Push message could not be sent because the request's deadline was exceeded.");
                return;
            case FAILED_PRECONDITION:
                LOG.warn("Push message could not be sent because a precondition failed.");
                return;
            case INTERNAL:
                LOG.warn("Push message could not be sent because an internal error was occurred in the FCM servers.");
                return;
            case INVALID_ARGUMENT:
                LOG.warn("Push message could not be sent because an invalid argument was provided.");
                return;
            case NOT_FOUND:
                LOG.warn("Received error '{}' when sending push message to '{}', removing registration ID.", e.getErrorCode(), notification.getSourceToken());
                registrationIDs.forEach(r -> removeRegistrations(notification, r));
                return;
            case OUT_OF_RANGE:
                LOG.warn("Push message could not be sent because an invalid range was specified.");
                return;
            case PERMISSION_DENIED:
                LOG.warn("Push message could not be sent due to insufficient permissions. This can happen because the OAuth token does not have the right scopes, the client doesn't have permission, or the API has not been enabled for the client project.");
                return;
            case RESOURCE_EXHAUSTED:
                LOG.warn("Push message could not be sent due to the resource either being out of quota or rate limited.");
                return;
            case UNAUTHENTICATED:
                LOG.warn("Push message could not be sent because the request was not authenticated due to missing, invalid or expired OAuth token.");
                return;
            case UNAVAILABLE:
                LOG.warn("Push message could not be sent because the FCM servers were not available.");
                return;
            case UNKNOWN:
                LOG.warn("Push message could not be sent because an unknown error was occurred in the FCM servers.");
                return;
            default:
                LOG.warn("", e);
        }
    }

    /**
     * Removes the registrations for the specified notification
     * 
     * @param notification The notification
     * @param registrationID the registration id
     */
    private void removeRegistrations(PushNotification notification, String registrationID) {
        try {
            //@formatter:off
            DefaultPushSubscription.Builder builder = DefaultPushSubscription.builder()
                .contextId(notification.getContextId())
                .token(registrationID)
                .transportId(ID)
                .userId(notification.getUserId());
            //@formatter:on
            DefaultPushSubscription subscriptionDesc = builder.build();

            boolean success = pushSubscriptionRegistry.unregisterSubscription(subscriptionDesc);
            if (success) {
                LOG.info("Successfully removed registration ID {}.", registrationID);
                return;
            }
        } catch (OXException e) {
            LOG.error("Error removing subscription", e);
        }
        LOG.warn("Registration ID {} not removed.", registrationID);
    }

    //////////////////////////////////// VALUE SETTERS ///////////////////////////////

    /**
     * Creates the multicast message to send to the devices
     * 
     * @param client The client
     * @param notification The notification to send
     * @param registrtationIDs The registration ids of the devices
     * @return The multicast message
     * @throws OXException when no message generator exists or there is an unsupported message class
     */
    @SuppressWarnings({ "unchecked" })
    private MulticastMessage createMessage(String client, PushNotification notification, List<String> registrtationIDs) throws OXException {
        PushMessageGenerator generator = generatorRegistry.getGenerator(client);
        if (null == generator) {
            throw PushExceptionCodes.NO_SUCH_GENERATOR.create(client);
        }

        com.openexchange.pns.Message<?> message = generator.generateMessageFor(ID, notification);
        Object object = message.getMessage();
        if (object instanceof MulticastMessage) {
            MulticastMessage m = (MulticastMessage) object;
            return checkMessage(m);
        }
        if (object instanceof Map) {
            Map<String, Object> m = (Map<String, Object>) object;
            return checkMessage(toMessage(m, registrtationIDs));
        }
        throw PushExceptionCodes.UNSUPPORTED_MESSAGE_CLASS.create(null == object ? "null" : object.getClass().getName());
    }

    /**
     * Checks whether the message's payload size exceeds the limits
     * 
     * @param message The message
     * @return The message for chained calls
     * @throws OXException if the message is too big
     */
    private MulticastMessage checkMessage(MulticastMessage message) throws OXException {
        int currentLength = PushNotifications.getPayloadLength(message.toString());
        if (currentLength > MAX_PAYLOAD_SIZE) {
            throw PushExceptionCodes.MESSAGE_TOO_BIG.create(I(MAX_PAYLOAD_SIZE), I(currentLength));
        }
        return message;
    }

    /**
     * Converts the specified map to a multicast message
     * Note: 'contentAvailable', 'delayWhileIdle' and 'badge' properties are not available in FCM
     * 
     * @param message the map containing the message information
     * @param registrationIds The registration ids
     * @return The multicast message
     * @throws OXException if the icon element is missing
     */
    private MulticastMessage toMessage(Map<String, Object> message, List<String> registrationIds) throws OXException {
        Map<String, Object> source = new HashMap<>(message);

        AndroidNotification.Builder notificationBuilder = AndroidNotification.builder();
        setValue(source, "icon", notificationBuilder, false, AndroidNotification.Builder::setIcon);
        setValue(source, "body", notificationBuilder, AndroidNotification.Builder::setBody);
        setValue(source, "click_action", notificationBuilder, AndroidNotification.Builder::setClickAction);
        setValue(source, "color", notificationBuilder, AndroidNotification.Builder::setColor);
        setValue(source, "sound", notificationBuilder, AndroidNotification.Builder::setSound);
        setValue(source, "tag", notificationBuilder, AndroidNotification.Builder::setTag);
        setValue(source, "title", notificationBuilder, AndroidNotification.Builder::setTitle);

        AndroidConfig.Builder androidConfigBuilder = AndroidConfig.builder();
        setValue(source, "collapse_key", androidConfigBuilder, AndroidConfig.Builder::setCollapseKey);
        setValue(source, "restricted_package_name", androidConfigBuilder, AndroidConfig.Builder::setRestrictedPackageName);
        setIntValue(source, "time_to_live", androidConfigBuilder, AndroidConfig.Builder::setTtl);
        setPriority(source, androidConfigBuilder);
        androidConfigBuilder.setNotification(notificationBuilder.build());

        MulticastMessage.Builder builder = MulticastMessage.builder();
        builder.setAndroidConfig(androidConfigBuilder.build());
        builder.addAllTokens(registrationIds);

        // Add the remaining data as raw data
        for (Map.Entry<String, Object> entry : message.entrySet()) {
            String value = String.valueOf(entry.getValue());
            builder.putData(entry.getKey(), value);
        }

        return builder.build();
    }

    /**
     * Sets the value to the specified builder
     * 
     * @param source The source containing the value
     * @param key the key for the value
     * @param builder The builder
     * @param valueSetter The value setter
     * @throws OXException if a mandatory value is empty
     */
    private void setValue(Map<String, Object> source, String key, AndroidNotification.Builder builder, BiConsumer<AndroidNotification.Builder, String> valueSetter) throws OXException {
        setValue(source, key, builder, true, valueSetter);
    }

    /**
     * Sets the value to the specified builder
     * 
     * @param source The source containing the value
     * @param key the key for the value
     * @param builder The builder
     * @param canBeEmpty Whether the value of the key is allowed to be empty
     * @param valueSetter The value setter
     * @throws OXException if a mandatory value is empty
     */
    private void setValue(Map<String, Object> source, String key, AndroidNotification.Builder builder, boolean canBeEmpty, BiConsumer<AndroidNotification.Builder, String> valueSetter) throws OXException {
        String value = (String) source.remove(key);
        if (value != null) {
            valueSetter.accept(builder, value);
            return;
        }
        if (canBeEmpty) {
            return;
        }
        throw PushExceptionCodes.MESSAGE_GENERATION_FAILED.create("Missing '" + key + "' element.");
    }

    /**
     * Sets the value to the specified builder
     * 
     * @param source The source containing the value
     * @param key the key for the value
     * @param builder The builder
     * @param valueSetter The value setter
     */
    private void setValue(Map<String, Object> source, String key, AndroidConfig.Builder builder, BiConsumer<AndroidConfig.Builder, String> valueSetter) {
        String value = (String) source.remove(key);
        if (value == null) {
            return;
        }
        valueSetter.accept(builder, value);
    }

    /**
     * Sets the integer value to the specified builder
     * 
     * @param source The source containing the value
     * @param key the key for the value
     * @param builder The builder
     * @param valueSetter The value setter
     */
    private void setIntValue(Map<String, Object> source, String key, AndroidConfig.Builder builder, ObjIntConsumer<AndroidConfig.Builder> valueSetter) {
        Integer value = (Integer) source.remove(key);
        if (value == null) {
            return;
        }
        valueSetter.accept(builder, value.intValue());
    }

    /**
     * Sets the priority to the specified builder
     * 
     * @param source The source containing the value
     * @param key the key for the value
     * @param builder The builder
     * @param valueSetter The value setter
     */
    private void setPriority(Map<String, Object> source, AndroidConfig.Builder builder) {
        String sPriority = (String) source.remove("priority");
        if (null == sPriority) {
            return;
        }
        if ("high".equalsIgnoreCase(sPriority)) {
            builder.setPriority(Priority.HIGH);
        } else if ("normal".equalsIgnoreCase(sPriority)) {
            builder.setPriority(Priority.NORMAL);
        }
    }

    /**
     * Invalidates the <i>enabled cache</i>.
     */
    public static void invalidateEnabledCache() {
        CACHE_AVAILABILITY.invalidateAll();
    }
}
