/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package gnu.trove.set.hash;

import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.concurrent.locks.StampedLock;
import gnu.trove.iterator.TIntIterator;
import gnu.trove.procedure.TIntProcedure;

/**
 * {@link ConcurrentTIntHashSet}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public final class ConcurrentTIntHashSet extends TIntHashSet implements Cloneable {

    private final TIntHashSet set;

    private StampedLock lock;

    /**
     * Initializes a new {@link ConcurrentTIntHashSet}.
     */
    public ConcurrentTIntHashSet() {
        super();
        set = new TIntHashSet();
        lock = new StampedLock();
    }

    /**
     * Initializes a new {@link ConcurrentTIntHashSet}.
     *
     * @param initialCapacity used to find a prime capacity for the table.
     * @param loadFactor used to calculate the threshold over which rehashing takes place.
     */
    public ConcurrentTIntHashSet(final int initialCapacity, final float loadFactor) {
        super();
        set = new TIntHashSet(initialCapacity, loadFactor);
        lock = new StampedLock();
    }

    /**
     * Initializes a new {@link ConcurrentTIntHashSet}.
     *
     * @param initialCapacity used to find a prime capacity for the table.
     */
    public ConcurrentTIntHashSet(final int initialCapacity) {
        super();
        set = new TIntHashSet(initialCapacity);
        lock = new StampedLock();
    }

    /**
     * Initializes a new {@link ConcurrentTIntHashSet}.
     *
     * @param array an array of int primitives
     */
    public ConcurrentTIntHashSet(final int[] array) {
        super();
        set = new TIntHashSet(array);
        lock = new StampedLock();
    }

    @Override
    public ConcurrentTIntHashSet clone() {
        try {
            final ConcurrentTIntHashSet clone = (ConcurrentTIntHashSet) super.clone();
            clone.lock = new StampedLock();
            return clone;
        } catch (CloneNotSupportedException e) {
            // Cannot occur
            throw new InternalError("Clone not supported although Cloneable implemented.");
        }
    }

    @Override
    public TIntIterator iterator() {
        long stamp = lock.tryOptimisticRead();
        int[] arr = set.toArray();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                arr = set.toArray();
            } finally {
                lock.unlockRead(stamp);
            }
        }

        int[] array = arr;
        return new TIntIterator() {

            private int index = 0;

            @Override
            public void remove() {
                throw new UnsupportedOperationException("ConcurrentTIntHashSet.TIntIterator().remove()");
            }

            @Override
            public boolean hasNext() {
                return index < array.length;
            }

            @Override
            public int next() {
                return array[index++];
            }
        };
    }

    @Override
    public boolean isEmpty() {
        long stamp = lock.tryOptimisticRead();
        boolean empty = set.isEmpty();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                empty = set.isEmpty();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return empty;
    }

    @Override
    public boolean add(final int val) {
        long stamp = lock.writeLock();
        try {
            return set.add(val);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public boolean contains(final int val) {
        long stamp = lock.tryOptimisticRead();
        boolean contains = set.contains(val);

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                contains = set.contains(val);
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return contains;
    }

    @Override
    public int size() {
        long stamp = lock.tryOptimisticRead();
        int size = set.size();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                size = set.size();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return size;
    }

    @Override
    public boolean forEach(final TIntProcedure procedure) {
        long stamp = lock.readLock();
        try {
            return set.forEach(procedure);
        } finally {
            lock.unlockRead(stamp);
        }
    }

    @Override
    public void ensureCapacity(final int desiredCapacity) {
        long stamp = lock.writeLock();
        try {
            set.ensureCapacity(desiredCapacity);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public void compact() {
        long stamp = lock.writeLock();
        try {
            set.compact();
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public int[] toArray() {
        long stamp = lock.tryOptimisticRead();
        int[] arr = set.toArray();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                arr = set.toArray();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return arr;
    }

    @Override
    public void clear() {
        long stamp = lock.writeLock();
        try {
            set.clear();
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public boolean equals(final Object other) {
        long stamp = lock.tryOptimisticRead();
        boolean equals = set.equals(other);

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                equals = set.equals(other);
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return equals;
    }

    @Override
    public void setAutoCompactionFactor(final float factor) {
        long stamp = lock.writeLock();
        try {
            set.setAutoCompactionFactor(factor);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public int hashCode() {
        long stamp = lock.tryOptimisticRead();
        int h = set.hashCode();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                h = set.hashCode();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return h;
    }

    @Override
    public float getAutoCompactionFactor() {
        long stamp = lock.tryOptimisticRead();
        float f = set.getAutoCompactionFactor();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                f = set.getAutoCompactionFactor();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return f;
    }

    @Override
    public boolean remove(final int val) {
        long stamp = lock.writeLock();
        try {
            return set.remove(val);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public boolean containsAll(final int[] array) {
        long stamp = lock.tryOptimisticRead();
        boolean containsAll = set.containsAll(array);

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                containsAll = set.containsAll(array);
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return containsAll;
    }

    @Override
    public boolean addAll(final int[] array) {
        long stamp = lock.writeLock();
        try {
            return set.addAll(array);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public boolean removeAll(final int[] array) {
        long stamp = lock.writeLock();
        try {
            return set.removeAll(array);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public String toString() {
        long stamp = lock.tryOptimisticRead();
        String s = set.toString();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                s = set.toString();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return s;
    }

    @Override
    public boolean retainAll(final int[] array) {
        long stamp = lock.writeLock();
        try {
            return set.retainAll(array);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public void writeExternal(final ObjectOutput out) throws IOException {
        long stamp = lock.readLock();
        try {
            set.writeExternal(out);
        } finally {
            lock.unlockRead(stamp);
        }
    }

    @Override
    public void readExternal(final ObjectInput in) throws IOException, ClassNotFoundException {
        long stamp = lock.writeLock();
        try {
            set.readExternal(in);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

}
