/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */


package com.openexchange.filestore.s3.internal;

import static com.openexchange.filestore.s3.internal.S3ExceptionCode.wrap;
import static com.openexchange.java.Autoboxing.L;
import java.io.InputStream;
import java.net.URI;
import java.util.OptionalLong;
import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.SdkClientException;
import com.amazonaws.auth.policy.Policy;
import com.amazonaws.auth.policy.Principal;
import com.amazonaws.auth.policy.Resource;
import com.amazonaws.auth.policy.Statement;
import com.amazonaws.auth.policy.actions.S3Actions;
import com.amazonaws.auth.policy.conditions.BooleanCondition;
import com.amazonaws.auth.policy.conditions.StringCondition;
import com.amazonaws.auth.policy.conditions.StringCondition.StringComparisonType;
import com.amazonaws.services.s3.Headers;
import com.amazonaws.services.s3.internal.BucketNameUtils;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.CreateBucketRequest;
import com.amazonaws.services.s3.model.MetadataDirective;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.Region;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.SetBucketPolicyRequest;
import com.openexchange.exception.OXException;
import com.openexchange.filestore.s3.internal.client.S3FileStorageClient;
import com.openexchange.java.Strings;

/**
 * {@link AbstractS3FileStorage} - The abstract S3 file storage.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v8.x
 */
public abstract class AbstractS3FileStorage implements S3FileStorage {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(AbstractS3FileStorage.class);

    /** The delimiter character (<code>'/'</code>) to separate the prefix from the keys. */
    public static final char DELIMITER = '/';

    /** The minimum part size required for a multipart upload */
    protected static final int MINIMUM_MULTIPART_SIZE = 5242880; /* 5 MB */

    /** Header for the original, unencrypted size of an encrypted object: {@value Headers#UNENCRYPTED_CONTENT_LENGTH} */
    protected static final String UNENCRYPTED_CONTENT_LENGTH = Headers.UNENCRYPTED_CONTENT_LENGTH;

    /**
     * The max. number of keys that are allowed being passed by a multiple delete objects request.
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjects.html">See here</a>
     */
    protected static final int MAX_NUMBER_OF_KEYS_TO_DELETE = 1000;

    // -------------------------------------------------------------------------------------------------------------------------------------

    protected final URI uri;
    protected final String prefix;
    protected final String bucketName;
    protected final S3FileStorageClient client;

    /**
     * Initializes a new {@link AbstractS3FileStorage}.
     *
     * @param uri The URI that fully qualifies this file storage
     * @param prefix The prefix to use; e.g. <code>"1337ctxstore"</code>
     * @param bucketName The bucket name to use
     * @param client The file storage client
     */
    protected AbstractS3FileStorage(URI uri, String prefix, String bucketName, S3FileStorageClient client) {
        super();
        BucketNameUtils.validateBucketName(bucketName);
        if (Strings.isEmpty(prefix) || prefix.indexOf(DELIMITER) >= 0) {
            throw new IllegalArgumentException(prefix);
        }
        this.uri = uri;
        this.prefix = prefix;
        this.bucketName = bucketName;
        this.client = client;
    }

    @Override
    public URI getUri() {
        return uri;
    }

    @Override
    public boolean isSpooling() {
        return true;
    }

    @Override
    public long getPreferredChunkSize() {
        return client.getChunkSize();
    }

    @Override
    public void recreateStateFile() throws OXException {
        // no
    }

    @Override
    public boolean stateFileIsCorrect() throws OXException {
        return true;
    }

    @Override
    public void ensureBucket() throws OXException {
        boolean bucketExists = false;
        try {
            bucketExists = withRetry(c -> Boolean.valueOf(c.doesBucketExist(bucketName))).booleanValue();
        } catch (AmazonClientException e) {
            throw S3ExceptionCode.wrap(e);
        } catch (RuntimeException e) {
            throw S3ExceptionCode.UNEXPECTED_ERROR.create(e, e.getMessage());
        }

        if (!bucketExists) {
            // Bucket does not yet exist
            String region = client.getSdkClient().getRegionName();
            try {
                withRetry(c -> c.createBucket(new CreateBucketRequest(bucketName, Region.fromValue(region))));
                if (client.getEncryptionConfig().isServerSideEncryptionEnabled()) {
                    withVoidRetry(c -> c.setBucketPolicy(new SetBucketPolicyRequest(bucketName, getSSEOnlyBucketPolicy(bucketName))));
                }
            } catch (AmazonS3Exception e) {
                if ("InvalidLocationConstraint".equals(e.getErrorCode())) {
                    // Failed to create such a bucket
                    throw S3ExceptionCode.BUCKET_CREATION_FAILED.create(bucketName, region);
                }
                throw S3ExceptionCode.wrap(e);
            } catch (AmazonClientException e) {
                throw S3ExceptionCode.wrap(e);
            } catch (RuntimeException e) {
                throw S3ExceptionCode.UNEXPECTED_ERROR.create(e, e.getMessage());
            }
        }
    }

    protected ObjectMetadata prepareMetadataForSSE(ObjectMetadata metadata) {
        if (client.getEncryptionConfig().isServerSideEncryptionEnabled()) {
            metadata.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
        }
        return metadata;
    }

    protected InputStream wrapperWithoutRangeSupport(S3ObjectInputStream objectContent, String key) {
        /*
         * Consume rather than skip on invocation of skip() if client-side encryption is enabled
         * since com.amazonaws.services.s3.internal.crypto.CipherLiteInputStream.skip(long) does
         * not read next chunk if more than currently buffered bytes are supposed to be skipped
         */
        boolean consumeBytesOnSkip = client.getEncryptionConfig().isClientEncryptionEnabled();
        return new ResumableAbortIfNotConsumedInputStream(objectContent, bucketName, key, consumeBytesOnSkip, client.getSdkClient());
    }

    protected InputStream wrapperWithRangeSupport(S3ObjectInputStream objectContent, long[] range, String key) {
        /*
         * Consume rather than skip on invocation of skip() if client-side encryption is enabled
         * since com.amazonaws.services.s3.internal.crypto.CipherLiteInputStream.skip(long) does
         * not read next chunk if more than currently buffered bytes are supposed to be skipped
         */
        boolean consumeBytesOnSkip = client.getEncryptionConfig().isClientEncryptionEnabled();
        return new RangeAcceptingResumableAbortIfNotConsumedInputStream(objectContent, range, bucketName, key, consumeBytesOnSkip, client.getSdkClient());
    }

    /**
     * Gets the bucket policy for a server side encryption only bucket.
     *
     * @param bucketName The name of the bucket
     * @return The encryption only policy
     */
    protected static String getSSEOnlyBucketPolicy(String bucketName) {
        Policy bucketPolicy = new Policy().withStatements(
            new Statement(Statement.Effect.Deny)
                .withId("DenyIncorrectEncryptionHeader")
                .withPrincipals(Principal.AllUsers)
                .withActions(S3Actions.PutObject)
                .withResources(new Resource("arn:aws:s3:::" + bucketName + "/*"))
                .withConditions(new StringCondition(StringComparisonType.StringNotEquals, "s3:x-amz-server-side-encryption", "AES256")),
            new Statement(Statement.Effect.Deny)
                .withId("DenyUnEncryptedObjectUploads")
                .withPrincipals(Principal.AllUsers)
                .withActions(S3Actions.PutObject)
                .withResources(new Resource("arn:aws:s3:::" + bucketName + "/*"))
                .withConditions(new BooleanCondition("s3:x-amz-server-side-encryption", true))
                );
        return bucketPolicy.toJson();
    }

    /**
     * Gets a stored S3 object.
     *
     * @param key The key of the file
     * @return The S3 object
     * @throws OXException
     */
    protected S3Object getObject(String key) throws OXException {
        try {
            return withRetry(c -> c.getObject(bucketName, key));
        } catch (AmazonClientException e) {
            throw wrap(e, key);
        }
    }

    /**
     * Gets metadata for an existing file.
     *
     * @param key The (full) key for the new file; no additional prefix will be prepended implicitly
     * @return The upload ID for the multipart upload
     * @throws OXException
     */
    protected ObjectMetadata getMetadata(String key) throws OXException {
        try {
            return withRetry(c -> c.getObjectMetadata(bucketName, key));
        } catch (AmazonClientException e) {
            throw wrap(e, key);
        }
    }

    /**
     * Retrieves the object metadata for an existing file and extracts the effective content length, which is the length of the
     * unencrypted content if specified, or the plain content length, otherwise.
     *
     * @param key The (full) key of the file to get the actual content length for; no additional prefix will be prepended implicitly
     * @return The length of the unencrypted content if specified, or the plain content length, otherwise
     */
    protected long getContentLength(String key) throws OXException {
        ObjectMetadata metadata = getMetadata(key);
        OptionalLong unencryptedContentLength = optUnencryptedContentLength(metadata);
        if (unencryptedContentLength.isPresent()) {
            return unencryptedContentLength.getAsLong();
        }
        if (client.getEncryptionConfig().isClientEncryptionEnabled() && Strings.isNotEmpty(metadata.getUserMetaDataOf(Headers.CRYPTO_IV))) {
            String logKey = UNENCRYPTED_CONTENT_LENGTH + ':' + bucketName;
            LOG.debug("No value for \"{}\" found in metadata of object \"{}\", but client-side encryption is enabled and \"{}\" is present in metadata - content length may not be accurate.",
                UNENCRYPTED_CONTENT_LENGTH, logKey, com.amazonaws.services.s3.Headers.CRYPTO_IV);
        }
        return metadata.getContentLength();
    }

    /**
     * Optionally gets and parses the value for special header {@value #UNENCRYPTED_CONTENT_LENGTH} from
     * the supplied object metadata.
     *
     * @param metadata The metadata to extract the value from
     * @return The value as {@link OptionalLong}, or {@link OptionalLong#empty()} if not present or parseable
     */
    protected static OptionalLong optUnencryptedContentLength(ObjectMetadata metadata) {
        String value = metadata.getUserMetaDataOf(UNENCRYPTED_CONTENT_LENGTH);
        long longValue = Strings.parseUnsignedLong(value);
        return -1 == longValue ? OptionalLong.empty() : OptionalLong.of(longValue);
    }

    /**
     * Ensures that the special {@value Headers#UNENCRYPTED_CONTENT_LENGTH} header is present in the object's
     * metadata and set to the given value if required, which is when client-side encryption is enabled and the value is missing or
     * different from the expected one.
     *
     * @param key The (full) key of the file to apply the unencrypted content length for; no additional prefix will be prepended implicitly
     * @param unencryptedContentLength The value to apply
     */
    protected void applyUnencryptedContentLengthIfNeeded(String key, long unencryptedContentLength) {
        if (client.getEncryptionConfig().isClientEncryptionEnabled()) {
            ObjectMetadata metadata = withRetry(c -> c.getObjectMetadata(bucketName, key));
            if (unencryptedContentLength != optUnencryptedContentLength(metadata).orElse(-1L)) {
                ObjectMetadata newMetadata = cloneMetadata(metadata);
                newMetadata.addUserMetadata(UNENCRYPTED_CONTENT_LENGTH, Long.toString(unencryptedContentLength));
                CopyObjectRequest request = new CopyObjectRequest(bucketName, key, bucketName, key)
                    .withMetadataDirective(MetadataDirective.REPLACE).withNewObjectMetadata(newMetadata);
                withVoidRetry(c -> c.copyObject(request));
                LOG.debug("Successfully applied \"{} = {}\" for object \"{}\"", UNENCRYPTED_CONTENT_LENGTH, L(unencryptedContentLength), key);
            }
        }
    }

    /**
     * Clones the given metadata, ensuring that some default headers are taken over as expected.
     *
     * @param source The metadata to clone
     * @return The cloned metadata
     */
    protected static ObjectMetadata cloneMetadata(ObjectMetadata source) {
        if (null == source) {
            return null;
        }
        ObjectMetadata clonedMetadata = source.clone();
        /*
         * auto-correct case in some default headers
         */
        for (String headerName : new String[] { Headers.CONTENT_LENGTH, Headers.CONTENT_TYPE, Headers.DATE, Headers.ETAG, Headers.LAST_MODIFIED }) {
            Object value = source.getRawMetadataValue(headerName);
            if (null != value) {
                clonedMetadata.setHeader(headerName, value);
            }
        }
        return clonedMetadata;
    }

    /**
     * Checks if "CopyPart" request can be used during a multipart upload or not.
     *
     * @param contentLength The content length of the object representing the part to be copied
     * @return <code>true</code> if "CopyPart" can be used, <code>false</code> otherwise
     */
    protected boolean isCopyPartAvailable(long contentLength) {
        return client.useUploadPartCopy() && !client.getEncryptionConfig().isClientEncryptionEnabled() && contentLength >= MINIMUM_MULTIPART_SIZE;
    }

    /**
     * Executes with retry the given closure with configured retry count.
     *
     * @param <R> The result type
     * @param closure The closure to invoke
     * @return The result
     * @throws SdkClientException If any errors are encountered in the client while making the request or handling the response
     * @throws AmazonServiceException If any errors occurred in Amazon S3 while processing the request
     */
    protected void withVoidRetry(S3VoidOperationClosure closure) throws SdkClientException, AmazonServiceException {
        int retryCount = client.getNumOfRetryAttemptsOnConnectionPoolTimeout();
        new RetryingS3OperationExecutor<Void>(closure, retryCount <= 0 ? 1 : (retryCount + 1)).execute(client.getSdkClient());
    }

    /**
     * Executes with retry the given closure with configured retry count.
     *
     * @param <R> The result type
     * @param closure The closure to invoke
     * @return The result
     * @throws SdkClientException If any errors are encountered in the client while making the request or handling the response
     * @throws AmazonServiceException If any errors occurred in Amazon S3 while processing the request
     */
    protected <R> R withRetry(S3OperationClosure<R> closure) throws SdkClientException, AmazonServiceException {
        int retryCount = client.getNumOfRetryAttemptsOnConnectionPoolTimeout();
        return new RetryingS3OperationExecutor<R>(closure, retryCount).execute(client.getSdkClient());
    }

}
