/*
* @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
* @license AGPL-3.0
*
* This code is free software: you can redistribute it and/or modify
* it under the terms of the GNU Affero General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU Affero General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public License
* along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
* 
* Any use of the work other than as authorized under this license or copyright law is prohibited.
*
*/

package com.openexchange.drive.events.fcm.groupware;

import static com.openexchange.java.Autoboxing.I;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.database.Databases;
import com.openexchange.database.Databases.ConnectionStatus;
import com.openexchange.drive.events.subscribe.rdb.DriveEventSubscriptionsCreateTableTask;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.update.PerformParameters;
import com.openexchange.groupware.update.UpdateExceptionCodes;
import com.openexchange.groupware.update.UpdateTaskAdapter;

/**
 * {@link RenameGCM2FCMUpdateTask}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class RenameGCM2FCMUpdateTask extends UpdateTaskAdapter {

    private static final Logger LOG = LoggerFactory.getLogger(RenameGCM2FCMUpdateTask.class);
    private static final String UPDATE_SQL = "UPDATE driveEventSubscriptions SET service='fcm' WHERE service='gcm'";

    /**
     * Initialises a new {@link RenameGCM2FCMUpdateTask}.
     */
    public RenameGCM2FCMUpdateTask() {
        super();
    }

    @Override
    public void perform(PerformParameters params) throws OXException {
        Connection connection = params.getConnection();
        ConnectionStatus status = ConnectionStatus.INITIALISED;
        try (PreparedStatement statement = connection.prepareStatement(UPDATE_SQL)) {
            connection.setAutoCommit(false);
            status = ConnectionStatus.FAILED;
            int updated = statement.executeUpdate();
            connection.commit();
            status = ConnectionStatus.SUCCEEDED;

            LOG.info("Renamed service ids in 'driveEventSubscriptions' from 'gcm' to 'fcm'. Total rows affected: {}", I(updated));
        } catch (SQLException e) {
            throw UpdateExceptionCodes.SQL_PROBLEM.create(e, e.getMessage());
        } catch (RuntimeException e) {
            throw UpdateExceptionCodes.OTHER_PROBLEM.create(e, e.getMessage());
        } finally {
            switch (status) {
                case FAILED:
                    Databases.rollback(connection);
                    //$FALL-THROUGH$
                case SUCCEEDED:
                    Databases.autocommit(connection);
                    break;
                case INITIALISED:
                default:
                    break;
            }
        }
    }

    @Override
    public String[] getDependencies() {
        return new String[] { DriveEventSubscriptionsCreateTableTask.class.getName() };
    }
}
