/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.drive.client.windows.service;

import com.openexchange.exception.Category;
import com.openexchange.exception.DisplayableOXExceptionCode;
import com.openexchange.exception.OXException;
import com.openexchange.exception.OXExceptionFactory;
import com.openexchange.exception.OXExceptionStrings;

/**
 * {@link UpdaterExceptionCodes}
 *
 * @author <a href="mailto:steffen.templin@open-xchange.com">Steffen Templin</a>
 */
public enum UpdaterExceptionCodes implements DisplayableOXExceptionCode {
    /**
     * Error while looking for available files in directory %1$s.
     */
    NO_FILES_ERROR("Error while looking for available files in directory %1$s.", Category.CATEGORY_ERROR, 1, null),

    /**
     * This service is not responsible for delivering file %1$s.
     */
    SERVICE_NOT_RESPONSIBLE("This service is not responsible for delivering file %1$s.", Category.CATEGORY_ERROR, 3, null),

    /**
     * No files are available. This service will not work correctly.
     */
    NO_FILES_AVAILABLE("No files are available. This service will not work correctly.", Category.CATEGORY_ERROR, 4, null),

    /**
     * The configuration property '%1$s' is missing but required!
     */
    MISSING_CONFIG_PROPERTY("The configuration property '%1$s' is missing but required!", Category.CATEGORY_ERROR, 5, null),

    /**
     * An IO error occurred: %1$s
     */
    IO_ERROR("An IO error occurred: %1$s", Category.CATEGORY_ERROR, 6, null),

    /**
     * Some or all necessary files of the brand \"%1$s\" are missing but required! Please provide all necessary files or change the configured brand.
     */
    BRANDING_ERROR("Some or all necessary files of the brand \"%1$s\" are missing but required! Please provide all necessary files or change the configured brand.", Category.CATEGORY_ERROR, 7, null);

    ;

    /**
     * The error code prefix.
     */
    public static final String PREFIX = "UPDATER";

    private final String message;

    private final int number;

    private final Category category;

    private final String displayMessage;

    private UpdaterExceptionCodes(String message, Category category, int detailNumber, String displayMessage) {
        this.message = message;
        number = detailNumber;
        this.category = category;
        this.displayMessage = displayMessage != null ? displayMessage : OXExceptionStrings.MESSAGE;
    }

    /**
     * Creates a new {@link OXException} instance pre-filled with this code's attributes.
     *
     * @return The newly created {@link OXException} instance
     */
    public OXException create() {
        return OXExceptionFactory.getInstance().create(this, new Object[0]);
    }

    /**
     * Creates a new {@link OXException} instance pre-filled with this code's attributes.
     *
     * @param args The message arguments in case of printf-style message
     * @return The newly created {@link OXException} instance
     */
    public OXException create(final Object... args) {
        return OXExceptionFactory.getInstance().create(this, (Throwable) null, args);
    }

    /**
     * Creates a new {@link OXException} instance pre-filled with this code's attributes.
     *
     * @param cause The optional initial cause
     * @param args The message arguments in case of printf-style message
     * @return The newly created {@link OXException} instance
     */
    public OXException create(final Throwable cause, final Object... args) {
        return OXExceptionFactory.getInstance().create(this, cause, args);
    }

    @Override
    public boolean equals(final OXException e) {
        return OXExceptionFactory.getInstance().equals(this, e);
    }

    @Override
    public String getDisplayMessage() {
        return displayMessage;
    }

    @Override
    public int getNumber() {
        return number;
    }

    @Override
    public Category getCategory() {
        return category;
    }

    @Override
    public String getPrefix() {
        return PREFIX;
    }

    @Override
    public String getMessage() {
        return message;
    }

}
