/**
 * plugin.js
 *
 * Copyright, Moxiecode Systems AB
 * Released under LGPL License.
 *
 * License: http://www.tinymce.com/license
 * Contributing: http://www.tinymce.com/contributing
 */

(function (tinymce) {

    var cache = {};

    tinymce.PluginManager.add('oximage', function (editor) {

        var self = this,
            imageLoader = editor.settings.oxImageLoader;

        this.getPendingDeferreds = function (ids) {
            return _(ids)
                .chain()
                .map(function (id) {
                    return cache[id];
                })
                .compact()
                .value();
        };

        // is called automatically, as soon as an image appears in the editor
        editor.settings.images_upload_handler = function (blobInfo, success, failure) {
            var id = blobInfo.id();
            require(['io.ox/core/yell'], function (yell) {
                cache[id] = new $.Deferred();
                imageLoader.upload(blobInfo.blob()).then(
                    function uploadSuccess(response) {
                        var url = imageLoader.getUrl(response);
                        success(url);
                        _.defer(function () {
                            // remove data-pending after image has been replaced
                            var img = $(editor.getBody()).find('img[data-id="' + id + '"]');
                            img.removeData('pending id').removeAttr('data-pending data-id');
                            cache[id].resolve();
                            // fire a keyup event such that the changes inside the editor can be tracked
                            editor.fire('keyup');
                        });
                    },
                    function fail(error) {
                        yell(error);
                        failure();
                        // remove image manually
                        $(editor.getBody()).find('img[data-id="' + id + '"]').remove();
                        cache[id].reject();
                    }
                );
                cache[id].always(function () {
                    delete cache[id];
                });
            });
        };

        /*
        * Uploads a blob image file and inserts it at the cursor position.
        */
        this.uploadBlob = function (blob, opt) {
            opt = opt || {};

            // store in tinymce blobcache
            var reader = new window.FileReader();
            reader.readAsDataURL(blob);
            reader.onloadend = function () {
                var id = _.uniqueId('upload-image-'),
                    blobcache = editor.editorUpload.blobCache,
                    blobInfo = blobcache.create(id, blob, reader.result);
                blobcache.add(blobInfo);

                self.addImage({ src: blobInfo.blobUri(), id: id, from_clipboard: opt.from_clipboard });
            };
            reader.onerror = function () {
                console.error(reader.error);
            };
        };

        this.addImage = function (opt) {
            editor.undoManager.transact(function () {
                editor.focus();
                // tinymce pasteplugin seems to strip ids so provide as class for now
                var attr = { src: opt.src, 'data-mce-src': opt.src, style: 'max-width: 100%;', class: 'aspect-ratio', alt: '', 'data-pending': 'true', 'data-id': opt.id };
                if (opt.from_clipboard) attr.from_clipboard = true;
                editor.selection.setContent(editor.dom.createHTML('img', attr));
            });
        };

        editor.addButton('image', {
            icon: 'image',
            title: 'Insert image',
            onPostRender: (function () {

                var change = _.debounce(function () {

                    var file = $(this);

                    require(['io.ox/core/notifications', 'gettext!io.ox/mail'], function (notifications, gt) {

                        if (!(/\.(gif|bmp|tiff|jpe?g|gmp|png)$/i).test(file.val())) {
                            notifications.yell('error', gt('Please select a valid image File to insert'));
                        } else if (file[0].files && editor) {
                            self.uploadBlob(file[0].files[0]);
                        }
                    });
                }, 100);

                return function onPostRender() {
                    var self = this;
                    var fileInput = $('<input type="file" name="file" accept="image/*" capture="camera">').css('display', 'none');

                    self.off('click')
                        .on('click touchstart', function () {
                            fileInput[0].value = '';
                            fileInput.trigger('click');
                        });

                    fileInput.on('change', change);

                    self.$el.find('button').before(fileInput);
                };
            }())
        });
    });
}(window.tinymce));
