/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.storage;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import com.openexchange.usm.api.ox.json.OXJSONAccess;
import com.openexchange.usm.api.session.assets.CompleteSessionID;
import com.openexchange.usm.api.session.assets.SessionID;
import com.openexchange.usm.api.session.assets.UserID;
import com.openexchange.usm.session.impl.SessionImpl;
import com.openexchange.usm.session.impl.SessionManagerImpl;
import com.openexchange.usm.session.impl.SessionReference;

public class SessionStorage {

    // Map from USM SessionID (user, protocol, device) to actual Session
    private final Map<SessionID, SessionReference> _map = new ConcurrentHashMap<SessionID, SessionReference>();

    // backwards Map from references of Sessions to combined USM and OX id, needed for cleanup of other Maps
    private final Map<SoftReference<SessionImpl>, CompleteSessionID> _backwardsMap = new ConcurrentHashMap<SoftReference<SessionImpl>, CompleteSessionID>();

    // Lookup Map from OX UserID to list of active SessionIDs for that user
    private final Map<UserID, SessionID[]> _userSessionMap = new HashMap<UserID, SessionID[]>();

    // Cleanup queue to remove session IDs of any kind from the Maps when Sessions are no longer used
    private final ReferenceQueue<SessionImpl> _referenceQueue = new ReferenceQueue<SessionImpl>();

    private final SessionManagerImpl _sessionManager;

    public SessionStorage(SessionManagerImpl sessionManager) {
        _sessionManager = sessionManager;
    }

    public void storeSession(SessionImpl session) {
        SessionID id = session.getCompleteSessionID().getSessionID();
        SessionReference ref = new SessionReference(session, _referenceQueue);
        SessionReference oldRef = _map.put(id, ref);
        _backwardsMap.put(ref.getReference(), session.getCompleteSessionID());
        if (oldRef != null)
            _backwardsMap.remove(oldRef.getReference());
        addToUserDeviceMap(session.getCompleteSessionID().getUserID(), id);
    }

    public void removeSession(SessionImpl session) {
        SessionReference oldRef = _map.remove(session.getCompleteSessionID().getSessionID());
        if (oldRef != null) {
            CompleteSessionID completeSessionId = _backwardsMap.remove(oldRef.getReference());
            if (completeSessionId != null)
                removeFromUserDeviceMap(completeSessionId.getUserID(), completeSessionId.getSessionID());
        }
        _sessionManager.getSyncStateCacheProvider().freeCache(session.getCompleteSessionID());
        _sessionManager.getOXAjaxAccess().logout(session);
    }

    public SessionImpl getSession(SessionID id) {
        SessionReference ref = _map.get(id);
        return (ref == null) ? null : ref.get();
    }

    public SessionID[] getActiveSessions(UserID userId) {
        synchronized (_userSessionMap) {
            return _userSessionMap.get(userId);
        }
    }

    /**
     * Store a device for a specific user. Since normally a user has only 1 device and only rarely a few (2-3), storage is optimized for
     * that case
     */
    private void addToUserDeviceMap(UserID userID, SessionID sessionID) {
        synchronized (_userSessionMap) {
            SessionID[] data = _userSessionMap.get(userID);
            if (data == null) {
                _userSessionMap.put(userID, new SessionID[] { sessionID });
            } else {
                for (int i = 0; i < data.length; i++) {
                    if (data[i].equals(sessionID))
                        return;
                }
                SessionID[] newData = new SessionID[data.length + 1];
                for (int i = 0; i < data.length; i++)
                    newData[i] = data[i];
                newData[data.length] = sessionID;
                _userSessionMap.put(userID, newData);
            }
        }
    }

    /**
     * Remove a device for a specific user. Since normally a user has only 1 device and only rarely a few (2-3), storage is optimized for
     * that case
     */
    private void removeFromUserDeviceMap(UserID userId, SessionID sessionId) {
        synchronized (_userSessionMap) {
            SessionID[] data = _userSessionMap.get(userId);
            if (data == null)
                return;
            for (int i = 0; i < data.length; i++) {
                if (data[i].equals(sessionId)) {
                    if (data.length == 1) {
                        _userSessionMap.remove(userId);
                    } else {
                        SessionID[] newData = new SessionID[data.length - 1];
                        for (int j = 0; j < newData.length; j++)
                            newData[j] = data[(j < i) ? j : j + 1];
                        _userSessionMap.put(userId, newData);
                    }
                    return;
                }
            }
        }
    }

    /**
     * Make sure that any SessionReferences that no longer point to data are removed from the map. Otherwise the map would get filled up
     * with empty SoftReferences. All SoftReferences that have been cleared by the JVM are stored in the ReferenceQueue. Normally a call to
     * this method returns instantly (first poll() returns null), only if any SoftReferences have been cleared, the associated entries in
     * the maps will be removed.
     * 
     * @return number of SessionReferences that were removed
     */
    public int removeClearedReferences() {
        int count = 0;
        for (Reference<? extends SessionImpl> ref = _referenceQueue.poll(); ref != null; ref = _referenceQueue.poll()) {
            CompleteSessionID id = _backwardsMap.remove(ref);
            if (id != null) {
                removeFromUserDeviceMap(id.getUserID(), id.getSessionID());
                _map.remove(id.getSessionID());
                _sessionManager.getSyncStateCacheProvider().freeCache(id);
                _sessionManager.getOXAjaxAccess().logout(id.getOXConnectionInformation());
                // SoftReference<SessionImpl> oldRef = _map.remove(id.getSessionID());
                // assert (ref == oldRef);
                count++;
            }
        }
        return count;
    }

    public void shutdown(OXJSONAccess ajaxAccess) {
        for (SessionReference ref : _map.values()) {
            SessionImpl session = ref.get();
            if (session != null)
                ajaxAccess.logout(session);
        }
    }

    public int getSessionCount() {
        removeClearedReferences();
        return _map.size();
    }

    public List<SessionImpl> getSessionList() {
        List<SessionImpl> list = new ArrayList<SessionImpl>();
        for (SessionReference ref : _map.values()) {
            SessionImpl session = ref.poll();
            if (session != null)
                list.add(session);
        }
        return list;
    }

    public Collection<SessionReference> getSessionReferences() {
        return _map.values();
    }
}
