/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.api.session;

import org.json.JSONObject;
import com.openexchange.usm.api.exceptions.USMException;

/**
 * Interface that can be provided to the USM system. It specifies a handler that will be called if a session is created for the first time,
 * i.e. if no persistent information is stored in the database. It will be also called on later accesses to update session-internal data
 * from any modified OX configuration and to verify that access to the session is still allowed.This handler can be used to configure the
 * session, e.g. to set which ContentTypes (and which fields) are used by the client. This handler also specifies which optional behavior of
 * the USM sync system should be enabled.
 * 
 * @author afe
 */
public interface SessionInitializer {

    /**
     * Called after a Session is created for the first time, i.e. if no persistent information is found in the database. It is guaranteed
     * that the login for the session already succeeded. The Session is internally already completely initialized, all methods may be
     * called.
     * 
     * @param session
     * @param ajaxConfiguration configuration as retrieved from the OX JSON interface
     * @throws USMException
     */
    void initializeSession(Session session, JSONObject ajaxConfiguration) throws USMException;

    /**
     * Called after a Session has been loaded from its persistent state. This method is not called if the Session is initialized for the
     * first time, in which case initializeSession is called. This method can be used to check configuration parameters that have changed
     * and update the persistent fields accordingly.<br>
     * It is guaranteed that the login for the session already succeeded. The Session is internally already completely initialized, all
     * methods may be called.
     * 
     * @param session
     * @param ajaxConfiguration
     * @throws USMException
     */
    void reinitializeSession(Session session, JSONObject ajaxConfiguration) throws USMException;

    /**
     * Called on regular intervals to make sure the access to the specific protocol is still enabled for that session.
     * 
     * @param session
     * @throws USMException
     */
    void checkAccess(Session session) throws USMException;

    /**
     * Called if the default mail folders have changed.
     * 
     * @param session
     * @param ajaxConfiguration
     * @throws USMException
     */
    void updateDefaultFolders(Session session, JSONObject ajaxConfiguration) throws USMException;

    /**
     * @return true if this protocol expects modified folders to be reported on each waitForChanges()-call. If this method returns false,
     *         the internal modified flag will be reset after the folder has been reported as modified for a few (internally configured)
     *         times
     */
    boolean allowsUnlimitedWaitForChangesOnModifiedFolders();

    /**
     * @return true if this protocols identifies objects by their UUID, false if they are identified by their OX ID. If they are identified
     *         by their UUID, the sync mechanism can implement internal mappings, e.g. if the OX-ID of standard-folders has changed.
     */
    boolean objectsSynchronizedByUUID();
}
