/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.git.conflict;

import static com.openexchange.Definitions.SPECFILE_DF;
import static com.openexchange.Definitions.SPEC_CHANGELOG_PATTERN;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import com.openexchange.data.SpecChangeLogEntry;
import com.openexchange.data.ChangeLogSorterForSpec;
import com.openexchange.git.MergeException;
import com.openexchange.git.Strings;
import com.openexchange.git.data.Conflict;

/**
 * This class tries to solve merge conflicts in RPM spec files. Currently it can only solve conflicts in the %changelog area of the spec
 * file.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class SpecChangelog implements Solver {

    public SpecChangelog() {
        super();
    }

    public boolean applies(Conflict conflict) {
        // Adding lines before or after the conflict into the conflicting area allows easier pattern matching if the conflict only occurs in
        // the changelog comment

        // Just check normal lines within conflict
        String here = Strings.append(conflict.getHere());
        String there = Strings.append(conflict.getThere());
        if (applies(here, there)) {
            return true;
        }
        // Include line before conflict
        List<String> before = conflict.getBefore();
        if (before.size() > 0) {
            // Only a single line from the context.
            before = before.subList(before.size() - 1, before.size());
            here = Strings.append(before, conflict.getHere());
            there = Strings.append(before, conflict.getThere());
            if (applies(here, there)) {
                return true;
            }
        }
        return false;
    }

    private static boolean applies(String here, String there) {
        Matcher matcher = SPEC_CHANGELOG_PATTERN.matcher(here);
        StringBuilder sb = new StringBuilder();
        while (matcher.find()) {
            sb.append(matcher.group());
            sb.append('\n');
        }
        if (!sb.toString().equals(here)) {
            return false;
        }
        sb.setLength(0);
        matcher = SPEC_CHANGELOG_PATTERN.matcher(there);
        while (matcher.find()) {
            sb.append(matcher.group());
            sb.append('\n');
        }
        if (!sb.toString().equals(there)) {
            return false;
        }
        return true;
    }

    public List<String> solve(Conflict conflict) throws MergeException {
        String here = Strings.append(conflict.getHere());
        String there = Strings.append(conflict.getThere());
        if (applies(here, there)) {
            return solve(here, there);
        }
        // Include line before conflict
        List<String> before = conflict.getBefore();
        if (before.size() > 0) {
            // Only a single line from the context.
            before = before.subList(before.size() - 1, before.size());
            here = Strings.append(before, conflict.getHere());
            there = Strings.append(before, conflict.getThere());
            if (applies(here, there)) {
                // Remove the first line from the replacement.
                List<String> replacement = solve(here, there);
                return replacement.subList(1, replacement.size());
            }
        }
        throw new MergeException();
    }

    private static List<String> solve(String here, String there) throws MergeException {
        List<SpecChangeLogEntry> entries = new ArrayList<SpecChangeLogEntry>();
        Matcher matcher = SPEC_CHANGELOG_PATTERN.matcher(here);
        while (matcher.find()) {
            final Date date;
            try {
                date = new SimpleDateFormat(SPECFILE_DF, Locale.ENGLISH).parse(matcher.group(1));
            } catch (ParseException e) {
                throw new MergeException("Unable to parse date for changelog entry: \"" + matcher.group(1) + "\"", e);
            }
            String author = matcher.group(2);
            String comment = matcher.group(3);
            SpecChangeLogEntry entry = new SpecChangeLogEntry(date, author, comment);
            entries.add(entry);
        }
        matcher = SPEC_CHANGELOG_PATTERN.matcher(there);
        while (matcher.find()) {
            final Date date;
            try {
                date = new SimpleDateFormat(SPECFILE_DF, Locale.ENGLISH).parse(matcher.group(1));
            } catch (ParseException e) {
                throw new MergeException("Unable to parse date for changelog entry: \"" + matcher.group(1) + "\"", e);
            }
            String author = matcher.group(2);
            String comment = matcher.group(3);
            SpecChangeLogEntry entry = new SpecChangeLogEntry(date, author, comment);
            entries.add(entry);
        }
        Collections.sort(entries, new ChangeLogSorterForSpec());
        List<String> replacement = new ArrayList<String>();
        for (SpecChangeLogEntry entry : entries) {
            for (String line : entry.toString().split("\n")) {
                replacement.add(line);
            }
        }
        return replacement;
    }
}
