/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.files;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.logging.Log;

import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.FileStorageFileAccess.SortDirection;
import com.openexchange.file.storage.FileStorageObjectPermission;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.ExceptionToErrorCode;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.iterator.SearchIterator;
import com.openexchange.tx.TransactionAwares;


/**
 * A collection of low-level file helper functions.
 *
 * @author Carsten Driesner
 *
 */
public class FileHelper {

    public static final String STR_EXT_PGP = "pgp";

    @SuppressWarnings("deprecation")
	private static final Log LOG = com.openexchange.log.Log.loggerFor(FileHelper.class);
    private FileHelper() {};

    /**
     * Gets the base name, minus the full path and extension, from a full
     * filename.
     *
     * @param fileName
     *  The file name where the base name should be retrieved from. The
     *  fileName can be null.
     *
     * @return
     *  The base name of the given filename or null if fileName is null.
     */
    static public String getBaseName(String fileName) {
        return getBaseName(fileName, false);
    }

    /**
     * Gets the base name, minus the full path and extension, from a full
     * filename.
     *
     * @param fileName
     *  The file name where the base name should be retrieved from. The
     *  fileName can be null.
     *
     * @param ignoreGuardExt
     *  Removes a possible guard file extension.
     *
     * @return
     *  The base name of the given filename or null if fileName is null.
     */
    static public String getBaseName(String fileName, boolean ignoreGuardExt) {
        if (ignoreGuardExt &&
            ((fileName != null) && (fileName.length() > STR_EXT_PGP.length() + 1)) &&
            (STR_EXT_PGP.equals(FileHelper.getExtension(fileName, false)))) {
            fileName = fileName.substring(0, fileName.length() - 1 - STR_EXT_PGP.length() - 1); // remove pgp extension including '.'
        }

        return FilenameUtils.getBaseName(fileName);
    }

    /**
     * Gets the file name including the extension, minus the full path, from
     * a full path filename.
     *
     * @param fullPathFileName
     *  The filename with  the full path including the extension. The
     *  fullPathFileName can be null.
     *
     * @return
     *  The file name of the given full path file name or null if the
     *  fullPathFileName is null.
     */
    static public String getFileName(String fullPathFileName) {
        return getFileName(fullPathFileName, false);
    }

    /**
     * Gets the file name including the extension, minus the full path, from
     * a full path filename.
     *
     * @param fullPathFileName
     *  The filename with  the full path including the extension. The
     *  fullPathFileName can be null.
     *
     * @param ignoreGuardExt
     *  Removes a possible guard file extension.
     *
     * @return
     *  The file name of the given full path file name or null if the
     *  fullPathFileName is null.
     */
    static public String getFileName(String fullPathFileName, boolean ignoreGuardExt) {
        String fileName  = FilenameUtils.getName(fullPathFileName);

        if (ignoreGuardExt) {
            final StringBuffer tmp = new StringBuffer(64);
            tmp.append(getBaseName(fileName, true));
            tmp.append(".");
            tmp.append(getExtension(fileName, true));
            fileName = tmp.toString();
        }

        return fileName;
    }

    /**
     * Determines, if the user associated with the provided session can
     * write to the file, checking folder/file attributes.
     *
     * @param services the service lookup instance to be used
     * @param session the session of the user
     * @param userId the user id of the user
     * @param folderId the folder id where the file is located
     * @param fileId the file to be checked for write permissions
     * @return TRUE, if the file can be written, FALSE otherwise
     */
    static public boolean canWriteToFile(final ServiceLookup servicesDEPRECATED, final Session session, final int userId, final String fileId) {
        Validate.notNull(session);
        Validate.notNull(fileId);

        boolean writeAccess = false;

        try {
            final IDBasedFileAccess fileAccess = FileHelper.getFileAccess(null, session);
            final File metaData = fileAccess.getFileMetadata(fileId, FileStorageFileAccess.CURRENT_VERSION);

            writeAccess = canWriteToFile(null, session, metaData, userId);
        } catch (Exception e) {
            LOG.error("RT connection: Exception caught while trying to determine file/folder access rights for user", e);
        }

        return writeAccess;
    }

    /**
     * Determines, if the user associated with the provided session can
     * write to the file, checking folder/file attributes.
     *
     * @param services the service lookup instance to be used
     * @param metaData the meta data of the file to be checked
     * @param userId the user id of the user
     * @return TRUE, if the file can be written, FALSE otherwise
     */
    static public boolean canWriteToFile(final ServiceLookup servicesDEPRECATED, final Session session, final File metaData, final int userId) {
        Validate.notNull(session);
        Validate.notNull(metaData);

        boolean writeAccess = false;

        try {
            final IDBasedFolderAccess folderAccess = FolderHelper.getFolderAccess(null, session);
            final int[] permissions = FolderHelper.getFolderPermissions(folderAccess, metaData.getFolderId());

            if (null != permissions) {
                int write  = permissions[FolderHelper.FOLDER_PERMISSIONS_WRITE];

                if (write >= com.openexchange.file.storage.FileStoragePermission.WRITE_ALL_OBJECTS) {
                    // permissions are sufficient to write the document
                    writeAccess = true;
                } else {
                    // We have again dig deeper and now check the creator to determine, if we
                    // have write permissions or not. This is necessary to check the WRITE_OWN_OBJECTS
                    // permission correctly.
                    writeAccess = ((write >= com.openexchange.file.storage.FileStoragePermission.WRITE_OWN_OBJECTS) &&
                                    metaData.getCreatedBy() == userId);
                }

                if (!writeAccess) {
                    // #40873 check guest permissions, which can provide write access via object
                    final List<FileStorageObjectPermission> perms = metaData.getObjectPermissions();

                    // #42905 object permissions can be null - must be checked
                    if (null != perms) {
                        // we have to check all permissions to find out that at least one has the needed permission
                        for(FileStorageObjectPermission perm : perms) {
                            if (!perm.isGroup() && (perm.getEntity() == userId) && (perm.getPermissions() >= FileStorageObjectPermission.WRITE)) {
                                // write access via a user permission
                                writeAccess = true;
                                break;
                            } else if (perm.isGroup() && (perm.getPermissions() >= FileStorageObjectPermission.WRITE)) {
                                // #52180 write access via a group permission
                                writeAccess = true;
                                break;
                            }
                        }
                    }
                }

                // check the locking attributes which can also result in no write permissions
                final Date now = new Date();
                final Date lockedUntil = metaData.getLockedUntil();
                int lockedByUserId = -1;

                if ((null != lockedUntil) && lockedUntil.after(now)) {
                    lockedByUserId = metaData.getModifiedBy();
                }
                writeAccess = ((writeAccess && (lockedByUserId == -1 )) || (writeAccess && (lockedByUserId != -1) && (userId == lockedByUserId)));
            }
        } catch (Exception e) {
            LOG.error("RT connection: Exception caught while trying to determine file/folder access rights for user", e);
        }

        return writeAccess;
    }

    /**
     * Determines, if the user associated with the provided session can
     * read the file, checking folder/file attributes.
     *
     * @param services the service lookup instance to be used
     * @param folderId the folder id where the file resides
     * @param fileId the file id that should be checked for read access rights
     * @param userId the user id of the user
     * @return TRUE, if the file can be read, FALSE otherwise
     */
    static public boolean canReadFile(final Session session, String folderId, String fileId, final int userId) {

        Validate.notNull(session);
        Validate.notNull(folderId);
        Validate.notNull(fileId);

        boolean readAccess = false;

        try {
            final IDBasedFolderAccess folderAccess = FolderHelper.getFolderAccess(null, session);
            final int[] permissions = FolderHelper.getFolderPermissions(folderAccess, folderId);
            final File metaData     = getFileMetaData(session, fileId);

            // We must have access to the meta data and be able to retrieve the folder permissions
            // otherwise this is a clear indication that no read access is available. We have to
            // check special access rights, too.
            if ((null != metaData) && (null != permissions)) {
                int read = permissions[FolderHelper.FOLDER_PERMISSIONS_READ];

                if (read >= com.openexchange.file.storage.FileStoragePermission.READ_ALL_OBJECTS) {
                    // permissions are sufficient to read the document
                    readAccess = true;
                } else {
                    // We have again dig deeper and now check the creator to determine, if we
                    // have read permissions or not. This is necessary to check the READ_OWN_OBJECTS
                    // permission correctly.
                    readAccess = ((read >= com.openexchange.file.storage.FileStoragePermission.READ_OWN_OBJECTS) &&
                                  ((metaData != null) && metaData.getCreatedBy() == userId));
                }

                if (!readAccess && (metaData != null)) {
                    // #40873 check guest permissions, which can provide read access via object
                    final List<FileStorageObjectPermission> perms = metaData.getObjectPermissions();

                    // #42905 object permissions can be null - must be checked
                    if (null != perms) {
                        // we have to check all permissions to find out that at least one has the needed permission
                        for(FileStorageObjectPermission perm : perms){
                            if (!perm.isGroup() && (perm.getEntity() == userId) && (perm.getPermissions() >= FileStorageObjectPermission.READ)) {
                                // read access via a user permission
                                readAccess = true;
                                break;
                            } else if (perm.isGroup() && (perm.getPermissions() >= FileStorageObjectPermission.READ)) {
                                // read access via a group permission
                                readAccess = true;
                                break;
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            LOG.error("RT connection: Exception caught while trying to determine file/folder access rights for user", e);
        }

        return readAccess;
    }

    /**
     * Retrieves the meta data of a file specified by the id.
     *
     * @param session
     * @param fileId
     * @return the meta data or NULL if the meta data cannot be retrieved
     *  which is normally related to missing access rights
     */
    static public File getFileMetaData(final Session session, String fileId) {
        Validate.notNull(session);
        Validate.notNull(fileId);

        File result = null;

        try
        {
            final IDBasedFileAccessFactory fileAccessFactory = ServiceLookupRegistry.get().getService(IDBasedFileAccessFactory.class);
            final IDBasedFileAccess        fileAccess        = fileAccessFactory.createAccess(session);
            result = fileAccess.getFileMetadata(fileId, FileStorageFileAccess.CURRENT_VERSION);
        }
        catch (OXException e) {
            final ErrorCode errorCode = ExceptionToErrorCode.map(e, ErrorCode.GENERAL_PERMISSION_READ_MISSING_ERROR, false);
            if (errorCode.getCode() != ErrorCode.GENERAL_PERMISSION_READ_MISSING_ERROR.getCode()) {
                LOG.error("RT connection: Unexpected exception caught trying to retrieve meta data for file : " + fileId, e);
            } else {
                LOG.debug("No read permissions detected for file : " + fileId, e);
            }
        }

        return result;
    }

    /**
     * Retrieves the extension of a file name.
     *
     * @param fileName
     *  The file name, the extension should be retrieved from.
     *
     * @return
     *  The extension of the given filename or an empty string in case of
     *  (null == fileName) or if no extension is set at all
     */
    static public String getExtension(String fileName) {
        return getExtension(fileName, false);
    }

    /**
     * Retrieves the extension of a file name.
     *
     * @param fileName
     *  The file name, the extension should be retrieved from.
     *
     * @param ignoreGuardExt
     *  Ignore the Guard pgp extension
     *
     * @return
     *  The extension of the given filename or an empty string in case of
     *  (null == fileName) or if no extension is set at all
     */
    static public String getExtension(String fileName, boolean ignoreGuardExt) {
        int index = (fileName != null) ? fileName.lastIndexOf('.') : -1;
        String ext = (index >= 0) ? fileName.substring(index + 1).toLowerCase() : "";
        if ((ignoreGuardExt == true) && ext.equals(STR_EXT_PGP)) {
            fileName = fileName.substring(0, index);
            ext = getExtension(fileName, false);
        }

        return ext;
    }

    /**
     * Creates a OX Drive file and writes the provided input stream to it.
     *
     * @param session
     *  The session of the user who requested to create a new file.
     *
     * @param fileAccess
     *  The file access instance to be used by this method. Can be null
     *  where the method creates it's own file access.
     *
     * @param file
     *  The new file instance, which is correctly initialized.
     *
     * @param inputStream
     *  The input stream of the document to be written. Attention: The stream
     *  is not closed by the method. So the caller is responsible to close
     *  the stream.
     *
     * @return
     *  The error code of the operation which is ErrorCode.NO_ERROR on success
     *  or any other code in case of an error.
     */
    public static ErrorCode createFileAndWriteStream(final ServiceLookup servicesDEPRECATED, final Session session, final IDBasedFileAccess fileAccess, final File file, final InputStream inputStream) {
        IDBasedFileAccess myFileAccess = fileAccess;
        if (null == myFileAccess) {
            final IDBasedFileAccessFactory fileFactory = ServiceLookupRegistry.get().getService(IDBasedFileAccessFactory.class);
            myFileAccess = fileFactory.createAccess(session);
        }
        return writeFileStream(myFileAccess, file, inputStream);
    }

    /**
     * Creates a OX Drive file and writes the provided input stream to the file.
     *
     * @param fileAccess
     *  The file access instance to be used by this method. Must NOT be null.
     *
     * @param folder_id
     *  The folder id where the file should be created.
     *
     * @param fileName
     *  The file name of the new file.
     *
     * @param mimeType
     *  The mime type of the new file.
     *
     * @param inputStream
     *  The input stream of the document to be written. Attention: The stream
     *  is not closed by the method. So the caller is responsible to close
     *  the stream.
     *
     * @return
     *  The error code of the operation which is ErrorCode.NO_ERROR on success
     *  or any other code in case of an error.
     */
    public static ErrorCode createFileWithStream(final IDBasedFileAccess fileAccess, final String folder_id, final String fileName, final String mimeType, final InputStream inputStream) {
        // create default file for creating new file
        final DefaultFile metaData = new DefaultFile();

        // set new file attributes - using old file as template
        metaData.setId(FileStorageFileAccess.NEW);
        metaData.setFolderId(folder_id);
        metaData.setFileName(fileName);
        metaData.setFileMIMEType(mimeType);

        return writeFileStream(fileAccess, metaData, inputStream);
    }

    /**
     * Creates a new file and writes the provided input stream, while optional
     * taking over some meta data from a present file.
     *
     * @param fileAccess
     *  The file access instance to be used by this method. Must NOT be null.
     *
     * @param fileName
     *  The file name of the new file.
     *
     * @param sourceFile
     *  The source file which is used to take over the meta data from. The following
     *  meta data is copied: createdBy, modifiedBy, created, lastModified, mimeType.
     *
     * @param inputStream
     *  The input stream of the document to be written. Attention: The stream
     *  is not closed by the method. So the caller is responsible to close
     *  the stream.
     *
     * @return
     *  The error code of the operation which is ErrorCode.NO_ERROR on success
     *  or any other code in case of an error.
     */
    public static ErrorCode createFileCopyWithStream(final IDBasedFileAccess fileAccess, final String folder_id, final String fileName, final File sourceFile, final InputStream inputStream, boolean copyMetaData) {
        // create default file for creating new file
        final DefaultFile metaData = new DefaultFile();

        // set new file attributes - using old file as template
        metaData.setId(FileStorageFileAccess.NEW);
        metaData.setFolderId(folder_id);
        metaData.setFileName(fileName);
        metaData.setFileMIMEType(sourceFile.getFileMIMEType());
        // take over modification, creator and dates from the source file
        if (copyMetaData) {
            metaData.setCreatedBy(sourceFile.getCreatedBy());
            metaData.setModifiedBy(sourceFile.getModifiedBy());
            metaData.setCreated(sourceFile.getCreated());
            metaData.setLastModified(sourceFile.getLastModified());
        }

        return writeFileStream(fileAccess, metaData, inputStream);
    }

    /**
     * Writes the provided stream to the target file and optional copies
     * given meta data to the file.
     *
     * @param fileAccess
     *  The file access instance to be used by this method. Must NOT be null.
     *
     * @param file
     *  Specifies the file that should be written to.
     *
     * @param inputStream
     *  The input stream of the document to be written. Attention: The stream
     *  is not closed by the method. So the caller is responsible to close
     *  the stream.
     *
     * @return
     *  The error code of the operation which is ErrorCode.NO_ERROR on success
     *  or any other code in case of an error.
     */
    public static ErrorCode writeStreamToFile(final IDBasedFileAccess fileAccess, final File targetFile, final InputStream inputStream, final File copyMetaData) {
        final DefaultFile metaData = new DefaultFile();

        // set necessary meta data to overwrite an existing file
        metaData.setId(targetFile.getId());
        metaData.setFolderId(targetFile.getFolderId());
        metaData.setFileName(targetFile.getFileName());
        metaData.setFileMIMEType(targetFile.getFileMIMEType());
        // take over modification, creator and dates from the source file
        if (null != copyMetaData) {
            metaData.setCreatedBy(copyMetaData.getCreatedBy());
            metaData.setModifiedBy(copyMetaData.getModifiedBy());
            metaData.setCreated(copyMetaData.getCreated());
            metaData.setLastModified(copyMetaData.getLastModified());
        }

        return writeFileStream(fileAccess, metaData, inputStream);
    }

    /**
     * Retrieves the meta data from a file specified by the folder and
     * file name.
     *
     * @param fileAccess
     *  The file access instance to be used by this method. Must NOT be null.
     *
     * @param folder_id
     *  The folder where the file is located retrieving the meta data from. Must NOT be null.
     *
     * @param fileName
     *  The file name of the file retrieving the meta data from. Must NOT be null.
     *
     * @return
     *  A File reference or null if no file could be found.
     *
     * @throws OXException
     */
    public static File getMetaDataFromFileName(final IDBasedFileAccess fileAccess, final String folder_id, final String fileName) throws OXException {
        final List<Field> fields = new ArrayList<Field>();
        fields.add(Field.ID);
        fields.add(Field.FOLDER_ID);
        fields.add(Field.FILENAME);
        fields.add(Field.FILE_MIMETYPE);
        fields.add(Field.VERSION);

        return getMetaDataFromFileName(fileAccess, folder_id, fileName, fields);
    }

    /**
     * Retrieves the meta data from a file specified by the folder and
     * file name.
     *
     * @param fileAccess
     *  The file access instance to be used by this method. Must NOT be null.
     *
     * @param folder_id
     *  The folder where the file is located retrieving the meta data from. Must NOT be null.
     *
     * @param fileName
     *  The file name of the file retrieving the meta data from. Must NOT be null.
     *
     * @param fields
     *  A list of fields which should be retrieved from the file.
     *
     * @return
     *  A File reference or null if no file could be found.
     *
     * @throws OXException
     */
    public static File getMetaDataFromFileName(final IDBasedFileAccess fileAccess, final String folder_id, final String fileName, final List<Field> fields) throws OXException {
        File result = null;

        final long time1 = System.currentTimeMillis();
        final SearchIterator<File> files = fileAccess.search(fileName, fields, folder_id, null, SortDirection.DEFAULT, FileStorageFileAccess.NOT_SET, FileStorageFileAccess.NOT_SET);

        while (files.hasNext()) {
            final File file = files.next();
            if (hasSameFileName(file, fileName)) {
                result = file;
                break;
            }
        }

        LOG.trace("RT connection: TIME to get meta data from file = " + (System.currentTimeMillis() - time1) + "ms");

        return result;
    }

    /**
     *
     * @param file
     *  A file reference which should be checked with the provided file name.
     * @param fileName
     *  The file name to be checked. Must NOT be null.
     *
     * @return
     */
    private static boolean hasSameFileName(final File file, final String fileName) {
        return (file != null) ? (file.getFileName().equals(fileName)) : false;
    }

    /**
     * Checks, if the given file exists or not.
     *
     * @param fileAccess
     *  The file access instance to be used by this method. Must NOT be null.
     *
     * @param folder_id
     * @param fileName
     * @return
     * @throws OXException
     */
    public static boolean fileNameExists(final IDBasedFileAccess fileAccess, final String folder_id, final String fileName) throws OXException {
        final List<Field> fields = new ArrayList<Field>();
        fields.add(Field.ID);
        fields.add(Field.FILENAME);

        final long time1 = System.currentTimeMillis();
        SearchIterator<File> iter = fileAccess.search(fileName, fields, folder_id, null, SortDirection.DEFAULT, -1, -1);
        LOG.trace("RT connection: TIME for file name exists = " + (System.currentTimeMillis() - time1) + "ms");
        return (iter != null) && (!iter.hasNext());
    }

    /**
     * Create a file name with postfix string part.
     *
     * @param fileName
     *  A file name with extension and without path part.
     *
     * @param postFix
     *  A text part which should be appended to the base file name.
     *
     * @return
     *  A file name with consists of: <filename><postFix>.<ext>
     */
    public static String createFilenameWithPostfix(final String fileName, final String postFix) {
        return createFilenameWithPostfix(fileName, postFix, false);
    }

    /**
     * Create a file name with postfix string part.
     *
     * @param fileName
     *  A file name with extension and without path part.
     *
     * @param postFix
     *  A text part which should be appended to the base file name.
     *
     * @param ignoreGuardExt
     *  Ignore the Guard pgp extension
     *
     * @return
     *  A file name with consists of: <filename><postFix>.<ext>
     */
    public static String createFilenameWithPostfix(final String fileName, final String postFix, boolean ignoreGuardExt) {
        final StringBuffer tmp = new StringBuffer(64);
        tmp.append(FileHelper.getBaseName(fileName, ignoreGuardExt));
        tmp.append(postFix);
        tmp.append(".");
        tmp.append(getExtension(fileName, ignoreGuardExt));
        return tmp.toString();
    }

    /**
     * Writes the stream to the provided file.
     *
     * @param fileAccess
     *   The FileAccess instance writing the file.
     *
     * @param newFile
     *   The new meta file instance. Must be filled with the necessary
     *   properties.
     *
     * @param inputStream
     *   The data to be written.
     *
     * @return
     */
    private static ErrorCode writeFileStream(final IDBasedFileAccess fileAccess, final File fileMetaData, final InputStream inputStream) {
        ErrorCode errorCode = ErrorCode.NO_ERROR;
        boolean rollback = true;

        final long time1 = System.currentTimeMillis();
        try {
            LOG.trace("RT connection: [writeFileStream] saveDocument using meta data: " + ((null != fileMetaData) ? fileMetaData.toString() : "null"));
            fileAccess.startTransaction();
            rollback = true;
            fileAccess.saveDocument(fileMetaData, inputStream, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
            fileAccess.commit();
            rollback = false;
        } catch (final Exception e) {
            errorCode = ErrorCode.GENERAL_PERMISSION_CREATE_MISSING_ERROR;
            if (e instanceof OXException) {
                errorCode = ExceptionToErrorCode.map((OXException)e, errorCode, false);
            }
            LOG.error(errorCode.getDescription(), e);
        } finally {
            // Roll-back (if needed) and finish
            if (rollback) {
                TransactionAwares.rollbackSafe(fileAccess);
            }

            TransactionAwares.finishSafe(fileAccess);
        }
        LOG.trace("RT connection: TIME to write file stream = " + (System.currentTimeMillis() - time1) + "ms");

        return errorCode;
    }

    /**
     * Creates a file access using the provided services and session.
     *
     * @param services
     *  The service lookup to used to create the file access.
     *
     * @param session
     *  The session of a client to be used to create the file access.
     *
     * @return
     *  A file access associated with the provided session or null.
     *
     * @throws OXException
     *  If the request cannot be successfully processed an exception is thrown
     *  which describes the cause.
     */
    public static IDBasedFileAccess getFileAccess(ServiceLookup servicesDEPRECATED, Session session) throws OXException {
        IDBasedFileAccessFactory factory = ServiceLookupRegistry.get().getService(IDBasedFileAccessFactory.class);
        return factory.createAccess(session);
    }

    /**
     * Tries to delete a file, depending on the parameter the file is moved
     * to the trash or completely deleted.
     *
     * @param fileAccess the file access instance to be used for the delete operation
     * @param fileId the file id of the file to be delete
     * @param hardDelete if true the file will be deleted without putting it into the trash
     *
     * @return
     *  The error code of the operation. In case of success, ErrorCode.NO_ERROR.
     */
    public static ErrorCode deleteFile(final IDBasedFileAccess fileAccess, final String fileId, boolean hardDelete) {
        ErrorCode errorCode = ErrorCode.NO_ERROR;

        if (null != fileAccess) {
            try {
                final List<String> ids = new ArrayList<String>();

                ids.add(fileId);
                fileAccess.removeDocument(ids, FileStorageFileAccess.DISTANT_FUTURE, hardDelete);
            } catch (OXException e) {
                errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
                errorCode = ExceptionToErrorCode.map(e, errorCode, false);
            }
        }

        return errorCode;
    }
}
