/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.doc;

import java.util.HashMap;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Helper class to provide certain document based mime types and
 * function to determine mime-types from file extensions.
 *
 * @author Carsten Driesner
 *
 */
public class MimeTypeHelper {
    private static AtomicReference<HashMap<String, String>> m_mapTemplateExtensionsToMimeType = new AtomicReference<HashMap<String, String>>();
    private static AtomicReference<HashMap<String, DocumentType>> m_mapTemplateMimeTypeToDocumentType = new AtomicReference<HashMap<String, DocumentType>>();
    private static AtomicReference<HashMap<String, String>> m_mapExtensionsToMimeType = new AtomicReference<HashMap<String, String>>();
    private static AtomicReference<HashMap<String, DocumentType>> m_mapMimeTypeToDocumentType = new AtomicReference<HashMap<String, DocumentType>>();

    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";
    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_MACRO = "application/vnd.ms-excel.sheet.macroEnabled.12";
    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE = "application/vnd.openxmlformats-officedocument.spreadsheetml.template";
    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO = "application/vnd.ms-excel.template.macroEnabled.12";

    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION = "application/vnd.openxmlformats-officedocument.presentationml.presentation";
    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_MACRO = "application/vnd.ms-powerpoint.presentation.macroEnabled.12";
    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE = "application/vnd.openxmlformats-officedocument.presentationml.template";
    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE_MACRO = "application/vnd.ms-powerpoint.presentation.macroEnabled.12";

    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING = "application/vnd.openxmlformats-officedocument.wordprocessingml.document";
    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_MACRO = "application/vnd.ms-word.document.macroEnabled.12";
    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE = "application/vnd.openxmlformats-officedocument.wordprocessingml.template";
    public static final String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE_MACRO = "application/vnd.ms-word.template.macroEnabled.12";

    public static final String MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT = "application/vnd.oasis.opendocument.text";
    public static final String MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE = "application/vnd.oasis.opendocument.text-template";

    public static final String MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET = "application/vnd.oasis.opendocument.spreadsheet";
    public static final String MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE = "application/vnd.oasis.opendocument.spreadsheet-template";

    public static final String MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION = "application/vnd.oasis.opendocument.presentation";
    public static final String MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION_TEMPLATE = "application/vnd.oasis.opendocument.presentation-template";

    public static final String MIMETYPE_MSEXCEL = "application/vnd.ms-excel";
    public static final String MIMETYPE_MSPOWERPOINT = "application/vnd.ms-powerpoint";
    public static final String MIMETYPE_MSWORD = "application/msword";

    // template extensions
    public static final String EXCEL_TEMPLATE_EXTENSIONS[] = { "xltx", "xltm", "xlt" };
    public static final String WORD_TEMPLATE_EXTENSIONS[] = { "dotx", "dotm", "dot" };
    public static final String POWERPOINT_TEMPLATE_EXTENSIONS[] = { "potx", "pptm", "pot" };
    public static final String ODF_CALC_TEMPLATE_EXTENSIONS[] = { "ots" };
    public static final String ODF_WRITER_TEMPLATE_EXTENSIONS[] = { "ott" };
    public static final String ODF_IMPRESS_TEMPLATE_EXTENSIONS[] = { "otp" };
    // document extensions
    public static final String EXCEL_DOC_EXTENSIONS[] = { "xlsx", "xlsm", "xls" };
    public static final String WORD_DOC_EXTENSIONS[] = { "docx", "docm", "doc" };
    public static final String POWERPOINT_DOC_EXTENSIONS[] = { "pptx", "pptm", "ppt" };
    public static final String ODF_CALC_DOC_EXTENSIONS[] = { "ods" };
    public static final String ODF_WRITER_DOC_EXTENSIONS[] = { "odt" };
    public static final String ODF_IMPRESS_DOC_EXTENSIONS[] = { "odp" };
    
    
    public static final String SPREADSHEET_EXTENSIONS[] = { "xlsx", "xlsm", "xltx", "xltm", "ods", "ots" };
    public static final String TEXT_EXTENSIONS[] = { "docx", "docm", "dotx", "dotm", "odt", "ott" };
    public static final String PRESENTATION_EXTENSIONS[] = { "pptx", "pptm", "potx", "potm", "odp", "otp" };

    /**
     * Provides the mime type of a known template file extension.
     *
     * @param templateExt
     *  A template document file extension where the mime type should be
     *  retrieved for.
     *
     * @return
     *  The mime type for the template document file extensions or null
     *  if the extension is unknown, null or not a template document extension.
     */
    public static String getMimeTypeFromTemplateExtension(String templateExt) {
    	return getStaticMimeTypeTemplateMap().get(templateExt);
    }

    /**
     * Provides the document type for a provided template mime type.
     *
     * @param templateMimeType
     *  A template document mime type.
     *
     * @return
     *  The document type or null if the mime type couldn't be recognized as
     *  a supported document template type.
     */
    public static DocumentType getDocumentTypeFromTemplateMimeType(String templateMimeType) {
        return getStaticDocumentTypeTemplateMap().get(templateMimeType);
    }

    /**
     * Provides the mime type of a known template file extension.
     *
     * @param templateExt
     *  A template document file extension where the mime type should be
     *  retrieved for.
     *
     * @return
     *  The mime type for the template document file extensions or null
     *  if the extension is unknown, null or not a template document extension.
     */
    public static String getMimeTypeFromExtension(String templateExt) {
    	return getStaticMimeTypeMap().get(templateExt);
    }

    /**
     * Provides the document type for a provided template mime type.
     *
     * @param templateMimeType
     *  A template document mime type.
     *
     * @return
     *  The document type or null if the mime type couldn't be recognized as
     *  a supported document template type.
     */
    public static DocumentType getDocumentTypeFromMimeType(String templateMimeType) {
        return getStaticDocumentTypeMap().get(templateMimeType);
    }

    // ------------------------------------------------------------------------
    // Templates
    // ------------------------------------------------------------------------
    
    /**
     * Retrieves the initialized static template mime type map to speed up the search.
     */
    private static HashMap<String, String> getStaticMimeTypeTemplateMap() {
    	HashMap<String, String> result = m_mapTemplateExtensionsToMimeType.get();

        if (result == null) {
        	final HashMap<String, String> mapExtensionsToMimeType = new HashMap<String, String>();

            // Excel
        	mapExtensionsToMimeType.put(EXCEL_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE);
        	mapExtensionsToMimeType.put(EXCEL_TEMPLATE_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO);
        	mapExtensionsToMimeType.put(EXCEL_TEMPLATE_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSEXCEL);

        	// Powerpoint
        	mapExtensionsToMimeType.put(POWERPOINT_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE);
        	mapExtensionsToMimeType.put(POWERPOINT_TEMPLATE_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE_MACRO);
        	mapExtensionsToMimeType.put(POWERPOINT_TEMPLATE_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSPOWERPOINT);

        	// Word
        	mapExtensionsToMimeType.put(WORD_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE);
        	mapExtensionsToMimeType.put(WORD_TEMPLATE_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE_MACRO);
        	mapExtensionsToMimeType.put(WORD_TEMPLATE_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSWORD);

        	// Calc
        	mapExtensionsToMimeType.put(ODF_CALC_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE);
            // Impress
        	mapExtensionsToMimeType.put(ODF_IMPRESS_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION_TEMPLATE);
            // Writer
        	mapExtensionsToMimeType.put(ODF_WRITER_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE);

        	m_mapTemplateExtensionsToMimeType.compareAndSet(null, mapExtensionsToMimeType);
        	result = mapExtensionsToMimeType;
        }
        
        return result;
    }

    /**
     * Retrieves the initialized static template document type map to speed up the search.
     */
    private static HashMap<String, DocumentType> getStaticDocumentTypeTemplateMap() {
    	HashMap<String, DocumentType> result = m_mapTemplateMimeTypeToDocumentType.get();

        if (result == null) {
        	final HashMap<String, DocumentType> mapMimeTypeToDocumentType = new HashMap<String, DocumentType>();

            // Excel/Calc
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE, DocumentType.SPREADSHEET);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO, DocumentType.SPREADSHEET);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_MSEXCEL, DocumentType.SPREADSHEET);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE, DocumentType.SPREADSHEET);

        	// Powerpoint/Impress
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE, DocumentType.PRESENTATION);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE_MACRO, DocumentType.PRESENTATION);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_MSPOWERPOINT, DocumentType.PRESENTATION);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION_TEMPLATE, DocumentType.PRESENTATION);

        	// Word/Writer
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE, DocumentType.TEXT);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE_MACRO, DocumentType.TEXT);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_MSWORD, DocumentType.TEXT);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE, DocumentType.TEXT);

        	m_mapTemplateMimeTypeToDocumentType.compareAndSet(null, mapMimeTypeToDocumentType);
        	result = mapMimeTypeToDocumentType;
        }

        return result;
    }

    // ------------------------------------------------------------------------
    // Documents & Templates
    // ------------------------------------------------------------------------
    
    /**
     * Retrieves the initialized static mime type map to speed up the search.
     */
    private static HashMap<String, String> getStaticMimeTypeMap() {
    	HashMap<String, String> result = m_mapExtensionsToMimeType.get();
    	
        if (result == null) {
        	final HashMap<String, String> mapExtensionsToMimeType = new HashMap<String, String>();

            // Excel
        	mapExtensionsToMimeType.put(EXCEL_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE);
        	mapExtensionsToMimeType.put(EXCEL_TEMPLATE_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO);
        	mapExtensionsToMimeType.put(EXCEL_TEMPLATE_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSEXCEL);
        	mapExtensionsToMimeType.put(EXCEL_DOC_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET);
        	mapExtensionsToMimeType.put(EXCEL_DOC_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_MACRO);
        	mapExtensionsToMimeType.put(EXCEL_DOC_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSEXCEL);

            // Powerpoint
        	mapExtensionsToMimeType.put(POWERPOINT_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE);
        	mapExtensionsToMimeType.put(POWERPOINT_TEMPLATE_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE_MACRO);
        	mapExtensionsToMimeType.put(POWERPOINT_TEMPLATE_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSPOWERPOINT);
        	mapExtensionsToMimeType.put(POWERPOINT_DOC_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION);
        	mapExtensionsToMimeType.put(POWERPOINT_DOC_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_MACRO);
        	mapExtensionsToMimeType.put(POWERPOINT_DOC_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSPOWERPOINT);

            // Word
        	mapExtensionsToMimeType.put(WORD_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE);
        	mapExtensionsToMimeType.put(WORD_TEMPLATE_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE_MACRO);
        	mapExtensionsToMimeType.put(WORD_TEMPLATE_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSWORD);
        	mapExtensionsToMimeType.put(WORD_DOC_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING);
        	mapExtensionsToMimeType.put(WORD_DOC_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_MACRO);
        	mapExtensionsToMimeType.put(WORD_DOC_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSWORD);

            // Calc
        	mapExtensionsToMimeType.put(ODF_CALC_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE);
        	mapExtensionsToMimeType.put(ODF_CALC_DOC_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET);
            // Impress
        	mapExtensionsToMimeType.put(ODF_IMPRESS_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION_TEMPLATE);
        	mapExtensionsToMimeType.put(ODF_IMPRESS_DOC_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION);
            // Writer
        	mapExtensionsToMimeType.put(ODF_WRITER_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE);
        	mapExtensionsToMimeType.put(ODF_WRITER_DOC_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT);

        	m_mapTemplateExtensionsToMimeType.compareAndSet(null, mapExtensionsToMimeType);
        	result = mapExtensionsToMimeType;
        }
        
        return result;
    }

    /**
     * Retrieves the initialized static document type map to speed up the search.
     */
    private static HashMap<String, DocumentType> getStaticDocumentTypeMap() {
    	HashMap<String, DocumentType> result = m_mapMimeTypeToDocumentType.get();

    	if (result == null) {
        	final HashMap<String, DocumentType> mapMimeTypeToDocumentType = new HashMap<String, DocumentType>();
    		
            // Excel
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET, DocumentType.SPREADSHEET);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_MACRO, DocumentType.SPREADSHEET);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE, DocumentType.SPREADSHEET);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO, DocumentType.SPREADSHEET);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_MSEXCEL, DocumentType.SPREADSHEET);
        	// Calc
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET, DocumentType.SPREADSHEET);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE, DocumentType.SPREADSHEET);

            // Powerpoint
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION, DocumentType.PRESENTATION);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_MACRO, DocumentType.PRESENTATION);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE, DocumentType.PRESENTATION);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE_MACRO, DocumentType.PRESENTATION);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_MSPOWERPOINT, DocumentType.PRESENTATION);
        	// Impress
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION, DocumentType.PRESENTATION);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION_TEMPLATE, DocumentType.PRESENTATION);

            // Word
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING, DocumentType.TEXT);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_MACRO, DocumentType.TEXT);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE, DocumentType.TEXT);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE_MACRO, DocumentType.TEXT);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_MSWORD, DocumentType.TEXT);
        	// Writer
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT, DocumentType.TEXT);
        	mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE, DocumentType.TEXT);

            m_mapMimeTypeToDocumentType.compareAndSet(null, mapMimeTypeToDocumentType);
            result = mapMimeTypeToDocumentType;
    	}

        return result;
    }
}
