/*
/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.realtime.impl;

import java.lang.ref.WeakReference;
import java.util.Date;
import java.util.concurrent.atomic.AtomicLong;

import com.openexchange.realtime.packet.ID;
import com.openexchange.session.Session;

public class AliveTimeoutRunnable  implements Runnable {

       // - Members ----------------------------------------------------------------
       static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(AliveTimeoutRunnable.class);
       private final WeakReference<DocumentConnection> m_connection;
       private final ID m_oldEditorID;
       private final Session m_session;
       private final AtomicLong m_nStartTime = new AtomicLong();
       private final long m_nMaxTimeout;

       /**
        * Initializes a new instance of a CompleteEditRightsTimeoutRunnable.
        *
        * @param connection The reference to the Connection instance that needs the time-out mechanism.
        * @param session The session of the client that loses the edit rights.
        * @param newEditorId The ID of the client that receives the edit rights.
        */
        public AliveTimeoutRunnable(final DocumentConnection connection, final Session session, final ID fromId, final long nMaxTimeout) {
           super();
           m_connection = new WeakReference<DocumentConnection>(connection);
           m_session = session;
           m_oldEditorID = fromId;
           m_nMaxTimeout = nMaxTimeout;
           m_nStartTime.set(new Date().getTime());

           LOG.debug("RT connection: Alive timeout runnable for client: " + fromId.toString() + ", max timeout: " + nMaxTimeout + "ms");
       }

       /**
        * Resets the time without any keep alive message.
        */
       public void resetTime() {
           m_nStartTime.set(new Date().getTime());
       }

       /**
        * Provides the current time span that elapsed without an
        * alive message received from the client.
        *
        * @return current time span in milliseconds.
        */
       public long getTimeSpanWithoutAliveMsg() {
           return (new Date().getTime() - m_nStartTime.get());
       }

       /**
        * The method that gets called by the timer service. It calls an internal method
        * of the Connection instance to handle a timeout case.
        */
       @Override
       public void run() {
           final long nElapsedTime = (new Date().getTime() - m_nStartTime.get());

           if (nElapsedTime > m_nMaxTimeout) {
               LOG.debug("RT connection: Max time-out reached for missing alive msgs while switching edit right. Client: " + m_oldEditorID.toString() + ", elapsed time w/o alive msgs: " + nElapsedTime);

               final DocumentConnection docConnection = m_connection.get();
               if (docConnection != null) {
                   docConnection.canLoseEditRights(m_session, m_oldEditorID, 0, Timeout.TM_NO_ALIVE_DURING_SWITCH_EDITRIGHTS);
               }
           }
       }
}
