/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.drawingml;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import org.docx4j.dml.CTAdjPoint2D;
import org.docx4j.dml.CTCustomGeometry2D;
import org.docx4j.dml.CTGeomGuide;
import org.docx4j.dml.CTGeomGuideList;
import org.docx4j.dml.CTGeomRect;
import org.docx4j.dml.CTPath2D;
import org.docx4j.dml.CTPath2DArcTo;
import org.docx4j.dml.CTPath2DClose;
import org.docx4j.dml.CTPath2DCubicBezierTo;
import org.docx4j.dml.CTPath2DLineTo;
import org.docx4j.dml.CTPath2DList;
import org.docx4j.dml.CTPath2DMoveTo;
import org.docx4j.dml.CTPath2DQuadBezierTo;
import org.docx4j.dml.CTPresetGeometry2D;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.IAvList;
import org.docx4j.dml.STPathFillMode;
import org.docx4j.dml.STShapeType;
import org.docx4j.jaxb.Context;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

final public class DMLGeometry {

	public static void applyGeometryPropertiesFromJson(JSONObject geometryAttrs, CTShapeProperties shapeProperties) throws JSONException {
		final String presetShape = geometryAttrs.optString("presetShape", null);
		if(presetShape!=null) {
			shapeProperties.setCustGeom(null);
			shapeProperties.getPrstGeom(true).setPrst(STShapeType.fromValue(presetShape));
			applyPresetGeometryPropertiesFromJson(geometryAttrs, shapeProperties.getPrstGeom(false));
		}
		else {
			shapeProperties.setPrstGeom(null);
			applyCustomGeometryPropertiesFromJson(geometryAttrs, shapeProperties.getCustGeom(true));
		}
	}

	public static void applyPresetGeometryPropertiesFromJson(JSONObject geometryAttrs, CTPresetGeometry2D presetGeometry) {
		final Object avList = geometryAttrs.opt("avList");
		if(avList!=null) {
			applyAvListFromJson(avList, presetGeometry);
		}
	}

	public static void applyCustomGeometryPropertiesFromJson(JSONObject geometryAttrs, CTCustomGeometry2D customGeometry) throws JSONException {
		final Object avList = geometryAttrs.opt("avList");
		if(avList!=null) {
			applyAvListFromJson(avList, customGeometry);
		}
		final Object gdList = geometryAttrs.opt("gdList");
		if(gdList!=null) {
		    applyGuideListFromJson(gdList, customGeometry);
		}
		final Object pathList = geometryAttrs.opt("pathList");
		if(pathList!=null) {
		    applyPathListFromJson(pathList, customGeometry);
		}
		final Object textRect = geometryAttrs.opt("textRect");
		if(textRect!=null) {
		    applyTextRectFromJson(textRect, customGeometry);
		}
	}

	public static void applyAvListFromJson(Object jsonObject, IAvList avList) {
		if(jsonObject instanceof JSONObject) {
			CTGeomGuideList geometryList = avList.getAvLst();
			if(geometryList==null) {
				geometryList = Context.getDmlObjectFactory().createCTGeomGuideList();
				avList.setAvLst(geometryList);
			}
			final List<CTGeomGuide> gdList = geometryList.getGd();
			final Iterator<Entry<String, Object>> avValueIter = ((JSONObject)jsonObject).entrySet().iterator();
			while(avValueIter.hasNext()) {
				final Entry<String, Object> avValueEntry = avValueIter.next();
				final String avName = avValueEntry.getKey();
				final Object avValue = avValueEntry.getValue();

				int i;
				for(i = 0; i < gdList.size(); i++) {
					final CTGeomGuide geomGuide = gdList.get(i);
					if(geomGuide.getName().equals(avName)) {
						if(avValue instanceof Number) {
							geomGuide.setFmla("val " + Long.valueOf(((Number)avValue).longValue()).toString());
						}
						else if(avValue == JSONObject.NULL) {
							gdList.remove(i);
						}
						break;
					}
				}
				if(i == gdList.size()) {
					if(avValue instanceof Number) {
						final CTGeomGuide geomGuide = Context.getDmlObjectFactory().createCTGeomGuide();
						geomGuide.setName(avName);
						geomGuide.setFmla("val " + Long.valueOf(((Number)avValue).longValue()).toString());
						gdList.add(geomGuide);
					}
				}
			}
		}
		else if(jsonObject == JSONObject.NULL) {
			avList.setAvLst(null);
		}
	}

    public static void applyGuideListFromJson(Object json, CTCustomGeometry2D customGeometry) throws JSONException {
        if(json instanceof JSONArray) {
            final List<CTGeomGuide> geomGuideList = new ArrayList<CTGeomGuide>(((JSONArray)json).length());
            for(int i=0; i<((JSONArray)json).length(); i++) {
                final CTGeomGuide geomGuide = new CTGeomGuide();
                final JSONObject o = ((JSONArray)json).getJSONObject(i);
                geomGuide.setName(o.getString("name"));
                final StringBuffer buffer = new StringBuffer();
                buffer.append(o.getString("op"));
                final Object p0 = o.get("p0");
                if(p0!=null) {
                    buffer.append(' ');
                    buffer.append(p0);
                    final Object p1 = o.get("p1");
                    if(p1!=null) {
                        buffer.append(' ');
                        buffer.append(p1);
                        final Object p2 = o.get("p2");
                        if(p2!=null) {
                            buffer.append(' ');
                            buffer.append(p2);
                        }
                    }
                }
                geomGuide.setFmla(buffer.toString());
                geomGuideList.add(geomGuide);
            }
            customGeometry.setGdLst(new CTGeomGuideList(geomGuideList));
        }
        else if(json == JSONObject.NULL) {
            customGeometry.setGdLst(null);
        }
    }

    public static void applyPathListFromJson(Object json, CTCustomGeometry2D customGeometry) throws JSONException {
        if(json instanceof JSONArray) {
            final List<CTPath2D> path2DList = new ArrayList<CTPath2D>(((JSONArray)json).length());
            for(int i=0; i<((JSONArray)json).length(); i++) {
                final JSONObject o = ((JSONArray)json).getJSONObject(i);
                final CTPath2D path2D = new CTPath2D();
                path2DList.add(path2D);
                final String fillMode = o.optString("fillMode");
                if(fillMode!=null) {
                	try {
                		path2D.setFill(STPathFillMode.fromValue(fillMode));
                	}
                	catch(IllegalArgumentException e) {
                		//
                	}
                }
                final Object width = o.opt("width");
                if(width instanceof Number) {
                	path2D.setW(((Number)width).longValue());
                }
                final Object height = o.opt("height");
                if(height instanceof Number) {
                	path2D.setH(((Number)height).longValue());
                }
                final boolean isStroke = o.optBoolean("isStroke", true);
                if(!isStroke) {
                	path2D.setStroke(false);
                }
                final boolean isExtrusionOk = o.optBoolean("isExtrusionOk", true);
    			if(!isExtrusionOk) {
    				path2D.setExtrusionOk(false);
    			}
    			final JSONArray commands = o.optJSONArray("commands");
    			if(commands!=null) {
    				final List<Object> cList = path2D.getCloseOrMoveToOrLnTo();
    				for(Object c:commands) {
    					if(c instanceof JSONObject) {
    						final JSONObject cObj = (JSONObject)c;
    						final String command = cObj.optString("c");
    						if(command!=null) {
    							switch(command) {
    								case "lineTo" : {
    									final CTPath2DLineTo path2DLineTo = new CTPath2DLineTo();
    									final CTAdjPoint2D point2D = new CTAdjPoint2D();
    									path2DLineTo.setPt(point2D);
    									point2D.setX(cObj.get("x").toString());
    									point2D.setY(cObj.get("y").toString());
    									cList.add(path2DLineTo);
    									break;
    								}
    								case "arcTo" : {
    									final CTPath2DArcTo path2DArcTo = new CTPath2DArcTo();
    									path2DArcTo.setHR(cObj.get("hr").toString());
    									path2DArcTo.setWR(cObj.get("wr").toString());
    									path2DArcTo.setStAng(cObj.get("stAng").toString());
    									path2DArcTo.setSwAng(cObj.get("swAng").toString());
    									cList.add(path2DArcTo);
    									break;
    								}
    								case "moveTo" : {
    									final CTPath2DMoveTo path2DMoveTo = new CTPath2DMoveTo();
    									final CTAdjPoint2D point2D = new CTAdjPoint2D();
    									path2DMoveTo.setPt(point2D);
    									point2D.setX(cObj.get("x").toString());
    									point2D.setY(cObj.get("y").toString());
    									cList.add(path2DMoveTo);
    									break;
    								}
    								case "close" : {
    									cList.add(new CTPath2DClose());
    									break;
    								}
    								case "quadBezierTo" : {
    									final CTPath2DQuadBezierTo path2DQuadBezierTo = new CTPath2DQuadBezierTo();
    									final JSONArray pts = cObj.getJSONArray("pts");
    									final List<CTAdjPoint2D> ptList = path2DQuadBezierTo.getPt();
    									for(Object pt:pts) {
        									final CTAdjPoint2D point2D = new CTAdjPoint2D();
        									point2D.setX(((JSONObject)pt).get("x").toString());
        									point2D.setY(((JSONObject)pt).get("y").toString());
        									ptList.add(point2D);
    									}
    									cList.add(path2DQuadBezierTo);
    									break;
    								}
    								case "cubicBezierTo" : {
    									final CTPath2DCubicBezierTo path2DCubicBezierTo = new CTPath2DCubicBezierTo();
    									final JSONArray pts = cObj.getJSONArray("pts");
    									final List<CTAdjPoint2D> ptList = path2DCubicBezierTo.getPt();
    									for(Object pt:pts) {
        									final CTAdjPoint2D point2D = new CTAdjPoint2D();
        									point2D.setX(((JSONObject)pt).get("x").toString());
        									point2D.setY(((JSONObject)pt).get("y").toString());
        									ptList.add(point2D);
    									}
    									cList.add(path2DCubicBezierTo);
    									break;
    								}
    							}
    						}
    					}
    				}
    			}
            }
            customGeometry.setPathLst(new CTPath2DList(path2DList));
        }
        else if(json==JSONObject.NULL) {
            customGeometry.setPathLst(null);
        }
    }

    public static void applyTextRectFromJson(Object json, CTCustomGeometry2D customGeometry) throws JSONException {
        if(json instanceof JSONObject) {
        	final CTGeomRect rect = new CTGeomRect();
			rect.setB(((JSONObject)json).get("b").toString());
			rect.setL(((JSONObject)json).get("l").toString());
			rect.setR(((JSONObject)json).get("r").toString());
			rect.setT(((JSONObject)json).get("t").toString());
        	customGeometry.setRect(rect);
        }
        else if(json == JSONObject.NULL) {
            customGeometry.setRect(null);
        }
    }

    public static void createJsonFromPresetGeometry(JSONObject attrs, CTPresetGeometry2D presetGeometry)
    	throws JSONException {

        if(presetGeometry!=null) {

        	final JSONObject initialGeometryAttrs = attrs.optJSONObject("geometry");
            final JSONObject geometryAttrs = initialGeometryAttrs!=null ? initialGeometryAttrs : new JSONObject(5);

            final STShapeType presetShapeType = presetGeometry.getPrst();
            if(presetShapeType!=null) {
            	geometryAttrs.put("presetShape", presetShapeType.value());
            }
            if(initialGeometryAttrs==null&&!geometryAttrs.isEmpty()) {
                attrs.put("geometry", geometryAttrs);
            }
            if(presetShapeType!=null) {
        		createJsonFromAvList(geometryAttrs, presetGeometry.getAvLst());
            }
        }
    }

    public static void createJsonFromCustomGeometry(JSONObject attrs, CTCustomGeometry2D customGeometry)
    	throws JSONException {

    	if(customGeometry!=null) {

    		final JSONObject initialGeometryAttrs = attrs.optJSONObject("geometry");
    		final JSONObject geometryAttrs = initialGeometryAttrs!=null ? initialGeometryAttrs : new JSONObject(4);

    		createJsonFromAvList(geometryAttrs, customGeometry.getAvLst());
    		createJsonFromGuideList(geometryAttrs, customGeometry.getGdLst());
    		createJsonFromPathList(geometryAttrs, customGeometry.getPathLst());
    		createJsonFromRect(geometryAttrs, customGeometry.getRect());
    		if(initialGeometryAttrs==null&&!geometryAttrs.isEmpty()) {
    			attrs.put("geometry", geometryAttrs);
    		}
    	}
    }

    public static JSONObject createJSPresetGeometries()
    	throws JSONException {

    	final JSONObject allPresets = new JSONObject(1);
    	final Iterator<Entry<STShapeType, CTCustomGeometry2D>> customShapePresetIter = Context.getCustomShapePresets().entrySet().iterator();
    	while(customShapePresetIter.hasNext()) {
    		final Entry<STShapeType, CTCustomGeometry2D> customShapePreset = customShapePresetIter.next();
    		final JSONObject preset = new JSONObject(4);
    		final CTCustomGeometry2D customGeometry = customShapePreset.getValue();
    		createJsonFromAvList(preset, customGeometry.getAvLst());
    		createJsonFromGuideList(preset, customGeometry.getGdLst());
    		createJsonFromPathList(preset, customGeometry.getPathLst());
    		createJsonFromRect(preset, customGeometry.getRect());
    		allPresets.put(customShapePreset.getKey().value(), preset);
    	}
    	return allPresets;
    }

    public static void createJsonFromAvList(JSONObject geometryAttrs, CTGeomGuideList geomGuides)
    	throws JSONException {

    	if(geomGuides!=null) {
    		final List<CTGeomGuide> geomGuideList = geomGuides.getGd();
    		JSONObject lst = geometryAttrs.optJSONObject("avList");
    		if(lst==null) {
    			lst = new JSONObject(geomGuideList.size());
        		geometryAttrs.put("avList", lst);
    		}
    		for(CTGeomGuide geomGuide:geomGuideList) {
    			if(geomGuide.getName()!=null&&geomGuide.getFmla()!=null) {
    				lst.put(geomGuide.getName(), Integer.valueOf(geomGuide.getFmla().substring(4)));
    			}
    		}
    	}
    }

    final private static String[] geomParameterNames = { "p0", "p1", "p2" };

    public static void createJsonFromGuideList(JSONObject geometryAttrs, CTGeomGuideList geomGuides)
    	throws JSONException {

    	if(geomGuides!=null) {
    		final List<CTGeomGuide> geomGuideList = geomGuides.getGd();
    		final JSONArray lst = new JSONArray(geomGuideList.size());
    		for(CTGeomGuide geomGuide:geomGuideList) {
    			final Object[] p = geomGuide.getParsedFmla();
    			final JSONObject gd = new JSONObject(1 + p.length);
    			if(geomGuide.getName()!=null) {
    				gd.put("name", geomGuide.getName());
    			}
   				gd.put("op", p[0]);
   				for(int i = 1; i < p.length; i++) {
					gd.put(geomParameterNames[i-1], p[i]);
   				}
   				lst.put(gd);
    		}
    		geometryAttrs.put("gdList", lst);
    	}
    }

    public static void createJsonFromPathList(JSONObject geometryAttrs, CTPath2DList pathes)
    	throws JSONException {

    	if(pathes!=null) {
    		final List<CTPath2D> pathList = pathes.getPath();
    		final JSONArray jsonPathList = new JSONArray(pathList.size());
    		for(CTPath2D path:pathList) {
    			final JSONObject jsonPath = new JSONObject();
    			final STPathFillMode pathFillMode = path.getFill();
    			if(pathFillMode!=null) {
    				jsonPath.put("fillMode", pathFillMode.value());
    			}
    			jsonPath.put("width", path.getW());
    			jsonPath.put("height", path.getH());
    			if(path.isStroke()==false) {
    				jsonPath.put("isStroke", false);
    			}
    			if(path.isExtrusionOk()==false) {
    				jsonPath.put("isExtrusionOk", false);
    			}
    			final List<Object> commands = path.getCloseOrMoveToOrLnTo();
    			final JSONArray jsonCommands = new JSONArray(commands.size());
    			for(Object command:commands) {
    				if(command instanceof CTPath2DLineTo) {
    					final CTPath2DLineTo c = (CTPath2DLineTo)command;
    					final CTAdjPoint2D pt2D = c.getPt();
    					final JSONObject jsonLineTo = new JSONObject(3);
    					jsonLineTo.put("c", "lineTo");
    					jsonLineTo.put("x", pt2D.getParsedX());
    					jsonLineTo.put("y", pt2D.getParsedY());
    					jsonCommands.put(jsonLineTo);
    				}
    				else if(command instanceof CTPath2DArcTo) {
    					final CTPath2DArcTo c = (CTPath2DArcTo)command;
    					final JSONObject jsonArcTo = new JSONObject(5);
    					jsonArcTo.put("c", "arcTo");
    					jsonArcTo.put("hr", c.getParsedHR());
    					jsonArcTo.put("wr", c.getParsedWR());
    					jsonArcTo.put("stAng", c.getParsedStAng());
    					jsonArcTo.put("swAng", c.getParsedSwAng());
    					jsonCommands.put(jsonArcTo);
    				}
    				else if(command instanceof CTPath2DMoveTo) {
    					final CTPath2DMoveTo c = (CTPath2DMoveTo)command;
    					final CTAdjPoint2D pt2D = c.getPt();
    					final JSONObject jsonMoveTo = new JSONObject(3);
    					jsonMoveTo.put("c", "moveTo");
    					jsonMoveTo.put("x", pt2D.getParsedX());
    					jsonMoveTo.put("y", pt2D.getParsedY());
    					jsonCommands.put(jsonMoveTo);
    				}
    				else if(command instanceof CTPath2DClose) {
    					final JSONObject jsonClose = new JSONObject(1);
    					jsonClose.put("c", "close");
    					jsonCommands.put(jsonClose);
    				}
    				else if(command instanceof CTPath2DQuadBezierTo) {
    					final CTPath2DQuadBezierTo c = (CTPath2DQuadBezierTo)command;
    					final List<CTAdjPoint2D> pts2D = c.getPt();
    					final JSONObject jsonQuadBezierTo = new JSONObject(2);
    					jsonQuadBezierTo.put("c", "quadBezierTo");
    					final JSONArray jsonPts = new JSONArray(pts2D.size());
    					for(CTAdjPoint2D pt:pts2D) {
        					final JSONObject jsonPt = new JSONObject(2);
    						jsonPt.put("x", pt.getParsedX());
    						jsonPt.put("y", pt.getParsedY());
    						jsonPts.put(jsonPt);
    					}
    					jsonQuadBezierTo.put("pts", jsonPts);
    					jsonCommands.put(jsonQuadBezierTo);
    				}
    				else if(command instanceof CTPath2DCubicBezierTo) {
    					final CTPath2DCubicBezierTo c = (CTPath2DCubicBezierTo)command;
    					final List<CTAdjPoint2D> pts2D = c.getPt();
    					final JSONObject jsonCubicBezierTo = new JSONObject(2);
    					jsonCubicBezierTo.put("c", "cubicBezierTo");
    					final JSONArray jsonPts = new JSONArray(pts2D.size());
    					for(CTAdjPoint2D pt:pts2D) {
        					final JSONObject jsonPt = new JSONObject(2);
    						jsonPt.put("x", pt.getParsedX());
    						jsonPt.put("y", pt.getParsedY());
    						jsonPts.put(jsonPt);
    					}
    					jsonCubicBezierTo.put("pts", jsonPts);
    					jsonCommands.put(jsonCubicBezierTo);
    				}
    			}
    			jsonPath.put("commands", jsonCommands);
    			jsonPathList.put(jsonPath);
    		}
    		geometryAttrs.put("pathList", jsonPathList);
    	}
    }

    public static void createJsonFromRect(JSONObject geometryAttrs, CTGeomRect rect)
    	throws JSONException {

    	if(rect!=null) {
    		final JSONObject textRect = new JSONObject(4);
    		if(rect.getL()!=null) {
    			textRect.put("l", CTAdjPoint2D.getLongIfPossible(rect.getL()));
    		}
    		if(rect.getT()!=null) {
    			textRect.put("t", CTAdjPoint2D.getLongIfPossible(rect.getT()));
    		}
    		if(rect.getR()!=null) {
    			textRect.put("r", CTAdjPoint2D.getLongIfPossible(rect.getR()));
    		}
    		if(rect.getB()!=null) {
    			textRect.put("b", CTAdjPoint2D.getLongIfPossible(rect.getB()));
    		}
    		if(!textRect.isEmpty()) {
    			geometryAttrs.put("textRect", textRect);
    		}
    	}
    }
}
