/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.xlsx.operations;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.docx4j.dml.CTRegularTextRun;
import org.docx4j.dml.CTTextField;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.JaxbXmlPart;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.SpreadsheetML.ChartsheetPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.BookViews;
import org.xlsx4j.sml.CTBookView;
import org.xlsx4j.sml.CTCalcPr;
import org.xlsx4j.sml.CTCellFormula;
import org.xlsx4j.sml.CTCellStyle;
import org.xlsx4j.sml.CTCellStyleXfs;
import org.xlsx4j.sml.CTCellStyles;
import org.xlsx4j.sml.CTCellXfs;
import org.xlsx4j.sml.CTDefinedName;
import org.xlsx4j.sml.CTDxf;
import org.xlsx4j.sml.CTFont;
import org.xlsx4j.sml.CTHyperlink;
import org.xlsx4j.sml.CTHyperlinks;
import org.xlsx4j.sml.CTNumFmt;
import org.xlsx4j.sml.CTNumFmts;
import org.xlsx4j.sml.CTRst;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTTableStyle;
import org.xlsx4j.sml.CTTableStyleElement;
import org.xlsx4j.sml.CTTableStyles;
import org.xlsx4j.sml.CTXf;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Col;
import org.xlsx4j.sml.Cols;
import org.xlsx4j.sml.DefinedNames;
import org.xlsx4j.sml.IDataValidation;
import org.xlsx4j.sml.IDataValidations;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STCellFormulaType;
import org.xlsx4j.sml.STCellType;
import org.xlsx4j.sml.STTableStyleType;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.Sheets;
import org.xlsx4j.sml.SmlUtils;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.WorkbookPr;
import org.xlsx4j.sml.Worksheet;
import com.openexchange.office.filter.api.DocumentProperties;
import com.openexchange.office.filter.api.FilterException;
import com.openexchange.office.filter.api.FilterException.ErrorCode;
import com.openexchange.office.filter.ooxml.components.Component;
import com.openexchange.office.filter.ooxml.components.IShapeType;
import com.openexchange.office.filter.ooxml.components.ShapeType;
import com.openexchange.office.filter.ooxml.operations.CreateOperationHelper;
import com.openexchange.office.filter.ooxml.tools.Commons;
import com.openexchange.office.filter.ooxml.xlsx.XlsxOperationDocument;
import com.openexchange.office.filter.ooxml.xlsx.components.ChartsheetComponent;
import com.openexchange.office.filter.ooxml.xlsx.components.HardBreakComponent;
import com.openexchange.office.filter.ooxml.xlsx.components.ParagraphComponent;
import com.openexchange.office.filter.ooxml.xlsx.components.ShapeGraphicComponent;
import com.openexchange.office.filter.ooxml.xlsx.components.SheetComponent;
import com.openexchange.office.filter.ooxml.xlsx.components.TextComponent;
import com.openexchange.office.filter.ooxml.xlsx.components.TextFieldComponent;
import com.openexchange.office.filter.ooxml.xlsx.tools.AutoFilterHelper;
import com.openexchange.office.filter.ooxml.xlsx.tools.CellUtils;
import com.openexchange.office.filter.ooxml.xlsx.tools.ColumnUtils;
import com.openexchange.office.filter.ooxml.xlsx.tools.ConditionalFormattings;
import com.openexchange.office.filter.ooxml.xlsx.tools.Drawings;
import com.openexchange.office.filter.ooxml.xlsx.tools.MergeCellHelper;
import com.openexchange.office.filter.ooxml.xlsx.tools.RowUtils;
import com.openexchange.office.filter.ooxml.xlsx.tools.SheetUtils;
import com.openexchange.office.filter.ooxml.xlsx.tools.TableHelper;
import com.openexchange.office.filter.ooxml.xlsx.tools.Utils;
import com.openexchange.office.filter.ooxml.xlsx.tools.ValidationUtils;
import com.openexchange.office.tools.config.ConfigurationHelper;
import com.openexchange.session.Session;

public class XlsxCreateOperationHelper extends CreateOperationHelper {

    private final SpreadsheetMLPackage spreadsheetMLPackage;
    private final Map<Long, String> indexToStyleId = new HashMap<Long, String>();
    final int maxCellCount;
    final int maxSheetCount;
    int currentCellCount = 0;

    public XlsxCreateOperationHelper(XlsxOperationDocument _operationDocument, JSONArray mOperationsArray)
        throws FilterException {

        super(_operationDocument, mOperationsArray);
        spreadsheetMLPackage = _operationDocument.getPackage();
        final Session session = _operationDocument.getSession();
        maxCellCount = ConfigurationHelper.getIntegerOfficeConfigurationValue(null, session, "//spreadsheet/maxCells", 500000);
        maxSheetCount = ConfigurationHelper.getIntegerOfficeConfigurationValue(null, session, "//spreadsheet/maxSheets", 256);

        final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
        final Workbook workbook = workbookPart.getJaxbElement();
        final Sheets sheets = workbook.getSheets();
        if(sheets!=null) {
            final List<Sheet> sheetList = sheets.getSheet();
            if(maxSheetCount>=0) {
                if(sheetList.size()>maxSheetCount) {
                    final FilterException filterException = new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
                    filterException.setSubType(FilterException.ST_MAX_SHEET_COUNT_EXCEEDED);
                    throw filterException;
                }
            }
        }
        final CTStylesheet stylesheet = getOperationDocument().getStylesheet(false);
        if(stylesheet!=null) {
            // creating styles
            final CTCellStyleXfs cellStyleXfs = stylesheet.getCellStyleXfs();
            if(cellStyleXfs!=null) {
                BuiltinIds.getUsedStyles(stylesheet, indexToStyleId, null);
            }
        }
    }

    @Override
    public XlsxOperationDocument getOperationDocument() {
        return (XlsxOperationDocument)operationDocument;
    }

    @Override
    public void createDocumentDefaults(String userLanguage) {

    }

    public void createStyleOperations()
        throws Exception {

        CTStylesheet stylesheet = getOperationDocument().getStylesheet(false);
        if(stylesheet!=null) {

            // create number formats
            final CTNumFmts numFmts = stylesheet.getNumberFormats(false);
            if(numFmts!=null) {
                for (CTNumFmt numFmt: numFmts.getNumFmt()) {
                    final JSONObject operation = new JSONObject();
                    operation.put("name", "insertNumberFormat");
                    operation.put("id", numFmt.getNumFmtId());
                    operation.put("code", numFmt.getFormatCode());
                    operationsArray.put(operation);
                }
            }

            // creating styles
            final CTCellStyleXfs cellStyleXfs = stylesheet.getCellStyleXfs();
            if(cellStyleXfs!=null) {
                final List<CTXf> cellStyleXfList = cellStyleXfs.getXf();
                for(int i=0;i<cellStyleXfList.size();i++) {
                    final String styleId = indexToStyleId.get(new Long(i));
                    if(styleId!=null) {
                        final CTXf cellStyleXf = cellStyleXfList.get(i);
                        String styleName = styleId;
                        JSONObject jsonProperties = new JSONObject();
                        Commons.mergeJsonObjectIfUsed("cell", jsonProperties, CellUtils.createCellProperties(cellStyleXf, stylesheet, true));
                        final CTFont ctFont = stylesheet.getFontByIndex(CellUtils.getAttributeIndex(cellStyleXf.getFontId(), cellStyleXf.isApplyFont(), true));
                        Commons.mergeJsonObjectIfUsed("character", jsonProperties, CellUtils.createCharacterProperties(new JSONObject(), stylesheet, ctFont));
                        final JSONObject applyProperties = createApplyProperties(cellStyleXf, true);
                        if(!applyProperties.isEmpty()) {
                            jsonProperties.put("apply", applyProperties);
                        }
                        JSONObject attrs = jsonProperties.length()>0?jsonProperties:null;
                        Boolean hidden = null;
                        Integer uiPriority = null;
                        Boolean isDefault = null;

                        // check for proper UI styleName
                        CTCellStyles cellStyles = stylesheet.getCellStyles();
                        if(cellStyles!=null) {
                            for(CTCellStyle cellStyle:cellStyles.getCellStyle()) {
                                if(cellStyle.getXfId()==i) {
                                    final String name = cellStyle.getName();
                                    if(name!=null&&!name.isEmpty()) {
                                        styleName = name;
                                    }
                                    if(i==0) {
                                        isDefault = new Boolean(true);
                                    }
                                    hidden = cellStyle.isHidden();
                                    break;
                                }
                            }
                        }
                        if(hidden==null||!hidden.booleanValue()) {
                            operationsArray.put(createInsertStyleSheetOperation("cell", styleId, styleName, attrs, null, hidden, uiPriority, isDefault, null));
                        }
                    }
                }
            }

            // creating autostyles
            final CTCellXfs cellXfs = stylesheet.getCellXfs();
            if(cellXfs!=null) {
                final List<CTXf> cellXfsList = cellXfs.getXf();
                for(int i=0;i<cellXfsList.size();i++) {
                    final CTXf xf = cellXfsList.get(i);
                    final String styleId = "a" + Integer.toString(i);
                    JSONObject jsonProperties = new JSONObject();
                    Commons.mergeJsonObjectIfUsed("cell", jsonProperties, CellUtils.createCellProperties(xf, stylesheet, false));
                    final CTFont ctFont = stylesheet.getFontByIndex(CellUtils.getAttributeIndex(xf.getFontId(), xf.isApplyFont(), false));
                    Commons.mergeJsonObjectIfUsed("character", jsonProperties, CellUtils.createCharacterProperties(new JSONObject(), stylesheet, ctFont));
                    final JSONObject applyProperties = createApplyProperties(xf, false);
                    if(!applyProperties.isEmpty()) {
                        jsonProperties.put("apply", applyProperties);
                   }
                   final Long xfId = xf.getXfId();
                    if(xfId!=null) {
                        final String parentId = indexToStyleId.get(xfId);
                        if(parentId!=null) {
                            jsonProperties.put("styleId", parentId);
                        }
                    }
                    operationsArray.put(createInsertAutoStyleOperation("cell", styleId, jsonProperties.length()>0?jsonProperties:null, i==0));
                }
            }
            final CTTableStyles tableStyles = stylesheet.getTableStyles(false);
            if(tableStyles!=null) {
                final Iterator<CTTableStyle> tableStyleIter = tableStyles.getTableStyle().iterator();
                while(tableStyleIter.hasNext()) {
                    final CTTableStyle tableStyle = tableStyleIter.next();
                    createInsertTableStyleOperation(stylesheet, tableStyle);
                }
            }
        }
    }

    final void createApplyProperty(JSONObject jsonApplyProps, String propName, Boolean xfProperty, boolean defaultValue)
        throws JSONException {

        // OOXML default value depends on XF type (true in styles, false in cells); operation default is always true
        boolean applyValue = (xfProperty == null) ? defaultValue : xfProperty.booleanValue();
        if (!applyValue) { jsonApplyProps.put(propName, false); }
    }

    final JSONObject createApplyProperties(CTXf cellStyleXf, boolean defaultValue)
        throws JSONException {

        final JSONObject jsonApplyProperties = new JSONObject(0);
        createApplyProperty(jsonApplyProperties, "border", cellStyleXf.isApplyBorder(), defaultValue);
        createApplyProperty(jsonApplyProperties, "align", cellStyleXf.isApplyAlignment(), defaultValue);
        createApplyProperty(jsonApplyProperties, "number", cellStyleXf.isApplyNumberFormat(), defaultValue);
        createApplyProperty(jsonApplyProperties, "protect", cellStyleXf.isApplyProtection(), defaultValue);
        createApplyProperty(jsonApplyProperties, "font", cellStyleXf.isApplyFont(), defaultValue);
        createApplyProperty(jsonApplyProperties, "fill", cellStyleXf.isApplyFill(), defaultValue);
        return jsonApplyProperties;
    }

    final void createInsertTableStyleOperation(CTStylesheet stylesheet, CTTableStyle tableStyle)
        throws JSONException {

        final JSONObject insertStylesheetObject = new JSONObject();
        insertStylesheetObject.put("name", "insertStyleSheet");
        insertStylesheetObject.put("type", "table");
        insertStylesheetObject.put("styleId", tableStyle.getName());
        insertStylesheetObject.put("styleName", tableStyle.getName());

        final JSONObject attrs = new JSONObject();
        final Iterator<CTTableStyleElement> tableStyleElementIter = tableStyle.getTableStyleElement().iterator();
        while(tableStyleElementIter.hasNext()) {
            final CTTableStyleElement tableStyleElement = tableStyleElementIter.next();
            final STTableStyleType tableStyleElementType = tableStyleElement.getType();
            if(tableStyleElementType!=null) {
                switch(tableStyleElementType) {
                    case WHOLE_TABLE: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "wholeTable", attrs);
                        break;
                    }
                    case FIRST_COLUMN_STRIPE: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "band1Vert", attrs);
                        break;
                    }
                    case SECOND_COLUMN_STRIPE: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "band2Vert", attrs);
                        break;
                    }
                    case FIRST_ROW_STRIPE: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "band1Hor", attrs);
                        break;
                    }
                    case SECOND_ROW_STRIPE: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "band2Hor", attrs);
                        break;
                    }
                    case LAST_COLUMN: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "lastCol", attrs);
                        break;
                    }
                    case FIRST_COLUMN: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "firstCol", attrs);
                        break;
                    }
                    case HEADER_ROW: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "firstRow", attrs);
                        break;
                    }
                    case TOTAL_ROW: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "lastRow", attrs);
                        break;
                    }
                    case FIRST_HEADER_CELL: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "northWestCell", attrs);
                        break;
                    }
                    case LAST_HEADER_CELL: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "northEastCell", attrs);
                        break;
                    }
                    case FIRST_TOTAL_CELL: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "southWestCell", attrs);
                        break;
                    }
                    case LAST_TOTAL_CELL: {
                        createTableStyleElementOperation(stylesheet, tableStyleElement, "southEastCell", attrs);
                        break;
                    }

                    // ignoring, used for pivot tables only:
                    case BLANK_ROW:
                    case FIRST_COLUMN_SUBHEADING:
                    case FIRST_ROW_SUBHEADING:
                    case FIRST_SUBTOTAL_COLUMN:
                    case FIRST_SUBTOTAL_ROW:
                    case PAGE_FIELD_LABELS:
                    case PAGE_FIELD_VALUES:
                    case SECOND_COLUMN_SUBHEADING:
                    case SECOND_ROW_SUBHEADING:
                    case SECOND_SUBTOTAL_COLUMN:
                    case SECOND_SUBTOTAL_ROW:
                    case THIRD_COLUMN_SUBHEADING:
                    case THIRD_ROW_SUBHEADING:
                    case THIRD_SUBTOTAL_COLUMN:
                    case THIRD_SUBTOTAL_ROW:
                    default:
                        break;

                }
            }
        }
        if(!attrs.isEmpty()) {
            insertStylesheetObject.put("attrs", attrs);
        }
        insertStylesheetObject.put("custom", true);
        operationsArray.put(insertStylesheetObject);
    }

    public void createTableStyleElementOperation(CTStylesheet stylesheet, CTTableStyleElement tableStyleElement, String conditionalTableStyle, JSONObject attrs)
        throws JSONException {

        final Long dxfId = tableStyleElement.getDxfId();
        if(dxfId!=null) {
            final CTDxf dxf = stylesheet.getDxfsByIndex(dxfId);
            if(dxf!=null) {
                attrs.put(conditionalTableStyle, Utils.createDxfAttrs(dxf, stylesheet));
            }
        }
    }

    public void createOperations()
        throws Exception {

        final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
        final Workbook workbook = workbookPart.getJaxbElement();

        createDocumentSettings(workbook);
        createThemeOperations(null, null);
        createStyleOperations();
        createInsertNameOperations(workbook, null);
        final Sheets sheets = workbook.getSheets();
        final List<Sheet> sheetList = sheets.getSheet();
        for(int sheetIndex = 0; sheetIndex < sheetList.size(); sheetIndex++) {
            createSheetOperations(true, true, workbook, sheetList.get(sheetIndex), sheetIndex);
        }
    }

    public void createMetaOperations(DocumentProperties documentProperties)
        throws Exception {

        final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
        final Workbook workbook = workbookPart.getJaxbElement();

        final int activeSheetIndex = createDocumentSettings(workbook);
        createThemeOperations(null, null);
        createStyleOperations();
        createInsertNameOperations(workbook, null);
        documentProperties.put(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX, activeSheetIndex);
        documentProperties.put(DocumentProperties.PROP_SPREADHSEET_CURRENT_SHEET_INDEX, Integer.valueOf(0));
    }

    public void createActivePartOperations(DocumentProperties documentProperties)
        throws Exception {

        final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
        final Workbook workbook = workbookPart.getJaxbElement();

        final int activeSheetIndex = (Integer)documentProperties.get(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX);
        final Sheets sheets = workbook.getSheets();
        final List<Sheet> sheetList = sheets.getSheet();
        for(int sheetIndex = 0; sheetIndex < sheetList.size(); sheetIndex++) {
            createSheetOperations(true, activeSheetIndex==sheetIndex, workbook, sheetList.get(sheetIndex), sheetIndex);
        }
        documentProperties.put(DocumentProperties.PROP_SPREADSHEET_SHEET_COUNT, sheetList.size());
    }

    public boolean createNextPartOperations(DocumentProperties documentProperties)
        throws Exception {

        final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
        final Workbook workbook = workbookPart.getJaxbElement();

        final int activeSheetIndex = (Integer)documentProperties.get(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX);
        int currentSheetIndex = (Integer)documentProperties.get(DocumentProperties.PROP_SPREADHSEET_CURRENT_SHEET_INDEX);

        final Sheets sheets = workbook.getSheets();
        final List<Sheet> sheetList = sheets.getSheet();
        for(; currentSheetIndex < sheetList.size();) {
            createSheetOperations(false, activeSheetIndex!=currentSheetIndex, workbook, sheetList.get(currentSheetIndex), currentSheetIndex);
            documentProperties.put(DocumentProperties.PROP_SPREADHSEET_CURRENT_SHEET_INDEX, Integer.valueOf(currentSheetIndex+1));
            return true;
        }
        return false;
    }

    private void updateCellCount(Worksheet worksheet)
        throws FilterException {

        if(maxCellCount>0) {
            final SheetData sheetData = worksheet.getSheetData();
            if(sheetData!=null) {
                final Iterator<Row> rowIterator = sheetData.createRowIterator();
                while(rowIterator.hasNext()) {
                    final Row row = rowIterator.next();
                    currentCellCount += row.getCellCount();
                    if(currentCellCount>maxCellCount) {
                        throw new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
                    }
                }
            }
        }
    }

    /* returns the active sheet */

    private int createDocumentSettings(Workbook workbook)
        throws JSONException {

        long activeSheet = 0;
        final JSONObject documentAttrs = new JSONObject();
        documentAttrs.put("fileFormat", "ooxml");
        documentAttrs.put("cols", 16384);
        documentAttrs.put("rows", 1048576);
        final BookViews bookViews = workbook.getBookViews();
        if(bookViews!=null) {
            final List<CTBookView> bookViewList = bookViews.getWorkbookView();
            if(!bookViewList.isEmpty()) {
                final CTBookView bookView = bookViewList.get(0);
                activeSheet = bookView.getActiveTab();
                documentAttrs.put("activeSheet", activeSheet);
            }
        }

        // property nullDate: serial number 0 represents 1904-01-01 in "date1904" mode
        final WorkbookPr wbPr = workbook.getWorkbookPr(false);
        if (wbPr!=null && wbPr.isDate1904()) {
            documentAttrs.put("nullDate", "1904-01-01");
        }

        // property calcOnLoad: recalculate all formulas once after import
        final CTCalcPr calcPr = workbook.getCalcPr(false);
        if (calcPr!=null && calcPr.isFullCalcOnLoad()) {
            documentAttrs.put("calcOnLoad", true);
        }

        final JSONObject attrs = new JSONObject(1);
        attrs.put("document", documentAttrs);
        addSetDocumentAttributesOperation(attrs);

        return (int)activeSheet;
    }

    private void createInsertNameOperations(Workbook workbook, Long sheetIndex)
        throws JSONException {

        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames==null) {
            return;
        }
        final HashMap<String, CTDefinedName> definedNameMap = definedNames.getDefinedNameMap();
        for(CTDefinedName definedName:definedNameMap.values()) {
            Long localSheetId = definedName.getLocalSheetId();
            if ((sheetIndex == null) ? (localSheetId == null) : sheetIndex.equals(localSheetId)) {
                addInsertNameOperation(sheetIndex, definedName.getName(), definedName.getValue()!=null ? definedName.getValue() : "", definedName.isHidden());
            }
        }
    }

    private JSONObject getCellObject(Cell cell)
        throws Exception {

        STCellType cellType = cell.getT();
        if(cellType==null) {
            cellType = STCellType.N;
        }
        final JSONObject cellObject = new JSONObject();
        final CTCellFormula cellFormula = cell.getF();
        if(cellFormula!=null) {
            STCellFormulaType cellFormulaType = cellFormula.getT();
            final String formulaExpr = cellFormula.getValue();
            if (formulaExpr!=null && !formulaExpr.isEmpty()) {
                cellObject.put("f", formulaExpr);
            }
            if(cellFormulaType==STCellFormulaType.SHARED) {
                final Long si = cellFormula.getSi();
                if(si != null) {
                    cellObject.put("si", si);
                }
                final String sr = cellFormula.getRef();
                if (sr != null) {
                    cellObject.put("sr", sr);
                }
            } else if (cellFormulaType==STCellFormulaType.ARRAY) {
                final String mr = cellFormula.getRef();
                if (mr != null) {
                    cellObject.put("mr", mr);
                }
            }
        }
        if(cellType==STCellType.INLINE_STR) {
            final CTRst is = cell.getIs();
            if(is!=null) {
                if(is.getT()!=null) {
                    cellObject.put("v", new String(is.getT().getValue()));;
                }
            }
        }
        else if(cell.getV()!=null) {
            if(cellType==STCellType.N) {
                try {
                    cellObject.put("v", new Double(Double.parseDouble(cell.getV())));
                }
                catch(NumberFormatException e) {
                    // TODO: ups
                }
            }
            else if(cellType==STCellType.S) {
                String sharedString = getOperationDocument().getSharedString(cell.getV());
                if(sharedString!=null) {
                    cellObject.put("v", sharedString);
                }
            }
            else if(cellType==STCellType.B) {
                if(cell.getV().equals("0")||cell.getV().toLowerCase().equals("false")) {
                    cellObject.put("v", new Boolean(false));
                }
                else if(cell.getV().equals("1")||cell.getV().toLowerCase().equals("true")) {
                    cellObject.put("v", new Boolean(true));
                }
            }
            else if(cellType==STCellType.E) {
                cellObject.put("e", cell.getV());
            }
            else {  // STCellType.STR, STCellType.F (ARRAY or DATA_TABLE format)
                cellObject.put("v", cell.getV());
            }
        }
        final long s = cell.getStyle();
        if ((s > 0) || cellObject.isEmpty()) {
            cellObject.put("s", "a" + Long.toString(s));
        }
        return cellObject;
    }

    private void createSheetOperations(boolean insertsheet, boolean sheetcontent, Workbook workbook, Sheet sheet, int sheetIndex)
        throws Exception {

        final Part part = XlsxOperationDocument.getSheetPart(spreadsheetMLPackage, sheet);
        final String sheetType = XlsxOperationDocument.getSheetType(part);

        if(sheetType.equals("worksheet")) {
            updateCellCount(((WorksheetPart)part).getJaxbElement());
        }
        if(insertsheet) {
            final JSONObject attrs = new JSONObject();
            final JSONObject sheetProps = SheetUtils.createSheetProperties(sheet);
            if(sheetProps!=null&&!sheetProps.isEmpty()) {
                attrs.put("sheet", sheetProps);
            }
            addInsertSheetOperation(sheetIndex, sheet.getName(), sheetType, attrs);
        }
        if(sheetcontent) {
            createInsertNameOperations(workbook, new Long(sheetIndex));
            if(sheetType.equals("worksheet")) {
                createSetSheetAttributesOperation(sheet, ((WorksheetPart)part).getJaxbElement(), sheetIndex);
                createWorksheetOperations(sheet, (WorksheetPart)part, sheetIndex);
            }
            else if(sheetType.equals("chartsheet")) {
                createChartsheetOperations(sheet, (ChartsheetPart)part, sheetIndex);
            }
        }
        if(part instanceof JaxbXmlPart) {
            ((JaxbXmlPart)part).setJaxbElement((Integer)null);
        }
    }

    private void createSetSheetAttributesOperation(Sheet sheet, Worksheet worksheet, int sheetIndex)
        throws JSONException {

        final JSONObject properties = new JSONObject();
        properties.put("sheet", SheetUtils.createWorksheetProperties(sheet, worksheet));
        properties.put("column", SheetUtils.createWorksheetColumnProperties(getOperationDocument(), worksheet));
        final JSONObject rowProperties = SheetUtils.createWorksheetRowProperties(worksheet);
        if(rowProperties!=null&&!rowProperties.isEmpty()) {
            properties.put("row", rowProperties);
        }
        addSetSheetAttributesOperation(sheetIndex, properties);
    }

    private void createWorksheetOperations(Sheet sheet, final WorksheetPart worksheetPart, int sheetIndex)
        throws Exception {

        final Worksheet worksheet = worksheetPart.getJaxbElement();

        // column attributes... row attributes ... cells...
        createColumnOperations(worksheet.getCols(), sheetIndex);
        createRowOperations(worksheet, sheetIndex);

        MergeCellHelper.createMergeCellOperations(operationsArray, sheetIndex, worksheet.getMergeCells());
        AutoFilterHelper.createAutoFilterOperations(operationsArray, sheetIndex, worksheet.getAutoFilter(false), worksheet.getSortState(false));
        TableHelper.createTableOperations(operationsArray, worksheetPart, sheetIndex, worksheet.getTableParts(false));

        // hyperlinks
        final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks!=null) {
            final List<CTHyperlink> hyperlinkList = hyperlinks.getHyperlink();
            for(int i=0; i<hyperlinkList.size();i++) {
                final CTHyperlink hyperlink = hyperlinkList.get(i);
                final SmlUtils.CellRefRange cellRefRange = hyperlink.getCellRefRange(false);
                if(cellRefRange!=null) {
                    final String hyperlinkUrl = Commons.getUrl(worksheetPart, hyperlink.getId());
                    if(hyperlinkUrl!=null&&!hyperlinkUrl.isEmpty()) {
                        final JSONObject insertHyperlinkOperation = new JSONObject(5);
                        insertHyperlinkOperation.put("name", "insertHyperlink");
                        insertHyperlinkOperation.put("sheet", sheetIndex);
                        insertHyperlinkOperation.put("start", cellRefRange.getStart().getJSONArray());
                        if(!cellRefRange.getStart().equals(cellRefRange.getEnd())) {
                            insertHyperlinkOperation.put("end", cellRefRange.getEnd().getJSONArray());
                        }
                        insertHyperlinkOperation.put("url", hyperlinkUrl);
                        operationsArray.put(insertHyperlinkOperation);
                    }
                }
            }
        }
        final SheetComponent sheetComponent = new SheetComponent(sheet, getOperationDocument(), sheetIndex);
        final List<Integer> sheetPosition = new ArrayList<Integer>(1);
        sheetPosition.add(sheetIndex);
        Component childComponent = sheetComponent.getNextChildComponent(null, null);
        while(childComponent!=null) {
            createDrawingOperations(childComponent, sheetPosition, null);
            childComponent = childComponent.getNextComponent();
        }

/*
        //drawings
        final CTDrawing drawing = worksheet.getDrawing();
        if(drawing!=null) {
            final String drawingId = drawing.getId();
            if(drawingId!=null&&!drawingId.isEmpty()) {
                final RelationshipsPart sourceRelationships = worksheetPart.getRelationshipsPart();
                if(sourceRelationships!=null) {
                    final Part part = sourceRelationships.getPart(drawingId);
                    if(part instanceof org.docx4j.openpackaging.parts.DrawingML.Drawing) {
                        Drawings.createDrawingOperations(this.getOperationDocument(), operationsArray, (org.docx4j.openpackaging.parts.DrawingML.Drawing)part, sheetIndex);
                    }
                }
            }
        }
*/

        // validations
        final List<IDataValidations> validations = XlsxOperationDocument.getDataValidations(worksheet);
        int validationIndex = 0;
        for(IDataValidations dataValidations:validations) {
            for(IDataValidation dataValidation:dataValidations.getDataValidation()) {
                createDataValidation(sheetIndex, validationIndex++, dataValidation);
            }
        }

        // conditional formattings
        ConditionalFormattings.createOperations(getOperationDocument(), operationsArray, worksheet, sheetIndex);
    }

    private void createChartsheetOperations(Sheet sheet, final ChartsheetPart chartsheetPart, int sheetIndex)
        throws Exception {

        final ChartsheetComponent sheetComponent = new ChartsheetComponent(sheet, getOperationDocument(), sheetIndex);
        final List<Integer> sheetPosition = new ArrayList<Integer>(1);
        sheetPosition.add(sheetIndex);
        Component childComponent = sheetComponent.getNextChildComponent(null, null);
        while(childComponent!=null) {
            createDrawingOperations(childComponent, sheetPosition, null);
            childComponent = childComponent.getNextComponent();
        }
    }

    private void createDrawingOperations(Component drawingComponent, List<Integer> parentPosition, String target)
        throws Exception {

        final List<Integer> position = new ArrayList<Integer>(parentPosition);
        position.add(drawingComponent.getComponentNumber());
        final JSONObject jsonInsertOperation = new JSONObject(5);
        jsonInsertOperation.put("name", "insertDrawing");
        jsonInsertOperation.put("type", ((IShapeType)drawingComponent).getType().toString());
        jsonInsertOperation.put("start", position);
        if(target!=null) {
            jsonInsertOperation.put("target", target);
        }
        final JSONObject attrs = drawingComponent.createJSONAttrs(this, new JSONObject());
        if(!attrs.isEmpty()) {
            jsonInsertOperation.put("attrs", attrs);
        }
        operationsArray.put(jsonInsertOperation);
        if(((IShapeType)drawingComponent).getType()==ShapeType.CHART) {
            Drawings.getChart(((ShapeGraphicComponent) drawingComponent).getDrawingProperties()).createChartSpaceOperations(operationsArray, position);
        }
        Component shapeChildComponent = drawingComponent.getNextChildComponent(null, null);

        while(shapeChildComponent!=null) {
            if(shapeChildComponent instanceof ParagraphComponent) {
                createParagraphOperations((ParagraphComponent)shapeChildComponent, position, null);
            }
            else {
                createDrawingOperations(shapeChildComponent, position, target);
            }
            shapeChildComponent = shapeChildComponent.getNextComponent();
        }
    }

    public void createParagraphOperations(ParagraphComponent paragraphComponent, List<Integer> parentPosition, String target)
        throws JSONException, ParseException {

        final List<Integer> position = new ArrayList<Integer>(parentPosition);
        position.add(paragraphComponent.getComponentNumber());
        final JSONObject jsonInsertParagraphOperation = new JSONObject(4);
        jsonInsertParagraphOperation.put("name", "insertParagraph");
        jsonInsertParagraphOperation.put("start", position);
        if(target!=null) {
            jsonInsertParagraphOperation.put("target", target);
        }
        final JSONObject attrs = paragraphComponent.createJSONAttrs(this, new JSONObject());
        if(!attrs.isEmpty()) {
            jsonInsertParagraphOperation.put("attrs", attrs);
        }
        operationsArray.put(jsonInsertParagraphOperation);
        Component paragraphChild = paragraphComponent.getNextChildComponent(null, null);
        while(paragraphChild!=null) {
            if(paragraphChild instanceof TextComponent) {
                int n = paragraphChild.getComponentNumber();
                final String[] tokens =
                    (((CTRegularTextRun)(((TextComponent)paragraphChild).getObject())).getT()).split("\\t", -1);
                for(int i=0; i < tokens.length; i++) {
                    final String t = tokens[i];
                    if(!t.isEmpty()) {
                        createTextInsertOperation(t, n, position, target);
                        n += t.length();
                    }
                    if(tokens.length!=1&&i!=tokens.length-1) {
                        createInsertTabOperation(n, position, target);
                        n++;
                    }
                }
            }
            else if(paragraphChild instanceof TextFieldComponent) {
                createInsertFieldOperation((TextFieldComponent)paragraphChild, position, target);
            }
            else if(paragraphChild instanceof HardBreakComponent) {
                createHardBreakOperation((HardBreakComponent)paragraphChild, position, target);
            }
            paragraphChild = paragraphChild.getNextComponent();
        }
        paragraphChild = paragraphComponent.getNextChildComponent(null, null);
        while(paragraphChild!=null) {
            if(paragraphChild instanceof TextComponent) {
                createTextAttributesOperation((TextComponent)paragraphChild, position, target);
            }
            else if(paragraphChild instanceof TextFieldComponent) {
                createTextFieldAttributesOperation((TextFieldComponent)paragraphChild, position, target);
            }
            else if(paragraphChild instanceof HardBreakComponent) {
                
            }
            paragraphChild = paragraphChild.getNextComponent();
        }
    }

    public void createTextInsertOperation(String text, int textPosition, List<Integer> parentPosition, String target)
        throws JSONException {

        final List<Integer> position = new ArrayList<Integer>(parentPosition);
        position.add(textPosition);

        final JSONObject jsonInsertTextOperation = new JSONObject(4);
        jsonInsertTextOperation.put("name", "insertText");
        jsonInsertTextOperation.put("start", position);
        if(target!=null) {
            jsonInsertTextOperation.put("target", target);
        }
        jsonInsertTextOperation.put("text", text);
        operationsArray.put(jsonInsertTextOperation);
    }

    public void createInsertTabOperation(int textPosition, List<Integer> parentPosition, String target)
        throws JSONException {

        final List<Integer> position = new ArrayList<Integer>(parentPosition);
        position.add(textPosition);

        final JSONObject jsonInsertTabOperation = new JSONObject(3);
        jsonInsertTabOperation.put("name", "insertTab");
        jsonInsertTabOperation.put("start", position);
        if(target!=null) {
            jsonInsertTabOperation.put("target", target);
        }
        operationsArray.put(jsonInsertTabOperation);
    }

    public void createTextAttributesOperation(TextComponent textComponent, List<Integer> parentPosition, String target)
        throws JSONException, ParseException {

        final JSONObject attrs = textComponent.createJSONAttrs(this, new JSONObject());
        if(!attrs.isEmpty()) {

            final JSONObject jsonSetAttributesOperation = new JSONObject(5);
            jsonSetAttributesOperation.put("name", "setAttributes");

            final List<Integer> start = new ArrayList<Integer>(parentPosition);
            start.add(textComponent.getComponentNumber());
            jsonSetAttributesOperation.put("start", start);

            final CTRegularTextRun textRun = (CTRegularTextRun)textComponent.getObject();
            if(textRun.getT().length()>1) {
                start.set(start.size()-1, (textComponent.getComponentNumber()+textRun.getT().length())-1);
                jsonSetAttributesOperation.put("end", start);
            }

            if(target!=null) {
                jsonSetAttributesOperation.put("target", target);
            }
            jsonSetAttributesOperation.put("attrs", attrs);
            operationsArray.put(jsonSetAttributesOperation);
        }
    }

    public void createTextFieldAttributesOperation(TextFieldComponent textFieldComponent, List<Integer> parentPosition, String target)
        throws JSONException {

        final JSONObject attrs = textFieldComponent.createJSONAttrs(this, new JSONObject());
        if(!attrs.isEmpty()) {

            final JSONObject jsonSetAttributesOperation = new JSONObject(4);
            jsonSetAttributesOperation.put("name", "setAttributes");

            final List<Integer> start = new ArrayList<Integer>(parentPosition);
            start.add(textFieldComponent.getComponentNumber());
            jsonSetAttributesOperation.put("start", start);
            if(target!=null) {
                jsonSetAttributesOperation.put("target", target);
            }
            jsonSetAttributesOperation.put("attrs", attrs);
            operationsArray.put(jsonSetAttributesOperation);
        }
    }

    public void createHardBreakAttributesOperation(HardBreakComponent hardBreakComponent, List<Integer> parentPosition, String target)
        throws JSONException {

        final JSONObject attrs = hardBreakComponent.createJSONAttrs(this, new JSONObject());
        if(!attrs.isEmpty()) {

            final JSONObject jsonSetAttributesOperation = new JSONObject(5);
            jsonSetAttributesOperation.put("name", "setAttributes");

            final List<Integer> start = new ArrayList<Integer>(parentPosition);
            start.add(hardBreakComponent.getComponentNumber());
            jsonSetAttributesOperation.put("start", start);

            if(target!=null) {
                jsonSetAttributesOperation.put("target", target);
            }
            jsonSetAttributesOperation.put("attrs", attrs);
            operationsArray.put(jsonSetAttributesOperation);
        }
    }

    public void createInsertFieldOperation(TextFieldComponent textFieldComponent, List<Integer> parentPosition, String target)
        throws JSONException {

        final List<Integer> position = new ArrayList<Integer>(parentPosition);
        position.add(textFieldComponent.getComponentNumber());

        final JSONObject jsonInsertSimpleFieldOperation = new JSONObject(6);
        jsonInsertSimpleFieldOperation.put("name", "insertField");
        jsonInsertSimpleFieldOperation.put("start", position);
        final CTTextField textField = (CTTextField)textFieldComponent.getObject();
        jsonInsertSimpleFieldOperation.put("type", textField.getType());
        jsonInsertSimpleFieldOperation.put("representation", textField.getT());
        if(target!=null) {
            jsonInsertSimpleFieldOperation.put("target", target);
        }
        final JSONObject attrs = textFieldComponent.createJSONAttrs(this, new JSONObject());
        if(!attrs.isEmpty()) {
            jsonInsertSimpleFieldOperation.put("attrs", attrs);
        }
        operationsArray.put(jsonInsertSimpleFieldOperation);
    }

    public void createHardBreakOperation(HardBreakComponent hardBreakComponent, List<Integer> parentPosition, String target)
        throws JSONException {

        final List<Integer> position = new ArrayList<Integer>(parentPosition);
        position.add(hardBreakComponent.getComponentNumber());

        final JSONObject jsonInsertHardBreakOperation = new JSONObject(4);
        jsonInsertHardBreakOperation.put("name", "insertHardBreak");
        jsonInsertHardBreakOperation.put("start", position);
        if(target!=null) {
            jsonInsertHardBreakOperation.put("target", target);
        }
        final JSONObject attrs = hardBreakComponent.createJSONAttrs(this, new JSONObject());
        if(!attrs.isEmpty()) {
            jsonInsertHardBreakOperation.put("attrs", attrs);
        }
        operationsArray.put(jsonInsertHardBreakOperation);
    }

    private void createDataValidation(int sheetIndex, int validationIndex, IDataValidation dataValidation)
        throws JSONException {

        final List<SmlUtils.CellRefRange> cellRefRangeList = Utils.createCellRefRangeListFromSqrefList(dataValidation.getsqref());
        ValidationUtils.addInsertValidationOperation(operationsArray, sheetIndex, validationIndex, cellRefRangeList, dataValidation.getType(), dataValidation.getOperator(), dataValidation.getFormula1(), dataValidation.getFormula2(), dataValidation.isShowInputMessage(),
            dataValidation.getPromptTitle(), dataValidation.getPrompt(), dataValidation.isShowErrorMessage(), dataValidation.getErrorTitle(), dataValidation.getError(), dataValidation.getErrorStyle(),
                dataValidation.isShowDropDown()==false, dataValidation.isAllowBlank());
    }

    private boolean compareCell(Cell source, Cell dest) {
        if(source.getF()!=null||dest.getF()!=null)
            return false;
        if(source.getIs()!=null||dest.getIs()!=null)
            return false;
        if(source.getT()!=dest.getT())
            return false;
        if(source.getVm()!=dest.getVm())
            return false;
        if(source.getCm()!=dest.getCm())
            return false;
        if(source.getStyle()!=dest.getStyle())
            return false;
        if(source.getV()==null&&dest.getV()==null)
            return true;
        if(source.getV()!=null&&dest.getV()!=null) {
            return source.getV().equals(dest.getV());
        }
        return false;
    }

    private void createRowOperations(Worksheet worksheet, int sheetIndex)
        throws Exception {

        final SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            return;
        }

        // first we will create row attributes
        Iterator<Row> rowIterator = sheetData.createRowIterator();

        Row lastRow = null;
        int lastRowNumber = 0;
        int rowCount = 0;

        while(rowIterator.hasNext()) {
            final Row row = rowIterator.next();

            if(lastRow!=null&&row.getRow()==lastRowNumber+rowCount&&RowUtils.compareRowProperties(lastRow, row)) {
                rowCount++;
            }
            else {
                if(lastRow!=null) {
                    final JSONObject attrs = RowUtils.createRowProperties(lastRow);
                    final Long style = lastRow.getS();
                    if(attrs!=null || style!=null) {
                        addChangeRows(sheetIndex, lastRowNumber, (lastRowNumber - 1) + rowCount, attrs, style);
                    }
                }
                lastRow = row;
                lastRowNumber = row.getRow();
                rowCount = 1;
            }
            if(!rowIterator.hasNext()&&lastRow!=null) {
                final JSONObject attrs = RowUtils.createRowProperties(lastRow);
                final Long style = lastRow.getS();
                if(attrs!=null || style!=null) {
                    addChangeRows(sheetIndex, lastRowNumber, (lastRowNumber - 1) + rowCount, attrs, style);
                }
            }
        }

        // create cell content for each row
        rowIterator = sheetData.createRowIterator();
        while(rowIterator.hasNext()) {
            final Row row = rowIterator.next();

            JSONArray cellArray = new JSONArray();
            SmlUtils.CellRef topLeftCellRef = null;

            Cell        lastCell = null;
            JSONObject  lastCellObject = null;
            int         lastCellX = -1;

            Iterator<Cell> cellIterator = row.createCellIterator();
            while(cellIterator.hasNext()) {
                boolean repeat = false;
                Cell cell = cellIterator.next();
                final SmlUtils.CellRef cellRef = new SmlUtils.CellRef(cell.getColumn(), row.getRow());
                if(topLeftCellRef==null) {
                    topLeftCellRef = cellRef;
                }
                if(lastCell!=null) {
                    // checking empty cells
                    final int cellDifference = cellRef.getColumn() - lastCellX;
                    if(cellDifference > 1) {
                        // we have empty cells ... at least one
                        cellArray.put(lastCellObject);
                        final JSONObject emptyCell = new JSONObject(1);
                        emptyCell.put("r", cellDifference-1);
                        cellArray.put(emptyCell);
                    }
                    else {
                        if(compareCell(lastCell, cell)) {
                            lastCellObject.put("r", lastCellObject.optInt("r", 1) + 1);
                            repeat = true;
                        }
                        else {
                            cellArray.put(lastCellObject);
                        }
                    }
                }
                lastCell = cell;
                lastCellX = cellRef.getColumn();
                if(repeat==false) {
                    lastCellObject = getCellObject(cell);
                }
            }
            if(lastCellObject!=null) {
                cellArray.put(lastCellObject);
            }
            if(!cellArray.isEmpty()) {
                final JSONObject rowObject = new JSONObject();
                rowObject.put("c", cellArray);
                final JSONArray rowArray = new JSONArray(1);
                rowArray.put(rowObject);
                addChangeCellsOperation(sheetIndex, topLeftCellRef, rowArray);
            }
        }
    }

    private void createColumnOperations(List<Cols> colsIter, int sheetIndex)
        throws Exception {

        if(colsIter==null)
            return;

        for(Cols cols:colsIter) {
            List<Col> colIter = cols.getCol();
            if(colIter!=null) {
                for(Col col:colIter) {
                    final JSONObject attrs = ColumnUtils.createColumnProperties(getOperationDocument(), col);
                    final Long style = col.getStyle();
                    if(attrs!=null || style!=null) {
                        addChangeColumns(sheetIndex, col.getMin()-1, col.getMax()-1, attrs, style);
                    }
                }
            }
        }
    }

    @Override
    public void addSetDocumentAttributesOperation(JSONObject attrs)
        throws JSONException {

        if(attrs!=null&&!attrs.isEmpty()) {
            final JSONObject setDocumentAttributesObject = new JSONObject(2);
            setDocumentAttributesObject.put("name", "setDocumentAttributes");
            setDocumentAttributesObject.put("attrs", attrs);
            operationsArray.put(setDocumentAttributesObject);
        }
    }

    public void addInsertSheetOperation(int sheetIndex, final String sheetName, final String type, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertSheetObject = new JSONObject();
        insertSheetObject.put("name", "insertSheet");
        insertSheetObject.put("sheet", sheetIndex);
        if(type!=null&&!type.equals("worksheet")) {
            insertSheetObject.put("type", type);
        }
        insertSheetObject.put("sheetName", sheetName);
        if(attrs!=null&&!attrs.isEmpty()) {
            insertSheetObject.put("attrs", attrs);
        }
        operationsArray.put(insertSheetObject);
    }

    public void addSetSheetAttributesOperation(int sheetIndex, final JSONObject attrs)
        throws JSONException {

        final JSONObject setSheetAttributesObject = new JSONObject();
        setSheetAttributesObject.put("name", "setSheetAttributes");
        setSheetAttributesObject.put("sheet", sheetIndex);
        setSheetAttributesObject.put("attrs", attrs);
        operationsArray.put(setSheetAttributesObject);
    }

    final static List<Object> objectList = new ArrayList<Object>();

    public void addChangeCellsOperation(int sheetIndex, SmlUtils.CellRef cellRef, JSONArray cellContent)
        throws JSONException {

        final JSONObject insertCellsObject = new JSONObject(4);
        insertCellsObject.put("name", "changeCells");
        insertCellsObject.put("sheet", sheetIndex);
        insertCellsObject.put("start", cellRef.getJSONArray());
        insertCellsObject.put("contents", cellContent);

        operationsArray.put(insertCellsObject);
    }

    public void addChangeRows(int sheetIndex, int start, int end, JSONObject attrs, Long style)
        throws JSONException {

        final JSONObject operation = new JSONObject();
        if (attrs!=null && attrs.length()>0) {
            operation.put("attrs", attrs);
        }
        if (style!=null && style.longValue()>0) {
            operation.put("s", "a" + Long.toString(style));
        }

        if (operation.length()>0) {
            operation.put("name", "changeRows");
            operation.put("sheet", sheetIndex);
            operation.put("start", start);
            if(end!=start) {
                operation.put("end", end);
            }
            operationsArray.put(operation);
        }
    }

    public void addChangeColumns(int sheetIndex, long start, long end, JSONObject attrs, Long style)
        throws JSONException {

        final JSONObject operation = new JSONObject();
        if (attrs!=null && attrs.length()>0) {
            operation.put("attrs", attrs);
        }
        if (style!=null && style.longValue()>0) {
            operation.put("s", "a" + Long.toString(style));
        }

        if (operation.length()>0) {
            operation.put("name", "changeColumns");
            operation.put("sheet", sheetIndex);
            operation.put("start", start);
            if(end>start) {
                operation.put("end", end);
            }

            operationsArray.put(operation);
        }
    }

    public void addInsertNameOperation(Long sheetIndex, String name, String formula, boolean hidden)
        throws JSONException {

        final JSONObject addInsertNameObject = new JSONObject(5);
        addInsertNameObject.put("name", "insertName");
        if(sheetIndex!=null) {
            addInsertNameObject.put("sheet", sheetIndex);
        }
        if(hidden) {
            addInsertNameObject.put("hidden", true);
        }
        addInsertNameObject.put("label", name);
        addInsertNameObject.put("formula", formula);

        operationsArray.put(addInsertNameObject);
    }
}
