/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.docx.tools;

import java.math.BigInteger;
import java.util.Iterator;

import org.docx4j.jaxb.Context;
import org.docx4j.wml.BooleanDefaultTrue;
import org.docx4j.wml.CTLanguage;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.CTVerticalAlignRun;
import org.docx4j.wml.Color;
import org.docx4j.wml.Highlight;
import org.docx4j.wml.HpsMeasure;
import org.docx4j.wml.RFonts;
import org.docx4j.wml.RPrBase;
import org.docx4j.wml.STVerticalAlignRun;
import org.docx4j.wml.U;
import org.docx4j.wml.UnderlineEnumeration;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.filter.api.FilterException;
import com.openexchange.office.filter.ooxml.docx.DocxOperationDocument;
import com.openexchange.office.filter.ooxml.tools.Commons;
import com.openexchange.office.filter.ooxml.tools.ThemeFonts;

public class Character {

    public static void applyCharacterProperties(DocxOperationDocument operationDocument, JSONObject characterProperties, RPrBase rPr )
        throws FilterException, JSONException {

        if(characterProperties==null)
            return;

        Iterator<String> keys = characterProperties.keys();
        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = characterProperties.get(attr);
            if(attr.equals("bold")) {
                if (value instanceof Boolean) {
                    final BooleanDefaultTrue booleanDefaultTrue = Context.getWmlObjectFactory().createBooleanDefaultTrue();
                    booleanDefaultTrue.setVal(((Boolean) value).booleanValue());
                    rPr.setB(booleanDefaultTrue);
                }
                else
                    rPr.setB(null);
            }
            else if(attr.equals("italic")) {
                if (value instanceof Boolean) {
                    final BooleanDefaultTrue booleanDefaultTrue = Context.getWmlObjectFactory().createBooleanDefaultTrue();
                    booleanDefaultTrue.setVal(((Boolean) value).booleanValue());
                    rPr.setI(booleanDefaultTrue);
                }
                else
                    rPr.setI(null);
            }
            else if(attr.equals("underline")){
                if (value instanceof Boolean) {
                    final U u = Context.getWmlObjectFactory().createU();
                    u.setParent(rPr);
                    u.setVal(((Boolean) value).booleanValue() ? UnderlineEnumeration.SINGLE : UnderlineEnumeration.NONE);
                    rPr.setU(u);
                }
                else rPr.setU(null);
            }
            else if(attr.equals("fontName")) {
                if (value instanceof String) {
                    RFonts fonts = rPr.getRFonts();
                    if (fonts==null) {
                        fonts = Context.getWmlObjectFactory().createRFonts();
                        fonts.setParent(rPr);
                        rPr.setRFonts(fonts);
                    }
                    // TODO: taking care of the character set which is used...
                    // only writing necessary fonts then...
                    fonts.setAscii((String)value);
                    fonts.setHAnsi((String)value);
                    fonts.setCs((String)value);
                    fonts.setEastAsia((String)value);
                }
                else
                    rPr.setRFonts(null);
            }
            else if(attr.equals("fontSize")) {
                if (value instanceof Number) {
                    HpsMeasure size = rPr.getSz();
                    if (size==null){
                        size = Context.getWmlObjectFactory().createHpsMeasure();
                        size.setParent(rPr);
                        rPr.setSz(size);
                    }
                    final Double hpsDouble = Double.valueOf((((Number)value).doubleValue()*2)+0.50);
                    size.setVal(BigInteger.valueOf(hpsDouble.longValue()));
                }
                else
                    rPr.setSz(null);
            }
            else if(attr.equals("color")) {
                if(value instanceof JSONObject) {
                    Color color = rPr.getColor();
                    if (color==null) {
                        color = Context.getWmlObjectFactory().createColor();
                        color.setParent(rPr);
                        rPr.setColor(color);
                    }
                    JSONObject colObject = (JSONObject)value;
                    Utils.initColorFromJSONColor(operationDocument, colObject, color);
                }
                else
                    rPr.setColor(null);
            }
            else if (attr.equals("fillColor")) {
                if (value instanceof JSONObject) {
                    CTShd shd = rPr.getShd();
                    if (shd==null) {
                        shd = Context.getWmlObjectFactory().createCTShd();
                        shd.setParent(rPr);
                        rPr.setShd(shd);
                    }
                    JSONObject shdObject = (JSONObject)value;
                    Utils.initShdFromJSONColor(operationDocument, shdObject, shd);
                    Highlight highlight = rPr.getHighlight();
                    if (highlight!=null) {
                        // reset possible highlight
                        rPr.setHighlight(null);
                    }
                }
                else {
                    rPr.setShd(null);
                    rPr.setHighlight(null);
                }
            }
            else if (attr.equals("noProof")) {
            	if(value instanceof Boolean) {
            		final BooleanDefaultTrue noProof = Context.getWmlObjectFactory().createBooleanDefaultTrue();
            		noProof.setVal(((Boolean)value).booleanValue());
            	}
            	else {
            		rPr.setNoProof(null);
            	}
            }
            else if(attr.equals("language")){
                if(value instanceof String){
                	rPr.getLang(true).setVal((String)value);
                }
                else {
                	final CTLanguage lang = rPr.getLang(false);
                    if(lang!=null) {
                        lang.setVal(null);
                    }
                }
            }
            else if(attr.equals("languageEa")){
                if( value instanceof String){
                	rPr.getLang(true).setEastAsia((String)value);
                }
                else {
                	final CTLanguage lang = rPr.getLang(false);
                    if(lang!=null) {
                        lang.setEastAsia(null);
                    }
                }
            }
            else if(attr.equals("languageBidi")){
                if(value instanceof String){
                	rPr.getLang(true).setBidi((String)value);
                }
                else {
                    final CTLanguage lang = rPr.getLang(false);
                    if(lang!=null) {
                        lang.setBidi(null);
                    }
                }
            }
            else if(attr.equals("vertAlign")){
                if( value instanceof String){
                    CTVerticalAlignRun vertAlign = rPr.getVertAlign();
                    if(vertAlign == null){
                        vertAlign = Context.getWmlObjectFactory().createCTVerticalAlignRun();
                        vertAlign.setParent(rPr);
                        rPr.setVertAlign(vertAlign);
                    }
                    String align = (String)value;
                    if( align.equals("sub"))
                        vertAlign.setVal(STVerticalAlignRun.SUBSCRIPT);
                    else if( align.equals("super"))
                        vertAlign.setVal(STVerticalAlignRun.SUPERSCRIPT);
                    else
                        vertAlign.setVal(STVerticalAlignRun.BASELINE);
                }
                else {
                    rPr.setVertAlign(null);
                }
            }
            else if(attr.equals("strike")) {
                if (value instanceof String) {
                    final BooleanDefaultTrue booleanDefaultTrue = Context.getWmlObjectFactory().createBooleanDefaultTrue();
                    String sValue = (String)value;
                    boolean isNone = sValue.equals("none");
                    booleanDefaultTrue.setVal(!isNone);
                    if(sValue.equals("single")){
                        rPr.setStrike(booleanDefaultTrue);
                        rPr.setDstrike(null);
                    }
                    else if( sValue.equals("double") ) {
                        rPr.setDstrike(booleanDefaultTrue);
                        rPr.setStrike(null);
                    } else {
                        if( rPr.getDstrike() != null )
                            rPr.setDstrike(null);
                        rPr.setStrike(booleanDefaultTrue);
                    }
                }
                else
                    rPr.setStrike(null);
                    rPr.setDstrike(null);
            }
            else if(attr.equals("caps")) {
            	if(value instanceof String) {
            		if(((String)value).equals("all")) {
            			rPr.setSmallCaps(null);
                        final BooleanDefaultTrue booleanDefaultTrue = Context.getWmlObjectFactory().createBooleanDefaultTrue();
                        booleanDefaultTrue.setVal(true);
            			rPr.setCaps(booleanDefaultTrue);
            		}
            		else if (((String)value).equals("small")) {
            			rPr.setCaps(null);
                        final BooleanDefaultTrue booleanDefaultTrue = Context.getWmlObjectFactory().createBooleanDefaultTrue();
                        booleanDefaultTrue.setVal(true);
            			rPr.setSmallCaps(booleanDefaultTrue);
            		}
            		else if (((String)value).equals("none")) {
            			rPr.setCaps(null);
            			rPr.setSmallCaps(null);
            		}
            	}
            	else {
            		rPr.setCaps(null);
            		rPr.setSmallCaps(null);
            	}
            }
        }
    }

    public static JSONObject createCharacterProperties(ThemeFonts themeFonts, RPrBase characterProperties)
        throws JSONException {

        JSONObject jsonCharacterProperties = null;
        if(characterProperties!=null) {
            jsonCharacterProperties = new JSONObject();
            BooleanDefaultTrue bold = characterProperties.getB();
            if(bold!=null)
                jsonCharacterProperties.put("bold", bold.isVal());
            BooleanDefaultTrue italic = characterProperties.getI();
            if(italic!=null)
                jsonCharacterProperties.put("italic", italic.isVal());
            U u = characterProperties.getU();
            if(u!=null) {
                boolean bUnderline = false;
                if (u.getVal()!=UnderlineEnumeration.NONE)
                    bUnderline = true;
                jsonCharacterProperties.put("underline", bUnderline);
            }
            Commons.jsonPut(jsonCharacterProperties, "fontName", themeFonts.getFont(characterProperties.getRFonts()));
            HpsMeasure size = characterProperties.getSz();
            if(size!=null) {
                jsonCharacterProperties.put("fontSize", size.getVal().doubleValue()/2);
            }
            Color color = characterProperties.getColor();
            if(color!=null)
                Commons.jsonPut(jsonCharacterProperties, "color", Utils.createColor(color));
            Highlight highlight = characterProperties.getHighlight();
            if(highlight!=null){
                Commons.jsonPut(
                    jsonCharacterProperties,
                    "fillColor",
                    Utils.createColor(null, Commons.mapHightlightColorToRgb(highlight.getVal())));
            }
            else {
                CTShd shd = characterProperties.getShd();
                if(shd!=null) {
                    Commons.jsonPut(
                        jsonCharacterProperties,
                        "fillColor",
                        Utils.createFillColor(shd));
                }
            }
            final BooleanDefaultTrue noProof = characterProperties.getNoProof();
            if(noProof != null) {
            	jsonCharacterProperties.put("noProof", noProof.isVal());
            }
            final CTLanguage lang = characterProperties.getLang(false);
            if(lang!= null){
                jsonCharacterProperties.put("language", lang.getVal());
                jsonCharacterProperties.put("languageEa", lang.getEastAsia());
                jsonCharacterProperties.put("languageBidi", lang.getBidi());
            }
            final CTVerticalAlignRun vertAlign = characterProperties.getVertAlign();
            if(vertAlign != null){
                STVerticalAlignRun eAlign = vertAlign.getVal();
                jsonCharacterProperties.put("vertAlign", eAlign == STVerticalAlignRun.SUBSCRIPT ? "sub" :
                        eAlign == STVerticalAlignRun.SUPERSCRIPT ? "super" : "baseline");
            }
            final BooleanDefaultTrue strike = characterProperties.getStrike();
            if(strike != null) {
                jsonCharacterProperties.put("strike", strike.isVal() ? "single" : "none");
            }
            final BooleanDefaultTrue dstrike = characterProperties.getDstrike();
            if(dstrike != null) {
                jsonCharacterProperties.put("strike", dstrike.isVal() ? "double" : "none");
            }
            final BooleanDefaultTrue caps = characterProperties.getCaps();
            if(caps!=null&&caps.isVal()) {
            	jsonCharacterProperties.put("caps", "all");
            }
            else {
            	final BooleanDefaultTrue smallCaps = characterProperties.getSmallCaps();
            	if(smallCaps!=null&&smallCaps.isVal()) {
            		jsonCharacterProperties.put("caps", "small");
            	}
            }
        }
        return jsonCharacterProperties!=null&&jsonCharacterProperties.length()>0 ? jsonCharacterProperties : null;
    }
}
