/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.docx.components;

import java.util.HashSet;
import javax.xml.bind.JAXBException;
import javax.xml.datatype.XMLGregorianCalendar;
import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.XmlUtils;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.wml.Br;
import org.docx4j.wml.CTBookmark;
import org.docx4j.wml.CTMarkupRange;
import org.docx4j.wml.CTPPrChange;
import org.docx4j.wml.CTParaRPrOriginal;
import org.docx4j.wml.CTTrackChange;
import org.docx4j.wml.CommentRangeEnd;
import org.docx4j.wml.CommentRangeStart;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.Drawing;
import org.docx4j.wml.FldChar;
import org.docx4j.wml.IText;
import org.docx4j.wml.P;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase;
import org.docx4j.wml.PPrBase.PStyle;
import org.docx4j.wml.ParaRPr;
import org.docx4j.wml.ParaRPrChange;
import org.docx4j.wml.R;
import org.docx4j.wml.RPr;
import org.docx4j.wml.RunDel;
import org.docx4j.wml.RunIns;
import org.docx4j.wml.STBrType;
import org.docx4j.wml.STFldCharType;
import org.docx4j.wml.SdtBlock;
import org.docx4j.wml.SdtRun;
import org.json.JSONException;
import org.json.JSONObject;
import org.jvnet.jaxb2_commons.ppp.Child;
import com.openexchange.office.filter.ooxml.components.Component;
import com.openexchange.office.filter.ooxml.components.ComponentContext;
import com.openexchange.office.filter.ooxml.components.IParagraph;
import com.openexchange.office.filter.ooxml.docx.tools.Character;
import com.openexchange.office.filter.ooxml.docx.tools.DrawingML;
import com.openexchange.office.filter.ooxml.docx.tools.DrawingML.DrawingML_root;
import com.openexchange.office.filter.ooxml.docx.tools.Paragraph;
import com.openexchange.office.filter.ooxml.docx.tools.TextUtils;
import com.openexchange.office.filter.ooxml.docx.tools.Utils;

public class ParagraphComponent extends DocxComponent implements IParagraph {

    public ParagraphComponent(ComponentContext parentContext, IndexedNode<Object> _node, int _componentNumber) {
        super(parentContext, _node, _componentNumber);
    }

    @Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
        final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
        final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

        Component nextComponent = null;
        final IndexedNode<Object> paragraphNode = getNode();
        final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)paragraphNode.getData()).getContent();
        for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
            final IndexedNode<Object> childNode = nodeList.getNode(i);
            final Object o = getContentModel(childNode, paragraphNode.getData());
            if(o instanceof R) {
                final TextRunContext textRunContext = new TextRunContext(this, childNode);
                nextComponent = textRunContext.getNextChildComponent(null, previousChildComponent);
            }
            else if(o instanceof P.Hyperlink) {
                final HyperlinkContext hyperlinkContext = new HyperlinkContext(this, childNode);
                nextComponent = hyperlinkContext.getNextChildComponent(null, previousChildComponent);
            }
            else if(o instanceof RunIns) {
                final RunInsContext runInsContext = new RunInsContext(this, childNode);
                nextComponent = runInsContext.getNextChildComponent(null, previousChildComponent);
            }
            else if(o instanceof RunDel) {
                final RunDelContext runDelContext = new RunDelContext(this, childNode);
                nextComponent = runDelContext.getNextChildComponent(null, previousChildComponent);
            }
            else if(o instanceof CommentRangeStart) {
            	nextComponent = new CommentRangeStartComponent(this, childNode, nextComponentNumber);
            }
            else if(o instanceof CommentRangeEnd) {
            	nextComponent = new CommentRangeEndComponent(this, childNode, nextComponentNumber);
            }
            else if(o.getClass()==CTBookmark.class) {
                nextComponent = new BookmarkStartComponent(this, childNode, nextComponentNumber);
            }
            else if(o.getClass()==CTMarkupRange.class) {
                nextComponent = new BookmarkEndComponent(this, childNode, nextComponentNumber);
            }
            else if(o instanceof SdtRun) {
                final SdtParagraphContext sdtParagraphContext = new SdtParagraphContext(this, childNode);
                nextComponent = sdtParagraphContext.getNextChildComponent(null, previousChildComponent);
            }
        }
        return nextComponent;
    }
    @Override
    public Component insertChildComponent(com.openexchange.office.filter.ooxml.OperationDocument operationDocument, int textPosition, JSONObject attrs, Type childType)
        throws JSONException, JAXBException, InvalidFormatException, PartUnrecognisedException {

    	boolean textRunRequired = true;
        Child newChild = null;
        switch(childType) {
            case TAB : {
                newChild = Context.getWmlObjectFactory().createRTab();
                break;
            }
            case HARDBREAK_DEFAULT :
            case HARDBREAK_PAGE :
            case HARDBREAK_COLUMN : {
                newChild = Context.getWmlObjectFactory().createBr();
                if(childType==Type.HARDBREAK_PAGE) {
                    ((Br)newChild).setType(STBrType.PAGE);
                }
                else if(childType==Type.HARDBREAK_COLUMN) {
                    ((Br)newChild).setType(STBrType.COLUMN);
                }
                break;
            }
            case COMPLEX_FIELD_START :
            case COMPLEX_FIELD_SEPARATE :
            case COMPLEX_FIELD_END : {
            	newChild = Context.getWmlObjectFactory().createFldChar();
            	((FldChar)newChild).setFldCharType(STFldCharType.BEGIN);
            	if(childType==Type.COMPLEX_FIELD_END) {
                	((FldChar)newChild).setFldCharType(STFldCharType.END);
            	}
            	else if(childType==Type.COMPLEX_FIELD_SEPARATE) {
                	((FldChar)newChild).setFldCharType(STFldCharType.SEPARATE);
            	}
            	else {
            		((FldChar)newChild).setFldLock(false);
            	}
            	break;
            }
            case COMMENT_REFERENCE : {
            	newChild = Context.getWmlObjectFactory().createRCommentReference();
            	break;
            }
            case COMMENT_RANGE_START : {
            	newChild = Context.getWmlObjectFactory().createCommentRangeStart();
            	textRunRequired = false;
            	break;
            }
            case COMMENT_RANGE_END : {
            	newChild = Context.getWmlObjectFactory().createCommentRangeEnd();
            	textRunRequired = false;
            	break;
            }
            case BOOKMARK_START: {
                newChild = new CTBookmark();
                textRunRequired = false;
                break;
            }
            case BOOKMARK_END: {
                newChild = new CTMarkupRange();
                textRunRequired = false;
                break;
            }
            case AC_IMAGE : {
                final Drawing drawing = Context.getWmlObjectFactory().createDrawing();
                drawing.getAnchorOrInline().add(DrawingML_root.createInline(false, "", 500000, 500000));
                newChild = drawing;
                break;
            }
            case AC_SHAPE : {
                newChild = DrawingML.createAlternateContentShape();
                break;
            }
            case AC_CONNECTOR : {
                newChild = DrawingML.createAlternateContentConnector();
                break;
            }
            case AC_GROUP : {
                newChild = DrawingML.createAlternateContentGroup();
                break;
            }
            default : {
                throw new UnsupportedOperationException();
            }
        }
        final P paragraph = (P)getObject();
        final IndexedNodeList<Object> paragraphContent = paragraph.getContent();

        Component childComponent = getNextChildComponent(null, null);
        if(childComponent!=null) {
            if(textPosition>0) {
                childComponent = childComponent.getComponent(textPosition-1);
                childComponent.splitEnd(textPosition-1, SplitMode.ATTRIBUTE);
            }
            if(textRunRequired) {

            	if(childComponent instanceof TextRun_Base) {

            		// the new child can be added into an existing textRun
                    final R run = ((TextRun_Base)childComponent).getTextRun();
                    newChild.setParent(run);
                    run.getContent().addNode(((TextRun_Base)childComponent).getNode(), new IndexedNode<Object>(newChild), textPosition==0);
                }
                else {

                	// we need to create a new textRun, check if we can get referenceAttributes that we will clone for the new run. this is
                	// only possible if the childComponent is a CommentRangeStartComponent or an CommentRangeEndComponent
                	final R newRun = Context.getWmlObjectFactory().createR();
                	newChild.setParent(newRun);
                	newRun.getContent().add(newChild);

                	final ContentAccessor parentContextObject = (ContentAccessor)childComponent.getParentContext().getNode().getData();
                	((IndexedNodeList<Object>)parentContextObject.getContent()).addNode(childComponent.getNode(), new IndexedNode<Object>(newRun), textPosition==0);
                }
            }
            else {
                final HashSet<Class<?>> parentContextList = new HashSet<Class<?>>(1);
                parentContextList.add(SdtParagraphContext.class);
                parentContextList.add(HyperlinkContext.class);
                parentContextList.add(RunInsContext.class);
                parentContextList.add(RunDelContext.class);
                final ComponentContext contextChild = childComponent.getContextChild(parentContextList);
                final ComponentContext parentContext = contextChild.getParentContext();
                final ContentAccessor parentObject = (ContentAccessor)parentContext.getNode().getData();
                newChild.setParent(parentObject);
                ((IndexedNodeList<Object>)parentObject.getContent()).addNode(contextChild.getNode(), new IndexedNode<Object>(newChild), textPosition==0);
            }
        }
        else {
        	if(textRunRequired) {
                final R newRun = Context.getWmlObjectFactory().createR();
                newRun.setParent(paragraph);
                paragraphContent.add(newRun);
                newChild.setParent(newRun);
                newRun.getContent().add(newChild);
                if(paragraph.getPPr()!=null) {
                    final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
                    if(rPr!=null) {
                        rPr.setParent(newRun);
                        newRun.setRPr(rPr);
                    }
                }
        	}
        	else {
        		newChild.setParent(paragraph);
        		paragraphContent.add(newChild);
        	}
        }
        if(textPosition>0) {
            childComponent = childComponent.getNextComponent();
        }
        else {
            childComponent = getNextChildComponent(null, null);
        }
        if(childComponent instanceof TextRun_Base) {
            if(((TextRun_Base)childComponent).getCT()!=TextRun_Base.CT_NONE) {
                attrs = TextUtils.forceRemoveChangeTrack(attrs);
            }
        }
        if(attrs!=null) {
            childComponent.splitStart(textPosition, SplitMode.ATTRIBUTE);
            childComponent.splitEnd(textPosition, SplitMode.ATTRIBUTE);
            childComponent.applyAttrsFromJSON(operationDocument, attrs);
        }
        return childComponent;
    }

    public static void splitAccessor(Component component, boolean splitStart, SplitMode splitMode) {
        IndexedNode<Object> node = component.getNode();
        ComponentContext parentContext = component.getParentContext();
        while(parentContext!=null) {
        	if(parentContext instanceof ParagraphComponent) {
        		break;
        	}
        	if(splitMode==SplitMode.ATTRIBUTE) {
        		if(parentContext instanceof SdtParagraphContext) {
        			break;
        		}
        	}

            final IndexedNode<Object> parentNode = parentContext.getNode();
            final IndexedNodeList<Object> parentNodeContent = (IndexedNodeList<Object>)((ContentAccessor)parentNode.getData()).getContent();
            if((splitStart&&node.getIndex()!=0)||(!splitStart&&node.getIndex()!=parentNodeContent.size()-1)) {
                IndexedNode<Object> newParentNode = null;
                if(parentContext instanceof TextRunContext) {
                    final R parent = (R)parentNode.getData();
                    final R newParent = Context.getWmlObjectFactory().createR();
                    newParentNode = new IndexedNode<Object>(newParent);
                    final RPr rPr = ((R)parent).getRPr();
                    if(rPr!=null) {
                    	final RPr rPrNew = XmlUtils.deepCopy(rPr);
                    	TextUtils.resetMarkupId(rPrNew);
                        rPrNew.setParent(newParent);
                        newParent.setRPr(rPrNew);
                    }
                    if(splitStart&&component instanceof TextRun_Base) {
                        ((TextRun_Base)component).textRunNode = newParentNode;
                    }
                }
                else if(parentContext instanceof HyperlinkContext) {
                    final P.Hyperlink parent = (P.Hyperlink)parentNode.getData();
                    final P.Hyperlink newParent = Context.getWmlObjectFactory().createPHyperlink();
                    newParentNode = new IndexedNode<Object>(newParent);
                    newParent.setAnchor(parent.getAnchor());
                    newParent.setDocLocation(parent.getDocLocation());
                    newParent.setHistory(parent.isHistory());
                    newParent.setId(parent.getId());
                    newParent.setTgtFrame(parent.getTgtFrame());
                    newParent.setTooltip(parent.getTooltip());
                    if(splitStart&&component instanceof TextRun_Base) {
                    	((TextRun_Base)component).hyperlinkNode = newParentNode;
                    }
                }
                else if(parentContext instanceof RunInsContext) {
                    final RunIns parent = (RunIns)parentNode.getData();
                    final RunIns newParent = Context.getWmlObjectFactory().createRunIns();
                    newParentNode = new IndexedNode<Object>(newParent);
                    newParent.setAuthor(parent.getAuthor());
                    final XMLGregorianCalendar date = parent.getDate();
                    if(date!=null) {
                        newParent.setDate((XMLGregorianCalendar) date.clone());
                    }
                    if(splitStart&&component instanceof TextRun_Base) {
                    	((TextRun_Base)component).runInsNode = newParentNode;
                    }
                }
                else if(parentContext instanceof RunDelContext) {
                    final RunDel parent = (RunDel)parentNode.getData();
                    final RunDel newParent = Context.getWmlObjectFactory().createRunDel();
                    newParentNode = new IndexedNode<Object>(newParent);
                    newParent.setAuthor(parent.getAuthor());
                    final XMLGregorianCalendar date = parent.getDate();
                    if(date!=null) {
                        newParent.setDate((XMLGregorianCalendar)date.clone());
                    }
                    if(splitStart&&component instanceof TextRun_Base) {
                    	((TextRun_Base)component).runDelNode = newParentNode;
                    }
                }
                else if(parentContext instanceof SdtParagraphContext&&splitMode==SplitMode.DELETE) {
                    final SdtRun parent = (SdtRun)parentNode.getData();
                    final SdtRun newParent = Context.getWmlObjectFactory().createSdtRun();
                    newParentNode = new IndexedNode<Object>(newParent);
                    newParent.setSdtEndPr(parent.getSdtEndPr());
                    newParent.setSdtPr(parent.getSdtPr());
                }
                if(newParentNode!=null) {
                    ((Child)newParentNode.getData()).setParent(parentNode.getData());
                    final IndexedNodeList<Object> rContent = (IndexedNodeList<Object>)((ContentAccessor)parentNode.getData()).getContent();
                    final IndexedNodeList<Object> rContentNew = (IndexedNodeList<Object>)((ContentAccessor)newParentNode.getData()).getContent();
                    final Object pParent = ((Child)parentNode.getData()).getParent();
                    final Object pNewParent = newParentNode.getData();
                    ((Child)pNewParent).setParent(pParent);
                    final IndexedNodeList<Object> pParentNode = (IndexedNodeList<Object>)((ContentAccessor)pParent).getContent();
                    pParentNode.addNode(parentNode, newParentNode, false);
                    int index = node.getIndex();
                    if(!splitStart) {
                        index++;
                    }
                    final int count = rContent.size() - index;
                    if(count>0) {
                        IndexedNodeList.moveNodes(rContent, rContentNew, index, rContentNew.size(), count, pNewParent);
                    }
                    if(splitStart) {
                        parentContext.setNode(newParentNode);
                    }
                }
            }
            node = parentContext.getNode();
            parentContext = parentContext.getParentContext();
        }
    }

    @Override
	public void applyAttrsFromJSON(com.openexchange.office.filter.ooxml.OperationDocument operationDocument, JSONObject attrs) {
		if(attrs==null) {
			return;
		}
		try {
	        final P paragraph = (P)getObject();
	        PPr pPr = paragraph.getPPr();
	        if (pPr==null) {
	            pPr = Context.getWmlObjectFactory().createPPr();
	            pPr.setParent(paragraph);
	            paragraph.setPPr(pPr);
	        }
	        if(attrs.has("styleId")) {
	            final Object styleId = attrs.get("styleId");
	            if(styleId instanceof String) {
	                PPrBase.PStyle pStyle = pPr.getPStyle();
	                if(pStyle==null) {
	                    pStyle = Context.getWmlObjectFactory().createPPrBasePStyle();
	                    pStyle.setParent(pPr);
	                }
	                pStyle.setVal((String)styleId);
	                pPr.setPStyle(pStyle);
	            }
	            else {
	                pPr.setPStyle(null);
	            }
	        }
	        Paragraph.applyParagraphProperties((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, attrs.optJSONObject("paragraph"), pPr);
	        final Object characterAttrs = attrs.opt("character");
	        if(characterAttrs instanceof JSONObject) {
	            ParaRPr paraRPr = pPr.getRPr();
	            if(paraRPr==null) {
	                paraRPr = Context.getWmlObjectFactory().createParaRPr();
	                paraRPr.setParent(pPr);
	                pPr.setRPr(paraRPr);
	            }
	            com.openexchange.office.filter.ooxml.docx.tools.Character.applyCharacterProperties((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)characterAttrs, paraRPr);
	        }
            final Object changes = attrs.opt("changes");
            if(changes!=null) {
	            if(changes instanceof JSONObject) {
            		ParaRPr paraRPr = pPr.getRPr();
            		if(paraRPr==null) {
            			paraRPr = Context.getWmlObjectFactory().createParaRPr();
            			paraRPr.setParent(pPr);
            			pPr.setRPr(paraRPr);
            		}
	            	final Object attrsInserted = ((JSONObject)changes).opt("inserted");
	            	if(attrsInserted!=null) {
	            		if(attrsInserted instanceof JSONObject) {
	            			CTTrackChange ins = paraRPr.getIns();
	            			if(ins==null) {
	            				ins = Context.getWmlObjectFactory().createCTTrackChange();
	            				ins.setParent(paraRPr);
	            				paraRPr.setIns(ins);
	            			}
                			Utils.applyTrackInfoFromJSON((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)attrsInserted, ins);
	            		}
	            		else {
	            			paraRPr.setIns(null);
	            		}
	            	}
	            	final Object attrsRemoved = ((JSONObject)changes).opt("removed");
	            	if(attrsRemoved!=null) {
	            		if(attrsRemoved instanceof JSONObject) {
	            			CTTrackChange del = paraRPr.getDel();
	            			if(del==null) {
	            				del = Context.getWmlObjectFactory().createCTTrackChange();
	            				del.setParent(paraRPr);
	            				paraRPr.setDel(del);
	            			}
                			Utils.applyTrackInfoFromJSON((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)attrsRemoved, del);
	            		}
	            		else {
	            			paraRPr.setDel(null);
	            		}
	            	}
	            	final Object modified = ((JSONObject)changes).opt("modified");
	            	if(modified!=null) {
	            		if(modified instanceof JSONObject) {
      				        ParaRPrChange rPrChange = paraRPr.getRPrChange();
		            		if(rPrChange==null) {
		            			rPrChange = Context.getWmlObjectFactory().createParaRPrChange();
		            			rPrChange.setParent(paraRPr);
		            			paraRPr.setRPrChange(rPrChange);
		            		}
        					CTPPrChange pPrChange = pPr.getPPrChange();
        					if(pPrChange==null) {
        						pPrChange = Context.getWmlObjectFactory().createCTPPrChange();
        						pPrChange.setParent(pPr);
        						pPr.setPPrChange(pPrChange);
        					}
                			Utils.applyTrackInfoFromJSON((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)modified, rPrChange);
                			Utils.applyTrackInfoFromJSON((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)modified, pPrChange);
                			final Object attrsModified = ((JSONObject)modified).opt("attrs");
                			if(attrsModified!=null) {
                				if(attrsModified instanceof JSONObject) {
		                			// Style
                					if(((JSONObject) attrsModified).has("styleId")) {
        					            final Object styleId = attrs.get("styleId");
        					            PPrBase pPrBase = pPrChange.getPPr();
        					            if(pPrBase==null) {
        					            	pPrBase = Context.getWmlObjectFactory().createPPrBase();
        					            	pPrBase.setParent(pPrChange);
        					            	pPrChange.setPPr(pPrBase);
        					            }
         					            if(styleId instanceof String) {
        					                PPrBase.PStyle pStyle = pPrBase.getPStyle();
        					                if(pStyle==null) {
        					                    pStyle = Context.getWmlObjectFactory().createPPrBasePStyle();
        					                    pStyle.setParent(pPrBase);
        					                }
        					                pStyle.setVal((String)styleId);
        					                pPrBase.setPStyle(pStyle);
        					            }
        					            else {
        					                pPrBase.setPStyle(null);
        					            }
                					}
                					final Object characterChanges = ((JSONObject)attrsModified).opt("character");
			            			if(characterChanges!=null) {
			            				if(characterChanges instanceof JSONObject) {
			            					CTParaRPrOriginal paraRPrOriginal = rPrChange.getRPr();
			            					if(paraRPrOriginal==null) {
			            						paraRPrOriginal = Context.getWmlObjectFactory().createCTParaRPrOriginal();
			            						paraRPrOriginal.setParent(rPrChange);
			            						rPrChange.setRPr(paraRPrOriginal);
			            					}
			            					Character.applyCharacterProperties((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)characterChanges, paraRPrOriginal);
			            				}
			            				else {
			            					rPrChange.setRPr(null);
			            				}
			            			}
			            			final Object paragraphChanges = ((JSONObject)attrsModified).opt("paragraph");
			            			if(paragraphChanges!=null) {
			            				if(paragraphChanges instanceof JSONObject) {
	            							PPrBase pPrBase = pPrChange.getPPr();
	            							if(pPrBase==null) {
	            								pPrBase = Context.getWmlObjectFactory().createPPrBase();
	            								pPrBase.setParent(pPrChange);
	            								pPrChange.setPPr(pPrBase);
	            							}
	            							Paragraph.applyParagraphProperties((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)paragraphChanges, pPrBase);
			            				}
			            				else {
			            					pPr.setPPrChange(null);
			            				}
			            			}
                				}
                				else {
                					paraRPr.setRPrChange(null);
                					pPr.setPPrChange(null);
                				}
                			}
	            		}
	            		else {
	            			paraRPr.setRPrChange(null);
	            			pPr.setPPrChange(null);
	            		}
	            	}
	            }
	            else {
	            	final ParaRPr paraRPr = pPr.getRPr();
	            	if(paraRPr!=null) {
	            		paraRPr.setIns(null);
	            		paraRPr.setDel(null);
	            		paraRPr.setRPrChange(null);
	            	}
	            }
            }
		}
        catch (Exception e) {
		}
	}

	@Override
	public JSONObject createJSONAttrs(com.openexchange.office.filter.ooxml.operations.CreateOperationHelper createOperationHelper, JSONObject attrs)
		throws JSONException {

		final PPr pPr = ((P)getObject()).getPPr();
        if(pPr!=null) {
            final PPrBase.PStyle pStyle = pPr.getPStyle();
            if(pStyle!=null) {
                attrs.put("styleId", pStyle.getVal());
            }
            final JSONObject paragraphProperties = Paragraph.createParagraphProperties(pPr);
            if(paragraphProperties!=null) {
                attrs.put("paragraph", paragraphProperties);
            }
            final ParaRPr paraRPr = pPr.getRPr();
            if(paraRPr!=null) {
            	// only creating paragraph character attributes if there are no child elements #35344#, #35571#
            	if(getNextChildComponent(null, null)==null) {
	                final JSONObject characterProperties = Character.createCharacterProperties(createOperationHelper.getOperationDocument().getThemeFonts(), paraRPr);
	                if(characterProperties!=null) {
	                    attrs.put("character", characterProperties);
	                }
            	}
            }

            CTTrackChange rPrIns = null;
        	CTTrackChange rPrDel = null;
        	ParaRPrChange rPrChange = null;
        	if(paraRPr!=null) {
        		rPrIns = paraRPr.getIns();
        		rPrDel = paraRPr.getDel();
        		rPrChange = paraRPr.getRPrChange();
        	}
        	final CTPPrChange pPrChange = pPr.getPPrChange();
            if(rPrIns!=null||rPrDel!=null||rPrChange!=null||pPrChange!=null) {
            	final JSONObject changes = new JSONObject(2);
            	if(rPrIns!=null) {
            		changes.put("inserted", Utils.createJSONFromTrackInfo((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)createOperationHelper.getOperationDocument(), rPrIns));
                }
                if(rPrDel!=null) {
                	changes.put("removed", Utils.createJSONFromTrackInfo((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)createOperationHelper.getOperationDocument(), rPrDel));
                }
                if(rPrChange!=null||pPrChange!=null) {
                	JSONObject modified = null;
                	JSONObject modifiedAttrs = new JSONObject(3);
                	if(rPrChange!=null) {
 	                	if(modified==null) {
 	                		modified = Utils.createJSONFromTrackInfo((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)createOperationHelper.getOperationDocument(), rPrChange);
 	                	}
	                	final CTParaRPrOriginal rPrOriginal  = rPrChange.getRPr();
                    	if(rPrOriginal!=null) {
	                    	final JSONObject changedJsonCharacterAttributes = Character.createCharacterProperties(createOperationHelper.getOperationDocument().getThemeFonts(), rPrOriginal);
	                    	if(changedJsonCharacterAttributes!=null) {
	                    		modifiedAttrs.put("character", changedJsonCharacterAttributes);
	                    	}
                    	}
                	}
                	if(pPrChange!=null) {
 	                	if(modified==null) {
 	                		modified = Utils.createJSONFromTrackInfo((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)createOperationHelper.getOperationDocument(), pPrChange);
 	                	}
 	                	final PPrBase pPrOriginal = pPrChange.getPPr();
 	                	if(pPrOriginal!=null) {
	 	                	final PStyle originalStyle = pPrOriginal.getPStyle();
		 	   	            if(originalStyle!=null) {
		 		                modifiedAttrs.put("styleId", originalStyle.getVal());
		 		            }
 	                	}
 	                	final JSONObject changedJsonParagraphAttributes = Paragraph.createParagraphProperties(pPrOriginal);
 	                	if(changedJsonParagraphAttributes!=null) {
 	                		modifiedAttrs.put("paragraph", changedJsonParagraphAttributes);
 	                	}
                	}

                	if(!modifiedAttrs.isEmpty()) {
                		modified.put("attrs", modifiedAttrs);
                	}
                	if(!modified.isEmpty()) {
                		changes.put("modified", modified);
                	}
                }
                if(!changes.isEmpty()) {
                	attrs.put("changes", changes);
                }
            }
        }
        return attrs;
	}

	@Override
	public void insertText(com.openexchange.office.filter.ooxml.OperationDocument operationDocument, int textPosition, String text, JSONObject attrs)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

    	if(text.length()>0) {
        	final P paragraph = (P)getObject();

            IText t = null;
            Component childComponent = getNextChildComponent(null, null);
            Component cRet = null;

            if(childComponent!=null) {
            	if(textPosition>0) {
            		childComponent = childComponent.getComponent(textPosition-1);
            	}
                // check if the character could be inserted into an existing text:
                if(childComponent instanceof TextComponent) {
                    t = (IText)childComponent.getObject();
                    final StringBuffer s = new StringBuffer(t.getValue());
                    s.insert(textPosition-((TextComponent)childComponent).getComponentNumber(), text);
                    t.setValue(s.toString());
                    cRet = childComponent;
                }
                else {

                	t = Context.getWmlObjectFactory().createText();
                    t.setValue(text);
                	if(childComponent instanceof TextRun_Base) {

                		// jackpot, we can insert the text into a existing run
                    	final R oldRun = ((TextRun_Base)childComponent).getTextRun();
                        t.setParent(oldRun);
                        oldRun.getContent().addNode(childComponent.getNode(), new IndexedNode<Object>(t), textPosition==0);
                        cRet = childComponent;
                	}
                	else {

                		// childComponent will be CommentRangeStartComponent, CommentRangeEndComponent or FldSimpleComponent
                        final R newRun = Context.getWmlObjectFactory().createR();
                        t.setParent(newRun);
                        newRun.getContent().add(t);

                        final IndexedNode<Object> parentNode = childComponent.getParentContext().getNode();
                		final ContentAccessor parentContent = (ContentAccessor)parentNode.getData();
                		newRun.setParent(parentNode.getData());
                		((IndexedNodeList<Object>)parentContent.getContent()).addNode(childComponent.getNode(), new IndexedNode<Object>(newRun), textPosition==0);
                	}
                    if(textPosition>0) {
                        cRet = childComponent.getNextComponent();
                    }
                    else {
                        cRet = getNextChildComponent(null, null);
                    }
                }
            }
            else {

            	// the paragraph is empty, we have to create R and its text
                final R newRun = Context.getWmlObjectFactory().createR();
                newRun.setParent(getObject());
                paragraph.getContent().add(newRun);
                if(paragraph.getPPr()!=null) {
                    final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
                    if(rPr!=null) {
                        rPr.setParent(newRun);
                        newRun.setRPr(rPr);
                    }
                }
                t = Context.getWmlObjectFactory().createText();
                t.setParent(newRun);
                t.setValue(text);
                newRun.getContent().add(t);
                cRet = getNextChildComponent(null, null);
            }
			if(t!=null) {
				TextComponent.preserveSpace(t);
			}
			if(cRet instanceof TextRun_Base&&((TextRun_Base)cRet).getCT()!=TextRun_Base.CT_NONE) {
				attrs = TextUtils.forceRemoveChangeTrack(attrs);
			}
            if(attrs!=null) {
            	cRet.splitStart(textPosition, SplitMode.ATTRIBUTE);
            	cRet.splitEnd(textPosition+text.length()-1, SplitMode.ATTRIBUTE);
            	cRet.applyAttrsFromJSON(operationDocument, attrs);
            }
        }
	}

    public void splitParagraph(int textPosition) {

    	final P paragraph = (P)getObject();

        RPr lastRPr = null;
    	Component component = getNextChildComponent(null, null);
    	while(component!=null&&component.getNextComponentNumber()<=textPosition) {
    		if(component instanceof TextRun_Base) {
    			lastRPr = ((TextRun_Base)component).getTextRun().getRPr();
    		}
    		component = component.getNextComponent();
    	}
        if(component!=null) {
        	component.splitStart(textPosition, SplitMode.ATTRIBUTE);
        }
        // creating our new paragraph
        final P destParagraph = Context.getWmlObjectFactory().createP();
        destParagraph.setParent(paragraph.getParent());
        final IndexedNode<Object> destParagraphNode = new IndexedNode<Object>(destParagraph);
        Utils.getContent(paragraph.getParent()).addNode(getNode(), destParagraphNode, false);
        final PPr sourceParagraphProperties = paragraph.getPPr();
        if(sourceParagraphProperties!=null){
            final PPr destParagraphProperties = XmlUtils.deepCopy(sourceParagraphProperties);
            // no inheritance for pageBreakBefore attribute... this attribute is always false

            TextUtils.resetMarkupId(destParagraphProperties);

            destParagraphProperties.setPageBreakBefore(null);
            destParagraphProperties.setParent(destParagraph);
            destParagraph.setPPr(destParagraphProperties);
        }

        // TODO: to be optimized.. the component is already the correct source component,
        // the iteration is only needed to get the previous RPr
        while(component!=null) {
            if(textPosition>=component.getComponentNumber()&&textPosition<component.getNextComponentNumber()) {
                boolean skipSdt = false;
                ComponentContext childContext = component;
                ComponentContext parentContext = component.getParentContext();
                while(!(childContext instanceof ParagraphComponent)) {
                    if(parentContext instanceof SdtParagraphContext) {
                        if(childContext.getNode().getIndex()>0) {
                            // moving end-part of the sdt into the next paragraph
                            final IndexedNodeList<Object> sourceContent = (IndexedNodeList<Object>)((ContentAccessor)parentContext.getNode().getData()).getContent();
                            final int index = childContext.getNode().getIndex();
                            final int count = sourceContent.size() - index;
                            IndexedNodeList.moveNodes(sourceContent, destParagraph.getContent(), index, destParagraph.getContent().size(), count, destParagraph);
                            skipSdt = true;
                        }
                    }
                    else if(parentContext instanceof ParagraphComponent) {
                        final IndexedNode<Object> contextChildNode = childContext.getNode();
                        final IndexedNodeList<Object> parentContent = (IndexedNodeList<Object>)((ContentAccessor)parentContext.getNode().getData()).getContent();
                        int index = contextChildNode.getIndex();
                        if(skipSdt) {
                            index++;
                        }
                        final int count = parentContent.size() - index;
                        if(count>0) {
                            IndexedNodeList.moveNodes(parentContent, destParagraph.getContent(), index, destParagraph.getContent().size(), count, destParagraph);
                        }
                    }
                    childContext = parentContext;
                    parentContext = childContext.getParentContext();
                }
                break;
            } else if (component instanceof TextRun_Base) {
                lastRPr = ((TextRun_Base)component).getTextRun().getRPr();
            }
            component = component.getNextComponent();
        }

        // taking care of paragraph attributes
        if(lastRPr!=null) {

            // if available, we have to get the character attributes from the last textrun
            PPr paraProps = destParagraph.getPPr();
            if(paraProps==null) {
                paraProps = Context.getWmlObjectFactory().createPPr();
                paraProps.setParent(destParagraph);
                destParagraph.setPPr(paraProps);
            }

            // 30793 ... only setting character attributes if no listStyle is used at the paragraph
            if(paraProps.getNumPr()==null||paraProps.getNumPr().getNumId()==null) {
                ParaRPr paraRPr = TextUtils.cloneRPrToParaRPr(lastRPr);
                if(paraRPr!=null) {
                    paraRPr.setParent(paraProps);
                    paraProps.setRPr(paraRPr);
                }
            }
        }
    }

    public void mergeParagraph() {
    	final Component nextParagraphComponent = getNextComponent();
    	if(nextParagraphComponent instanceof ParagraphComponent) {
        	final P paragraph = (P)getObject();
        	final IndexedNodeList<Object> sourceContent = ((P)nextParagraphComponent.getObject()).getContent();
            final IndexedNodeList<Object> destContent = paragraph.getContent();
            if(sourceContent.size()>0) {
            	IndexedNodeList.moveNodes(sourceContent, destContent, 0, destContent.size(), sourceContent.size(), paragraph);
            }
            IndexedNode<Object> parentContextNode = nextParagraphComponent.getParentContext().getNode();
        	Utils.getContent(parentContextNode.getData()).removeNodes(nextParagraphComponent.getNode(), null);
        	if(parentContextNode.getData() instanceof SdtBlock) {
        	    final ComponentContext sdtRootContext = nextParagraphComponent.getParentContext();
        	    final ComponentContext parentOfSdtRootContext = sdtRootContext.getParentContext();
                if(((ContentAccessor)sdtRootContext.getNode().getData()).getContent().isEmpty()) {
                    Utils.getContent(parentOfSdtRootContext.getNode().getData()).removeNodes(sdtRootContext.getNode(), null);
                }
        	}
    	}
    }
}
