/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odt.dom;

import java.util.Iterator;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.OdfTextDocument;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.api.FilterException;
import com.openexchange.office.filter.api.FilterException.ErrorCode;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.IParagraph;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.Settings;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.styles.MasterStyles;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleHeaderFooter;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.odf.styles.StyleMasterPage;
import com.openexchange.office.filter.odf.styles.StylePageLayout;
import com.openexchange.office.filter.odf.styles.TextListStyle;
import com.openexchange.office.filter.odt.dom.components.ParagraphComponent;
import com.openexchange.office.filter.odt.dom.components.RootComponent;
import com.openexchange.office.filter.odt.dom.components.RowComponent;
import com.openexchange.office.filter.odt.dom.components.TableComponent;

public class JsonOperationConsumer {

	private final OdfOperationDoc opsDoc;
    private final OdfTextDocument doc;
    private final TextStyles styles;
    private final TextContent content;
    private final StyleManager styleManager;
    private final Settings settings;

    public JsonOperationConsumer(OdfOperationDoc opsDoc)
		throws SAXException {

		this.opsDoc = opsDoc;
		doc = (OdfTextDocument)opsDoc.getDocument();
		styles = (TextStyles)doc.getStylesDom();
		content = (TextContent)doc.getContentDom();
		styleManager = doc.getStyleManager();
	    settings = doc.getSettingsDom();

	    // only creating target nodes for the correct master page, so this is possible only after loading styles + content
	    createHeaderFooterTargetNodes(content.getBody().getMasterPageName(true));
	}

    public int applyOperations(JSONArray operations)
    	throws Exception {

    	for (int i = 0; i < operations.length(); i++) {
        	OdfOperationDoc.setSuccessfulAppliedOperations(i);
            final JSONObject op = (JSONObject) operations.get(i);
            final String opName = op.getString("name");
            switch(opName)
            {
            	case "insertText" : {
		            insertText(op.getJSONArray("start"), op.optString("target"), op.optJSONObject("attrs"), op.getString("text").replaceAll("\\p{Cc}", " "));
		            break;
            	}
            	case "insertTab": {
            		insertTab(op.getJSONArray("start"), op.optString("target"), op.optJSONObject("attrs"));
            		break;
            	}
            	case "insertHardBreak": {
            		insertHardBreak(op.getJSONArray("start"), op.optString("target"), /* op.optString("type"), */ op.optJSONObject("attrs"));
            		break;
            	}
            	case "insertField": {
            		insertField(op.getJSONArray("start"), op.optString("target"), op.getString("type"), op.getString("representation"), op.optJSONObject("attrs"));
            		break;
            	}
            	case "delete" : {
            		delete(op.getJSONArray("start"), op.optString("target"), op.optJSONArray("end"));
            		break;
            	}
            	case "setAttributes": {
            		setAttributes(op.getJSONObject("attrs"), op.optString("target"), op.getJSONArray("start"), op.optJSONArray("end"));
            		break;
            	}
            	case "insertParagraph": {
                	insertParagraph(op.getJSONArray("start"), op.optString("target"), op.optJSONObject("attrs"));
                	break;
            	}
            	case "splitParagraph": {
            		splitParagraph(op.getJSONArray("start"), op.optString("target"));
            		break;
            	}
            	case "mergeParagraph": {
            		mergeParagraph(op.getJSONArray("start"), op.optString("target"));
            		break;
            	}
            	case "move": {
            		move(op.getJSONArray("start"), op.optString("target"), op.optJSONArray("end"), op.getJSONArray("to"));
            		break;
            	}
            	case "insertDrawing": {
            		insertDrawing(op.getJSONArray("start"), op.optString("target"), op.getString("type"), op.optJSONObject("attrs"));
            		break;
            	}
            	case "insertTable": {
            		insertTable(op.getJSONArray("start"), op.optString("target"), op.optJSONObject("attrs"));
            		break;
            	}
                case "splitTable": {
                    splitTable(op.getJSONArray("start"), op.optString("target"));
                    break;
                }
                case "mergeTable": {
                    mergeTable(op.getJSONArray("start"), op.optString("target"));
                    break;
                }
            	case "insertRows": {
            		insertRows(op.getJSONArray("start"), op.optString("target"), op.optInt("count", 1), op.optBoolean("insertDefaultCells", false), op.optInt("referenceRow", -1), op.optJSONObject("attrs"));
            		break;
            	}
            	case "insertCells": {
            		insertCells(op.getJSONArray("start"), op.optString("target"), op.optInt("count", 1), op.optJSONObject("attrs"));
            		break;
            	}
            	case "insertColumn": {
            		insertColumn(op.getJSONArray("start"), op.optString("target"), op.getJSONArray("tableGrid"), op.getInt("gridPosition"), op.optString("insertMode", "before"));
            		break;
            	}
            	case "deleteColumns": {
            		deleteColumns(op.getJSONArray("start"), op.optString("target"), op.getInt("startGrid"), op.optInt("endGrid", op.getInt("startGrid")));
            		break;
            	}
            	case "insertListStyle": {
            		insertListStyle(op.getString("listStyleId"), op.getJSONObject("listDefinition"));
            		break;
            	}
            	case "insertComment": {
            		insertComment(op.getJSONArray("start"), op.optString("target"), op.getString("id"), op.optString("uid"), op.optString("author"), op.optString("date"), op.optJSONObject("attrs"));
            		break;
            	}
            	case "insertRange": {
            		insertRange(op.getJSONArray("start"), op.optString("target"), op.getString("id"), op.optString("type", "comment"), op.optString("position", "start"), op.optJSONObject("attrs"));
            		break;
            	}
            	case "insertHeaderFooter": {
            		insertHeaderFooter(op.getString("id"), op.getString("type"));
            		break;
            	}
            	case "deleteHeaderFooter": {
            		deleteHeaderFooter(op.getString("id"));
            		break;
            	}
                case "insertStyleSheet": {
                    styleManager.insertStyleSheet(op.getString("type"), op.getString("styleId"), op.optString("styleName"), op.optJSONObject("attrs"), op.optString("parent"), op.optBoolean("default"), op.optBoolean("hidden"));
                    break;
                }
                case "setDocumentAttributes": {
                	setDocumentAttributes(op.getJSONObject("attrs"));
                	break;
                }
                case "insertFontDescription": {
	                insertFontDescription(op.getString("fontName"), op.getJSONObject("attrs"));
	                break;
                }
            	case "noOp": {
            		break;
            	}
                case "createError": {
                    throw new FilterException("createError operation detected: " + opName, ErrorCode.UNSUPPORTED_OPERATION_USED);
                }
            }
        }
    	final Body body = content.getBody();
    	final String masterPageName = body.getMasterPageName(false);
    	if(masterPageName!=null) {
    	    String currentMasterPageName = null;
            final Paragraph paragraph = ParagraphComponent.getFirstParagraph(new RootComponent(body), true);
            final String firstParaStyle = paragraph.getStyleName();
            if(firstParaStyle!=null&&!firstParaStyle.isEmpty()) {
                final StyleBase paraStyle = styleManager.getStyle(firstParaStyle, "paragraph", true);
                if(paraStyle!=null) {
                    currentMasterPageName = paraStyle.getAttribute("style:master-page-name");
                }
            }
            if(!masterPageName.equals(currentMasterPageName)) {
                final JSONObject attrs = new JSONObject(1);
                attrs.put("masterPageName", masterPageName);
                paragraph.setStyleName(styleManager.applyAttributes("paragraph", paragraph.getStyleName(), true, attrs));
            }
    	}
        OdfOperationDoc.setSuccessfulAppliedOperations(operations.length());
        return 1;
    }

    public void createHeaderFooterTargetNodes(String masterPageName) {
        final MasterStyles masterStyles = styleManager.getMasterStyles();
        final Iterator<Object> styleMasterPageIter = masterStyles.getContent().iterator();
        while(styleMasterPageIter.hasNext()) {
            final Object o = styleMasterPageIter.next();
            if(o instanceof StyleMasterPage) {
                if(((StyleMasterPage)o).getName().equals(masterPageName)) {
                    for(StyleHeaderFooter styleHeaderFooter:((StyleMasterPage)o).getStyleHeaderFooters()) {
                        content.getTargetNodes().put(styleHeaderFooter.getId(), styleHeaderFooter);
                    }
                    return;
                }
            }
        }
    }

    public void insertParagraph(JSONArray start, String target, JSONObject attrs)
        throws JSONException, SAXException {

    	Component.getComponent(content.getRootComponent(target), start, start.length()-1)
    		.insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.PARAGRAPH);
    }

    public void splitParagraph(JSONArray start, String target)
        throws JSONException {

        ((IParagraph)Component.getComponent(content.getRootComponent(target), start, start.length()-1))
        	.splitParagraph(opsDoc, start.getInt(start.length()-1));
    }

    public void mergeParagraph(JSONArray start, String target) {

    	((IParagraph)Component.getComponent(content.getRootComponent(target), start, start.length()))
    		.mergeParagraph();
    }

    public void insertText(JSONArray start, String target, JSONObject attrs, String text)
    	throws IndexOutOfBoundsException, JSONException, SAXException {

    	if(text.length()>0) {
    		final Component component = Component.getComponent(content.getRootComponent(target), start, start.length()-1);
    		((IParagraph)component).insertText(opsDoc, start.getInt(start.length()-1), text, attrs);
    	}
    }

    public void insertTab(JSONArray start, String target, JSONObject attrs)
        throws JSONException, SAXException {

    	Component.getComponent(content.getRootComponent(target), start, start.length()-1)
    	    .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.TAB);
    }

    public void insertHardBreak(JSONArray start, String target, JSONObject attrs)
        throws JSONException, SAXException {

    	Component.getComponent(content.getRootComponent(target), start, start.length()-1)
			.insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.HARDBREAK);
    }

    public void splitTable(JSONArray start, String target)
        throws JSONException, UnsupportedOperationException, SAXException {

        final TableComponent splitTableComponent = (TableComponent)Component.getComponent(content.getRootComponent(target), start, start.length()-2)
            .insertChildComponent(opsDoc, start.getInt(start.length()-2) + 1, null, Component.Type.TABLE);

        ((TableComponent)Component.getComponent(content.getRootComponent(target), start, start.length()-1))
            .splitTable(opsDoc, start.getInt(start.length()-1), splitTableComponent);
    }

    public void mergeTable(JSONArray start, String target) {
        ((TableComponent)Component.getComponent(content.getRootComponent(target), start, start.length())).mergeTable(opsDoc);
    }

    public void insertField(JSONArray start, String target, String type, String representation, JSONObject attrs)
    	throws JSONException, SAXException {

    	final TextField textField = (TextField)(Component.getComponent(content.getRootComponent(target), start, start.length()-1)
			.insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.FIELD)).getObject();

    	if(TextField.TYPES.contains(type)) {
    		textField.setType(type);
    	}
    	textField.setRepresentation(representation);
    }

    public void delete(JSONArray start, String target, JSONArray end)
    	throws JSONException {

        if(start==null||start.length()==0)
            return;

        int startComponent = start.getInt(start.length()-1);
        int endComponent = startComponent;
        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endComponent = end.getInt(end.length()-1);
        }
        final Component component = Component.getComponent(content.getRootComponent(target), start, start.length());
		component.splitStart(startComponent);
        component.delete(opsDoc, (endComponent-startComponent)+1);
    }

    public void setAttributes(JSONObject attrs, String target, JSONArray start, JSONArray end)
        throws JSONException, SAXException {

    	if(attrs==null) {
    		return;
    	}
        int startIndex = start.getInt(start.length()-1);
        int endIndex = startIndex;

        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endIndex = end.getInt(end.length()-1);
        }
        Component component = Component.getComponent(content.getRootComponent(target), start, start.length());
		component.splitStart(startIndex);
        while(component!=null&&component.getComponentNumber()<=endIndex) {
        	if(component.getNextComponentNumber()>=endIndex+1) {
        		component.splitEnd(endIndex);
        	}
        	component.applyAttrsFromJSON(opsDoc, attrs);
            component = component.getNextComponent();
        }
    }

    public void move(JSONArray start, String target, JSONArray end, JSONArray to)
    	throws JSONException {

        Component.move(content.getRootComponent(target), start, end, to);
    }

    public void insertDrawing(JSONArray start, String target, String type, JSONObject attrs)
    	throws UnsupportedOperationException, JSONException, SAXException {

    	Component.Type childComponentType = Component.Type.AC_IMAGE;
    	if(type.equals("shape")||type.equals("connector")) {
    		if(attrs==null||!attrs.has("geometry")) {
    			childComponentType = Component.Type.AC_FRAME;
    		}
    		else {
    			childComponentType = Component.Type.AC_SHAPE;
    		}
    	}
    	else if(type.equals("group")) {
    	    childComponentType = Component.Type.AC_GROUP;
    	}
    	else if(type.equals("image")) {
    	    childComponentType = Component.Type.AC_IMAGE;
    	}
    	Component.getComponent(content.getRootComponent(target), start, start.length()-1)
    		.insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, childComponentType);
    }

    public void insertTable(JSONArray start, String target, JSONObject attrs)
    	throws UnsupportedOperationException, JSONException, SAXException {

    	Component.getComponent(content.getRootComponent(target), start, start.length()-1)
    		.insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.TABLE);
    }

    public void insertRows(JSONArray start, String target, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs)
    	throws JSONException, SAXException {

    	((TableComponent)Component.getComponent(content.getRootComponent(target), start, start.length()-1))
    		.insertRows(opsDoc, start.getInt(start.length()-1), count, insertDefaultCells, referenceRow, attrs);
    }

    public void insertCells(JSONArray start, String target, int count, JSONObject attrs)
        throws JSONException, SAXException {

    	((RowComponent)Component.getComponent(content.getRootComponent(target), start, start.length()-1))
    		.insertCells(opsDoc, start.getInt(start.length()-1), count, attrs);
    }

    public void insertColumn(JSONArray start, String target, JSONArray tableGrid, int gridPosition, String insertMode)
    	throws JSONException {

    	((TableComponent)Component.getComponent(content.getRootComponent(target), start, start.length()))
    		.insertColumn(opsDoc, tableGrid, gridPosition, insertMode);
    }

    public void deleteColumns(JSONArray position, String target, int gridStart, int gridEnd) {

    	((TableComponent)Component.getComponent(content.getRootComponent(target), position, position.length()))
    		.deleteColumns(opsDoc, gridStart, gridEnd);
    }

    public void insertListStyle(String listStyleId, JSONObject listDefinition) {
    	// TODO: listStyle is only created in contentAutoStyles... have to clone them for header/footer usage
    	final TextListStyle textListStyle = new TextListStyle(listStyleId, true, true);
    	styleManager.addStyle(textListStyle);
    	textListStyle.applyAttrs(styleManager, listDefinition);
    }

    public void insertComment(JSONArray start, String target, String id, String uid, String author, String date, JSONObject attrs)
    	throws UnsupportedOperationException, JSONException, SAXException {

    	final Component component = Component.getComponent(content.getRootComponent(target), start, start.length()-1)
    		.insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.COMMENT_REFERENCE);

    	final Annotation annotation = (Annotation)component.getObject();
    	annotation.setId(id);
    	annotation.setCreator(author);
    	annotation.setDate(date);
    	content.getDocument().getTargetNodes().put(id, annotation);
    }

    public void insertRange(JSONArray start, String target, String id, String type, String position, JSONObject attrs)
    	throws UnsupportedOperationException, JSONException, SAXException {

    	if(type.equals("comment")&&position.equals("end")) {
	    	final Component component = Component.getComponent(content.getRootComponent(target), start, start.length()-1)
	    		.insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.COMMENT_RANGE_END);

	    	((AnnotationEnd)component.getObject()).setId(id);
    	}
    }

    public void insertHeaderFooter(String id, String type) {

        final String masterPageName = content.getBody().getMasterPageName(true);
    	final DLList<Object> styleMasterPages = styleManager.getMasterStyles().getContent();
    	StyleMasterPage standardMasterPage = null;
    	final Iterator<Object> styleMasterPageIter = styleMasterPages.iterator();
    	while(styleMasterPageIter.hasNext()) {
    		final Object o = styleMasterPageIter.next();
    		if(o instanceof StyleMasterPage) {
        		if(((StyleMasterPage)o).getName().equals(masterPageName)) {
        			standardMasterPage = (StyleMasterPage)o;
        			break;
        		}
        	}
    	}
    	if(standardMasterPage==null) {
    		standardMasterPage = new StyleMasterPage(masterPageName, "");
    		styleMasterPages.add(standardMasterPage);
    	}
    	final StyleHeaderFooter styleHeaderFooter = new StyleHeaderFooter(id, type);
    	standardMasterPage.getStyleHeaderFooters().add(styleHeaderFooter);
    	content.getTargetNodes().put(id, styleHeaderFooter);
    }

    public void deleteHeaderFooter(String id) {
    	content.getTargetNodes().remove(id);
    	final DLList<Object> styleMasterPages = styleManager.getMasterStyles().getContent();
    	final Iterator<Object> styleMasterPageIter = styleMasterPages.iterator();
		while(styleMasterPageIter.hasNext()) {
			final Object o = styleMasterPageIter.next();
			if(o instanceof StyleMasterPage) {
				final DLList<StyleHeaderFooter> styleHeaderFooters = ((StyleMasterPage)o).getStyleHeaderFooters();
				for(StyleHeaderFooter styleHeaderFooter:styleHeaderFooters) {
					if(styleHeaderFooter.getId().equals(id)) {
						styleHeaderFooters.remove(styleHeaderFooter);
						return;
					}
				}
			}
		}
    }

    public void setDocumentAttributes(JSONObject attrs)
    	throws JSONException {

    	final MasterStyles masterStyles = styleManager.getMasterStyles();
    	final Iterator<Object> masterStyleContentIter = masterStyles.getContent().iterator();
    	StyleMasterPage standardMasterPage = null;
    	final String standardMasterPageName = content.getBody().getMasterPageName(true);
    	while(masterStyleContentIter.hasNext()) {
    		final Object c = masterStyleContentIter.next();
    		if(c instanceof StyleMasterPage) {
    			if(standardMasterPageName.equals(((StyleMasterPage)c).getName())) {
    				standardMasterPage = (StyleMasterPage)c;
    				break;
    			}
    		}
    	}
    	if(standardMasterPage!=null) {
    		String pageLayoutName = standardMasterPage.getPageLayoutName();
    		if(pageLayoutName==null||pageLayoutName.isEmpty()) {
    			pageLayoutName = styleManager.getUniqueStyleName("page-layout", false);
        		styleManager.insertAutoStyle("page", pageLayoutName, attrs, false);
        		standardMasterPage.setPageLayoutName(pageLayoutName);
    		}
    		else {
    			final StyleBase styleBase = styleManager.getStyle(pageLayoutName, "page-layout", false);
    			if(styleBase instanceof StylePageLayout) {
    				((StylePageLayout)styleBase).applyAttrs(styleManager, attrs);
    			}
    		}
    	}
    	else {
    		String pageLayoutName = styleManager.getUniqueStyleName("page-layout", false);
    		styleManager.insertAutoStyle("page", pageLayoutName, attrs, false);
    		standardMasterPage = new StyleMasterPage(standardMasterPageName, pageLayoutName);
    		styleManager.addStyle(standardMasterPage);
    	}
    }

    public void insertFontDescription(String fontName, JSONObject attrs) {

    	if(!fontName.isEmpty()) {
	    	final JSONArray panose1 = attrs.optJSONArray("panose1");
	    	String panose1Value = panose1!=null ? panose1.toString() : null;
	    	final String[] altNames = (String[]) attrs.opt("altNames");
	        final String family = attrs.optString("family");
	        final String familyGeneric = attrs.optString("familyGeneric");
	        final String pitch = attrs.optString("pitch");
	        styleManager.insertFontDescription(fontName, altNames, family, familyGeneric, pitch, panose1Value);
    	}
    }
}
