/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ods.dom.components;

import java.util.Set;
import java.util.logging.Logger;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.odftoolkit.odfdom.doc.OdfChartDocument;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.pkg.OdfPackage;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.DLNode;
import com.openexchange.office.filter.odf.ElementNS;
import com.openexchange.office.filter.odf.INodeAccessor;
import com.openexchange.office.filter.odf.MetaData;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.components.ComponentContext;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.DrawImage;
import com.openexchange.office.filter.odf.draw.DrawObject;
import com.openexchange.office.filter.odf.draw.DrawTextBox;
import com.openexchange.office.filter.odf.draw.GroupShape;
import com.openexchange.office.filter.odf.draw.Shape;
import com.openexchange.office.filter.ods.dom.Drawing;
import com.openexchange.office.filter.ods.dom.Sheet;
import com.openexchange.office.filter.ods.dom.SpreadsheetContent;
import com.openexchange.office.filter.ods.dom.chart.ChartContent;

public class SheetComponent extends Component {

    private final Sheet sheet;

    public SheetComponent(ComponentContext parentContext, DLNode<Object> _node, int _componentNumber) {
        super(parentContext, _node, _componentNumber);
        this.sheet = (Sheet)getObject();
    }

    public SheetComponent(Sheet sheet, int _componentNumber) {
        super(null, new DLNode<Object>(sheet), _componentNumber);
        this.sheet = (Sheet)getObject();
    }

    @Override
    public String simpleName() {
        return "Sheet";
    }

    @Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
        final int nextComponentNumber = previousChildComponent != null ? previousChildComponent.getNextComponentNumber() : 0;
        DLNode<Object> nextNode = previousChildContext != null ? previousChildContext.getNode().next : ((Sheet)getNode().getObject()).getContent().getFirstNode();
        while(nextNode!=null) {
            final Object child = nextNode.getObject();
            if(child instanceof Drawing) {
                return new DrawingComponent(this, nextNode, nextComponentNumber);
            }
            nextNode = nextNode.next;
        }
        return null;
    }

    @Override
    public Component insertChildComponent(ComponentContext parentContext, DLNode<Object> contextNode, int number, Component child, Type type, JSONObject attrs) {

        DLList<Object> indexedNodeList = ((INodeAccessor)contextNode.getObject()).getContent();
        DLNode<Object> referenceNode = child != null && child.getComponentNumber()== number ? child.getNode() : null;

        switch(type) {
            case AC_SHAPE: {
                final Shape shape = Shape.createShape("spreadsheet", attrs, true);
                final DLNode<Object> drawingNode = new DLNode<Object>(new Drawing(sheet, null, shape));
                indexedNodeList.addNode(referenceNode, drawingNode, true);
                return new DrawingComponent(parentContext, drawingNode, number);
            }
            case AC_GROUP: {
                final GroupShape groupShape = new GroupShape("spreadsheet", true);
                final DLNode<Object> drawingNode = new DLNode<Object>(new Drawing(sheet, null, groupShape));
                indexedNodeList.addNode(referenceNode, drawingNode, true);
                return new DrawingComponent(parentContext, drawingNode, number);
            }
            case AC_IMAGE: {
                final DrawFrame drawFrame = new DrawFrame("spreadsheet", true);
                final DLNode<Object> drawingNode = new DLNode<Object>(new Drawing(sheet, null, drawFrame));
                indexedNodeList.addNode(referenceNode, drawingNode, true);
                final DrawImage drawImage = new DrawImage(drawFrame);
                drawFrame.addContent(drawImage);
                return new DrawingComponent(parentContext, drawingNode, number);
            }
            case AC_FRAME: {
                final DrawFrame drawFrame = new DrawFrame("spreadsheet", true);
                final DLNode<Object> drawingNode = new DLNode<Object>(new Drawing(sheet, null, drawFrame));
                indexedNodeList.addNode(referenceNode, drawingNode, true);
                final DrawTextBox drawTextBox = new DrawTextBox(drawFrame);
                drawFrame.addContent(drawTextBox);
                return new DrawingComponent(parentContext, drawingNode, number);
            }
            case AC_CHART: {
                final SpreadsheetContent spreadsheetContent = ((SpreadsheetContent)parentContext.getParentContext().getObject());
                final OdfDocument document = spreadsheetContent.getDocument();
                // the default is two cell anchor ...
                final DrawFrame drawFrame = new DrawFrame("spreadsheet", true);
                final DrawObject drawObject= new DrawObject(drawFrame);
                drawFrame.getContent().add(drawObject);
                OdfPackage pkg = document.getPackage();

                long index = 1;
                final String prefix = "Object ";

                Set<String> filePaths = pkg.getFilePaths();
                for (String path : filePaths) {
                    if (path.startsWith(prefix) && path.endsWith("/content.xml")) {
                        try {
                            long other = Long.parseLong(path.replace(prefix, "").replace("/content.xml", ""));

                            index = Math.max(other + 1, index);

                        } catch (NumberFormatException e) {
                            Logger.getAnonymousLogger().info(e.toString());
                        }
                    }
                }

                String name = prefix + index;

                final StringBuilder bContent = new StringBuilder();
                bContent.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?><office:document-content");
                for (OdfDocumentNamespace nameSpace : OdfDocumentNamespace.values()) {
                    bContent.append(" xmlns:" + nameSpace.getPrefix() + "=\"" + nameSpace.getUri() + "\"");
                }
                bContent.append("><office:body><office:chart></office:chart></office:body></office:document-content>");
                byte[] contentBytes = bContent.toString().getBytes();

                pkg.insert("".getBytes(), name + "/", OdfChartDocument.OdfMediaType.CHART.getMediaTypeString());
                pkg.insert(contentBytes, name + "/content.xml", "text/xml");

                pkg.insert("<?xml version=\"1.0\" encoding=\"UTF-8\"?><office:document-styles xmlns:office=\"urn:oasis:names:tc:opendocument:xmlns:office:1.0\" xmlns:style=\"urn:oasis:names:tc:opendocument:xmlns:style:1.0\" xmlns:text=\"urn:oasis:names:tc:opendocument:xmlns:text:1.0\" xmlns:table=\"urn:oasis:names:tc:opendocument:xmlns:table:1.0\" xmlns:draw=\"urn:oasis:names:tc:opendocument:xmlns:drawing:1.0\" xmlns:fo=\"urn:oasis:names:tc:opendocument:xmlns:xsl-fo-compatible:1.0\" xmlns:xlink=\"http://www.w3.org/1999/xlink\" xmlns:dc=\"http://purl.org/dc/elements/1.1/\" xmlns:number=\"urn:oasis:names:tc:opendocument:xmlns:datastyle:1.0\" xmlns:svg=\"urn:oasis:names:tc:opendocument:xmlns:svg-compatible:1.0\" xmlns:chart=\"urn:oasis:names:tc:opendocument:xmlns:chart:1.0\" xmlns:dr3d=\"urn:oasis:names:tc:opendocument:xmlns:dr3d:1.0\" xmlns:script=\"urn:oasis:names:tc:opendocument:xmlns:script:1.0\" xmlns:ooo=\"http://openoffice.org/2004/office\" xmlns:ooow=\"http://openoffice.org/2004/writer\" xmlns:oooc=\"http://openoffice.org/2004/calc\" xmlns:dom=\"http://www.w3.org/2001/xml-events\" xmlns:rpt=\"http://openoffice.org/2005/report\" xmlns:of=\"urn:oasis:names:tc:opendocument:xmlns:of:1.2\" xmlns:xhtml=\"http://www.w3.org/1999/xhtml\" xmlns:grddl=\"http://www.w3.org/2003/g/data-view#\" xmlns:tableooo=\"http://openoffice.org/2009/table\" xmlns:chartooo=\"http://openoffice.org/2010/chart\" xmlns:drawooo=\"http://openoffice.org/2010/draw\" xmlns:calcext=\"urn:org:documentfoundation:names:experimental:calc:xmlns:calcext:1.0\" xmlns:loext=\"urn:org:documentfoundation:names:experimental:office:xmlns:loext:1.0\" xmlns:field=\"urn:openoffice:names:experimental:ooo-ms-interop:xmlns:field:1.0\" xmlns:css3t=\"http://www.w3.org/TR/css3-text/\" office:version=\"1.2\"><office:styles/></office:document-styles>".getBytes(), name + "/styles.xml", "text/xml");

                OdfChartDocument chartDoc = (OdfChartDocument)document.loadSubDocument("./" + name);

                try {
                    final ChartContent chart = (ChartContent) chartDoc.getContentDom();
                    drawObject.setChart(chart);

                    ElementNS officeChart = (ElementNS) chart.getElementsByTagName("office:chart").item(0);

                    ElementNS chartArea = chart.newChartElement("chart");
                    ElementNS plotArea = chart.newChartElement("plot-area");
                    ElementNS wall = chart.newChartElement("wall");

                    JSONObject fillNone = new JSONObject(new JSONTokener("{fill:{type:'none'}}"));

                    chart.applyStyleAttrs(fillNone, wall);

                    officeChart.appendChild(chartArea);
                    chartArea.appendChild(plotArea);
                    plotArea.appendChild(wall);

                    chart.setChartArea(chartArea);
                    chart.setPlotArea(plotArea);

                    chartDoc.getOfficeMetaData(true);
                    chartDoc.getStylesDom();

                    MetaData docMeta = document.getOfficeMetaData(false);
                    if (docMeta != null) {
                        String generator = docMeta.getGenerator();
                        if (generator != null && !generator.isEmpty()) {
                            chartDoc.getOfficeMetaData(false).setGenerator(generator);
                        }

                    }

                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
                final AttributesImpl drawObjectAttributes = drawObject.getAttributes();
                drawObjectAttributes.setValue(Namespaces.XLINK, "href", "xlink:href", "./" + name);
                drawObjectAttributes.setValue(Namespaces.XLINK, "actuate", "xlink:actuate", "onLoad");
                drawObjectAttributes.setValue(Namespaces.XLINK, "show", "xlink:show", "embed");
                drawObjectAttributes.setValue(Namespaces.XLINK, "type", "xlink:type", "simple");
                drawObjectAttributes.setValue(Namespaces.DRAW, "name", "draw:name", name);
                final DLNode<Object> drawingNode = new DLNode<Object>(new Drawing(sheet, null, drawFrame));
                indexedNodeList.addNode(referenceNode, drawingNode, true);
                return new DrawingComponent(parentContext, drawingNode, number);
            }
            default : {
                throw new UnsupportedOperationException();
            }
        }
    }

    @Override
    public void applyAttrsFromJSON(OdfOperationDoc operationDocument, JSONObject attrs) {

    }

    @Override
    public void createAttrs(OdfOperationDoc createOperationHelper, OpAttrs attrs)
    	throws SAXException {

    }
}
