/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.ods.dom.components;

import java.util.Iterator;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.DLNode;
import com.openexchange.office.filter.odf.INodeAccessor;
import com.openexchange.office.filter.odf.IParagraph;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.components.ComponentContext;
import com.openexchange.office.filter.odf.components.TextComponent;
import com.openexchange.office.filter.odf.components.TextSpan_Base;
import com.openexchange.office.filter.odf.listlevel.ListLevelStyleEntry;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.odf.styles.TextListStyle;
import com.openexchange.office.filter.odt.dom.Paragraph;
import com.openexchange.office.filter.odt.dom.Text;
import com.openexchange.office.filter.odt.dom.TextField;
import com.openexchange.office.filter.odt.dom.TextLineBreak;
import com.openexchange.office.filter.odt.dom.TextList;
import com.openexchange.office.filter.odt.dom.TextListItem;
import com.openexchange.office.filter.odt.dom.TextSpan;
import com.openexchange.office.filter.odt.dom.TextTab;

public class ParagraphComponent extends Component implements IParagraph {

	final Paragraph paragraph;

	public ParagraphComponent(ComponentContext parentContext, DLNode<Object> paragraphNode, int componentNumber) {
		super(parentContext, paragraphNode, componentNumber);

		paragraph = (Paragraph)getObject();
	}

    @Override
    public String simpleName() {
        return "Para";
    }

    /*
	 * returns the object type the paragraph parent
	 * 
	 * it can be "title", "body" or "other"
	 */
	public String getPresentationType() {
        if(getParentComponent() instanceof FrameComponent) {
            final String presentationClass = ((FrameComponent)getParentComponent()).getDrawFrame().getPresentationClass();
            if("title".equals(presentationClass)) {
                return "title";
            }
            if("outline".equals(presentationClass)) {
                return "body";
            }
        }
        return "other";
	}

	@Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {

        DLNode<Object> nextNode = previousChildContext != null ? previousChildContext.getNode().next : ((Paragraph)getObject()).getContent().getFirstNode();
        while(nextNode!=null) {
			final Object child = nextNode.getObject();
			if(child instanceof TextSpan) {
				final TextSpanContext textSpanContext =  new TextSpanContext(this, nextNode, contentAutoStyle);
				final Component childComponent = textSpanContext.getNextChildComponent(null, previousChildComponent);
				if(childComponent!=null) {
					return childComponent;
				}
			}
			nextNode = nextNode.next;
		}
		return null;
	}

	@Override
	public void applyAttrsFromJSON(OdfOperationDoc operationDocument, JSONObject attrs)
		throws JSONException {

		if(attrs!=null) {
		    // applying standard paragraph attributes ...
            final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
            paragraph.setStyleName(styleManager.applyAttributes("paragraph", paragraph.getStyleName(), isContentAutoStyle(), attrs));

            final JSONObject paragraphAttrs = attrs.optJSONObject("paragraph");
            if(paragraphAttrs!=null) {
                // TODO: applying list item ....
                // first determine if bullet/numbering is on or off
                String buType = null;
                final Object bullet = paragraphAttrs.opt("bullet");
                if(bullet!=null) {
                    if(bullet instanceof JSONObject) {
                        final Object type = ((JSONObject)bullet).opt("type");
                        if(type instanceof String) {
                            buType = (String)type;
                        }
                    }
                    else if (bullet == JSONObject.NULL) {
                        if(isContentAutoStyle()) {
                            // inherit bullet style from master

                        }
                        else {
                            buType = "none";
                        }
                    }
                }
                else {
                    
                }
                final Object listLevel = paragraphAttrs.opt("level");
                if(listLevel!=null) {
                    if(listLevel instanceof Integer) {
                        paragraph.setListLevel(((Integer)listLevel).intValue());;
                    }
                    else if(listLevel==JSONObject.NULL) {
                        paragraph.setListLevel(-1);
                    }
                }
            }
		}
	}

	@Override
	public void createAttrs(OdfOperationDoc operationDocument, OpAttrs attrs) {
		final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
		if(paragraph.getStyleName()!=null&&!paragraph.getStyleName().isEmpty()) {
			styleManager.createAutoStyleAttributes(attrs, paragraph.getStyleName(), "paragraph", isContentAutoStyle());
		}

		final OpAttrs paragraphAttrs  = attrs.getMap("paragraph", true);
		TextListItem textListItem = paragraph.getTextListItem();
        if(textListItem!=null) {
            final TextList textList = textListItem.getParentTextList();
            if(textList!=null) {
                final String styleName = textList.getStyleName(true);
                if(styleName!=null&&!styleName.isEmpty()) {
                    final StyleBase listStyle = styleManager.getStyle(styleName, "list-style", isContentAutoStyle());
                    if(listStyle instanceof TextListStyle) {
                        // TODO: direct access to ListLevelStyleEntry ... only 10 are available
                        final Iterator<IElementWriter> listStyleIter = ((TextListStyle)listStyle).getContent().iterator();
                        while(listStyleIter.hasNext()) {
                            final IElementWriter n = listStyleIter.next();
                            if(n instanceof ListLevelStyleEntry) {
                                if(((ListLevelStyleEntry)n).getListLevel(0)-1==textListItem.getListLevel()) {
                                    ((ListLevelStyleEntry)n).createPresentationAttrs(styleManager, "other", isContentAutoStyle(), paragraphAttrs);
                                    ListLevelStyleEntry.finalizeParagraphAttrs(paragraphAttrs);
                                    break;
                                }
                            }
                        }
                    }
                }
            }
			paragraphAttrs.put("level", textListItem.getListLevel());
			final Integer startValue = paragraph.getStartValue();
			if(startValue!=null) {
				paragraphAttrs.put("listStartValue", startValue);
			}
			final Integer outlineLevel = paragraph.getOutlineLevel();
			if(outlineLevel!=null) {
				paragraphAttrs.put("outlineLevel", outlineLevel);
			}
		}
        else {
            final OpAttrs bulletAttrs = paragraphAttrs.getMap("bullet", true);
            bulletAttrs.put("type", "none");
            if(this.getParentComponent() instanceof FrameComponent) {
                final String presentationStyleName = ((FrameComponent)getParentComponent()).getDrawFrame().getPresentationStyleName();
                if(presentationStyleName!=null&&!presentationStyleName.isEmpty()) {
                    final int level = paragraph.getListLevel();
                    if(level>=0) {
                        paragraphAttrs.put("level", level);
                    }
                }
            }
        }
		if(paragraphAttrs.isEmpty()) {
			attrs.remove("paragraph");
		}
	}

	@Override
	public void insertText(OdfOperationDoc operationDocument, int textPosition, String text, JSONObject attrs)
		throws JSONException, SAXException {

    	if(text.length()>0) {
            Text t = null;
            Component childComponent = getNextChildComponent(null, null);
            Component cRet = null;

            if(childComponent!=null) {
            	if(textPosition>0) {
            		childComponent = childComponent.getComponent(textPosition-1);
            	}
                // check if the character could be inserted into an existing text:
                if(childComponent instanceof TextComponent) {
                    t = (Text)childComponent.getObject();
                    final StringBuffer s = new StringBuffer(t.getText());
                    s.insert(textPosition-((TextComponent)childComponent).getComponentNumber(), text);
                    t.setText(s.toString());
                    cRet = childComponent;
                }
                else {
                	t = new Text(text, true);
                	final TextSpanContext spanContext = (TextSpanContext)childComponent.getParentContext();
                    ((TextSpan)spanContext.getObject()).getContent().addNode(childComponent.getNode(), new DLNode<Object>(t), textPosition==0);
                    cRet = childComponent;

                    if(textPosition>0) {
                        cRet = childComponent.getNextComponent();
                    }
                    else {
                        cRet = getNextChildComponent(null, null);
                    }
                }
            }
            else {

            	// the paragraph is empty, we have to create R and its text
                final TextSpan newRun = new TextSpan();
                paragraph.getContent().add(newRun);
/*
                if(paragraph.getPPr()!=null) {
                    final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
                    if(rPr!=null) {
                        rPr.setParent(newRun);
                        newRun.setRPr(rPr);
                    }
                }
*/
                t = new Text(text, true);
                newRun.getContent().add(t);
                cRet = getNextChildComponent(null, null);
            }
            if(attrs!=null) {
            	cRet.splitStart(textPosition);
            	cRet.splitEnd(textPosition+text.length()-1);
            	cRet.applyAttrsFromJSON(operationDocument, attrs);
            }
        }
	}

    @Override
    public Component insertChildComponent(OdfOperationDoc operationDocument, int textPosition, JSONObject attrs, Type childType)
        throws JSONException, SAXException {

        Object newChild = null;
        switch(childType) {
            case TAB : {
                newChild = new TextTab(null);
                break;
            }
            case HARDBREAK : {
            	newChild = new TextLineBreak();
            	break;
            }
            case FIELD : {
                newChild = new TextField();
                break;
            }
            default : {
                throw new UnsupportedOperationException();
            }
        }
        Component childComponent = getNextChildComponent(null, null);
        if(childComponent!=null) {
            if(textPosition>0) {
                childComponent = childComponent.getComponent(textPosition-1);
                childComponent.splitEnd(textPosition-1);
            }
    		// the new child can be added into an existing textRun
            final TextSpan textSpan = ((TextSpan_Base)childComponent).getTextSpan();
            textSpan.getContent().addNode(childComponent.getNode(), new DLNode<Object>(newChild), textPosition==0);
        }
        else {
            final TextSpan newTextSpan = new TextSpan();
            paragraph.getContent().add(newTextSpan);
            newTextSpan.getContent().add(newChild);
        }
        if(textPosition>0) {
            childComponent = childComponent.getNextComponent();
        }
        else {
            childComponent = getNextChildComponent(null, null);
        }
        if(attrs!=null) {
            childComponent.splitStart(textPosition);
            childComponent.splitEnd(textPosition);
            childComponent.applyAttrsFromJSON(operationDocument, attrs);
        }
        return childComponent;
    }

	@Override
    public void splitParagraph(OdfOperationDoc operationDocument, int textPosition) {
        final Paragraph destParagraph = new Paragraph(
        	paragraph.getTextListItem()!=null
        	? new TextListItem(paragraph.getTextListItem().getParentTextList())
        	: null);

        // taking care of paragraphs sharing the same textListItem
        if(destParagraph.getTextListItem()!=null) {
        	DLNode<Object> nextParaNode = getNode().next;
        	while(nextParaNode!=null&&nextParaNode.getObject() instanceof Paragraph) {
	        	final Paragraph nextPara = (Paragraph)getNode().next.getObject();
	        	if(nextPara.getTextListItem()!=paragraph.getTextListItem()) {
	        		break;
	        	}
	        	nextPara.setTextListItem(destParagraph.getTextListItem());
    	        nextParaNode = nextParaNode.next;
        	}
        }

        final DLNode<Object> destParagraphNode = new DLNode<Object>(destParagraph);
        ((INodeAccessor)getParentContext().getObject()).getContent().addNode(getNode(), destParagraphNode, false);
        destParagraph.setStyleName(paragraph.getStyleName());

        Component component = getNextChildComponent(null, null);
    	while(component!=null&&component.getNextComponentNumber()<=textPosition) {
    		component = component.getNextComponent();
    	}
        if(component!=null) {
        	component.splitStart(textPosition);

        	// moving text spans into the new paragraph
        	paragraph.getContent().moveNodes(component.getParentContext().getNode(), -1, destParagraph.getContent(), null, true);
        }
    }

	@Override
	public void mergeParagraph() {
    	final Component nextParagraphComponent = getNextComponent();
    	if(nextParagraphComponent instanceof ParagraphComponent) {
        	final DLList<Object> sourceContent = ((Paragraph)nextParagraphComponent.getObject()).getContent();
            paragraph.getContent().mergeNodes(sourceContent);
            ((INodeAccessor)getParentContext().getObject()).getContent().removeNode(nextParagraphComponent.getNode());
    	}
	}
}
