/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ods.dom.chart;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.pkg.NamespaceName;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.odftoolkit.odfdom.pkg.OdfPackageDocument;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import com.openexchange.office.filter.core.chart.AttributeSet;
import com.openexchange.office.filter.core.chart.Chart;
import com.openexchange.office.filter.core.chart.ChartAxis;
import com.openexchange.office.filter.core.chart.ChartSeries;
import com.openexchange.office.filter.odf.ElementNS;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.styles.AutomaticStyles;
import com.openexchange.office.filter.odf.styles.StyleChart;
import com.openexchange.office.filter.odf.styles.StyleManager;

public class ChartContent extends OdfFileDom {

    public static final String USERDATA   = "userdata";
    public static final String ODF_STYLE  = "chart:style-name";
    public static final String ODF_LEGEND = "legend-position";

    public static final String ODF_VALUES           = "values-cell-range-address";
    public static final String ODF_TITLE            = "label-cell-address";
    public static final String ODF_CELL_RANGE       = "cell-range-address";
    public static final String ODF_TABLE_CELL_RANGE = "table:cell-range-address";

    public static final String CHARTSTYLEID = "ooxml-style";

    public static final Map<String, String> LEGENDPOS_ODF_TO_OP;
    public static final Map<String, String> LEGENDPOS_OP_TO_ODF;

    public static final List<String> MARKERS = Collections.unmodifiableList(Arrays.asList(new String[] { "circle", "square", "arrow-up", "x" }));

    static {
        HashMap<String, String> odfToOp = new HashMap<>();
        HashMap<String, String> opToOdf = new HashMap<>();

        opToOdf.put("top", "top");
        opToOdf.put("bottom", "bottom");
        opToOdf.put("left", "start");
        opToOdf.put("right", "end");
        opToOdf.put("topright", "top-end");
        opToOdf.put("topleft", "top-start");
        opToOdf.put("bottomright", "bottom-end");
        opToOdf.put("bottomleft", "bottom-start");

        for (Entry<String, String> e : opToOdf.entrySet()) {
            odfToOp.put(e.getValue(), e.getKey());
        }

        LEGENDPOS_ODF_TO_OP = Collections.unmodifiableMap(odfToOp);
        LEGENDPOS_OP_TO_ODF = Collections.unmodifiableMap(opToOdf);
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////

    private Chart     chart;
    private ElementNS plotArea;
    private ElementNS chartArea;

    public ChartContent(OdfPackageDocument packageDocument, String packagePath) throws SAXException {
        super(packageDocument, packagePath);
    }

    @Override
    protected void initialize() throws SAXException {
        for (NamespaceName name : OdfDocumentNamespace.values()) {
            setNamespace(name.getPrefix(), name.getUri());
        }

        final StyleManager styleManager = getDocument().getStyleManager();
        styleManager.setAutomaticStyles(new AutomaticStyles(this), true);

        final XMLReader xmlReader = mPackage.getXMLReader();
        super.initialize(new ChartContentHandler(this, xmlReader), xmlReader);

        final Node root = getRootElement();
        root.insertBefore(styleManager.getAutomaticStyles(true), root.getFirstChild());

    }

    public Chart getChart() {
        if (chart == null) {
            chart = new Chart();
        }
        return chart;
    }

    public StyleChart getPlotAreaStyle() {
        return getStyleChart(getOrCreateStyle(plotArea));
    }

    private ElementNS createSeries(String styleName) {
        return createBoundElement(new ChartSeries(), "series", styleName);
    }

    private ElementNS createAxis(String styleName, String axisId) {
        ChartAxis ax = chart.getAxis(axisId);
        ElementNS res = createBoundElement(ax, "axis", styleName);
        setAttribute(res, "class", "major");
        setAttribute(res, "dimension", axisId.substring(0, 1));
        setAttribute(res, "name", getAxisName(ax));
        return res;
    }

    private ElementNS createBoundElement(AttributeSet attrs, String name, String styleName) {
        ElementNS el = newChartElement(name);
        if (attrs != null) {
            el.setUserData(USERDATA, attrs, null);
        }

        if (!StringUtils.isEmpty(styleName)) {
            setAttribute(el, "style-name", styleName);
        }
        return el;
    }

    public ElementNS newChartElement(String name) {
        return newElement(OdfDocumentNamespace.CHART, name);
    }

    public ElementNS newElement(OdfDocumentNamespace namespace, String name) {
        return new ElementNS(this, namespace.getUri(), namespace.getPrefix() + ":" + name);
    }

    public void insertDataSeries(int seriesIndex, JSONObject attrs) {
        ElementNS seriesEl = createSeries(null);
        ChartSeries serie = (ChartSeries) seriesEl.getUserData(USERDATA);

        ChartSeries before = getChart().insertDataSeries(seriesIndex, serie);
        if (before != null) {
            Node bef = getChildByUserData(plotArea, before);
            plotArea.insertBefore(seriesEl, bef);
        } else {
            plotArea.appendChild(seriesEl);
        }

        setDataSeriesAttributes(seriesIndex, attrs);

        handleAxisDirections();
    }

    public ElementNS addDataSeries(String type, String values, String title, String styleName, int axisXIndex, int axisYIndex) {
        ElementNS seriesEl = createSeries(styleName);

        ChartSeries serie = (ChartSeries) seriesEl.getUserData(USERDATA);
        serie.setSeriesAttributes(type, values, title, axisXIndex, axisYIndex);

        plotArea.appendChild(seriesEl);

        getChart().addDataSeries(serie);

        implSetDataSeriesAttributes(getChart().getSeries().size() - 1, getStyleAttrs(seriesEl), null);

        return seriesEl;
    }

    private ElementNS implSetDataSeriesAttributes(int seriesIndex, JSONObject attrs, String symbol) {
        ChartSeries serie = getChart().setDataSeriesAttributes(seriesIndex, attrs);
        ElementNS node = getChildByUserData(plotArea, serie);

        applyStyleAttrs(serie.getAttributes(), node);
        applySeriesAttrs(serie.getOrCreate("series"), node, symbol);

        return node;
    }

    public void setDataSeriesAttributes(int seriesIndex, JSONObject attrs) {

        String symbol = "automatic";

        if (attrs.hasAndNotNull("series")) {
            JSONObject series = attrs.optJSONObject("series");

            if (series.hasAndNotNull("dataPoints")) {
                // workaround for Bug 51731
                JSONObject fdp = series.optJSONArray("dataPoints").optJSONObject(0);
                if (fdp != null && fdp.hasAndNotNull("fill")) {
                    JSONObject fill = fdp.optJSONObject("fill");
                    if (StringUtils.equals(fill.optString("type"), "none")) {
                        symbol = "none";
                    }
                }
            }

            if (series.has("values") && !series.has("names")) {
                series.putSafe("names", "");
            }
        }

        ElementNS seriesNode = implSetDataSeriesAttributes(seriesIndex, attrs, symbol);
        setChartStyleAttribute(seriesNode, "symbol-type", symbol);

        handleAxisDirections();
    }

    public ElementNS addAxis(String axisId, String styleName) {
        ElementNS axisEl = createAxis(styleName, axisId);

        ChartAxis axis = (ChartAxis) axisEl.getUserData(USERDATA);
        axis.setAxis(axisId);

        NodeList otherAxes = plotArea.getElementsByTagName("chart:axis");
        Node refChild = null;
        if (otherAxes.getLength() > 0) {
            refChild = otherAxes.item(otherAxes.getLength() - 1).getNextSibling();
        }
        if (refChild != null) {
            plotArea.insertBefore(axisEl, refChild);
        } else {
            plotArea.appendChild(axisEl);
        }

        getChart().setAxis(axis);

        setAxisAttributes(axisId, getStyleAttrs(axisEl));

        return axisEl;
    }

    public ElementNS addGrid(ElementNS axis, String styleName) {
        ChartAxis ax = (ChartAxis) axis.getUserData(USERDATA);

        ElementNS gridEl = createBoundElement(ax.getGridLine(), "grid", styleName);
        setAttribute(gridEl, "class", "major");

        axis.appendChild(gridEl);

        setGridlineAttributes(ax.getAxis(), getStyleAttrs(gridEl));

        return gridEl;
    }

    public ElementNS addTitle(ElementNS axis, String styleName) {
        ChartAxis ax = (ChartAxis) axis.getUserData(USERDATA);

        ElementNS titleEl = createBoundElement(ax.getTitle(), "title", styleName);

        axis.appendChild(titleEl);

        setTitleAttributes(ax.getAxis(), getStyleAttrs(titleEl));

        return titleEl;
    }

    public ElementNS addMainTitle(String styleName) {
        ElementNS titleEl = createBoundElement(getChart().getTitle("main"), "title", styleName);

        chartArea.appendChild(titleEl);

        setTitleAttributes("main", getStyleAttrs(titleEl));

        return titleEl;
    }

    public ElementNS addLegend(String styleName) {
        ElementNS legendEl = createBoundElement(getChart().getLegend(), "legend", styleName);

        // AttributeSet legend = (AttributeSet) legendEl.getUserData(USERDATA);

        chartArea.appendChild(legendEl);

        setLegendAttributes(getStyleAttrs(legendEl));

        return legendEl;
    }

    public void deleteDataSeries(int seriesIndex) {
        ChartSeries old = getChart().deleteDataSeries(seriesIndex);
        ElementNS node = getChildByUserData(plotArea, old);
        handleDataPoints(node, new JSONArray(), null);
        deleteElement(node);
    }

    public void setPlotArea(ElementNS plotArea) {
        this.plotArea = plotArea;
    }

    public void setChartArea(ElementNS chartArea) {
        this.chartArea = chartArea;
    }

    public ElementNS getPlotArea() {
        return plotArea;
    }

    public ElementNS getChartArea() {
        return chartArea;
    }

    public void setAxisAttributes(String axisId, JSONObject attrs) {
        AttributeSet ax = getAxis(axisId);
        ax.setAttributes(attrs);
        ElementNS node = getChildByUserData(plotArea, ax);

        applyAxisAttrs(ax.getOrCreate("axis"), node);
        applyStyleAttrs(ax.getAttributes(), node);
    }

    private ChartAxis getAxis(String axisId) {
        ChartAxis ax = getChart().getAxis(axisId);
        ElementNS node = getChildByUserData(plotArea, ax);
        if (node == null) {
            node = addAxis(axisId, null);
        }
        return ax;
    }

    public void setGridlineAttributes(String axisId, JSONObject attrs) {
        ChartAxis ax = getAxis(axisId);
        AttributeSet grid = ax.getGridLine();
        grid.setAttributes(attrs);

        ElementNS axEl = getChildByUserData(plotArea, ax);
        ElementNS gridEl = getChildByUserData(axEl, grid);

        if (gridEl == null) {
            gridEl = addGrid(axEl, null);
        }

        applyStyleAttrs(grid.getAttributes(), gridEl);
    }

    public void setTitleAttributes(String axisId, JSONObject attrs) {

        AttributeSet title = getChart().getTitle(axisId);
        title.setAttributes(attrs);

        ElementNS titleEl = null;

        if ("main".equals(axisId)) {
            titleEl = getChildByUserData(chartArea, title);
            if (titleEl == null) {
                titleEl = addMainTitle(null);
            }
        } else {
            ChartAxis ax = getAxis(axisId);
            ElementNS axEl = getChildByUserData(plotArea, ax);
            titleEl = getChildByUserData(axEl, title);

            if (titleEl == null) {
                titleEl = addTitle(axEl, null);
            }
        }

        applyTitleAttrs(title.getOrCreate("title"), titleEl);
        applyStyleAttrs(title.getAttributes(), titleEl);

        if (attrs.has("text")) {
            JSONObject text = attrs.optJSONObject("text");

            Node child = titleEl.getFirstChild();
            if (text == null) {
                if (child != null) {
                    titleEl.removeChild(child);
                }
            } else {
                if (child == null) {
                    child = new ElementNS(this, "", "text:p");
                    titleEl.appendChild(child);
                }
                JSONArray link = text.optJSONArray("link");
                if (link != null && link.length() > 0) {
                    child.setTextContent(link.optString(0));
                } else {
                    child.setTextContent("");
                }
            }
        }

    }

    public void setLegendAttributes(JSONObject attrs) {
        AttributeSet legend = getChart().getLegend();

        ElementNS legendEl = getChildByUserData(chartArea, legend);
        if (legendEl == null) {
            legendEl = createBoundElement(legend, "legend", null);
            chartArea.appendChild(legendEl);
        }

        legend.setAttributes(attrs);
        JSONObject legendAttrs = legend.getOrCreate("legend");
        if ("off".equals(legendAttrs.optString("pos"))) {
            deleteElement(legendEl);
        } else {
            applyLegendAttrs(legendAttrs, legendEl);
            applyStyleAttrs(legend.getAttributes(), legendEl);
        }
    }

    private void deleteElement(ElementNS el) {
        el.getParentNode().removeChild(el);

        String styleName = el.getAttribute(ChartContent.ODF_STYLE);
        if (StringUtils.isNotEmpty(styleName)) {
            StyleManager styleManager = getDocument().getStyleManager();
            styleManager.removeStyle("chart", styleName, true, false);
        }
    }

    public JSONObject getStyleAttrs(String styleName) {
        if (StringUtils.isEmpty(styleName)) {
            // throw new RuntimeException("stylename is null!!!");
            return new JSONObject();
        }

        StyleChart style = getStyleChart(styleName);

        if (style == null) {
            throw new RuntimeException("no style found for " + styleName);
        }

        OpAttrs attrs = new OpAttrs();
        style.createAttrs(getDocument().getStyleManager(), attrs);

        //FIXME: hacks hacks hacks
        if (attrs.containsKey("fill") && !attrs.getMap("fill", false).containsKey("type")) {
            attrs.getMap("fill", false).put("type", "solid");
        }
        if (attrs.containsKey("line") && !attrs.getMap("line", false).containsKey("type")) {
            attrs.getMap("line", false).put("type", "solid");
        }

        return new JSONObject(attrs);
    }

    public StyleChart getStyleChart(String styleName) {
        StyleManager styleManager = getDocument().getStyleManager();
        return (StyleChart) styleManager.getStyle(styleName, "chart", true);
    }

    public void applyDrawingAttributes(JSONObject attrs) {
        JSONObject json = attrs.optJSONObject("chart");
        for (Entry<String, Object> e : json.entrySet()) {
            String key = e.getKey();
            String value = e.getValue().toString();

            if ("chartStyleId".equals(key)) {
                setAttribute(plotArea, CHARTSTYLEID, value);
            } else if ("stacking".equals(key)) {
                if ("percentStacked".equals(value)) {
                    setChartStyleAttribute(plotArea, "percentage", true);
                    setChartStyleAttribute(plotArea, "stacked", null);
                } else if ("stacked".equals(value)) {
                    setChartStyleAttribute(plotArea, "percentage", null);
                    setChartStyleAttribute(plotArea, "stacked", true);
                } else {
                    setChartStyleAttribute(plotArea, "percentage", null);
                    setChartStyleAttribute(plotArea, "stacked", null);
                }
            } else if ("curved".equals(key)) {
                if ("true".equals(value)) {
                    setChartStyleAttribute(plotArea, "interpolation", "cubic-spline");
                } else {
                    setChartStyleAttribute(plotArea, "interpolation", null);
                }
            } else if ("varyColors".equals(key)) {
                // ignore varyColors
            } else {
                //System.out.println("ChartContent.applyAttributes() " + key);
            }
        }
        applyStyleAttrs(attrs, chartArea);
        setChartStyleAttribute(plotArea, "auto-position", "true");
    }

    private void applySeriesAttrs(JSONObject json, ElementNS el, String symbol) {
        for (Entry<String, Object> e : json.entrySet()) {
            String key = e.getKey();
            String value = null;
            if (e.getValue() != JSONObject.NULL) {
                value = e.getValue().toString();
            }

            if ("type".equals(key)) {
                final String chartAreaType;
                boolean vertical = false;

                if ("column".equals(value)) {
                    value = "bar";
                    chartAreaType = "bar";
                } else if ("bar".equals(value)) {
                    chartAreaType = "bar";
                    vertical = true;
                } else if ("pie".equals(value)) {
                    value = "circle";
                    chartAreaType = value;
                } else if ("donut".equals(value)) {
                    value = "circle";
                    chartAreaType = "ring";
                } else if ("stock".equals(value)) {
                    chartAreaType = value;
                    value = null;
                } else if ("line".equals(value)) {
                    handleLineType(el);
                    chartAreaType = value;
                } else if ("scatter".equals(value)) {
                    handleLineType(el);
                    chartAreaType = value;
                } else if ("bubble".equals(value)) {
                    chartAreaType = value;
                } else {
                    chartAreaType = null;
                }
                if (chartAreaType != null) {
                    setAttribute(chartArea, "class", "chart:" + chartAreaType);
                } else {
                    setAttribute(chartArea, "class", null);
                }
                if (value != null) {
                    setAttribute(el, "class", "chart:" + value);
                } else {
                    setAttribute(el, "class", null);
                }

                getPlotAreaStyle().getChartProperties().getAttributes().setBooleanValue(OdfDocumentNamespace.CHART.getUri(), "vertical", "chart:vertical", vertical);

            } else if ("dataPoints".equals(key)) {
                handleDataPoints(el, json.optJSONArray(key), symbol);
            } else if ("values".equals(key)) {
                if (json.has("bubbles")) {
                    setBubbleAttribute(el, fromFormula(value), 0);
                    String bubbles = json.optString("bubbles");
                    setAttribute(el, ODF_VALUES, fromFormula(bubbles));
                } else {
                    setAttribute(el, ODF_VALUES, fromFormula(value));
                }
            } else if ("title".equals(key)) {
                setAttribute(el, ODF_TITLE, fromFormula(value));
            } else if ("bubbles".equals(key)) {
                // handled in "values" part
            } else if ("names".equals(key)) {
                setNamesAttribute(el, fromFormula(value));
            } else if ("dataLabel".equals(key)) {
                setChartStyleAttribute(el, "data-label-text", "false");
                setChartStyleAttribute(el, "data-label-symbol", "false");

                if ("value;percent".equals(value)) {
                    setChartStyleAttribute(el, "data-label-number", "value-and-percentage");
                } else if ("value".equals(value)) {
                    setChartStyleAttribute(el, "data-label-number", "value");
                } else if ("percent".equals(value)) {
                    setChartStyleAttribute(el, "data-label-number", "percentage");
                } else {
                    setChartStyleAttribute(el, "data-label-number", "none");
                }
            } else if ("format".equals(key)) {
                //ignore series format!
            } else if ("axisXIndex".equals(key)) {
                //ignore axisXIndex!
            } else if ("axisYIndex".equals(key)) {
                if ("1".equals(value)) {
                    setAttribute(el, "attached-axis", "secondary-y");
                } else {
                    setAttribute(el, "attached-axis", "primary-y");
                }
            } else {
                //System.out.println("ChartContent.applySeriesAttrs() " + key);
            }
        }

        setChartStyleAttribute(el, "link-data-style-to-source", "true");
        setGraficStyleAttribute(el, OdfDocumentNamespace.DR3D, "edge-rounding", "5%");
    }

    private void handleLineType(ElementNS el) {
        setChartStyleAttribute(plotArea, "three-dimensional", false);
    }

    private void handleAxisDirections() {
        ElementNS xAxis = getChildByUserData(plotArea, getAxis("x"));
        ElementNS yAxis = getChildByUserData(plotArea, getAxis("y"));

        setChartStyleAttribute(xAxis, "reverse-direction", chart.isPieOrDonut() + "");
        setChartStyleAttribute(yAxis, "reverse-direction", "false");

        setChartStyleAttribute(xAxis, "axis-position", "0");
        setChartStyleAttribute(yAxis, "axis-position", "start");
    }

    private void setNamesAttribute(ElementNS el, String formula) {
        ElementNS axis = getChildByTag(plotArea, "chart:axis");
        if (axis == null) {
            if (StringUtils.isEmpty(formula)) {
                return;
            }
            axis = addAxis("x", null);
        }

        ElementNS cat = getChildByTag(axis, "chart:categories");
        if (cat == null) {
            if (StringUtils.isEmpty(formula)) {
                return;
            }
            cat = newChartElement("categories");
            axis.appendChild(cat);
        }

        setAttribute(cat, OdfDocumentNamespace.TABLE, ODF_CELL_RANGE, formula);
    }

    public String getNamesFormula() {
        ElementNS axis = getChildByTag(plotArea, "chart:axis");
        if (axis == null) {
            return null;
        }
        ElementNS cat = getChildByTag(axis, "chart:categories");
        if (cat == null) {
            return null;
        }
        return getFormula(cat.getAttribute(ODF_TABLE_CELL_RANGE));
    }

    private String getBubblesFormula() {
        if (chart.isScatter() || chart.isBubble()) {
            ChartSeries serie = getChart().getSeries().get(0);
            ElementNS node = getChildByUserData(plotArea, serie);

            ElementNS cat = getChildByTag(node, "chart:domain", 1);
            if (cat == null) {
                cat = getChildByTag(node, "chart:domain", 0);
            }
            if (cat == null) {
                return null;
            }
            return getFormula(cat.getAttribute(ODF_TABLE_CELL_RANGE));

        }
        return null;
    }

    private void setBubbleAttribute(ElementNS el, String formula, int index) {
        ElementNS domain = getChildByTag(el, "chart:domain", index);
        while (domain == null) {
            if (formula == null) {
                return;
            }
            el.appendChild(newChartElement("domain"));
            domain = getChildByTag(el, "chart:domain", index);
        }
        setAttribute(domain, OdfDocumentNamespace.TABLE, ODF_CELL_RANGE, formula);
    }

    private void applyAxisAttrs(JSONObject json, ElementNS el) {
        for (Entry<String, Object> e : json.entrySet()) {
            String key = e.getKey();
            String value = e.getValue().toString();

            if ("label".equals(key)) {
                setChartStyleAttribute(el, "display-label", value);
            } else {
                //System.out.println("ChartContent.applyAxisAttrs() " + key);
            }
        }
    }

    private void applyTitleAttrs(JSONObject json, ElementNS el) {
        for (Entry<String, Object> e : json.entrySet()) {
            String key = e.getKey();
            String value = e.getValue().toString();
            System.out.println("ChartContent.applyTitleAttrs() " + key);
        }
    }

    private void applyLegendAttrs(JSONObject json, ElementNS el) {
        for (Entry<String, Object> e : json.entrySet()) {
            String key = e.getKey();
            String value = e.getValue().toString();
            if ("pos".equals(key)) {
                setAttribute(el, ODF_LEGEND, LEGENDPOS_OP_TO_ODF.get(value));
            } else {
                //System.out.println("ChartContent.applyTitleAttrs() " + key);
            }
        }
    }

    private JSONObject handleDataPoints(ElementNS seriesEl, JSONArray dataPoints, String symbol) {
        NodeList oldDataPoints = seriesEl.getChildNodes();
        for (int i = oldDataPoints.getLength() - 1; i >= 0; i--) {
            deleteElement((ElementNS) oldDataPoints.item(i));
        }

        for (int i = 0; i < dataPoints.length(); i++) {
            JSONObject dataPoint = dataPoints.optJSONObject(i);

            ElementNS dataPointEl = createBoundElement(null, "data-point", null);
            seriesEl.appendChild(dataPointEl);

            if (dataPoint != null) {
                applyStyleAttrs(dataPoint, dataPointEl);
            }

            if (symbol != null) {
                setChartStyleAttribute(dataPointEl, "symbol-type", symbol);

                if (!StringUtils.equals(symbol, "none")) {
                    setChartStyleAttribute(dataPointEl, "symbol-width", "0.25cm");
                    setChartStyleAttribute(dataPointEl, "symbol-height", "0.25cm");
                    setChartStyleAttribute(dataPointEl, "symbol-name", MARKERS.get(i % MARKERS.size()));
                }
            }

        }
        return dataPoints.optJSONObject(dataPoints.length() - 1);
    }

    public void applyStyleAttrs(JSONObject attrs, ElementNS el) {
        if (attrs.isEmpty()) {
            return;
        }
        if (!attrs.has("line") && !attrs.has("fill") && !attrs.has("character")) {
            return;
        }

        StyleManager styleManager = getDocument().getStyleManager();

        String styleName = getOrCreateStyle(el);

        StyleChart style = getStyleChart(styleName);

        try {
            style.applyAttrs(styleManager, attrs);
        } catch (JSONException e) {
            throw new RuntimeException(e);
        }
    }

    public JSONObject getStyleAttrs(ElementNS el) {
        String styleName = getOrCreateStyle(el);
        return getStyleAttrs(styleName);
    }

    private String getOrCreateStyle(Element element) {
        String styleName = element.getAttribute(ChartContent.ODF_STYLE);

        if (StringUtils.isEmpty(styleName)) {
            StyleManager styleManager = getDocument().getStyleManager();
            styleName = getUniqueStyleName();
            styleManager.addStyle(new StyleChart(styleName, true, true));
            setAttribute(element, "style-name", styleName);
        }
        return styleName;
    }

    private String getUniqueStyleName() {
        StyleManager styleManager = getDocument().getStyleManager();
        // String styleName = styleManager.getUniqueStyleName("chart", true);

        int index = 1;
        String styleName = "ch" + index++;
        while (styleManager.getStyle(styleName, "chart", true) != null) {
            styleName = "ch" + index++;
        }

        return styleName;
    }

    private void setChartStyleAttribute(Element element, String name, Object value) {
        setChartStyleAttribute(element, OdfDocumentNamespace.CHART, name, value);
    }

    private void setChartStyleAttribute(Element element, OdfDocumentNamespace namespace, String name, Object value) {
        String styleName = getOrCreateStyle(element);

        setAttribute(element, "style-name", styleName);

        StyleChart style = (StyleChart) getDocument().getStyleManager().getAutoStyle(styleName, "chart", true);

        if (value == null) {
            style.getChartProperties().getAttributes().remove(namespace.getPrefix() + ":" + name);
        } else {
            style.getChartProperties().getAttributes().setValue(namespace.getUri(), name, namespace.getPrefix() + ":" + name, value.toString());
        }
    }

    private void setGraficStyleAttribute(Element element, OdfDocumentNamespace namespace, String name, Object value) {
        String styleName = getOrCreateStyle(element);

        setAttribute(element, "style-name", styleName);

        StyleChart style = (StyleChart) getDocument().getStyleManager().getAutoStyle(styleName, "chart", true);

        if (value == null) {
            style.getGraphicProperties().getAttributes().remove(namespace.getPrefix() + ":" + name);
        } else {
            style.getGraphicProperties().getAttributes().setValue(namespace.getUri(), name, namespace.getPrefix() + ":" + name, value.toString());
        }
    }

    public static String getFormula(String formula) {
        if (StringUtils.isBlank(formula)) {
            return null;
        } else {
            return "[" + formula + "]";
        }
    }

    public static String fromFormula(String value) {
        if (StringUtils.isBlank(value)) {
            return null;
        } else {
            return value.substring(1, value.length() - 1);
        }
    }

    private static void setAttribute(Element element, String name, String value) {
        setAttribute(element, OdfDocumentNamespace.CHART, name, value);
    }

    private static void setAttribute(Element element, OdfDocumentNamespace namespace, String name, String value) {
        if (element == null) {
            throw new RuntimeException("element is null!");
        }
        String qName = namespace.getPrefix() + ":" + name;
        if (StringUtils.isBlank(value)) {
            element.removeAttribute(qName);
        } else {
            ElementNS.setAttribute(element, namespace.getUri(), qName, name, value);
        }
    }

    private static ElementNS getChildByUserData(Node parent, Object userData) {
        NodeList children = parent.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node child = children.item(i);

            if (child.getUserData(USERDATA) == userData) {
                return (ElementNS) child;
            }
        }
        return null;
    }

    private static ElementNS getChildByTag(ElementNS parent, String tag) {
        return getChildByTag(parent, tag, 0);
    }

    private static ElementNS getChildByTag(ElementNS parent, String tag, int index) {
        NodeList tagList = parent.getElementsByTagName(tag);
        if (tagList.getLength() > index) {
            return (ElementNS) tagList.item(index);
        }
        return null;
    }

    private static String getAxisName(ChartAxis axis) {
        String axId = axis.getAxis();

        String xy = axId.substring(0, 1);
        String subType = "primary";
        if (axId.length() > 1 && "2".equals(axId.substring(1, 2))) {
            subType = "secondary";
        }
        return subType + "-" + xy;
    }

    public void createInsertOperations(int sheetIndex, JSONArray operationQueue, int drawingIndex) {

        String names = getNamesFormula();
        String bubbles = getBubblesFormula();

        if (names != null || bubbles != null) {
            for (ChartSeries serie : chart.getSeries()) {
                if (names != null) {
                    serie.setNames(names);
                }
                if (bubbles != null) {
                    serie.setBubbles(serie.getValues());
                    serie.setValues(bubbles);
                }
            }
        }

        chart.createInsertOperations(sheetIndex, operationQueue, drawingIndex);
    }

}
