/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ods.dom;

import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.draw.DrawingType;
import com.openexchange.office.filter.odf.draw.IDrawing;
import com.openexchange.office.filter.odf.draw.Shape;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleGraphic;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.ods.dom.SmlUtils.CellRef;

public class Drawing implements IDrawing {

    private Sheet sheet;
    private DrawingAnchor drawingAnchor;
	final private Shape shape;

	public Drawing(Sheet sheet, DrawingAnchor anchor, Shape shape) {
		drawingAnchor = anchor;
		this.sheet = sheet;
		this.shape = shape;
	}

	@Override
    public DrawingType getType() {
	    return shape.getType();
	}

    @Override
    public DLList<Object> getContent() {
        return shape.getContent();
    }

	public DrawingAnchor getAnchor() {
		return drawingAnchor;
	}

    @Override
    public AttributesImpl getAttributes() {
        return shape.getAttributes();
    }

    public Shape getShape() {
        return shape;
    }

    // changes the DrawingAnchor, if the Drawing is already added to Drawings, then
	// this DrawingAnchor should be changed via Drawings.setDrawingAnchor() only
	public void setDrawingAnchor(DrawingAnchor anchor) {
		drawingAnchor = anchor;
	}

    @Override
    public void writeObject(SerializationHandler output) throws SAXException {
        shape.writeObject(output);
    }

    @Override
    public void createAttrs(OdfOperationDoc operationDocument, OpAttrs attrs, boolean contentAutoStyle) {

        shape.createAttrs(operationDocument, attrs, contentAutoStyle);
        boolean protectedSize = false;
		boolean protectedPosition = false;

		final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
		final String graphicStyleName = shape.getAttributes().getValue("draw:style-name");
		if(graphicStyleName!=null) {
			final StyleBase styleBase = styleManager.getStyle(graphicStyleName, "graphic", true);
			if(styleBase!=null) {
				final String styleProtect = ((StyleGraphic)styleBase).getGraphicProperties().getAttribute("style:protect");
				if(styleProtect!=null) {
					if(styleProtect.contains("size")) {
						protectedSize = true;
					}
					if(styleProtect.contains("position")) {
						protectedPosition = true;
					}
				}
			}
		}
        String anchorType = "absolute";
        final OpAttrs drawingAttrs = attrs.getMap("drawing", true);
		if(drawingAnchor!=null) {
		    anchorType = "oneCell";
		    int startCol = drawingAnchor.getColumn();
		    int startRow = drawingAnchor.getRow();
			Object left = drawingAttrs.remove("left");
			Object top = drawingAttrs.remove("top");
			if(!protectedSize) {
                final String tableEndCellAddress = shape.getAttributes().getValue("table:end-cell-address");
                if(tableEndCellAddress!=null&&!tableEndCellAddress.isEmpty()) {
                    anchorType = "twoCell";
                    final CellRef endAddress = SmlUtils.createCellRefWithSheet(tableEndCellAddress);
                    int endCol = endAddress.getColumn();
                    int endRow = endAddress.getRow();
                    Object endX = shape.getAttributes().getLength100thmm("table:end-x", false);
                    Object endY = shape.getAttributes().getLength100thmm("table:end-y", false);
                    if(endCol < startCol) {
                        endCol = startCol;
                        startCol = endAddress.getColumn();
                        final Object l = left;
                        left = endX;
                        endX = l;
                    }
                    if(endRow < startRow) {
                        endRow = startRow;
                        startRow = endAddress.getRow();
                        final Object t = top;
                        top = endY;
                        endY = t;
                    }
                    drawingAttrs.put("endCol", endCol);
                    drawingAttrs.put("endRow", endRow);
                    if(endX!=null) {
                        drawingAttrs.put("endColOffset", endX);
                    }
                    if(endY!=null) {
                        drawingAttrs.put("endRowOffset", endY);
                    }
                }
			}
            drawingAttrs.put("startCol", startCol);
            drawingAttrs.put("startRow", startRow);
            if(left!=null) {
                drawingAttrs.put("startColOffset", left);
            }
            if(top!=null) {
                drawingAttrs.put("startRowOffset", top);
            }
		}
        drawingAttrs.put("anchorType", anchorType);
		attrs.removeEmpty("line");
		attrs.removeEmpty("fill");
		attrs.removeEmpty("image");
    }

    // anchorType and startCol/startRow is already set

    @Override
    public void applyAttrsFromJSON(OdfOperationDoc operationDocument, JSONObject attrs, boolean contentAutoStyle)
        throws JSONException {

        final AttributesImpl shapeAttributes = shape.getAttributes();
        final JSONObject drawingAttrs = attrs.optJSONObject("drawing");
        if(drawingAttrs!=null) {

            // first updating the drawingAnchor -> sheet / cell anchoring
            final Object newAnchorType = drawingAttrs.opt("anchorType");
            if("absolute".equals(newAnchorType)) {
                sheet.getDrawings().setDrawingAnchor(null, this);
                shapeAttributes.remove("table:end-cell-address");
                shapeAttributes.remove("table:end-x");
                shapeAttributes.remove("table:end-y");
            }
            else {
                // even if JSONObject.NULL we will have an anchor ... -> updating / creating new anchor...
                final DrawingAnchor oldDrawingAnchor = getAnchor();
                int oldCol = 0;
                int oldRow = 0;
                if(oldDrawingAnchor!=null) {
                    oldCol = oldDrawingAnchor.getColumn();
                    oldRow = oldDrawingAnchor.getRow();
                }
                int newCol = oldCol;
                int newRow = oldRow;
                if(drawingAttrs.has("startCol")) {
                    newCol = drawingAttrs.getInt("startCol");
                }
                if(drawingAttrs.has("startRow")) {
                    newRow = drawingAttrs.getInt("startRow");
                }
                if(newCol!=oldCol||newRow!=oldRow) {
                    // ensure that the cell where the drawing is attached to is physically available ...
                    sheet.getRow(newRow, true, true, true).getCell(newCol, true, true, true);
                    sheet.getDrawings().setDrawingAnchor(new DrawingAnchor(newCol, newRow), this);
                }
            }
            if(drawingAnchor!=null) {
                // col/row offsets are needed at the transformer
                drawingAttrs.remove("left");
                drawingAttrs.remove("top");
            }
        }

        shape.applyAttrsFromJSON(operationDocument, attrs, contentAutoStyle);

		if(drawingAttrs!=null&&drawingAnchor!=null) {

		    // applying new endCellAddress
            final Object endColO = drawingAttrs.opt("endCol");
            final Object endRowO = drawingAttrs.opt("endRow");
            if(endColO!=null||endRowO!=null) {

                // initializing endAddress with defaults
    		    CellRef endAddress = null;
                final String tableEndCellAddress = shapeAttributes.getValue("table:end-cell-address");
                if(tableEndCellAddress!=null&&!tableEndCellAddress.isEmpty()) {
                    endAddress = SmlUtils.createCellRefWithSheet(tableEndCellAddress);
                }
                if(endAddress==null) {
                    endAddress = new CellRef(drawingAnchor.getColumn(), drawingAnchor.getRow());
                }
    
                if(endColO instanceof Integer) {
                    endAddress.setColumn((Integer)endColO);
                }
                if(endRowO instanceof Integer) {
                    endAddress.setRow((Integer)endRowO);
                }
                shapeAttributes.setValue(Namespaces.TABLE, "end-cell-address", "table:end-cell-address", sheet.getSheetName() + "." + SmlUtils.getCellRef(endAddress));
            }

            if(drawingAttrs.has("startColOffset")) {
                shape.getTransformer().setX(drawingAttrs.optInt("startColOffset"));
            }
            if(drawingAttrs.has("startRowOffset")) {
                shape.getTransformer().setY(drawingAttrs.optInt("startRowOffset"));
            }
		    Object endColOffset = drawingAttrs.opt("endColOffset");
		    if(endColOffset!=null) {
		        shapeAttributes.setLength100thmm(Namespaces.TABLE, "end-x", "table:end-x", endColOffset instanceof Number ? Integer.valueOf(((Number)endColOffset).intValue()) : 0);
		    }
		    Object endRowOffset = drawingAttrs.opt("endRowOffset");
		    if(endRowOffset!=null) {
		        shapeAttributes.setLength100thmm(Namespaces.TABLE, "end-y", "table:end-y", endRowOffset instanceof Number ? Integer.valueOf(((Number)endRowOffset).intValue()) : 0);
		    }
		}
    }
}
