/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.draw;

import java.awt.geom.Rectangle2D;
import java.util.Map;
import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.OdfTextDocument;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.INodeAccessor;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.SaxContextHandler;
import com.openexchange.office.filter.odt.dom.TextContentHelper;

public class Shape implements IDrawing, INodeAccessor {

	protected final String uri;
	protected final String localName;
	protected final String qName;
    protected final boolean rootShape;
	protected final AttributesImpl attributes;
	protected final Transformer transformer;
	protected final DLList<Object> childs = new DLList<Object>();

	public Shape(Attributes attributes, String uri, String localName, String qName, boolean rootShape) {
		this.attributes = new AttributesImpl(attributes);
		this.transformer = new Transformer(this.attributes, isLineMode());
		this.uri = uri;
		this.localName = localName;
		this.qName = qName;
		this.rootShape = rootShape;
	}

	public Shape(String documentType, String uri, String localName, String qName, boolean rootShape) {
		this.attributes = new AttributesImpl();
        this.transformer = new Transformer(isLineMode());
		this.uri = uri;
		this.localName = localName;
		this.qName = qName;
		this.rootShape = rootShape;

		if(rootShape&&documentType.equals("text")) {
		    this.attributes.setValue(Namespaces.TEXT, "anchor-type", "text:anchor-type", "as-char");
		}
	}

    public Transformer getTransformer() {
	    return transformer;
	}

	public boolean isLineMode() {
	    return false;
	}

	@Override
	public DLList<Object> getContent() {
		return childs;
	}

	@Override
    public DrawingType getType() {
		return DrawingType.SHAPE;
	}

	@Override
    public AttributesImpl getAttributes() {
		return attributes;
	}

	public String getStyleName() {
		return attributes.getValue("draw:style-name");
	}

	public void setStyleName(String style) {
		attributes.setValue(Namespaces.DRAW, "style-name", "draw:style-name", style);
	}

	public void rotate(double rotation) {
        final Double currentRotation = transformer.getRotation();
        transformer.setRotation(currentRotation==null ? rotation : currentRotation + rotation);
	}

	public void scaleX(Rectangle2D parent, double scaleX) {
	    transformer.setWidth(transformer.getWidth() * scaleX);
	    transformer.setX(((transformer.getX()-parent.getX()) * scaleX) + parent.getX());
	}

    public void scaleY(Rectangle2D parent, double scaleY) {
        transformer.setHeight(transformer.getHeight() * scaleY);
        transformer.setY(((transformer.getY()-parent.getY()) * scaleY) + parent.getY());
    }

    public void translateX(double transX) {
        transformer.setX(transformer.getX() + transX);
    }

    public void translateY(double transY) {
        transformer.setY(transformer.getY() + transY);
    }

    public Rectangle2D getChildRect() {
        return transformer.getRect2D();
    }

	@Override
	public void writeObject(SerializationHandler output)
		throws SAXException {

		output.startElement(uri, localName, qName);
		transformer.updateAttributes(attributes);
		attributes.write(output);
		TextContentHelper.write(output, getContent());
		output.endElement(uri, localName, qName);
	}

    public SaxContextHandler getContextHandler(SaxContextHandler parentHandler) {
        if(getType()==DrawingType.SHAPE) {
            return new ShapeHandler(parentHandler, this);
        }
        return new ShapeUnknownChildsHandler(parentHandler, this);
    }

	@Override
	public void applyAttrsFromJSON(OdfOperationDoc operationDocument, JSONObject attrs, boolean contentAutoStyle)
		throws JSONException {

		setStyleName(operationDocument.getDocument().getStyleManager().applyAttributes("graphic", getStyleName(), contentAutoStyle, attrs));

		final JSONObject drawingProps = attrs.optJSONObject("drawing");
        if(drawingProps!=null) {
            if (drawingProps.has("name")) {
                final String name = drawingProps.optString("name");
                if (name!=null && !name.isEmpty()) {
                	attributes.setValue(Namespaces.DRAW, "name", "draw:name", name);
                }
            }
            if(rootShape && operationDocument.getDocument() instanceof OdfTextDocument) {
                if (drawingProps.has("anchorHorOffset")) {
                    transformer.setX(drawingProps.optInt("anchorHorOffset"));
                }
                if (drawingProps.has("anchorVertOffset")) {
                    transformer.setY(drawingProps.optInt("anchorVertOffset"));
                }
                applyTextRootShapeProps(drawingProps);
            }
            else {
                if (drawingProps.has("left")) {
                    transformer.setX(drawingProps.optInt("left"));
                }
                if (drawingProps.has("top")) {
                    transformer.setY(drawingProps.optInt("top"));
                }
            }
            if (drawingProps.has("width")) {
                transformer.setWidth(drawingProps.optInt("width"));
            }
            if (drawingProps.has("height")) {
                transformer.setHeight(drawingProps.optInt("height"));
            }
            final Object rotation = drawingProps.opt("rotation");
            if(rotation!=null) {
                transformer.setRotation(rotation instanceof Number ? ((Number)rotation).doubleValue() : null);
            }
        }
        final JSONObject shapeProps = attrs.optJSONObject("shape");
        if(shapeProps!=null) {
            if (shapeProps.has("autoResizeHeight") && shapeProps.optBoolean("autoResizeHeight") == true) {
            	attributes.remove("svg:height");
            }
        }
	}

	public void applyTextRootShapeProps(JSONObject drawingProps)
	    throws JSONException {

	    if (drawingProps.hasAndNotNull("inline") && drawingProps.optBoolean("inline")) {
            attributes.setValue(Namespaces.TEXT, "anchor-type", "text:anchor-type", "as-char");
        } else if (drawingProps.has("anchorHorBase") && drawingProps.has("anchorVertBase")) {
            String anchorHorBase = drawingProps.optString("anchorHorBase");
            String anchorVertBase = drawingProps.optString("anchorVertBase");
            if (anchorHorBase != null && anchorVertBase != null) {
                if (anchorHorBase.equals("page") && anchorVertBase.equals("page")) {
                    attributes.setValue(Namespaces.TEXT, "anchor-type", "text:anchor-type", "paragraph");

                } else if (anchorHorBase.equals("column") && anchorVertBase.equals("margin")) {
                    attributes.setValue(Namespaces.TEXT, "anchor-type", "text:anchor-type", "frame");

                } else if (anchorHorBase.equals("column") && anchorVertBase.equals("paragraph")) {
                    attributes.setValue(Namespaces.TEXT, "anchor-type", "text:anchor-type", "paragraph");
                    //apply related default wrapping, if not part of the attributes:
                    if (!drawingProps.has("textWrapMode") && !drawingProps.has("textWrapSide")) {
                        drawingProps.put("textWrapMode", "topAndBottom");
                    }
                } else if (anchorHorBase.equals("character") && anchorVertBase.equals("paragraph")) {
                    attributes.setValue(Namespaces.TEXT, "anchor-type", "text:anchor-type", "char");
                } else { // the default is "inline" a
                    attributes.setValue(Namespaces.TEXT, "anchor-type", "text:anchor-type", "as-char");
                }
            }
        } else {
            if (drawingProps.has("anchorHorBase")) {
                String anchorHorBase = drawingProps.optString("anchorHorBase");
                if (anchorHorBase != null) {
                    if (anchorHorBase.equals("page")) {
                        attributes.setValue(Namespaces.TEXT, "anchor-type", "text:anchor-type", "page");

                    } else if (anchorHorBase.equals("column")) {
                        attributes.setValue(Namespaces.TEXT, "anchor-type", "text:anchor-type", "paragraph");

                    } else if (anchorHorBase.equals("character")) {
                        attributes.setValue(Namespaces.TEXT, "anchor-type", "text:anchor-type", "char");
                    }
                }
            }
        }
        if (drawingProps.has("anchorLayerOrder")) {
            int anchorLayerOrder = drawingProps.optInt("anchorLayerOrder", 0);
            attributes.setIntValue(Namespaces.DRAW, "z-index", "draw:z-index", anchorLayerOrder);
        }
	}

	@Override
	public void createAttrs(OdfOperationDoc operationDocument, OpAttrs attrs, boolean contentAutoStyle) {
	    final String docType = operationDocument.getDocument().getDocumentType();
	    if(getStyleName()!=null&&!getStyleName().isEmpty()) {
			operationDocument.getDocument().getStyleManager().
			    collectAllGraphicPropertiesFromStyle(attrs, getStyleName(), "graphic", contentAutoStyle, true);
		}
		final OpAttrs drawingAttrs = attrs.getMap("drawing", true);
		final boolean isTextRootShape = rootShape && docType.equals("text");

		int x = Double.valueOf(transformer.getX()).intValue();
		int y = Double.valueOf(transformer.getY()).intValue();
		int width = Double.valueOf(transformer.getWidth()).intValue();
		int height = Double.valueOf(transformer.getHeight()).intValue();
		if(isLineMode()) {
		    if(width<0) {
                x += width;
		        width = -width;
		        drawingAttrs.put("flipH", true);
		    }
		    if(height<0) {
		        y += height;
		        height = -height;
		        drawingAttrs.put("flipV", true);
		    }
		}
        drawingAttrs.put("width", width);
        drawingAttrs.put("height", height);
        drawingAttrs.put(isTextRootShape ? "anchorHorOffset" : "left", x);
        drawingAttrs.put(isTextRootShape ? "anchorVertOffset" : "top", y);

        final Double rotation = transformer.getRotation();
        if(rotation!=null) {
            drawingAttrs.put("rotation", rotation);
        }
        if(isTextRootShape) {
            createTextRootShapeProps(drawingAttrs);
        }
        if(!drawingAttrs.containsKey("height")) {
    		final Map<String, Object> shapeAttrs = attrs.getMap("shape", true);
    		shapeAttrs.put("autoResizeHeight", true);
        }
	}

	public void createTextRootShapeProps(OpAttrs drawingAttrs) {
	    final Integer zIndex = attributes.getIntValue("draw:z-index");
        if(zIndex!=null) {
            drawingAttrs.put("anchorLayerOrder", zIndex);
        }
        final String anchorType = attributes.getValue("text:anchor-type");
        if(anchorType!=null) {
            String anchorVertBase = null;
            String anchorHorBase = null;
            if (anchorType.equals("page")) {
                // OX API: true: image as character, false: floating mode
                drawingAttrs.put("inline", Boolean.FALSE);
                //page anchor requires page relation
                drawingAttrs.put("anchorHorBase", "page");
                drawingAttrs.put("anchorVertBase", "page");
            } else if (anchorType.equals("frame")) {
                // OX API: true: image as character, false: floating mode
                drawingAttrs.put("inline", Boolean.FALSE);
                anchorVertBase = "column";
                anchorVertBase= "margin";
            } else if (anchorType.equals("paragraph")) {
                // OX API: true: image as character, false: floating mode
                drawingAttrs.put("inline", Boolean.FALSE);
                anchorHorBase = "column";
                anchorVertBase = "paragraph";
            } else if (anchorType.equals("char")) {
                // OX API: true: image as character, true: floating mode
                drawingAttrs.put("inline", Boolean.FALSE);
                anchorHorBase = "character";
                anchorVertBase = "paragraph";
            } else if (anchorType.equals("as-char")) {
                // OX API: true: image as character, false: floating mode
                drawingAttrs.put("inline", Boolean.TRUE);
            }
            if(anchorVertBase != null && !drawingAttrs.containsKey("anchorVertBase")) {
                drawingAttrs.put("anchorVertBase", anchorVertBase);
            }
            if(anchorHorBase != null && !drawingAttrs.containsKey("anchorHorBase")) {
                drawingAttrs.put("anchorHorBase", anchorHorBase);
            }
        }
	}

	public static Shape createShape(String documentType, JSONObject attrs, boolean rootShape) {
	    if(attrs!=null) {
	        final JSONObject geometryAttrs = attrs.optJSONObject("geometry");
	        if(geometryAttrs!=null) {
	            final String presetShapeType = geometryAttrs.optString("presetShape", null);
	            final String hostData = geometryAttrs.optString("hostData", null);
	            if(presetShapeType!=null||hostData!=null) {
	                return new CustomShape(documentType, rootShape);
	            }
	        }
	    }
	    return new Shape(documentType, Namespaces.DRAW, "rect", "draw:rect", rootShape);
	}
}
