/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.draw;

import java.awt.geom.Rectangle2D;
import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.OdfTextDocument;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.SaxContextHandler;

public class GroupShape extends Shape {

    private Rectangle2D rect2D;

    // following values are to be applied to the group and its childs when saved
    private Double newX;
    private Double newY;
    private Double newWidth;
    private Double newHeight;
    private Double rotation;

    public GroupShape(String documentType, boolean rootShape) {
        super(documentType, Namespaces.DRAW, "g", "draw:g", rootShape);
        rect2D = null;
        newX = null;
        newY = null;
        newWidth = null;
        newHeight = null;
        rotation = null;
    }

    public GroupShape(Attributes attributes, boolean rootShape) {
	    super(attributes, Namespaces.DRAW, "g", "draw:g", rootShape);
	    rect2D = null;
        newX = null;
        newY = null;
        newWidth = null;
        newHeight = null;
        rotation = null;
	}

    @Override
    public DrawingType getType() {
        return DrawingType.GROUP;
    }

    @Override
    public void rotate(double rotate) {
        for(Object child:childs) {
            ((Shape)child).rotate(rotate);
        }
    }

    @Override
    public void scaleX(Rectangle2D parent, double scaleX) {
        for(Object child:childs) {
            ((Shape)child).scaleX(parent, scaleX);
        }
    }

    @Override
    public void scaleY(Rectangle2D parent, double scaleY) {
        for(Object child:childs) {
            ((Shape)child).scaleY(parent, scaleY);
        }
    }

    @Override
    public void translateX(double translateX) {
        for(Object child:childs) {
            ((Shape)child).translateX(translateX);
        }
    }

    @Override
    public void translateY(double translateY) {
        for(Object child:childs) {
            ((Shape)child).translateY(translateY);
        }
    }

    @Override
    public void writeObject(SerializationHandler output)
        throws SAXException {

        finalizeGroup();
        super.writeObject(output);
    }

    public void finalizeGroup() {
        if(rotation!=null) {
            rotate(rotation);
            rotation = null;
        }
        if(newX!=null||newY!=null||newWidth!=null||newHeight!=null) {
            if(rect2D==null) {
                getChildRect();
            }
            if(newWidth!=null&&rect2D.getWidth()!=0) {
                scaleX(rect2D, newWidth / rect2D.getWidth());
            }
            if(newHeight!=null&&rect2D.getHeight()!=0) {
                scaleY(rect2D, newHeight / rect2D.getHeight());
            }
            if(newX!=null) {
                translateX(newX - rect2D.getX());
            }
            if(newY!=null) {
                translateY(newY - rect2D.getY());
            }
            newX = null;
            newY = null;
            newWidth = null;
            newHeight = null;
        }
    }

    @Override
    public SaxContextHandler getContextHandler(SaxContextHandler parentHandler) {
        return new ShapeGroupHandler(parentHandler, this);
    }

    @Override
    public void applyAttrsFromJSON(OdfOperationDoc operationDocument, JSONObject attrs, boolean contentAutoStyle) throws JSONException {

        final boolean isTextRootShape = rootShape && operationDocument.getDocument() instanceof OdfTextDocument;

        final JSONObject drawingProps = attrs.optJSONObject("drawing");
        if(drawingProps != null) {
            if (drawingProps.has("name")) {
                String name = drawingProps.optString("name");
                if (name != null && !name.isEmpty()) {
                    attributes.setValue(Namespaces.DRAW, "name", "draw:name", name);
                }
            }
            if(rootShape && operationDocument.getDocument() instanceof OdfTextDocument) {
                if (drawingProps.has("anchorHorOffset")) {
                    newX = drawingProps.optDouble("anchorHorOffset", 0);
                }
                if (drawingProps.has("anchorVertOffset")) {
                    newY = drawingProps.optDouble("anchorVertOffset", 0);
                }
                applyTextRootShapeProps(drawingProps);
            }
            else {
                if (drawingProps.has("left")) {
                    newX = drawingProps.optDouble("left", 0);
                }
                if (drawingProps.has("top")) {
                    newY = drawingProps.optDouble("top", 0);
                }
            }
            if (drawingProps.has("width")) {
                newWidth = drawingProps.optDouble("width", 1);
            }
            if (drawingProps.has("height")) {
                newHeight = drawingProps.optDouble("height", 1);
            }
            final Object rot = drawingProps.opt("rotation");
            if(rot instanceof Number) {
                rotation = (((Number)rot).doubleValue());
            }
            if(isTextRootShape) {
                applyTextRootShapeProps(drawingProps);
            }
        }
    }

    @Override
    public void createAttrs(OdfOperationDoc operationDocument, OpAttrs attrs, boolean contentAutoStyle) {

        final String docType = operationDocument.getDocument().getDocumentType();
        final OpAttrs drawingAttrs = attrs.getMap("drawing", true);
        final boolean isTextRootShape = rootShape && docType.equals("text");

        int x = 0;
        int y = 0;
        int width = 0;
        int height = 0;

        final Rectangle2D childRect = getChildRect();
        if(!childRect.isEmpty()) {
            x = Double.valueOf(childRect.getX()).intValue();
            y = Double.valueOf(childRect.getY()).intValue();
            width = Double.valueOf(childRect.getWidth()).intValue();
            height = Double.valueOf(childRect.getHeight()).intValue();
        }
        drawingAttrs.put("width", width);
        drawingAttrs.put("height", height);
        drawingAttrs.put(isTextRootShape ? "anchorHorOffset" : "left", x);
        drawingAttrs.put(isTextRootShape ? "anchorVertOffset" : "top", y);

        drawingAttrs.put("childWidth", width);
        drawingAttrs.put("childHeight", height);
        drawingAttrs.put("childLeft", x);
        drawingAttrs.put("childTop", y);

        if(isTextRootShape) {
            createTextRootShapeProps(drawingAttrs);
        }
        if(drawingAttrs.isEmpty()) {
            attrs.remove("drawing");
        }
    }

    @Override
    public Rectangle2D getChildRect() {
        if(rect2D!=null) {
            return rect2D;
        }
        rect2D = new Rectangle2D.Double();
        for(Object child:getContent()) {
            if(child instanceof Shape) {
                final Rectangle2D childRect = ((Shape)child).getChildRect();
                if(rect2D.isEmpty()) {
                    rect2D = childRect;
                }
                else {
                    rect2D = rect2D.createUnion(childRect);
                }
            }
        }
        return rect2D;
    }
}
