/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/selection/selection',
    'io.ox/office/textframework/utils/dom'
], function (AppHelper, DrawingFrame, Selection, DOM) {

    'use strict';

    // class Selection ===========================================================

    describe('Textframework class Selection', function () {

        it('should exist', function () {
            expect(Selection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { defaultTabStop: 1270, zoom: { value: 100 } }, page: { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 } } },
            { name: 'insertParagraph', start: [0] },
            { name: 'splitParagraph', start: [0, 0] },
            { name: 'insertText', text: 'Hello World', start: [1, 0] },
            { name: 'splitParagraph', start: [1, 0] },
            { name: 'setAttributes', start: [2], attrs: { paragraph: { pageBreakBefore: true } } },
            { name: 'splitParagraph', start: [2, 0] },
            { name: 'setAttributes', start: [3], attrs: { paragraph: { pageBreakBefore: true } } }
        ];

        var model, selection, pageLayout;
        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            selection = model.getSelection();
            pageLayout = model.getPageLayout();
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a Selection class instance', function () {
                expect(selection).to.be.an['instanceof'](Selection);
            });
        });

        // public methods -----------------------------------------------------

        before(function () {
            pageLayout.callInitialPageBreaks();
        });

        describe('method "getFirstDocumentPosition"', function () {
            it('should return correct first oxo position in document', function () {
                expect(selection.getFirstDocumentPosition()).to.eql([0, 0]);
            });
        });

        describe('method "getLastDocumentPosition"', function () {
            it('should return correct last oxo position in document', function () {
                expect(selection.getLastDocumentPosition()).to.eql([3, 11]);
            });
        });

        describe('method "isAllSelected"', function () {
            it('should not be true when not all is selected in document', function () {
                expect(selection.isAllSelected()).to.equal(false);
            });
        });

        describe('method "selectAll"', function () {
            before(function () {
                selection.selectAll();
            });

            it('should be true when all is selected in document', function () {
                expect(selection.isAllSelected()).to.equal(true);
            });
        });

    });

    // ========================================================================

    describe('Presentation class Selection', function () {

        it('should exist', function () {
            expect(Selection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var model = null,
            selection = null,
            layoutId_1 = 'layout1',
            masterId_1 = 'master1',
            activeSlide = null,
            activeSlideId = null,
            drawings = null,
            slide_1_id = 'slide_1', // the ID of the first slide in document view
            text_para1_drawing1 = 'Paragraph 1 in the shape smileyFace',

            // the operations to be applied by the document model
            OPERATIONS = [
                {
                    name: 'setDocumentAttributes',
                    attrs: {
                        page: { width: 33866, height: 19050, orientation: 'landscape' },
                        defaultTextListStyles: {
                            l1: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 0 } },
                            l2: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 1270 } },
                            l3: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 2540 } }
                        }
                    }
                },
                { name: 'insertMasterSlide', id: masterId_1 },

                { name: 'insertLayoutSlide', id: layoutId_1, target: masterId_1 },
                { name: 'insertDrawing', start: [0, 0], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'ctrTitle' }, drawing: { name: 'Titel 1', left: 1905, top: 5918, width: 21590, height: 4083 } } },
                { name: 'insertParagraph', start: [0, 0, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 0, 0, 0], target: layoutId_1, text: 'Title' },
                { name: 'insertDrawing', start: [0, 1], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'subTitle', phIndex: 1 }, drawing: { name: 'Untertitel 2', left: 3810, top: 10795, width: 17780, height: 4868 } } },
                { name: 'insertParagraph', start: [0, 1, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 1, 0, 0], target: layoutId_1, text: 'Subtitle' },

                { name: 'insertSlide', start: [0], target: layoutId_1 },

                // inserting a drawing of type 'shape', preset shape attribute is 'smileyFace'
                { name: 'insertDrawing', attrs: { fill: { type: 'solid', color: { type: 'scheme', value: 'accent1', fallbackValue: '4F81BD' } }, line: { type: 'solid', style: 'solid', width: 26, color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 50000 }], fallbackValue: '254061' } }, geometry: { presetShape: 'smileyFace' }, character: { color: { type: 'scheme', value: 'light1' } }, drawing: { name: 'smileyFace', left: 4286, top: 8969, width: 6853, height: 5874 }, shape: { anchor: 'centered', anchorCentered: false, wordWrap: true, horzOverflow: 'overflow', vertOverflow: 'overflow', paddingRight: 685, paddingLeft: 685, paddingBottom: 587, paddingTop: 587 } }, start: [0, 0], type: 'shape' },
                { name: 'setAttributes', attrs: { shape: { anchor: 'centered' } }, start: [0, 0] },
                { name: 'insertParagraph', start: [0, 0, 0], attrs: { character: { color: { type: 'scheme', value: 'light1' } }, paragraph: { alignment: 'center', bullet: { type: 'none' }, indentFirstLine: 0 } } },
                { name: 'insertText', text: text_para1_drawing1, start: [0, 0, 0, 0] },

                // inserting a drawing of type 'connector', preset shape attribute is 'line'
                { name: 'insertDrawing', attrs: { line: { type: 'solid', style: 'solid', width: 26, color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 50000 }], fallbackValue: '254061' } }, geometry: { presetShape: 'line' }, character: { color: { type: 'scheme', value: 'light1' } }, drawing: { name: 'line', left: 14287, top: 4762, width: 8229, height: 4657 }, shape: { anchor: 'centered', anchorCentered: false, wordWrap: true, horzOverflow: 'overflow', vertOverflow: 'overflow', paddingRight: 823, paddingLeft: 823, paddingBottom: 466, paddingTop: 466 } }, start: [0, 1], type: 'connector' },
                { name: 'setAttributes', attrs: { shape: { anchor: 'centered' } }, start: [0, 1] }
            ];

        AppHelper.createPresentationApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            selection = model.getSelection();
            // drawingStyles = model.getDrawingStyles();
        });

        // constructor --------------------------------------------------------

        describe('drawing selection preparations', function () {

            it('should create a Selection class instance', function () {
                expect(selection).to.be.an['instanceof'](Selection);
            });

            it('should have the first slide activated', function () {
                activeSlideId = model.getActiveSlideId();
                expect(activeSlideId).to.equal(slide_1_id);
            });

            it('should find two drawings on the active slide', function () {
                activeSlide = model.getSlideById(activeSlideId);
                expect(activeSlide.length).to.equal(1);
                drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(drawings.length).to.equal(2);
            });

            it('should have the correct drawing types specified at the drawing nodes', function () {
                expect(DrawingFrame.getDrawingType(drawings[0])).to.equal('shape');  // the smiley face
                expect(DrawingFrame.getDrawingType(drawings[1])).to.equal('connector');  // the line
            });

        });

        describe('method "isTextCursor"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('isTextCursor');
            });

            it('should be a text cursor selection after loading the document', function () {
                expect(selection.isTextCursor()).to.equal(true); // in a slide selection
            });

        });

        describe('method "isTopLevelTextCursor"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('isTopLevelTextCursor');
            });

            it('should be a text cursor selection after loading the document', function () {
                expect(selection.isTopLevelTextCursor()).to.equal(true); // a slide selection
            });

        });

        describe('method "isSlideSelection"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('isSlideSelection');
            });

            it('should be a slide selection after loading the document', function () {
                expect(selection.isSlideSelection()).to.equal(true);
            });

        });

        describe('method "isDrawingSelection"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('isDrawingSelection');
            });

            it('should not be a drawing selection after loading the document', function () {
                expect(selection.isDrawingSelection()).to.equal(false);
            });

        });

        describe('method "getSelectedDrawing"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('getSelectedDrawing');
            });

            it('should return no drawing after loading the document (empty jQuery collection)', function () {
                expect(selection.getSelectedDrawing().length).to.equal(0);
            });

        });

        describe('method "getSelectedTextFrameDrawing"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('getSelectedTextFrameDrawing');
            });

            it('should return no drawing after loading the document (empty jQuery collection)', function () {
                expect(selection.getSelectedTextFrameDrawing().length).to.equal(0);
            });

        });

        describe('method "isAdditionalTextframeSelection"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('isAdditionalTextframeSelection');
            });

            // selecting the text in the drawing
            it('should be an additional text frame selection, if the cursor is set into the first drawing', function () {

                expect(selection.isAdditionalTextframeSelection()).to.equal(false);

                selection.setTextSelection([0, 0, 0, 0]);  // setting the cursor into the first drawing on the first slide

                expect(selection.isAdditionalTextframeSelection()).to.equal(true);
                expect(selection.isDrawingSelection()).to.equal(false);  // no drawing selection
                expect(selection.isSlideSelection()).to.equal(false); // no longer a slide selection
                expect(selection.isTextCursor()).to.equal(true);  // a text cursor selection inside the drawing
                expect(selection.isTopLevelTextCursor()).to.equal(false); // no longer a slide selection
            });

            it('should be the first drawing on the slide that is additionally selected', function () {
                var selectedDrawing = selection.getSelectedTextFrameDrawing();
                expect(selectedDrawing.length).to.equal(1);
                expect(selectedDrawing[0] === drawings[0]).to.equal(true);  // checking that drawing nodes are identical

                selectedDrawing = selection.getSelectedDrawing();
                expect(selectedDrawing.length).to.equal(0); // no drawing selection
            });

            it('should have 8 resizer nodes at the additional drawing selection', function () {

                // getting the drawing node in the drawing selection overlay node
                var drawingSelectionNode = selection.getSelectedTextFrameDrawing().data('selection');
                expect(drawingSelectionNode.length).to.equal(1);

                // the 'selection' node is a child of the drawingSelectionNode
                var selectionNode = drawingSelectionNode.children('.selection');
                expect(selectionNode.length).to.equal(1);

                // the 'resizers' node is a child of the selection
                var resizersNode = selectionNode.children('.resizers');
                expect(resizersNode.length).to.equal(1);

                // the 'data-pos' nodes are children of the resizers node
                expect(resizersNode.children('div[data-pos]').length).to.equal(8);  // 8 resizers
            });

        });

        // selecting the 'line' drawing
        describe('selecting the line drawing', function () {

            // selecting the drawing
            it('should be a drawing selection', function () {

                expect(selection.isAdditionalTextframeSelection()).to.equal(true);
                expect(selection.isDrawingSelection()).to.equal(false);  // no drawing selection

                selection.setTextSelection([0, 1], [0, 2]);  // selecting the line drawing on the first slide

                expect(selection.isAdditionalTextframeSelection()).to.equal(false);
                expect(selection.isDrawingSelection()).to.equal(true);  // drawing selection
                expect(selection.isSlideSelection()).to.equal(false); // no longer a slide selection
                expect(selection.isTextCursor()).to.equal(false);  // a text cursor selection inside the drawing
                expect(selection.isTopLevelTextCursor()).to.equal(false); // no longer a slide selection
            });

            it('should be the second drawing on the slide that is selected', function () {

                var selectedDrawing = selection.getSelectedTextFrameDrawing();
                expect(selectedDrawing.length).to.equal(0);

                selectedDrawing = selection.getSelectedDrawing();
                expect(selectedDrawing.length).to.equal(1); // no drawing selection
                expect(selectedDrawing[0] === drawings[1]).to.equal(true);  // checking that drawing nodes are identical
            });

            it('should have only 2 resizer nodes at the additional drawing selection', function () {

                // getting the drawing node in the drawing selection overlay node
                var drawingSelectionNode = selection.getSelectedDrawing().data('selection');
                expect(drawingSelectionNode.length).to.equal(1);

                // the 'selection' node is a child of the drawingSelectionNode
                var selectionNode = drawingSelectionNode.children('.selection');
                expect(selectionNode.length).to.equal(1);

                // the 'resizers' node is a child of the selection
                var resizersNode = selectionNode.children('.resizers');
                expect(resizersNode.length).to.equal(1);

                // the 'data-pos' nodes are children of the resizers node
                expect(resizersNode.children('div[data-pos]').length).to.equal(2);  // only 2 resizers
            });

        });

        // selecting the 'smileyFace' drawing
        describe('selecting the line drawing', function () {

            it('should be a drawing selection', function () {

                expect(selection.isAdditionalTextframeSelection()).to.equal(false);
                expect(selection.isDrawingSelection()).to.equal(true);  // drawing selection

                selection.setTextSelection([0, 0], [0, 1]);  // selecting the smiley face drawing on the first slide

                expect(selection.isAdditionalTextframeSelection()).to.equal(false);
                expect(selection.isDrawingSelection()).to.equal(true);  // drawing selection
                expect(selection.isSlideSelection()).to.equal(false); // no longer a slide selection
                expect(selection.isTextCursor()).to.equal(false);  // a text cursor selection inside the drawing
                expect(selection.isTopLevelTextCursor()).to.equal(false); // no longer a slide selection
            });

            it('should be the first drawing on the slide that is selected', function () {

                var selectedDrawing = selection.getSelectedTextFrameDrawing();
                expect(selectedDrawing.length).to.equal(0);

                selectedDrawing = selection.getSelectedDrawing();
                expect(selectedDrawing.length).to.equal(1); // no drawing selection
                expect(selectedDrawing[0] === drawings[0]).to.equal(true);  // checking that drawing nodes are identical
            });

            it('should have 8 resizer nodes at the additional drawing selection', function () {

                // getting the drawing node in the drawing selection overlay node
                var drawingSelectionNode = selection.getSelectedDrawing().data('selection');
                expect(drawingSelectionNode.length).to.equal(1);

                // the 'selection' node is a child of the drawingSelectionNode
                var selectionNode = drawingSelectionNode.children('.selection');
                expect(selectionNode.length).to.equal(1);

                // the 'resizers' node is a child of the selection
                var resizersNode = selectionNode.children('.resizers');
                expect(resizersNode.length).to.equal(1);

                // the 'data-pos' nodes are children of the resizers node
                expect(resizersNode.children('div[data-pos]').length).to.equal(8);  // 8 resizers
            });

        });

        // multi selection tests

        describe('method "isMultiSelectionSupported"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('isMultiSelectionSupported');
            });

            it('should return true in presentation application', function () {
                expect(selection.isMultiSelectionSupported()).to.equal(true);
            });

        });

        describe('method "isMultiSelection"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('isMultiSelection');
            });

            it('should return false, if only one drawing is selected', function () {
                expect(selection.isDrawingSelection()).to.equal(true);
                expect(selection.isMultiSelection()).to.equal(false);
            });

        });

        describe('method "selectAllDrawingsOnSlide"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('selectAllDrawingsOnSlide');
            });

            it('should return the number of selected drawings on the slide', function () {
                expect(selection.selectAllDrawingsOnSlide()).to.equal(2);
            });

            it('should be a multi selection and a drawing selection', function () {
                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.isDrawingSelection()).to.equal(true);
                expect(selection.isAdditionalTextframeSelection()).to.equal(false);
            });

            it('should generate two drawing selection nodes in the selection overlay node', function () {

                var drawingSelection1 = $(drawings[0]).data('selection');
                var drawingSelection2 = $(drawings[1]).data('selection');

                expect(drawingSelection1.length).to.equal(1);
                expect(drawingSelection2.length).to.equal(1);

                var selectionOverlayNode = drawingSelection1.parent();
                expect(selectionOverlayNode.hasClass('drawingselection-overlay')).to.equal(true);

                expect(drawingSelection1.parent()[0] === drawingSelection2.parent()[0]).to.equal(true);

                expect(selectionOverlayNode.parent()[0] === model.getNode()[0]).to.equal(true); // the parent node is the page

                expect(selectionOverlayNode.children().length).to.equal(2); // two active drawing selections

            });

            it('should remove all drawing selections from the overlay node in a text selection', function () {

                selection.setTextSelection([0, 0, 0, 0]);  // setting the cursor into the first drawing on the first slide

                expect(selection.isAdditionalTextframeSelection()).to.equal(true);
                expect(selection.isDrawingSelection()).to.equal(false);  // no drawing selection
                expect(selection.isSlideSelection()).to.equal(false); // no slide selection
                expect(selection.isTextCursor()).to.equal(true);  // a text cursor selection inside the drawing
                expect(selection.isMultiSelection()).to.equal(false); // no multi selection

                // the first drawing is still selected as additional text frame selection
                var drawingSelection = $(drawings[0]).data('selection');
                expect(drawingSelection.length).to.equal(1);

                var selectionOverlayNode = drawingSelection.parent();
                expect(selectionOverlayNode.hasClass('drawingselection-overlay')).to.equal(true);

                expect(selectionOverlayNode.children().length).to.equal(1); // one active drawing selection

                // the 'data-pos' nodes in the selection ndoe
                expect(drawingSelection.find('div[data-pos]').length).to.equal(12); // 12 'data-pos' nodes at all

                // the 'data-pos' nodes are children of the resizers node
                expect(drawingSelection.find('.borders > div[data-pos]').length).to.equal(4); // 4 'data-pos' nodes as borders

                // the 'data-pos' nodes are children of the resizers node
                expect(drawingSelection.find('.resizers > div[data-pos]').length).to.equal(8); // 8 'data-pos' nodes as resizers
            });

        });

    });

});
