/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/textframework/model/attributeoperationmixin',
    'io.ox/office/textframework/utils/position'
], function (AppHelper, Utils, LineHeight, AttributeOperationMixin, Position) {

    'use strict';

    // class AttributeOperationMixin ==========================================

    describe('TextFramework class AttributeOperationMixin', function () {

        var LINEHEIGHT__NORMAL       = LineHeight.NORMAL,
            LINEHEIGHT__SINGLE       = LineHeight.SINGLE,
            LINEHEIGHT___115         = LineHeight._115,
            LINEHEIGHT__ONE_HALF     = LineHeight.ONE_HALF,
            LINEHEIGHT__DOUBLE       = LineHeight.DOUBLE,

            PARAGRAPHSPACING__NONE   = 0,
            PARAGRAPHSPACING__NORMAL = 1,
            PARAGRAPHSPACING__WIDE   = 2,

            defaultMarginBottom;

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes',
                attrs: {
                    document:  { defaultTabStop: 1270, zoom: { value: 100 } },
                    page:      { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 },
                    character: { fontName: 'Arial', fontSize: 11, language: 'en-US', languageEa: 'en-US', languageBidi: 'ar-SA' },
                    paragraph: { lineHeight: { type: 'percent', value: 119 }, marginBottom: 352 }
                }
            },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading1', styleName: 'heading 1',
                attrs: {
                    character: { bold: true, fontName: 'Times New Roman', fontSize: 14, color: { transformations: [{ type: 'shade', value: 74902 }], type: 'scheme', value: 'accent1' } },
                    paragraph: { marginTop: 846, outlineLevel: 0, nextStyleId: 'Normal' }
                },
                parent: 'Normal',
                uiPriority: 9
            },
            { name: 'insertParagraph', start: [0] },
            { name: 'insertText', text: 'Hello World', start: [0, 0] },

            { name: 'splitParagraph', start: [0, 11] },
            { name: 'insertText', start: [1, 0], text: [
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet.'
            ].join('') },

            { name: 'splitParagraph', start: [1, 240] },
            { name: 'insertText', start: [2, 0], text: [
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet.'
            ].join('') },

            { name: 'splitParagraph', start: [2, 240] },
            { name: 'insertText', start: [3, 0], text: [
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet.'
            ].join('') },

            { name: 'splitParagraph', start: [3, 240] }
        ];

        var model, firstParagraph, secondParagraph, thirdParagraph, fourthParagraph;
        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            var node = model.getNode();
            firstParagraph  = Position.getParagraphElement(node, [0]);
            secondParagraph = Position.getParagraphElement(node, [1]);
            thirdParagraph  = Position.getParagraphElement(node, [2]);
            fourthParagraph = Position.getParagraphElement(node, [3]);

            defaultMarginBottom = model.getDefaultMarginBottom();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(AttributeOperationMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "setAttributes"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('setAttributes');
            });
            it('should set the specified paragraph style', function () {

                function compareEMWithPT(elementEM, comparePT) {
                    var compare = Utils.convertToEMValue(comparePT, 'pt');
                    expect(parseFloat(elementEM, 10)).to.be.closeTo(compare, 2);
                }

                function expectParagraphSpacing(position, exp) {
                    model.getSelection().setTextSelection(position);
                    var attributes = model.getAttributes('paragraph').paragraph;
                    expect(model.getParagraphSpacing(attributes)).to.equal(exp);
                }

                expect($(firstParagraph)).to.have.property('hasClass');
                expect($(firstParagraph).hasClass('p')).to.equal(true);
                compareEMWithPT($(firstParagraph).children().first().css('font-size'), 11);

                // setting styleId 'Heading1' to one and only paragraph
                model.getSelection().setTextSelection([0, 2]); // not necessary
                model.setAttributes('paragraph', { styleId: 'Heading1' }, { clear: true });

                compareEMWithPT($(firstParagraph).children().first().css('font-size'), 14);

                /**
                 *  'Paragraph and LineHeight spacing' test - User Story #99693312:
                 *  https://www.pivotaltracker.com/projects/534975/stories/99693312
                 */
                // setting both the paragraph spacing and the line height, covering the whole selection ...
                //
                // ... that should be equal to the following operations ...
                // { name: 'setAttributes', start: [1], attrs: { paragraph: { marginBottom: 500 }} },
                // { name: 'setAttributes', start: [2], attrs: { paragraph: { marginBottom: 500 }} },
                // { name: 'setAttributes', start: [3], attrs: { paragraph: { marginBottom: 500 }} }

                //
                // 1st COMBINED ROUND (lineHeight + marginBottom)
                //
                model.getSelection().setTextSelection([1, 0], [3, 240]);
                model.setAttribute('paragraph', 'lineHeight', LINEHEIGHT___115);
                model.setAttribute('paragraph', 'marginBottom', (PARAGRAPHSPACING__WIDE * defaultMarginBottom));

                expect(parseInt($(secondParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(139);
                expect(parseInt($(thirdParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(139);
                expect(parseInt($(fourthParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(139);

                expectParagraphSpacing([1, 0], PARAGRAPHSPACING__WIDE);
                expectParagraphSpacing([2, 0], PARAGRAPHSPACING__WIDE);
                expectParagraphSpacing([3, 0], PARAGRAPHSPACING__WIDE);

                //
                // 2nd COMBINED ROUND (lineHeight + marginBottom)
                //
                model.getSelection().setTextSelection([1, 0], [2, 240]);
                model.setAttribute('paragraph', 'lineHeight', LINEHEIGHT__ONE_HALF);
                model.setAttribute('paragraph', 'marginBottom', (PARAGRAPHSPACING__NONE * defaultMarginBottom));

                expect(parseInt($(secondParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(180);
                expect(parseInt($(thirdParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(180);
                expect(parseInt($(fourthParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(139);

                expectParagraphSpacing([1, 0], PARAGRAPHSPACING__NONE);
                expectParagraphSpacing([2, 0], PARAGRAPHSPACING__NONE);
                expectParagraphSpacing([3, 0], PARAGRAPHSPACING__WIDE);

                //
                // 3rd COMBINED ROUND (lineHeight + marginBottom)
                //
                model.getSelection().setTextSelection([2, 0], [3, 240]);
                model.setAttribute('paragraph', 'lineHeight', LINEHEIGHT__SINGLE);
                model.setAttribute('paragraph', 'marginBottom', (PARAGRAPHSPACING__NORMAL * defaultMarginBottom));

                expect(parseInt($(secondParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(180);
                // fontCollection.getNormalLineHeight() in characterStyles.updateElementLineHeight() causes 119 or 126 to be possible (1pt difference)
                expect(parseInt($(thirdParagraph).find('span')[0].style.lineHeight, 10)).to.be.oneOf([119, 126]);
                expect(parseInt($(fourthParagraph).find('span')[0].style.lineHeight, 10)).to.be.oneOf([119, 126]);

                expectParagraphSpacing([1, 0], PARAGRAPHSPACING__NONE);
                expectParagraphSpacing([2, 0], PARAGRAPHSPACING__NORMAL);
                expectParagraphSpacing([3, 0], PARAGRAPHSPACING__NORMAL);

                //
                // 4th COMBINED ROUND (lineHeight + marginBottom)
                //
                model.getSelection().setTextSelection([1, 5]);
                model.setAttribute('paragraph', 'lineHeight', LINEHEIGHT__DOUBLE);
                model.setAttribute('paragraph', 'marginBottom', (PARAGRAPHSPACING__WIDE * defaultMarginBottom));

                // fontCollection.getNormalLineHeight() in characterStyles.updateElementLineHeight() causes 119 or 126 (235 or 242) to be possible (1pt difference)
                expect(parseInt($(secondParagraph).find('span')[0].style.lineHeight, 10)).to.be.oneOf([235, 242]);
                expect(parseInt($(thirdParagraph).find('span')[0].style.lineHeight, 10)).to.be.oneOf([119, 126]);
                expect(parseInt($(fourthParagraph).find('span')[0].style.lineHeight, 10)).to.be.oneOf([119, 126]);

                expectParagraphSpacing([1, 0], PARAGRAPHSPACING__WIDE);
                expectParagraphSpacing([2, 0], PARAGRAPHSPACING__NORMAL);
                expectParagraphSpacing([3, 0], PARAGRAPHSPACING__NORMAL);

                //
                // 5th COMBINED ROUND (lineHeight + marginBottom)
                //
                model.getSelection().setTextSelection([1, 5], [2, 12]);
                model.setAttribute('paragraph', 'lineHeight', LINEHEIGHT__NORMAL);
                model.setAttribute('paragraph', 'marginBottom', (PARAGRAPHSPACING__NONE * defaultMarginBottom));

                // fontCollection.getNormalLineHeight() in characterStyles.updateElementLineHeight() causes 119 or 126 to be possible (1pt difference)
                expect(parseInt($(secondParagraph).find('span')[0].style.lineHeight, 10)).to.be.oneOf([119, 126]);
                expect(parseInt($(thirdParagraph).find('span')[0].style.lineHeight, 10)).to.be.oneOf([119, 126]);
                expect(parseInt($(fourthParagraph).find('span')[0].style.lineHeight, 10)).to.be.oneOf([119, 126]);

                expectParagraphSpacing([1, 0], PARAGRAPHSPACING__NONE);
                expectParagraphSpacing([2, 0], PARAGRAPHSPACING__NONE);
                expectParagraphSpacing([3, 0], PARAGRAPHSPACING__NORMAL);
                //
                // END OF 'Paragraph and LineHeight spacing' test
            });

            it('should avoid setting character attributes to trailing whitespace(s)', function () {
                expect($(firstParagraph).find('span')[0].style.textDecoration).to.equal('none');

                model.getSelection().setTextSelection([0, 0], [0, 6]);
                model.setAttribute('character', 'underline', true);

                expect($(firstParagraph).find('span')[0].style.textDecoration).to.equal('underline');
                expect($(firstParagraph).find('span').first().text().length).to.be.equal(5);
                expect($(firstParagraph).find('span').first().text()).to.equal('Hello');
            });

            it('should expand character attributes to whole surrounding word', function () {
                expect($(firstParagraph).find('span')[1].style.textDecoration).to.equal('none');

                // first setting cursor at the begining of word - should not expand
                model.getSelection().setTextSelection([0, 6]);
                model.setAttribute('character', 'underline', true);

                expect($(firstParagraph).find('span')[1].style.textDecoration).to.equal('none');

                // now setting cursor inside word - should expand
                model.getSelection().setTextSelection([0, 7]);
                model.setAttribute('character', 'underline', true);

                expect($(firstParagraph).find('span').last()[0].style.textDecoration).to.equal('underline');
                expect($(firstParagraph).find('span').last().text().length).to.be.equal(5);
                expect($(firstParagraph).find('span').last().text()).to.equal('World');
            });

            it('should set small caps to a specific word', function () {

                var allParagraphChildren = $(secondParagraph).children();
                var secondSpan = null;

                expect(allParagraphChildren.length).to.be.equal(1); // only one span in the second paragraph
                // Text in second paragraph: 'lorem ipsum dolor ...'
                model.getSelection().setTextSelection([1, 6], [1, 11]); // selecting 'ipsum'
                model.setAttribute('character', 'caps', 'small'); // setting small caps to the word ipsum

                allParagraphChildren = $(secondParagraph).children();  // refresh after applying the operation
                expect(allParagraphChildren.length).to.be.equal(3);

                secondSpan = $(allParagraphChildren[1]);

                expect(secondSpan.text()).to.equal('ipsum');
                // checking the data object at the span node
                expect(secondSpan.data().attributes.character.caps).to.equal('small');
                // checking if the correct class has been set
                expect(secondSpan.hasClass('lowercase')).to.equal(true);
            });

        });
    });

    // ========================================================================
});
