/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/textframework/components/field/fieldmanager',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/editframework/utils/attributeutils'
], function (AppHelper, FieldManager, Position, DOM, AttributeUtils) {

    'use strict';

    // class FieldManager ===========================================================

    describe('Text class FieldManager', function () {

        it('should exist', function () {
            expect(FieldManager).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var FILENAMECONST = 'unnamed (1).docx';

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { defaultTabStop: 1270, zoom: { value: 100 } }, page: { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 } } },

            { name: 'insertParagraph', start: [0] },
            { name: 'insertRange', start: [0, 0], type: 'field', position: 'start' },
            { name: 'insertComplexField', start: [0, 1], instruction: 'FILENAME' },
            { name: 'insertText', start: [0, 2], text: FILENAMECONST },
            { name: 'insertRange', start: [0, 18], type: 'field', position: 'end' },
            // checkbox
            { name: 'insertRange', start: [0, 19], type: 'field', position: 'start' },
            { name: 'insertRange', start: [0, 20], type: 'field', position: 'end' },
            { name: 'setAttributes', start: [0, 19], attrs: { character: { field: { formFieldType: 'checkBox', checked: true, enabled: true } } } },
            { name: 'insertRange', start: [0, 21], type: 'field', position: 'start' },
            { name: 'insertRange', start: [0, 22], type: 'field', position: 'end' },
            { name: 'setAttributes', start: [0, 21], attrs: { character: { field: { formFieldType: 'checkBox', checked: false, enabled: true } } } },

            { name: 'insertParagraph', start: [1], attrs: { paragraph: { pageBreakBefore: true } } },
            { name: 'insertParagraph', start: [2], attrs: { paragraph: { pageBreakBefore: true } } }
        ];

        // the ODF operations to be applied by the document model
        var ODF_OPERATIONS = [
            { name: 'insertParagraph', start: [0] },
            { name: 'insertField', start: [0, 0], type: 'date', representation: '1/12/2016', attrs: { character: { field: { dateFormat: 'M/D/YYYY' } } } },
            { name: 'updateField', start: [0, 0], type: 'date', representation: '1/12/2016', attrs: { character: { field: { dateFormat: 'M/D/YYYY', 'text:fixed': 'true' } } } }
        ];

        var model, selection, fieldManager;
        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            app.getFullFileName = function () { return FILENAMECONST; };
            model = app.getModel();
            selection = model.getSelection();
            fieldManager = model.getFieldManager();
        });

        var odfModel, odfSelection, odfFieldManager;
        AppHelper.createTextApp('odf', ODF_OPERATIONS).done(function (app) {
            app.getFullFileName = function () { return FILENAMECONST; };
            odfModel = app.getModel();
            odfSelection = odfModel.getSelection();
            odfFieldManager = odfModel.getFieldManager();
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a FieldManager class instance', function () {
                expect(fieldManager).to.be.an['instanceof'](FieldManager);
                expect(odfFieldManager).to.be.an['instanceof'](FieldManager);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "fieldsAreInDocument"', function () {
            it('should exist', function () {
                expect(fieldManager).to.respondTo('fieldsAreInDocument');
            });
            it('should return that there are fields in document', function () {
                expect(fieldManager.fieldsAreInDocument()).to.equal(true);
            });

            it('should exist odf', function () {
                expect(odfFieldManager).to.respondTo('fieldsAreInDocument');
            });
            it('should return that there are odf fields in document', function () {
                expect(odfFieldManager.fieldsAreInDocument()).to.equal(true);
            });
        });

        // Simple fields tests
        describe('method "getSimpleField"', function () {
            it('should exist', function () {
                expect(odfFieldManager).to.respondTo('getSimpleField');
            });
            it('should return simple odf field node', function () {
                expect(odfFieldManager.getSimpleField('sf0')).to.be.an['instanceof']($)
                    .and.to.have.className('field');
            });
        });

        describe('method "isAutomaticDateField" for odf', function () {
            it('should return simple odf field node', function () {
                expect(odfFieldManager.isFixedSimpleField(odfFieldManager.getSimpleField('sf0'), true)).to.equal(true);
            });
        });

        describe('testing changind date field to automatic from fixed, for odf', function () {
            before(function () {
                odfSelection.setTextSelection(odfSelection.getFirstDocumentPosition());
                odfFieldManager.setDateFieldToAutoUpdate(null, true);
            });
            it('should return that odf field node is not fixed anymore', function () {
                expect(odfFieldManager.isFixedSimpleField(odfFieldManager.getSimpleField('sf0'), true)).to.equal(false);
            });
        });

        // Complex field tests
        describe('method "isComplexFieldId"', function () {
            it('should exist', function () {
                expect(fieldManager).to.respondTo('isComplexFieldId');
            });
            it('should return that asked id is id of complex field', function () {
                expect(fieldManager.isComplexFieldId('fld0')).to.equal(true);
            });
        });

        describe('method "getComplexField"', function () {
            it('should exist', function () {
                expect(fieldManager).to.respondTo('getComplexField');
            });
            it('should return complex field node', function () {
                expect(fieldManager.getComplexField('fld0')).to.be.an['instanceof']($)
                    .and.to.have.className('complexfield');
            });
        });

        describe('method "isHighlightState"', function () {
            before(function () {
                selection.setTextSelection(selection.getFirstDocumentPosition());
            });
            it('should exist', function () {
                expect(fieldManager).to.respondTo('isHighlightState');
            });
            it('should return that complex field is highlighted', function () {
                expect(fieldManager.isHighlightState()).to.equal(true);
            });
        });

        describe('method "getSelectedFieldFormat"', function () {
            before(function () {
                selection.setTextSelection(selection.getFirstDocumentPosition());
            });
            it('should exist', function () {
                expect(fieldManager).to.respondTo('getSelectedFieldFormat');
            });
            it('should return default complex field format of Filename field', function () {
                expect(fieldManager.getSelectedFieldFormat()).to.equal('default');
            });
        });

        describe('method "updateFieldFormatting"', function () {
            before(function () {
                selection.setTextSelection(selection.getFirstDocumentPosition());
                fieldManager.updateFieldFormatting('Upper');
                selection.setTextSelection(selection.getFirstDocumentPosition());
            });
            it('should exist', function () {
                expect(fieldManager).to.respondTo('updateFieldFormatting');
            });
            it('should return new, changed complex field format', function () {
                expect(fieldManager.getSelectedFieldFormat()).to.equal('Upper');
            });
        });

        describe('Checking the rendering of checkboxes inside rangestart marker', function () {
            it('should render checkbox character with state "checked"', function () {
                var rangemarkerStartNodePoint = Position.getDOMPosition(model.getNode(), [0, 19], true);

                expect(rangemarkerStartNodePoint).not.to.equal(null);
                expect(rangemarkerStartNodePoint.node).not.to.equal(null);
                expect(DOM.isRangeStartWithCheckboxAttr(rangemarkerStartNodePoint.node)).to.equal(true);
                expect(DOM.isRangeStartWithCheckboxEnabledState(rangemarkerStartNodePoint.node)).to.equal(true);
            });

            it('should render checkbox character with state "unchecked"', function () {
                var rangemarkerStartNodePoint = Position.getDOMPosition(model.getNode(), [0, 21], true);

                expect(rangemarkerStartNodePoint).not.to.equal(null);
                expect(rangemarkerStartNodePoint.node).not.to.equal(null);
                expect(DOM.isRangeStartWithCheckboxAttr(rangemarkerStartNodePoint.node)).to.equal(true);
                expect(DOM.isRangeStartWithCheckboxEnabledState(rangemarkerStartNodePoint.node)).to.equal(false);
            });
        });

        describe('method "dispatchInsertField" for pageNum field on first page', function () {
            before(function () {
                selection.setTextSelection(selection.getFirstDocumentPosition());
                fieldManager.dispatchInsertField('pagenumber', 'default');
                selection.setTextSelection(selection.getFirstDocumentPosition());
            });
            it('should exist', function () {
                expect(fieldManager).to.respondTo('dispatchInsertField');
            });
            it('should return newly inserted complex field with id fld3', function () {
                expect(fieldManager.getSelectedFieldFormat()).to.equal('default');
                expect(fieldManager.getSelectedFieldNode()).to.be.an['instanceof']($)
                    .and.to.have.className('complexfield');
                expect(fieldManager.getSelectedFieldNode().attr('data-field-id')).to.equal('fld3');
                expect(fieldManager.getSelectedFieldNode().next().text()).to.equal('1');
            });
        });

        describe('method "dispatchInsertField" for pageNum field on last page', function () {
            before(function () {
                var lastPosCached = selection.getLastDocumentPosition();
                selection.setTextSelection(lastPosCached);
                fieldManager.dispatchInsertField('pagenumber', 'ALPHABETIC');
                selection.setTextSelection(lastPosCached);
            });
            it('should return newly inserted complex field with id fld4 and ALPHABETIC format', function () {
                expect(fieldManager.getSelectedFieldFormat()).to.equal('ALPHABETIC');
                expect(fieldManager.getSelectedFieldNode()).to.be.an['instanceof']($)
                .and.to.have.className('complexfield');
                expect(fieldManager.getSelectedFieldNode().attr('data-field-id')).to.equal('fld4');
                expect(fieldManager.getSelectedFieldNode().next().text()).to.equal('C'); // page 3 -> C in alphabetic upper format
            });
        });
    });

    describe('Text class FieldManager: update TOC.', function () {

        // private helpers ----------------------------------------------------

        // update toc field instructions
        var instruction1 = 'TOC \\o "1-3" \\h';
        var instruction2 = 'TOC \\o "5-7" \\n 6-7 \\p "_"';

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { defaultTabStop: 1270, zoom: { value: 100 } }, page: { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 } } },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading1', styleName: 'heading·1', attrs: { character: { bold: true, fontName: 'Times·New·Roman', fontSize: 14, color: { transformations: [{ type: 'shade', value: 74902 }], type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 846, outlineLevel: 0, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading2', styleName: 'heading·2', attrs: { character: { bold: true, fontName: 'Times·New·Roman', fontSize: 13, color: { type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 352, outlineLevel: 1, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },

            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading4', styleName: 'heading·4', attrs: { character: { bold: true, italic: true, fontName: 'Times·New·Roman', color: { type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 352, outlineLevel: 3, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading5', styleName: 'heading·5', attrs: { character: { bold: true, italic: true, fontName: 'Times·New·Roman', color: { type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 352, outlineLevel: 4, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading6', styleName: 'heading·6', attrs: { character: { bold: true, italic: true, fontName: 'Times·New·Roman', color: { type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 352, outlineLevel: 5, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading7', styleName: 'heading·7', attrs: { character: { bold: true, italic: true, fontName: 'Times·New·Roman', color: { type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 352, outlineLevel: 6, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },

            // TOC field
            { name: 'insertParagraph', start: [0], attrs: { paragraph: { tabStops: [{ value: 'right', pos: 15984, fillChar: 'dot' }] } } },
            { name: 'insertRange', start: [0, 0], type: 'field', position: 'start' },
            { name: 'insertComplexField', start: [0, 1], instruction: instruction1 },
            { name: 'insertText', start: [0, 2], text: 'Test text 1' },
            { name: 'insertRange', start: [0, 13], type: 'field', position: 'end' },

            // content for toc
            { name: 'insertParagraph', start: [1], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading1' } },
            { name: 'insertText', start: [1, 0], text: 'Aaa' },
            { name: 'insertParagraph', start: [2], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading4' } },
            { name: 'insertText', start: [2, 0], text: 'Bbb' },
            { name: 'insertParagraph', start: [3], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading2' } },
            { name: 'insertText', start: [3, 0], text: 'Ccc' },

            // second TOC field
            { name: 'insertParagraph', start: [4] },
            { name: 'insertParagraph', start: [5], attrs: { paragraph: { tabStops: [{ value: 'right', pos: 15984, fillChar: 'dot' }] } } },
            { name: 'insertRange', start: [5, 0], type: 'field', position: 'start' },
            { name: 'insertComplexField', start: [5, 1], instruction: instruction2 },
            { name: 'insertText', start: [5, 2], text: 'Test text 1' },
            { name: 'insertRange', start: [5, 13], type: 'field', position: 'end' },

            // content for second toc
            { name: 'insertParagraph', start: [6], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading5' } },
            { name: 'insertText', start: [6, 0], text: 'Second Aaa' },
            { name: 'insertParagraph', start: [7], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading6' } },
            { name: 'insertText', start: [7, 0], text: 'Second Bbb' },
            { name: 'insertParagraph', start: [8], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading7' } },
            { name: 'insertText', start: [8, 0], text: 'Second Ccc' },

            { name: 'insertParagraph', start: [9] }
        ];

        var model, selection, fieldManager;
        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            selection = model.getSelection();
            fieldManager = model.getFieldManager();
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a FieldManager class instance', function () {
                expect(fieldManager).to.be.an['instanceof'](FieldManager);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "updateTableOfContents" for first TOC in document', function () {
            var updateTOCPromise = null;
            before(function (done) {
                selection.setTextSelection([0, 2]);
                var node = fieldManager.getSelectedFieldNode();
                updateTOCPromise = fieldManager.dispatchUpdateTOC(node, instruction1, { testMode: true });
                updateTOCPromise.always(function () {
                    done();
                });
            });
            it('should update first TOC correctly', function () {
                var par1 = Position.getParagraphElement(model.getNode(), [0]);
                var par2 = Position.getParagraphElement(model.getNode(), [1]);
                var explAttrs = AttributeUtils.getExplicitAttributes($(par2).children('span').first());
                expect($(par1).text()).to.equal('Aaa2');
                expect($(par2).text()).not.to.equal('Bbb3');
                expect($(par2).text()).to.equal('Ccc4');
                expect(explAttrs.character).to.be.an('object');
                expect(explAttrs.character.anchor).to.be.a('String')
                    .and.not.to.be.empty;
            });
        });

        describe('method "updateTableOfContents" for second TOC in document', function () {
            var updateTOCPromise = null;
            before(function (done) {
                selection.setTextSelection([7, 2]);
                var node = fieldManager.getSelectedFieldNode();
                updateTOCPromise = fieldManager.dispatchUpdateTOC(node, instruction2, { testMode: true });
                updateTOCPromise.always(function () {
                    done();
                });
            });
            it('should update second TOC correctly', function () {
                var par1 = Position.getParagraphElement(model.getNode(), [7]);
                var par2 = Position.getParagraphElement(model.getNode(), [8]);
                var par3 = Position.getParagraphElement(model.getNode(), [9]);
                var explAttrs = AttributeUtils.getExplicitAttributes($(par2).children('span').first());
                expect($(par1).text()).to.equal('Second Aaa_5');
                expect($(par2).text()).to.equal('Second Bbb');
                expect($(par3).text()).to.equal('Second Ccc');
                expect(explAttrs.character).to.be.an('object');
                expect(explAttrs.character.anchor).to.be.empty;
            });
            it('should update second TOC correctly after heading text deletion', function () {
                selection.setTextSelection([13, 8], [13, 10]);
                model.deleteSelected();
                selection.setTextSelection([7, 2]);
                fieldManager.updateHighlightedField();
                var par = Position.getParagraphElement(model.getNode(), [9]);
                expect($(par).text()).to.equal('Second C');
            });
        });
    });

    describe('Text class FieldManager: insert TOC.', function () {
        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { defaultTabStop: 1270, zoom: { value: 100 } }, page: { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 } } },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading1', styleName: 'heading·1', attrs: { character: { bold: true, fontName: 'Times·New·Roman', fontSize: 14, color: { transformations: [{ type: 'shade', value: 74902 }], type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 846, outlineLevel: 0, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading2', styleName: 'heading·2', attrs: { character: { bold: true, fontName: 'Times·New·Roman', fontSize: 13, color: { type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 352, outlineLevel: 1, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading3', styleName: 'heading·3', attrs: { character: { bold: true, fontName: 'Times·New·Roman', color: { type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 352, outlineLevel: 2, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading4', styleName: 'heading·4', attrs: { character: { bold: true, italic: true, fontName: 'Times·New·Roman', color: { type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 352, outlineLevel: 3, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading5', styleName: 'heading·5', attrs: { character: { bold: true, italic: true, fontName: 'Times·New·Roman', color: { type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 352, outlineLevel: 4, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading6', styleName: 'heading·6', attrs: { character: { bold: true, italic: true, fontName: 'Times·New·Roman', color: { type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 352, outlineLevel: 5, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading7', styleName: 'heading·7', attrs: { character: { bold: true, italic: true, fontName: 'Times·New·Roman', color: { type: 'scheme', value: 'accent1' } }, paragraph: { marginTop: 352, outlineLevel: 6, nextStyleId: 'Normal' } }, parent: 'Normal', uiPriority: 9 },
            { name: 'insertParagraph', start: [0] },
            { name: 'insertParagraph', start: [1] },

            // content for toc
            { name: 'insertParagraph', start: [2], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading1' } },
            { name: 'insertText', start: [2, 0], text: 'Aaa' },
            { name: 'insertParagraph', start: [3], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading4' } },
            { name: 'insertText', start: [3, 0], text: 'Bbb' },
            { name: 'insertParagraph', start: [4], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading2' } },
            { name: 'insertText', start: [4, 0], text: 'Ccc' },

            { name: 'insertParagraph', start: [5] },

            // content for second toc
            { name: 'insertParagraph', start: [6], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading5' } },
            { name: 'insertText', start: [6, 0], text: 'Second Aaa' },
            { name: 'insertParagraph', start: [7], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading6' } },
            { name: 'insertText', start: [7, 0], text: 'Second Bbb' },
            { name: 'insertParagraph', start: [8], attrs: { paragraph: { pageBreakBefore: true }, styleId: 'Heading7' } },
            { name: 'insertText', start: [8, 0], text: 'Second Ccc' },

            { name: 'insertParagraph', start: [9] }
        ];

        var model, selection, fieldManager;
        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            selection = model.getSelection();
            fieldManager = model.getFieldManager();
        });

        describe('method "insertTableOfContents" with page numbers and dots in tab stops', function () {
            var insertTOCPromise = null;
            before(function (done) {
                selection.setTextSelection(selection.getLastDocumentPosition());
                insertTOCPromise = fieldManager.dispatchInsertToc(1, { testMode: true });
                insertTOCPromise.always(function () {
                    done();
                });
            });

            it('should exist', function () {
                expect(fieldManager).to.respondTo('dispatchInsertToc');
            });

            it('should insert new Table of Contents with page numbers and dots in tab stops', function () {
                var toc1FieldNode = fieldManager.getComplexField('fld0');
                var firstTextSpanInField = toc1FieldNode.next();
                var secondTextSpanInField = firstTextSpanInField.next().next(); // tabStop node is in between
                var paraAttrs = model.getParagraphStyles().getElementAttributes(toc1FieldNode.parent());

                expect(fieldManager.getCountOfAllFields()).to.be.equal(1);
                expect(toc1FieldNode.length).to.equal(1);
                expect(DOM.getComplexFieldInstruction(toc1FieldNode)).to.be.a('String')
                    .and.to.equal('TOC \\o "1-3" \\h \\u');
                expect(firstTextSpanInField.text()).to.equal('Aaa');
                expect(paraAttrs).to.be.an('Object');
                expect(paraAttrs.paragraph).to.be.an('Object');
                expect(paraAttrs.paragraph.tabStops).to.be.an('Array');
                expect(paraAttrs.paragraph.tabStops[0].fillChar).to.equal('dot');
                expect(secondTextSpanInField.text()).to.equal('2'); // Heading1 is on second page (one pageBreakBefore property before it)
            });
        });

        describe('method "insertTableOfContents" with page numbers and tab stops with fill type "none"', function () {
            var insertTOCPromise = null;

            before(function (done) {
                selection.setTextSelection(selection.getLastDocumentPosition());
                insertTOCPromise = fieldManager.dispatchInsertToc(2, { testMode: true });
                insertTOCPromise.always(function () {
                    done();
                });
            });

            it('should insert new Table of Contents with page numbers and tab stops with fill type "none"', function () {
                var toc2FieldNode = fieldManager.getComplexField('fld1');
                var firstTextSpanInField = toc2FieldNode.next();
                var secondTextSpanInField = firstTextSpanInField.next().next(); // tabStop node is in between
                var paraAttrs = model.getParagraphStyles().getElementAttributes(toc2FieldNode.parent());

                expect(fieldManager.getCountOfAllFields()).to.be.equal(2);
                expect(toc2FieldNode.length).to.equal(1);
                expect(DOM.getComplexFieldInstruction(toc2FieldNode)).to.be.a('String')
                    .and.to.equal('TOC \\o "1-3" \\h \\u');
                expect(firstTextSpanInField.text()).to.equal('Aaa');
                expect(paraAttrs).to.be.an('Object');
                expect(paraAttrs.paragraph).to.be.an('Object');
                expect(paraAttrs.paragraph.tabStops).to.be.an('Array');
                expect(paraAttrs.paragraph.tabStops[0].fillChar).to.equal('none');
                expect(secondTextSpanInField.text()).to.equal('2'); // Heading1 is on second page (one pageBreakBefore property before it)
            });
        });

        describe('method "insertTableOfContents" only with headings', function () {
            var insertTOCPromise = null;

            before(function (done) {
                selection.setTextSelection(selection.getLastDocumentPosition());
                insertTOCPromise = fieldManager.dispatchInsertToc(3, { testMode: true });
                insertTOCPromise.always(function () {
                    done();
                });
            });

            it('should insert new Table of Contents only with headings', function () {
                var toc3FieldNode = fieldManager.getComplexField('fld2');
                var firstTextSpanInField = toc3FieldNode.next();
                var secondTextSpanInField = firstTextSpanInField.next().next(); // tabStop node is in between

                expect(fieldManager.getCountOfAllFields()).to.be.equal(3);
                expect(toc3FieldNode.length).to.equal(1);
                expect(DOM.getComplexFieldInstruction(toc3FieldNode)).to.be.a('String')
                    .and.to.equal('TOC \\o "1-3" \\n \\h \\u');
                expect(firstTextSpanInField.text()).to.equal('Aaa');
                expect(secondTextSpanInField.text()).to.equal(''); // Heading1 is on second page (one pageBreakBefore property before it)
            });
        });

        describe('method "insertTableOfContents" without headings', function () {
            var insertTOCPromise = null;

            before(function (done) {
                selection.setTextSelection(selection.getFirstDocumentPosition(), selection.getLastDocumentPosition());
                var deletePromise = model.deleteSelected();
                deletePromise.always(function () {
                    selection.setTextSelection(selection.getLastDocumentPosition());
                    insertTOCPromise = fieldManager.dispatchInsertToc(3, { testMode: true });
                    insertTOCPromise.always(function () {
                        done();
                    });
                });
            });

            it('should insert new Table of Contents without headings', function () {
                var toc3FieldNode = fieldManager.getComplexField('fld3');
                var firstTextSpanInField = toc3FieldNode.next();

                expect(fieldManager.getCountOfAllFields()).to.be.equal(1);
                expect(toc3FieldNode.length).to.equal(1);
                expect(DOM.getComplexFieldInstruction(toc3FieldNode)).to.be.a('String')
                    .and.to.equal('TOC \\o "1-3" \\n \\h \\u');
                expect(firstTextSpanInField.text()).to.equal('No table of contents entries in the document.');
            });

        });
    });

    // ========================================================================
});
