/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/range3d'
], function (SheetHelper, Range, Range3D) {

    'use strict';

    // class Range ============================================================

    describe('Spreadsheet class Range3D', function () {

        it('should exist', function () {
            expect(Range3D).to.be.a('function');
        });
        it('should be subclassed from Range', function () {
            expect(Range3D).to.have.property('__super__', Range.prototype);
        });

        // private helpers ----------------------------------------------------

        var i = SheetHelper.i;
        var a = SheetHelper.a;
        var r = SheetHelper.r;
        var r3d = SheetHelper.r3d;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            var a1 = a('A2'), a2 = a('C4');
            it('should create a range address', function () {
                var r1 = new Range3D(6, 8, a1, a2);
                expect(r1).to.have.property('sheet1', 6);
                expect(r1).to.have.property('sheet2', 8);
                expect(r1).to.have.property('start', a1);
                expect(r1).to.have.property('end', a2);
            });
            it('should create a range address from a single address', function () {
                var r1 = new Range3D(6, 8, a1);
                expect(r1).to.have.property('sheet1', 6);
                expect(r1).to.have.property('sheet2', 8);
                expect(r1).to.have.property('start', a1);
                expect(r1).to.have.property('end');
                expect(r1.end).to.deep.equal(a1);
                expect(r1.end).to.not.equal(a1);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(Range3D).itself.to.respondTo('create');
            });
            it('should return the adjusted range address', function () {
                var r1 = new Range3D(6, 8, a('A2'), a('C4'));
                expect(Range3D.create(0, 1, 2, 3, 6, 8)).to.deep.equal(r1);
                expect(Range3D.create(2, 1, 0, 3, 8, 6)).to.deep.equal(r1);
                expect(Range3D.create(0, 3, 2, 1, 6, 8)).to.deep.equal(r1);
                expect(Range3D.create(2, 3, 0, 1, 8, 6)).to.deep.equal(r1);
            });
            it('should accept single sheet index', function () {
                var r1 = new Range3D(6, 6, a('A2'), a('C4'));
                expect(Range3D.create(0, 1, 2, 3, 6) + '!').to.deep.equal(r1 + '!');
            });
        });

        describe('method "createFromAddress"', function () {
            it('should exist', function () {
                expect(Range3D).itself.to.respondTo('createFromAddress');
            });
            var a1 = a('A2'), r1 = new Range3D(6, 8, a1);
            it('should return a range', function () {
                expect(Range3D.createFromAddress(a1, 6, 8)).to.be.an.instanceof(Range3D);
            });
            it('should return a range with adjusted sheet indexes', function () {
                expect(Range3D.createFromAddress(a1, 6, 8)).to.deep.equal(r1);
                expect(Range3D.createFromAddress(a1, 8, 6)).to.deep.equal(r1);
            });
            it('should accept single sheet index', function () {
                var r2 = Range3D.createFromAddress(a1, 6);
                expect(r2).to.have.property('sheet1', 6);
                expect(r2).to.have.property('sheet2', 6);
            });
            it('should clone the address', function () {
                var r2 = Range3D.createFromAddress(a1, 6, 8);
                expect(r2.start).to.deep.equal(a1);
                expect(r2.start).to.not.equal(a1);
                expect(r2.end).to.deep.equal(a1);
                expect(r2.end).to.not.equal(a1);
            });
        });

        describe('method "createFromAddresses"', function () {
            it('should exist', function () {
                expect(Range3D).itself.to.respondTo('createFromAddresses');
            });
            var a1 = a('A2'), a2 = a('C4'), r1 = new Range3D(6, 8, a1, a2);
            it('should return a range', function () {
                expect(Range3D.createFromAddresses(a1, a2, 6, 8)).to.be.an.instanceof(Range3D);
            });
            it('should return a range with adjusted indexes', function () {
                expect(Range3D.createFromAddresses(a1, a2, 6, 8)).to.deep.equal(r1);
                expect(Range3D.createFromAddresses(a2, a1, 8, 6)).to.deep.equal(r1);
            });
            it('should accept single sheet index', function () {
                var r2 = Range3D.createFromAddresses(a1, a2, 6);
                expect(r2).to.have.property('sheet1', 6);
                expect(r2).to.have.property('sheet2', 6);
            });
            it('should clone the addresses', function () {
                var r2 = Range3D.createFromAddresses(a1, a2, 6, 8);
                expect(r2.start).to.deep.equal(a1);
                expect(r2.start).to.not.equal(a1);
                expect(r2.end).to.deep.equal(a2);
                expect(r2.end).to.not.equal(a2);
            });
        });

        describe('method "createFromRange"', function () {
            it('should exist', function () {
                expect(Range3D).itself.to.respondTo('createFromRange');
            });
            var r1 = r('A2:C4'), r2 = new Range3D(6, 8, r1.start, r1.end);
            it('should return a range', function () {
                expect(Range3D.createFromRange(r1, 6, 8)).to.be.an.instanceof(Range3D);
            });
            it('should return a range with adjusted sheet indexes', function () {
                expect(Range3D.createFromRange(r1, 6, 8)).to.deep.equal(r2);
                expect(Range3D.createFromRange(r1, 8, 6)).to.deep.equal(r2);
            });
            it('should accept single sheet index', function () {
                var r3 = Range3D.createFromRange(r1, 6);
                expect(r3).to.have.property('sheet1', 6);
                expect(r3).to.have.property('sheet2', 6);
            });
            it('should clone addresses', function () {
                var r3 = Range3D.createFromRange(r1, 6, 8);
                expect(r3.start).to.deep.equal(r2.start);
                expect(r3.start).to.not.equal(r2.start);
                expect(r3.end).to.deep.equal(r2.end);
                expect(r3.end).to.not.equal(r2.end);
            });
        });

        describe('method "createFromIntervals"', function () {
            it('should exist', function () {
                expect(Range3D).itself.to.respondTo('createFromIntervals');
            });
            it('should create range address from intervals', function () {
                var r1 = Range3D.createFromIntervals(i('B:C'), i('3:4'), 6, 8);
                expect(r1.start).to.deep.equal(a('B3'));
                expect(r1.end).to.deep.equal(a('C4'));
                expect(r1).to.have.property('sheet1', 6);
                expect(r1).to.have.property('sheet2', 8);
            });
            it('should accept single sheet index', function () {
                var r1 = Range3D.createFromIntervals(i('B:C'), i('3:4'), 6);
                expect(r1.start).to.deep.equal(a('B3'));
                expect(r1.end).to.deep.equal(a('C4'));
                expect(r1).to.have.property('sheet1', 6);
                expect(r1).to.have.property('sheet2', 6);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "key"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('key');
            });
            it('should return unique key for range address', function () {
                expect(r3d('0:0!A2:A2').key()).to.equal('0:0!0,1:0,1');
                expect(r3d('5:6!B3:D5').key()).to.equal('5:6!1,2:3,4');
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('clone');
            });
            it('should return a clone', function () {
                var r1 = r3d('6:8!A2:C4'), r2 = r1.clone();
                expect(r2).to.be.an.instanceof(Range3D);
                expect(r2).to.not.equal(r1);
                expect(r2.start).to.not.equal(r1.start);
                expect(r2.end).to.not.equal(r1.end);
                expect(r2).to.deep.equal(r1);
            });
        });

        describe('method "equals"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('equals');
            });
            var range = r3d('6:8!A2:C4');
            it('should return true for equal ranges', function () {
                expect(r3d('1:1!A2:A4').equals(r3d('1:1!A2:A4'))).to.equal(true);
                expect(r3d('2:3!A2:C2').equals(r3d('2:3!A2:C2'))).to.equal(true);
                expect(r3d('6:8!A2:C4').equals(r3d('6:8!A2:C4'))).to.equal(true);
                expect(range.equals(range)).to.equal(true);
            });
            it('should return false for different ranges', function () {
                expect(range.equals(r3d('5:8!A2:C4'))).to.equal(false);
                expect(range.equals(r3d('6:7!A2:C4'))).to.equal(false);
                expect(range.equals(r3d('6:8!A2:C5'))).to.equal(false);
                expect(range.equals(r3d('6:8!A2:D4'))).to.equal(false);
                expect(range.equals(r3d('6:8!A3:C4'))).to.equal(false);
                expect(range.equals(r3d('6:8!B2:C4'))).to.equal(false);
            });
        });

        describe('method "differs"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('differs');
            });
            var range = r3d('6:8!A2:C4');
            it('should return false for equal ranges', function () {
                expect(r3d('1:1!A2:A4').differs(r3d('1:1!A2:A4'))).to.equal(false);
                expect(r3d('2:3!A2:C2').differs(r3d('2:3!A2:C2'))).to.equal(false);
                expect(r3d('6:8!A2:C4').differs(r3d('6:8!A2:C4'))).to.equal(false);
                expect(range.differs(range)).to.equal(false);
            });
            it('should return true for different ranges', function () {
                expect(range.differs(r3d('5:8!A2:C4'))).to.equal(true);
                expect(range.differs(r3d('6:7!A2:C4'))).to.equal(true);
                expect(range.differs(r3d('6:8!A2:C5'))).to.equal(true);
                expect(range.differs(r3d('6:8!A2:D4'))).to.equal(true);
                expect(range.differs(r3d('6:8!A3:C4'))).to.equal(true);
                expect(range.differs(r3d('6:8!B2:C4'))).to.equal(true);
            });
        });

        describe('method "sheets"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('sheets');
            });
            it('should return the correct sheet count', function () {
                expect(r3d('0:0!A2:C4').sheets()).to.equal(1);
                expect(r3d('0:1!A2:C4').sheets()).to.equal(2);
                expect(r3d('1:1!A2:C4').sheets()).to.equal(1);
                expect(r3d('1:2!A2:C4').sheets()).to.equal(2);
                expect(r3d('1:3!A2:C4').sheets()).to.equal(3);
                expect(r3d('1:4!A2:C4').sheets()).to.equal(4);
            });
        });

        describe('method "cells"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('cells');
            });
            it('should return the correct count', function () {
                expect(r3d('0:0!A1:A1').cells()).to.equal(1);
                expect(r3d('1:2!C3:C3').cells()).to.equal(2);
                expect(r3d('2:4!C3:D3').cells()).to.equal(6);
                expect(r3d('2:5!C3:E5').cells()).to.equal(36);
            });
        });

        describe('method "singleSheet"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('singleSheet');
            });
            it('should return true for equals sheets', function () {
                expect(r3d('1:1!A2:A4').singleSheet()).to.equal(true);
                expect(r3d('6:6!A2:A4').singleSheet()).to.equal(true);
            });
            it('should return false for different sheets', function () {
                expect(r3d('1:2!A2:A4').singleSheet()).to.equal(false);
                expect(r3d('6:8!A2:A4').singleSheet()).to.equal(false);
            });
        });

        describe('method "isSheet"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('isSheet');
            });
            it('should return true for correct sheet', function () {
                expect(r3d('1:1!A2:A4').isSheet(1)).to.equal(true);
                expect(r3d('6:6!A2:A4').isSheet(6)).to.equal(true);
            });
            it('should return false for other sheets', function () {
                expect(r3d('1:1!A2:A4').isSheet(0)).to.equal(false);
                expect(r3d('1:1!A2:A4').isSheet(2)).to.equal(false);
                expect(r3d('6:6!A2:A4').isSheet(5)).to.equal(false);
                expect(r3d('6:6!A2:A4').isSheet(7)).to.equal(false);
            });
            it('should return false for mult-sheet ranges', function () {
                var r1 = r3d('6:8!A2:A4');
                expect(r1.isSheet(5)).to.equal(false);
                expect(r1.isSheet(6)).to.equal(false);
                expect(r1.isSheet(7)).to.equal(false);
                expect(r1.isSheet(8)).to.equal(false);
                expect(r1.isSheet(9)).to.equal(false);
            });
        });

        describe('method "containsSheet"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('containsSheet');
            });
            var range = r3d('6:8!A2:A4');
            it('should return true for inner sheet indexes', function () {
                expect(range.containsSheet(6)).to.equal(true);
                expect(range.containsSheet(7)).to.equal(true);
                expect(range.containsSheet(8)).to.equal(true);
            });
            it('should return false for outer sheet indexes', function () {
                expect(range.containsSheet(4)).to.equal(false);
                expect(range.containsSheet(5)).to.equal(false);
                expect(range.containsSheet(9)).to.equal(false);
                expect(range.containsSheet(10)).to.equal(false);
            });
        });

        describe('method "contains"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('contains');
            });
            var range = r3d('6:8!C3:F6');
            it('should return false for ranges outside the range', function () {
                expect(range.contains(r3d('7:7!A1:B2'))).to.equal(false);
                expect(range.contains(r3d('4:5!D4:E5'))).to.equal(false);
            });
            it('should return false for ranges partly overlapping the range', function () {
                expect(range.contains(r3d('7:7!B2:D4'))).to.equal(false);
                expect(range.contains(r3d('5:7!D4:E5'))).to.equal(false);
                expect(range.contains(r3d('7:9!D4:E5'))).to.equal(false);
            });
            it('should return true for ranges inside the range', function () {
                expect(range.contains(r3d('7:7!D4:E5'))).to.equal(true);
                expect(range.contains(r3d('6:7!D4:E5'))).to.equal(true);
                expect(range.contains(r3d('7:8!D4:E5'))).to.equal(true);
                expect(range.contains(r3d('6:8!D4:E5'))).to.equal(true);
            });
        });

        describe('method "overlaps"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('overlaps');
            });
            var range = r3d('6:8!C3:F6');
            it('should return false for distinct ranges', function () {
                expect(range.overlaps(r3d('5:7!A1:B2'))).to.equal(false);
                expect(range.overlaps(r3d('7:9!G7:H8'))).to.equal(false);
                expect(range.overlaps(r3d('5:5!B2:D4'))).to.equal(false);
                expect(range.overlaps(r3d('9:9!E5:G7'))).to.equal(false);
            });
            it('should return true for ranges partly overlapping the range', function () {
                expect(range.overlaps(r3d('5:7!B2:D4'))).to.equal(true);
                expect(range.overlaps(r3d('7:9!E5:G7'))).to.equal(true);
            });
            it('should return true for ranges containing each other', function () {
                expect(range.overlaps(r3d('7:7!D4:E5'))).to.equal(true);
                expect(range.overlaps(r3d('5:9!B2:G7'))).to.equal(true);
            });
        });

        describe('method "boundary"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('boundary');
            });
            it('should return the bounding interval', function () {
                expect(r3d('1:1!B3:C4').boundary(r3d('1:1!C4:D5'))).to.deep.equal(r3d('1:1!B3:D5'));
                expect(r3d('1:1!C3:D4').boundary(r3d('3:3!B4:C5'))).to.deep.equal(r3d('1:3!B3:D5'));
                expect(r3d('3:3!B4:C5').boundary(r3d('1:1!C3:D4'))).to.deep.equal(r3d('1:3!B3:D5'));
                expect(r3d('1:3!C4:D5').boundary(r3d('2:4!B3:C4'))).to.deep.equal(r3d('1:4!B3:D5'));
                expect(r3d('2:4!C4:C4').boundary(r3d('1:3!B3:D5'))).to.deep.equal(r3d('1:4!B3:D5'));
                expect(r3d('1:4!B3:D5').boundary(r3d('2:3!C4:C4'))).to.deep.equal(r3d('1:4!B3:D5'));
            });
        });

        describe('method "intersect"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('intersect');
            });
            it('should return null for distinct ranges', function () {
                expect(r3d('6:8!B3:C4').intersect(r3d('6:8!D3:E4'))).to.equal(null);
                expect(r3d('6:8!D3:E4').intersect(r3d('6:8!B3:C4'))).to.equal(null);
                expect(r3d('6:8!B3:C4').intersect(r3d('6:8!B5:C6'))).to.equal(null);
                expect(r3d('6:8!B5:C6').intersect(r3d('6:8!B3:C4'))).to.equal(null);
                expect(r3d('1:2!B3:C4').intersect(r3d('6:8!B3:C4'))).to.equal(null);
            });
            it('should return intersection range for overlapping ranges', function () {
                var range = r3d('6:8!B3:E5');
                expect(range.intersect(r3d('5:7!A2:B3'))).to.deep.equal(r3d('6:7!B3:B3'));
                expect(range.intersect(r3d('7:9!A2:B6'))).to.deep.equal(r3d('7:8!B3:B5'));
                expect(range.intersect(r3d('7:7!A4:C6'))).to.deep.equal(r3d('7:7!B4:C5'));
                expect(range.intersect(r3d('5:9!A4:F4'))).to.deep.equal(r3d('6:8!B4:E4'));
                expect(range.intersect(r3d('6:8!C1:D8'))).to.deep.equal(r3d('6:8!C3:D5'));
                expect(range.intersect(r3d('6:8!A1:F8'))).to.deep.equal(r3d('6:8!B3:E5'));
                expect(range.intersect(r3d('6:8!B3:E5'))).to.deep.equal(r3d('6:8!B3:E5'));
                expect(range.intersect(r3d('6:8!C3:D5'))).to.deep.equal(r3d('6:8!C3:D5'));
                expect(range.intersect(r3d('6:8!C4:C4'))).to.deep.equal(r3d('6:8!C4:C4'));
                expect(range.intersect(range)).to.deep.equal(range);
                expect(range.intersect(range)).to.not.equal(range);
            });
        });

        describe('method "toRange"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('toRange');
            });
            it('should convert the range to a 2D range', function () {
                var r1 = r3d('6:8!A2:C4').toRange();
                expect(r1).to.be.an.instanceof(Range);
                expect(r1).to.deep.equal(r('A2:C4'));
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('toString');
            });
            it('should stringify the range', function () {
                expect(r3d('6:8!A2:C4').toString()).to.equal('6:8!A2:C4');
            });
            it('should stringify implicitly', function () {
                expect('<' + r3d('6:8!A2:C4') + '>').to.equal('<6:8!A2:C4>');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(Range3D).to.respondTo('toJSON');
            });
            var r1 = r3d('6:8!A3:B4'), r2 = { sheet1: 6, sheet2: 8, start: [0, 2], end: [1, 3] };
            it('should convert to JSON data', function () {
                expect(r1.toJSON()).to.deep.equal(r2);
            });
            it('should stringify implicitly', function () {
                expect(JSON.parse(JSON.stringify(r1))).to.deep.equal(r2);
            });
        });
    });

    // ========================================================================
});
