/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/utils/dimension',
    'io.ox/office/spreadsheet/model/formula/utils/matrix'
], function (ErrorCode, Dimension, Matrix) {

    'use strict';

    // class Matrix ===========================================================

    describe('Spreadsheet class Matrix', function () {

        it('should exist', function () {
            expect(Matrix).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var a1 = [[1]];
        var a2 = [[1, 2], ['a', 'b'], [false, true]];

        // constants ----------------------------------------------------------

        describe('constant "MAX_SIZE"', function () {
            it('should exist', function () {
                expect(Matrix).to.have.a.property('MAX_SIZE').that.is.a('number').and.above(0);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "isValidDim"', function () {
            it('should exist', function () {
                expect(Matrix).itself.to.respondTo('isValidDim');
            });
            var rows = Math.floor(Math.sqrt(Matrix.MAX_SIZE));
            var cols = Math.floor(Matrix.MAX_SIZE / rows);
            it('should return true for valid matrix sizes', function () {
                expect(Matrix.isValidDim(1, 1)).to.equal(true);
                expect(Matrix.isValidDim(Matrix.MAX_SIZE, 1)).to.equal(true);
                expect(Matrix.isValidDim(1, Matrix.MAX_SIZE)).to.equal(true);
                expect(Matrix.isValidDim(rows, cols)).to.equal(true);
                expect(Matrix.isValidDim(new Dimension(rows, cols))).to.equal(true);
            });
            it('should return false for invalid matrix sizes', function () {
                expect(Matrix.isValidDim(0, 1)).to.equal(false);
                expect(Matrix.isValidDim(1, 0)).to.equal(false);
                expect(Matrix.isValidDim(Matrix.MAX_SIZE + 1, 1)).to.equal(false);
                expect(Matrix.isValidDim(1, Matrix.MAX_SIZE + 1)).to.equal(false);
                expect(Matrix.isValidDim(rows + 1, cols)).to.equal(false);
                expect(Matrix.isValidDim(rows, cols + 1)).to.equal(false);
                expect(Matrix.isValidDim(new Dimension(rows, cols + 1))).to.equal(false);
            });
        });

        describe('method "create"', function () {
            it('should exist', function () {
                expect(Matrix).itself.to.respondTo('create');
            });
            it('should create a filled matrix', function () {
                var m1 = Matrix.create(1, 1, 1);
                expect(m1).to.be.an.instanceof(Matrix);
                expect(m1.values()).to.deep.equal([[1]]);
                var m2 = Matrix.create(3, 2, 'a');
                expect(m2).to.be.an.instanceof(Matrix);
                expect(m2.values()).to.deep.equal([['a', 'a'], ['a', 'a'], ['a', 'a']]);
            });
        });

        describe('method "generate"', function () {
            it('should exist', function () {
                expect(Matrix).itself.to.respondTo('generate');
            });
            it('should create a filled matrix', function () {
                var m1 = Matrix.generate(1, 1, _.constant(1));
                expect(m1).to.be.an.instanceof(Matrix);
                expect(m1.values()).to.deep.equal([[1]]);
                function generator(row, col) { return a2[row][col]; }
                var spy = sinon.spy(generator), m2 = Matrix.generate(3, 2, spy, a1);
                expect(m2).to.be.an.instanceof(Matrix);
                expect(m2.values()).to.deep.equal(a2);
                sinon.assert.callCount(spy, 6);
                sinon.assert.alwaysCalledOn(spy, a1);
            });
        });

        describe('method "identity"', function () {
            it('should exist', function () {
                expect(Matrix).itself.to.respondTo('identity');
            });
            it('should create an identity matrix', function () {
                var m1 = Matrix.identity(1);
                expect(m1).to.be.an.instanceof(Matrix);
                expect(m1.values()).to.deep.equal([[1]]);
                var m3 = Matrix.identity(3);
                expect(m3).to.be.an.instanceof(Matrix);
                expect(m3.values()).to.deep.equal([[1, 0, 0], [0, 1, 0], [0, 0, 1]]);
            });
        });

        describe('method "createRowVector"', function () {
            it('should exist', function () {
                expect(Matrix).itself.to.respondTo('createRowVector');
            });
            it('should create a row vector', function () {
                var m1 = Matrix.createRowVector([42, 'a', false]);
                expect(m1).to.be.an.instanceof(Matrix);
                expect(m1.values()).to.deep.equal([[42, 'a', false]]);
            });
        });

        describe('method "createColVector"', function () {
            it('should exist', function () {
                expect(Matrix).itself.to.respondTo('createColVector');
            });
            it('should create a column vector', function () {
                var m1 = Matrix.createColVector([42, 'a', false]);
                expect(m1).to.be.an.instanceof(Matrix);
                expect(m1.values()).to.deep.equal([[42], ['a'], [false]]);
            });
        });

        describe('method "getAddress"', function () {
            it('should exist', function () {
                expect(Matrix).itself.to.respondTo('getAddress');
            });
        });

        // public methods -----------------------------------------------------

        describe('method "values"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('values');
            });
            var m2 = new Matrix(a2);
            it('should return the internal arrays', function () {
                expect(m2.values()).to.equal(a2);
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('clone');
            });
            it('should return a deep clone of a matrix', function () {
                var m1 = new Matrix(a1);
                var m2 = m1.clone();
                expect(m2).to.be.an.instanceof(Matrix);
                expect(m2.values()).to.deep.equal(a1);
                expect(m2.values()).to.not.equal(a1);
            });
        });

        describe('method "rows"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('rows');
            });
            it('should return the number of rows', function () {
                expect(new Matrix(a1).rows()).to.equal(1);
                expect(new Matrix(a2).rows()).to.equal(3);
            });
        });

        describe('method "cols"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('cols');
            });
            it('should return the number of rows', function () {
                expect(new Matrix(a1).cols()).to.equal(1);
                expect(new Matrix(a2).cols()).to.equal(2);
            });
        });

        describe('method "size"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('size');
            });
            it('should return the number of rows', function () {
                expect(new Matrix(a1).size()).to.equal(1);
                expect(new Matrix(a2).size()).to.equal(6);
            });
        });

        describe('method "get"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('get');
            });
            var m1 = new Matrix(a1), m2 = new Matrix(a2), m3 = new Matrix([[2, 3]]), m4 = new Matrix([[2], [3]]);
            it('should return the specified element', function () {
                expect(m1.get(0, 0)).to.equal(a1[0][0]);
                expect(m2.get(0, 0)).to.equal(a2[0][0]);
                expect(m2.get(0, 1)).to.equal(a2[0][1]);
                expect(m2.get(1, 0)).to.equal(a2[1][0]);
                expect(m2.get(1, 1)).to.equal(a2[1][1]);
                expect(m2.get(2, 0)).to.equal(a2[2][0]);
                expect(m2.get(2, 1)).to.equal(a2[2][1]);
            });
            it('should ignore row index for single-row vector', function () {
                expect(m3.get(0, 0)).to.equal(2);
                expect(m3.get(0, 1)).to.equal(3);
                expect(m3.get(1, 0)).to.equal(2);
                expect(m3.get(1, 1)).to.equal(3);
                expect(m3.get(9, 0)).to.equal(2);
                expect(m3.get(9, 1)).to.equal(3);
            });
            it('should ignore column index for single-column vector', function () {
                expect(m4.get(0, 0)).to.equal(2);
                expect(m4.get(0, 1)).to.equal(2);
                expect(m4.get(0, 9)).to.equal(2);
                expect(m4.get(1, 0)).to.equal(3);
                expect(m4.get(1, 1)).to.equal(3);
                expect(m4.get(1, 9)).to.equal(3);
            });
            it('should return #N/A on invalid indexes', function () {
                expect(m2.get(2, 2)).to.equal(ErrorCode.NA);
                expect(m2.get(3, 1)).to.equal(ErrorCode.NA);
                expect(m2.get(3, 2)).to.equal(ErrorCode.NA);
                expect(m3.get(0, 2)).to.equal(ErrorCode.NA);
                expect(m3.get(9, 2)).to.equal(ErrorCode.NA);
                expect(m4.get(2, 0)).to.equal(ErrorCode.NA);
                expect(m4.get(2, 9)).to.equal(ErrorCode.NA);
            });
        });

        describe('method "getRow"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('getRow');
            });
            var m2 = new Matrix(a2);
            it('should return row vector', function () {
                expect(m2.getRow(0)).to.deep.equal([1, 2]);
                expect(m2.getRow(1)).to.deep.equal(['a', 'b']);
                expect(m2.getRow(2)).to.deep.equal([false, true]);
            });
        });

        describe('method "getCol"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('getCol');
            });
            var m2 = new Matrix(a2);
            it('should return column vector', function () {
                expect(m2.getCol(0)).to.deep.equal([1, 'a', false]);
                expect(m2.getCol(1)).to.deep.equal([2, 'b', true]);
            });
        });

        describe('method "getByIndex"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('getByIndex');
            });
            var m2 = new Matrix(a2);
            it('should return the specified element', function () {
                expect(m2.getByIndex(0)).to.equal(a2[0][0]);
                expect(m2.getByIndex(1)).to.equal(a2[0][1]);
                expect(m2.getByIndex(2)).to.equal(a2[1][0]);
                expect(m2.getByIndex(3)).to.equal(a2[1][1]);
                expect(m2.getByIndex(4)).to.equal(a2[2][0]);
                expect(m2.getByIndex(5)).to.equal(a2[2][1]);
            });
        });

        describe('method "set"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('set');
            });
            var m1 = Matrix.create(3, 2, 1);
            it('should change the specified element', function () {
                m1.set(0, 0, 'a');
                expect(m1.values()[0][0]).to.equal('a');
                m1.set(2, 1, true);
                expect(m1.values()[2][1]).to.equal(true);
            });
            it('should return itself', function () {
                expect(m1.set(0, 0, 1)).to.equal(m1);
            });
        });

        describe('method "setByIndex"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('setByIndex');
            });
            var m1 = Matrix.create(3, 2, 1);
            it('should change the specified element', function () {
                m1.setByIndex(0, 'a');
                expect(m1.values()[0][0]).to.equal('a');
                m1.setByIndex(5, true);
                expect(m1.values()[2][1]).to.equal(true);
            });
            it('should return itself', function () {
                expect(m1.setByIndex(0, 1)).to.equal(m1);
            });
        });

        describe('method "update"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('update');
            });
            var m1 = Matrix.create(3, 2, 1);
            it('should update the specified element', function () {
                m1.update(0, 0, function (n) { return n + 1; });
                expect(m1.values()[0][0]).to.equal(2);
                m1.update(2, 1, function (n) { return n - 1; });
                expect(m1.values()[2][1]).to.equal(0);
            });
            it('should return itself', function () {
                expect(m1.update(0, 0, _.identity)).to.equal(m1);
            });
        });

        describe('method "updateByIndex"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('updateByIndex');
            });
            var m1 = Matrix.create(3, 2, 1);
            it('should update the specified element', function () {
                m1.updateByIndex(0, function (n) { return n + 1; });
                expect(m1.values()[0][0]).to.equal(2);
                m1.updateByIndex(5, function (n) { return n - 1; });
                expect(m1.values()[2][1]).to.equal(0);
            });
            it('should return itself', function () {
                expect(m1.updateByIndex(0, _.identity)).to.equal(m1);
            });
        });

        describe('method "add"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('add');
            });
            var m1 = Matrix.create(3, 2, 1);
            it('should update the specified element', function () {
                m1.add(0, 0, 1);
                expect(m1.values()[0][0]).to.equal(2);
                m1.add(2, 1, -1);
                expect(m1.values()[2][1]).to.equal(0);
            });
            it('should return itself', function () {
                expect(m1.add(0, 0, 0)).to.equal(m1);
            });
        });

        describe('method "addByIndex"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('addByIndex');
            });
            var m1 = Matrix.create(3, 2, 1);
            it('should update the specified element', function () {
                m1.addByIndex(0, 1);
                expect(m1.values()[0][0]).to.equal(2);
                m1.addByIndex(5, -1);
                expect(m1.values()[2][1]).to.equal(0);
            });
            it('should return itself', function () {
                expect(m1.addByIndex(0, 0)).to.equal(m1);
            });
        });

        describe('method "fill"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('fill');
            });
            var m1 = Matrix.create(4, 4, 1);
            it('should change the specified elements', function () {
                m1.fill(0, 1, 2, 3, 42);
                expect(m1.values()).to.deep.equal([[1, 42, 42, 42], [1, 42, 42, 42], [1, 42, 42, 42], [1, 1, 1, 1]]);
            });
            it('should return itself', function () {
                expect(m1.fill(0, 0, 0, 0, 1)).to.equal(m1);
            });
        });

        describe('method "iterator"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('iterator');
            });
            it('should create an iterator that visits all matrix elements', function () {
                var m = new Matrix(a2), it = m.iterator();
                expect(it.next()).to.deep.equal({ done: false, value: 1, row: 0, col: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, row: 0, col: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 'a', row: 1, col: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 'b', row: 1, col: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: false, row: 2, col: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: true, row: 2, col: 1 });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('forEach');
            });
            it('should visit all matrix elements', function () {
                var m = new Matrix(a2), spy = sinon.spy();
                expect(m.forEach(spy, a1)).to.equal(m);
                sinon.assert.callCount(spy, 6);
                sinon.assert.alwaysCalledOn(spy, a1);
                sinon.assert.calledWithExactly(spy.getCall(0), a2[0][0], 0, 0);
                sinon.assert.calledWithExactly(spy.getCall(1), a2[0][1], 0, 1);
                sinon.assert.calledWithExactly(spy.getCall(2), a2[1][0], 1, 0);
                sinon.assert.calledWithExactly(spy.getCall(3), a2[1][1], 1, 1);
                sinon.assert.calledWithExactly(spy.getCall(4), a2[2][0], 2, 0);
                sinon.assert.calledWithExactly(spy.getCall(5), a2[2][1], 2, 1);
            });
        });

        describe('method "map"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('map');
            });
            it('should create a new matrix', function () {
                var m1 = new Matrix([[1, 2], [3, 4]]), m2 = m1.map(function (e) { return e + 1; });
                expect(m2).to.be.an.instanceof(Matrix);
                expect(m1.values()).to.not.equal(m2.values());
                expect(m2.values()).to.deep.equal([[2, 3], [4, 5]]);
            });
        });

        describe('method "reduce"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('reduce');
            });
            it('should reduce all matrix elements to a value', function () {
                var m = new Matrix([[1, 2], [3, 4]]), spy = sinon.spy(function (a, e) { return a + e; });
                expect(m.reduce(0, spy, a1)).to.equal(10);
                sinon.assert.callCount(spy, 4);
                sinon.assert.alwaysCalledOn(spy, a1);
                sinon.assert.calledWithExactly(spy.getCall(0), 0, 1, 0, 0);
                sinon.assert.calledWithExactly(spy.getCall(1), 1, 2, 0, 1);
                sinon.assert.calledWithExactly(spy.getCall(2), 3, 3, 1, 0);
                sinon.assert.calledWithExactly(spy.getCall(3), 6, 4, 1, 1);
            });
        });

        describe('method "transform"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('transform');
            });
            it('should modify all matrix elements', function () {
                var m = new Matrix([[1, 2], [3, 4]]), spy = sinon.spy(function (e) { return e * e; });
                expect(m.transform(spy, a1)).to.equal(m);
                expect(m.values()).to.deep.equal([[1, 4], [9, 16]]);
                sinon.assert.callCount(spy, 4);
                sinon.assert.alwaysCalledOn(spy, a1);
                sinon.assert.calledWithExactly(spy.getCall(0), 1, 0, 0);
                sinon.assert.calledWithExactly(spy.getCall(1), 2, 0, 1);
                sinon.assert.calledWithExactly(spy.getCall(2), 3, 1, 0);
                sinon.assert.calledWithExactly(spy.getCall(3), 4, 1, 1);
            });
        });

        describe('method "swapRows"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('swapRows');
            });
            it('should swap the row elements of a matrix', function () {
                var m = new Matrix([[1, 2, 3], [4, 5, 6], [7, 8, 9]]);
                expect(m.swapRows(1, 2)).to.equal(m);
                expect(m.values()).to.deep.equal([[1, 2, 3], [7, 8, 9], [4, 5, 6]]);
            });
        });

        describe('method "scaleRow"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('scaleRow');
            });
            it('should scale the row elements of a matrix', function () {
                var m = new Matrix([[1, 2, 3], [4, 5, 6], [7, 8, 9]]);
                expect(m.scaleRow(1, 2)).to.equal(m);
                expect(m.values()).to.deep.equal([[1, 2, 3], [8, 10, 12], [7, 8, 9]]);
            });
        });

        describe('method "addScaledRow"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('addScaledRow');
            });
            it('should add the scaled row elements of a matrix', function () {
                var m = new Matrix([[1, 2, 3], [4, 5, 6], [7, 8, 9]]);
                expect(m.addScaledRow(1, 2, -2)).to.equal(m);
                expect(m.values()).to.deep.equal([[1, 2, 3], [-10, -11, -12], [7, 8, 9]]);
            });
        });

        describe('method "transpose"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('transpose');
            });
            it('should transpose a matrix', function () {
                var m1 = new Matrix(_.copy(a2, true)), m2 = m1.transpose();
                expect(m2).to.be.an.instanceOf(Matrix);
                expect(m2).to.not.equal(m1);
                expect(m1.values()).to.deep.equal(a2);
                expect(m2.values()).to.deep.equal([[1, 'a', false], [2, 'b', true]]);
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                var m = new Matrix(a2);
                expect(m.toString()).to.equal('{1;2|"a";"b"|FALSE;TRUE}');
                expect('<' + m + '>').to.equal('<{1;2|"a";"b"|FALSE;TRUE}>');
            });
        });
    });

    // ========================================================================
});
