/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/parser/formulagrammar'
], function (SheetHelper, FormulaUtils, FormulaGrammar) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var a = SheetHelper.a;
    var CellRef = FormulaUtils.CellRef;
    var SheetRef = FormulaUtils.SheetRef;

    // class FormulaGrammar ===================================================

    describe('Spreadsheet class FormulaGrammar', function () {

        it('should exist', function () {
            expect(FormulaGrammar).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // create a CellRef instance on-the-fly from $1.$1 notation
        function cell(text) {
            var m = /^(\$)?(\d+)\.(\$)?(\d+)$/i.exec(text);
            return new CellRef(parseInt(m[2], 10), parseInt(m[4], 10), !!m[1], !!m[3]);
        }

        // create a SheetRef instance on-the-fly
        function sh(sheet, abs) {
            return new SheetRef(sheet, abs !== false);
        }

        var opOoxA1Grammar = null; // grammar=op, format=ooxml, style=a1
        var uiOoxA1Grammar = null; // grammar=ui, format=ooxml, style=a1
        var uiOoxRCGrammar = null; // grammar=ui, format=ooxml, style=rc
        var enOoxA1Grammar = null; // grammar=en, format=ooxml, style=a1
        var enOoxRCGrammar = null; // grammar=en, format=ooxml, style=rc
        var opOdfA1Grammar = null; // grammar=op, format=odf, style=a1
        var uiOdfA1Grammar = null; // grammar=ui, format=odf, style=a1
        var uiOdfRCGrammar = null; // grammar=ui, format=odf, style=rc
        var enOdfA1Grammar = null; // grammar=en, format=odf, style=a1
        var enOdfRCGrammar = null; // grammar=en, format=odf, style=rc

        // maximum column/row index for this test
        var MAXCOL = 255;
        var MAXROW = 65535;

        // simple mock of a document model, used for sheet names, and maximum column/row checks
        var docModel = {
            getSheetName: function (index) {
                return ['Sheet1', 'Sheet2', 'Sheet3', 'Sheet 4'][index] || null;
            },
            isValidAddress: function (address) {
                return (address[0] >= 0) && (address[0] <= MAXCOL) && (address[1] >= 0) && (address[1] <= MAXROW);
            },
            isColRange: function (range) {
                return (range.start[1] === 0) && (range.end[1] === MAXROW);
            },
            isRowRange: function (range) {
                return (range.start[0] === 0) && (range.end[0] === MAXCOL);
            }
        };

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a formula grammar instance', function () {
                opOoxA1Grammar = new FormulaGrammar('op', 'ooxml', false);
                expect(opOoxA1Grammar).to.be.an('object');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(FormulaGrammar).itself.to.respondTo('create');
            });
            it('should create a new configuration', function () {
                uiOoxA1Grammar = FormulaGrammar.create('ui', 'ooxml', false);
                expect(uiOoxA1Grammar).to.be.an.instanceof(FormulaGrammar);
                expect(uiOoxA1Grammar).to.not.equal(opOoxA1Grammar);
                uiOoxRCGrammar = FormulaGrammar.create('ui', 'ooxml', true);
                expect(uiOoxRCGrammar).to.be.an.instanceof(FormulaGrammar);
                expect(uiOoxRCGrammar).to.not.equal(opOoxA1Grammar);
                expect(uiOoxRCGrammar).to.not.equal(uiOoxA1Grammar);
                enOoxA1Grammar = FormulaGrammar.create('en', 'ooxml', false);
                expect(enOoxA1Grammar).to.be.an.instanceof(FormulaGrammar);
                expect(enOoxA1Grammar).to.not.equal(opOoxA1Grammar);
                expect(enOoxA1Grammar).to.not.equal(uiOoxA1Grammar);
                expect(enOoxA1Grammar).to.not.equal(uiOoxRCGrammar);
                enOoxRCGrammar = FormulaGrammar.create('en', 'ooxml', true);
                expect(enOoxRCGrammar).to.be.an.instanceof(FormulaGrammar);
                expect(enOoxRCGrammar).to.not.equal(opOoxA1Grammar);
                expect(enOoxRCGrammar).to.not.equal(uiOoxA1Grammar);
                expect(enOoxRCGrammar).to.not.equal(uiOoxRCGrammar);
                expect(enOoxRCGrammar).to.not.equal(enOoxA1Grammar);
                opOdfA1Grammar = FormulaGrammar.create('op', 'odf', false);
                expect(opOdfA1Grammar).to.be.an.instanceof(FormulaGrammar);
                expect(opOdfA1Grammar).to.not.equal(opOoxA1Grammar);
                expect(opOdfA1Grammar).to.not.equal(uiOoxA1Grammar);
                expect(opOdfA1Grammar).to.not.equal(uiOoxRCGrammar);
                expect(opOdfA1Grammar).to.not.equal(enOoxA1Grammar);
                expect(opOdfA1Grammar).to.not.equal(enOoxRCGrammar);
                uiOdfA1Grammar = FormulaGrammar.create('ui', 'odf', false);
                expect(uiOdfA1Grammar).to.be.an.instanceof(FormulaGrammar);
                expect(uiOdfA1Grammar).to.not.equal(opOoxA1Grammar);
                expect(uiOdfA1Grammar).to.not.equal(uiOoxA1Grammar);
                expect(uiOdfA1Grammar).to.not.equal(uiOoxRCGrammar);
                expect(uiOdfA1Grammar).to.not.equal(enOoxA1Grammar);
                expect(uiOdfA1Grammar).to.not.equal(enOoxRCGrammar);
                expect(uiOdfA1Grammar).to.not.equal(opOdfA1Grammar);
                uiOdfRCGrammar = FormulaGrammar.create('ui', 'odf', true);
                expect(uiOdfRCGrammar).to.be.an.instanceof(FormulaGrammar);
                expect(uiOdfRCGrammar).to.not.equal(opOoxA1Grammar);
                expect(uiOdfRCGrammar).to.not.equal(uiOoxA1Grammar);
                expect(uiOdfRCGrammar).to.not.equal(uiOoxRCGrammar);
                expect(uiOdfRCGrammar).to.not.equal(enOoxA1Grammar);
                expect(uiOdfRCGrammar).to.not.equal(enOoxRCGrammar);
                expect(uiOdfRCGrammar).to.not.equal(opOdfA1Grammar);
                expect(uiOdfRCGrammar).to.not.equal(uiOdfA1Grammar);
                enOdfA1Grammar = FormulaGrammar.create('en', 'odf', false);
                expect(enOdfA1Grammar).to.be.an.instanceof(FormulaGrammar);
                expect(enOdfA1Grammar).to.not.equal(opOoxA1Grammar);
                expect(enOdfA1Grammar).to.not.equal(uiOoxA1Grammar);
                expect(enOdfA1Grammar).to.not.equal(uiOoxRCGrammar);
                expect(enOdfA1Grammar).to.not.equal(enOoxA1Grammar);
                expect(enOdfA1Grammar).to.not.equal(enOoxRCGrammar);
                expect(enOdfA1Grammar).to.not.equal(opOdfA1Grammar);
                expect(enOdfA1Grammar).to.not.equal(uiOdfA1Grammar);
                expect(enOdfA1Grammar).to.not.equal(uiOdfRCGrammar);
                enOdfRCGrammar = FormulaGrammar.create('en', 'odf', true);
                expect(enOdfRCGrammar).to.be.an.instanceof(FormulaGrammar);
                expect(enOdfRCGrammar).to.not.equal(opOoxA1Grammar);
                expect(enOdfRCGrammar).to.not.equal(uiOoxA1Grammar);
                expect(enOdfRCGrammar).to.not.equal(uiOoxRCGrammar);
                expect(enOdfRCGrammar).to.not.equal(enOoxA1Grammar);
                expect(enOdfRCGrammar).to.not.equal(enOoxRCGrammar);
                expect(enOdfRCGrammar).to.not.equal(opOdfA1Grammar);
                expect(enOdfRCGrammar).to.not.equal(uiOdfA1Grammar);
                expect(enOdfRCGrammar).to.not.equal(uiOdfRCGrammar);
                expect(enOdfRCGrammar).to.not.equal(enOdfA1Grammar);
            });
            it('should return an existing configuration', function () {
                var config = FormulaGrammar.create('ui', 'ooxml', false);
                expect(config).to.equal(uiOoxA1Grammar);
            });
        });

        // constants ----------------------------------------------------------

        describe('constant "GRAMMAR"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('GRAMMAR', 'op');
                expect(uiOoxA1Grammar).to.have.a.property('GRAMMAR', 'ui');
                expect(uiOoxRCGrammar).to.have.a.property('GRAMMAR', 'ui');
                expect(enOoxA1Grammar).to.have.a.property('GRAMMAR', 'en');
                expect(enOoxRCGrammar).to.have.a.property('GRAMMAR', 'en');
                expect(opOdfA1Grammar).to.have.a.property('GRAMMAR', 'op');
                expect(uiOdfA1Grammar).to.have.a.property('GRAMMAR', 'ui');
                expect(uiOdfRCGrammar).to.have.a.property('GRAMMAR', 'ui');
                expect(enOdfA1Grammar).to.have.a.property('GRAMMAR', 'en');
                expect(enOdfRCGrammar).to.have.a.property('GRAMMAR', 'en');
            });
        });

        describe('constant "UI"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('UI', false);
                expect(uiOoxA1Grammar).to.have.a.property('UI', true);
                expect(uiOoxRCGrammar).to.have.a.property('UI', true);
                expect(enOoxA1Grammar).to.have.a.property('UI', true);
                expect(enOoxRCGrammar).to.have.a.property('UI', true);
                expect(opOdfA1Grammar).to.have.a.property('UI', false);
                expect(uiOdfA1Grammar).to.have.a.property('UI', true);
                expect(uiOdfRCGrammar).to.have.a.property('UI', true);
                expect(enOdfA1Grammar).to.have.a.property('UI', true);
                expect(enOdfRCGrammar).to.have.a.property('UI', true);
            });
        });

        describe('constant "REF_SYNTAX"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('REF_SYNTAX', 'ooxml');
                expect(uiOoxA1Grammar).to.have.a.property('REF_SYNTAX', 'ooxml');
                expect(uiOoxRCGrammar).to.have.a.property('REF_SYNTAX', 'ooxml');
                expect(enOoxA1Grammar).to.have.a.property('REF_SYNTAX', 'ooxml');
                expect(enOoxRCGrammar).to.have.a.property('REF_SYNTAX', 'ooxml');
                expect(opOdfA1Grammar).to.have.a.property('REF_SYNTAX', 'of');
                expect(uiOdfA1Grammar).to.have.a.property('REF_SYNTAX', 'odfui');
                expect(uiOdfRCGrammar).to.have.a.property('REF_SYNTAX', 'odfui');
                expect(enOdfA1Grammar).to.have.a.property('REF_SYNTAX', 'odfui');
                expect(enOdfRCGrammar).to.have.a.property('REF_SYNTAX', 'odfui');
            });
        });

        describe('constant "RC_STYLE"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('RC_STYLE', false);
                expect(uiOoxA1Grammar).to.have.a.property('RC_STYLE', false);
                expect(uiOoxRCGrammar).to.have.a.property('RC_STYLE', true);
                expect(enOoxA1Grammar).to.have.a.property('RC_STYLE', false);
                expect(enOoxRCGrammar).to.have.a.property('RC_STYLE', true);
                expect(opOdfA1Grammar).to.have.a.property('RC_STYLE', false);
                expect(uiOdfA1Grammar).to.have.a.property('RC_STYLE', false);
                expect(uiOdfRCGrammar).to.have.a.property('RC_STYLE', true);
                expect(enOdfA1Grammar).to.have.a.property('RC_STYLE', false);
                expect(enOdfRCGrammar).to.have.a.property('RC_STYLE', true);
            });
        });

        describe('constant "DEC"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('DEC', '.');
                expect(uiOoxA1Grammar).to.have.a.property('DEC', ',');
                expect(uiOoxRCGrammar).to.have.a.property('DEC', ',');
                expect(enOoxA1Grammar).to.have.a.property('DEC', '.');
                expect(enOoxRCGrammar).to.have.a.property('DEC', '.');
                expect(opOdfA1Grammar).to.have.a.property('DEC', '.');
                expect(uiOdfA1Grammar).to.have.a.property('DEC', ',');
                expect(uiOdfRCGrammar).to.have.a.property('DEC', ',');
                expect(enOdfA1Grammar).to.have.a.property('DEC', '.');
                expect(enOdfRCGrammar).to.have.a.property('DEC', '.');
            });
        });

        describe('constant "SEP"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('SEP', ',');
                expect(uiOoxA1Grammar).to.have.a.property('SEP', ';');
                expect(uiOoxRCGrammar).to.have.a.property('SEP', ';');
                expect(enOoxA1Grammar).to.have.a.property('SEP', ',');
                expect(enOoxRCGrammar).to.have.a.property('SEP', ',');
                expect(opOdfA1Grammar).to.have.a.property('SEP', ';');
                expect(uiOdfA1Grammar).to.have.a.property('SEP', ';');
                expect(uiOdfRCGrammar).to.have.a.property('SEP', ';');
                expect(enOdfA1Grammar).to.have.a.property('SEP', ',');
                expect(enOdfRCGrammar).to.have.a.property('SEP', ',');
            });
        });

        describe('constant "MAT_OPEN"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('MAT_OPEN', '{');
                expect(uiOoxA1Grammar).to.have.a.property('MAT_OPEN', '{');
                expect(uiOoxRCGrammar).to.have.a.property('MAT_OPEN', '{');
                expect(enOoxA1Grammar).to.have.a.property('MAT_OPEN', '{');
                expect(enOoxRCGrammar).to.have.a.property('MAT_OPEN', '{');
                expect(opOdfA1Grammar).to.have.a.property('MAT_OPEN', '{');
                expect(uiOdfA1Grammar).to.have.a.property('MAT_OPEN', '{');
                expect(uiOdfRCGrammar).to.have.a.property('MAT_OPEN', '{');
                expect(enOdfA1Grammar).to.have.a.property('MAT_OPEN', '{');
                expect(enOdfRCGrammar).to.have.a.property('MAT_OPEN', '{');
            });
        });

        describe('constant "MAT_ROW"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('MAT_ROW', ';');
                expect(uiOoxA1Grammar).to.have.a.property('MAT_ROW', '|');
                expect(uiOoxRCGrammar).to.have.a.property('MAT_ROW', '|');
                expect(enOoxA1Grammar).to.have.a.property('MAT_ROW', '|');
                expect(enOoxRCGrammar).to.have.a.property('MAT_ROW', '|');
                expect(opOdfA1Grammar).to.have.a.property('MAT_ROW', '|');
                expect(uiOdfA1Grammar).to.have.a.property('MAT_ROW', '|');
                expect(uiOdfRCGrammar).to.have.a.property('MAT_ROW', '|');
                expect(enOdfA1Grammar).to.have.a.property('MAT_ROW', '|');
                expect(enOdfRCGrammar).to.have.a.property('MAT_ROW', '|');
            });
        });

        describe('constant "MAT_COL"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('MAT_COL', ',');
                expect(uiOoxA1Grammar).to.have.a.property('MAT_COL', ';');
                expect(uiOoxRCGrammar).to.have.a.property('MAT_COL', ';');
                expect(enOoxA1Grammar).to.have.a.property('MAT_COL', ';');
                expect(enOoxRCGrammar).to.have.a.property('MAT_COL', ';');
                expect(opOdfA1Grammar).to.have.a.property('MAT_COL', ';');
                expect(uiOdfA1Grammar).to.have.a.property('MAT_COL', ';');
                expect(uiOdfRCGrammar).to.have.a.property('MAT_COL', ';');
                expect(enOdfA1Grammar).to.have.a.property('MAT_COL', ';');
                expect(enOdfRCGrammar).to.have.a.property('MAT_COL', ';');
            });
        });

        describe('constant "MAT_CLOSE"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('MAT_CLOSE', '}');
                expect(uiOoxA1Grammar).to.have.a.property('MAT_CLOSE', '}');
                expect(uiOoxRCGrammar).to.have.a.property('MAT_CLOSE', '}');
                expect(enOoxA1Grammar).to.have.a.property('MAT_CLOSE', '}');
                expect(enOoxRCGrammar).to.have.a.property('MAT_CLOSE', '}');
                expect(opOdfA1Grammar).to.have.a.property('MAT_CLOSE', '}');
                expect(uiOdfA1Grammar).to.have.a.property('MAT_CLOSE', '}');
                expect(uiOdfRCGrammar).to.have.a.property('MAT_CLOSE', '}');
                expect(enOdfA1Grammar).to.have.a.property('MAT_CLOSE', '}');
                expect(enOdfRCGrammar).to.have.a.property('MAT_CLOSE', '}');
            });
        });

        describe('constant "TABLE_SEP"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('TABLE_SEP', ',');
                expect(uiOoxA1Grammar).to.have.a.property('TABLE_SEP', ';');
                expect(uiOoxRCGrammar).to.have.a.property('TABLE_SEP', ';');
                expect(enOoxA1Grammar).to.have.a.property('TABLE_SEP', ',');
                expect(enOoxRCGrammar).to.have.a.property('TABLE_SEP', ',');
                expect(opOdfA1Grammar).to.have.a.property('TABLE_SEP', ';');
                expect(uiOdfA1Grammar).to.have.a.property('TABLE_SEP', ';');
                expect(uiOdfRCGrammar).to.have.a.property('TABLE_SEP', ';');
                expect(enOdfA1Grammar).to.have.a.property('TABLE_SEP', ',');
                expect(enOdfRCGrammar).to.have.a.property('TABLE_SEP', ',');
            });
        });

        describe('constant "PREFIX_CHARS"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('PREFIX_CHARS', 'RC');
                expect(uiOoxA1Grammar).to.have.a.property('PREFIX_CHARS', 'ZS');
                expect(uiOoxRCGrammar).to.have.a.property('PREFIX_CHARS', 'ZS');
                expect(enOoxA1Grammar).to.have.a.property('PREFIX_CHARS', 'RC');
                expect(enOoxRCGrammar).to.have.a.property('PREFIX_CHARS', 'RC');
                expect(opOdfA1Grammar).to.have.a.property('PREFIX_CHARS', 'RC');
                expect(uiOdfA1Grammar).to.have.a.property('PREFIX_CHARS', 'RC');
                expect(uiOdfRCGrammar).to.have.a.property('PREFIX_CHARS', 'RC');
                expect(enOdfA1Grammar).to.have.a.property('PREFIX_CHARS', 'RC');
                expect(enOdfRCGrammar).to.have.a.property('PREFIX_CHARS', 'RC');
            });
        });

        describe('constant "RE"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.have.a.property('RE').that.is.an('object');
                expect(uiOoxA1Grammar).to.have.a.property('RE').that.is.an('object');
                expect(uiOoxRCGrammar).to.have.a.property('RE').that.is.an('object');
                expect(enOoxA1Grammar).to.have.a.property('RE').that.is.an('object');
                expect(enOoxRCGrammar).to.have.a.property('RE').that.is.an('object');
                expect(opOdfA1Grammar).to.have.a.property('RE').that.is.an('object');
                expect(uiOdfA1Grammar).to.have.a.property('RE').that.is.an('object');
                expect(uiOdfRCGrammar).to.have.a.property('RE').that.is.an('object');
                expect(enOdfA1Grammar).to.have.a.property('RE').that.is.an('object');
                expect(enOdfRCGrammar).to.have.a.property('RE').that.is.an('object');
            });
            it('should contain regular expressions', function () {
                expect(_.every(opOoxA1Grammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(uiOoxA1Grammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(uiOoxRCGrammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(enOoxA1Grammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(enOoxRCGrammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(opOdfA1Grammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(uiOdfA1Grammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(uiOdfRCGrammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(enOdfA1Grammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(enOdfRCGrammar.RE, _.isRegExp)).to.equal(true);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "isReservedSymbol"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('isReservedSymbol');
                expect(uiOoxA1Grammar).to.respondTo('isReservedSymbol');
            });
            it('should recognize boolean literals', function () {
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'FALSE')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'FALSE')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'FALSE')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'TRUE')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'TRUE')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'TRUE')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'FALSCH')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'FALSCH')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'FALSCH')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'WAHR')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'WAHR')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'WAHR')).to.equal(false);
            });
            it('should recgonize A1 references', function () {
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'A1')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'A1')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'A1')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'iv65536')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'iv65536')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'iv65536')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'IW65536')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'IW65536')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'IW65536')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'IV65537')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'IV65537')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'IV65537')).to.equal(false);
            });
            it('should recgonize R1C1 references', function () {
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'R1C1')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'R1C1')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'R1C1')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'R65536C256')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'R65536C256')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'R65536C256')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'R65537C256')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'R65537C256')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'R65537C256')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'R65536C257')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'R65536C257')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'R65536C257')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'R1C')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'R1C')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'R1C')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'RC1')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'RC1')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'RC1')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'R1')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'R1')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'R1')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'C1')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'C1')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'C1')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'R')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'R')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'R')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'C')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'C')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'C')).to.equal(true);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'RC')).to.equal(true);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'RC')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'RC')).to.equal(true);
                // localized R1C1 references (not used in ODF)
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(true);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
                expect(opOdfA1Grammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
                expect(uiOdfA1Grammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
                expect(enOdfA1Grammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
            });
            it('should return false for other strings', function () {
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'A')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'A')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'A')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, '1')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, '1')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, '1')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, ' A1')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, ' A1')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, ' A1')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'A 1')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'A 1')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'A 1')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'R1 C1')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'R1 C1')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'R1 C1')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'C1R1')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'C1R1')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'C1R1')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, 'S1Z1')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, 'S1Z1')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, 'S1Z1')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, '')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, '')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, '')).to.equal(false);
                expect(opOoxA1Grammar.isReservedSymbol(docModel, ' TRUE')).to.equal(false);
                expect(uiOoxA1Grammar.isReservedSymbol(docModel, ' WAHR')).to.equal(false);
                expect(enOoxA1Grammar.isReservedSymbol(docModel, ' WAHR')).to.equal(false);
            });
        });

        describe('method "isSimpleSheetName"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('isSimpleSheetName');
                expect(uiOoxA1Grammar).to.respondTo('isSimpleSheetName');
            });
            it('should recognize simple sheet names', function () {
                // simple sheet name
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                expect(opOdfA1Grammar.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                expect(uiOdfA1Grammar.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                // whitespace
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                expect(opOdfA1Grammar.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                expect(uiOdfA1Grammar.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                // sheet range
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                expect(opOdfA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                expect(uiOdfA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                // booleans
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'TRUE')).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'TRUE')).to.equal(true);
                expect(enOoxA1Grammar.isSimpleSheetName(docModel, 'TRUE')).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'WAHR')).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'WAHR')).to.equal(false);
                expect(enOoxA1Grammar.isSimpleSheetName(docModel, 'WAHR')).to.equal(true);
                // A1 notation
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                expect(opOdfA1Grammar.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                expect(uiOdfA1Grammar.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                // native R1C1 notation
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(enOoxA1Grammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(opOdfA1Grammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(uiOdfA1Grammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(enOdfA1Grammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                // localized R1C1 notation
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(false);
                expect(enOoxA1Grammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                expect(opOdfA1Grammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                expect(uiOdfA1Grammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                expect(enOdfA1Grammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                // external path
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
                expect(opOdfA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
                expect(uiOdfA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
            });
            it('should recognize simple sheet ranges', function () {
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1', { range: true })).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1', { range: true })).to.equal(true);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet 1', { range: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet 1', { range: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true })).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true })).to.equal(true);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Sheet 2', { range: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Sheet 2', { range: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:TRUE', { range: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:TRUE', { range: true })).to.equal(true);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:WAHR', { range: true })).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:WAHR', { range: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:A1', { range: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:A1', { range: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:R1C1', { range: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:R1C1', { range: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Z1S1', { range: true })).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Z1S1', { range: true })).to.equal(false);
            });
            it('should recognize simple sheet names with document name', function () {
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1', { external: true })).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1', { external: true })).to.equal(true);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet 1', { external: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet 1', { external: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { external: true })).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { external: true })).to.equal(true);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { external: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { external: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { external: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { external: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1', { external: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1', { external: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, '[path/doc1.xlsx]Sheet1', { external: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, '[path/doc1.xlsx]Sheet1', { external: true })).to.equal(false);
            });
            it('should recognize simple sheet ranges with document name', function () {
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1', { range: true, external: true })).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1', { range: true, external: true })).to.equal(true);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet 1', { range: true, external: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet 1', { range: true, external: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { range: true, external: true })).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { range: true, external: true })).to.equal(true);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { range: true, external: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { range: true, external: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet 2', { range: true, external: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet 2', { range: true, external: true })).to.equal(false);
                expect(opOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(false);
            });
        });

        describe('method "validateNameLabel"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('validateNameLabel');
                expect(uiOoxA1Grammar).to.respondTo('validateNameLabel');
            });
            it('should reject invalid strings', function () {
                expect(opOoxA1Grammar.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(opOdfA1Grammar.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(opOoxA1Grammar.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(opOdfA1Grammar.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(opOoxA1Grammar.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(opOdfA1Grammar.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(opOoxA1Grammar.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(opOdfA1Grammar.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(opOoxA1Grammar.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(opOdfA1Grammar.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(opOoxA1Grammar.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
                expect(opOdfA1Grammar.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
            });
            it('should handle boolean literals', function () {
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'true')).to.equal('name:invalid');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'true')).to.equal('');
                expect(enOoxA1Grammar.validateNameLabel(docModel, 'true')).to.equal('name:invalid');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'true')).to.equal('');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'true')).to.equal('');
                expect(enOdfA1Grammar.validateNameLabel(docModel, 'true')).to.equal('');
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'wahr')).to.equal('');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'wahr')).to.equal('name:invalid');
                expect(enOoxA1Grammar.validateNameLabel(docModel, 'wahr')).to.equal('');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'wahr')).to.equal('');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'wahr')).to.equal('');
                expect(enOdfA1Grammar.validateNameLabel(docModel, 'wahr')).to.equal('');
            });
            it('should reject A1 cell references', function () {
                // first cell in sheet
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                // last cell in sheet
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                // outside of sheet range
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'iw65536')).to.equal('');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'iw65536')).to.equal('');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'iw65536')).to.equal('');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'iw65536')).to.equal('');
            });
            it('should reject R1C1 cell references', function () {
                // native absolute R1C1
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                // native relative R1C1
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                // localized absolute R1C1
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'z1s1')).to.equal('');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'z1s1')).to.equal('name:address');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'z1s1')).to.equal('');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'z1s1')).to.equal('');
                // localized relative R1C1
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'zs')).to.equal('');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'zs')).to.equal('name:address');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'zs')).to.equal('');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'zs')).to.equal('');
            });
            it('should accept valid names', function () {
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'true1')).to.equal('');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'wahr1')).to.equal('');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'true1')).to.equal('');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'wahr1')).to.equal('');
                expect(opOoxA1Grammar.validateNameLabel(docModel, 'c1r1')).to.equal('');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, 'c1r1')).to.equal('');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'c1r1')).to.equal('');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'c1r1')).to.equal('');
            });
            it('should handle backslash and question mark', function () {
                expect(opOoxA1Grammar.validateNameLabel(docModel, '\\name?name.name\\name')).to.equal('');
                expect(uiOoxA1Grammar.validateNameLabel(docModel, '\\name?name.name\\name')).to.equal('');
                expect(opOdfA1Grammar.validateNameLabel(docModel, '\\name')).to.equal('name:invalid');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, '\\name')).to.equal('name:invalid');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'name?a')).to.equal('');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'name?a')).to.equal('');
                expect(opOdfA1Grammar.validateNameLabel(docModel, 'name.a')).to.equal('name:invalid');
                expect(uiOdfA1Grammar.validateNameLabel(docModel, 'name.a')).to.equal('name:invalid');
            });
        });

        describe('method "formatScalar"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('formatScalar');
                expect(uiOoxA1Grammar).to.respondTo('formatScalar');
            });
            it('should convert values to native text', function () {
                expect(opOoxA1Grammar.formatScalar(42)).to.equal('42');
                expect(opOoxA1Grammar.formatScalar(-12.5)).to.equal('-12.5');
                expect(opOoxA1Grammar.formatScalar(1e300)).to.equal('1E+300');
                expect(opOoxA1Grammar.formatScalar(1e-300)).to.equal('1E-300');
                expect(opOoxA1Grammar.formatScalar(1e-310)).to.equal('0');
                expect(opOoxA1Grammar.formatScalar(Number.POSITIVE_INFINITY)).to.equal('#NUM!');
                expect(opOoxA1Grammar.formatScalar(Number.NaN)).to.equal('#NUM!');
                expect(opOoxA1Grammar.formatScalar('abc')).to.equal('"abc"');
                expect(opOoxA1Grammar.formatScalar('a"b"c')).to.equal('"a""b""c"');
                expect(opOoxA1Grammar.formatScalar('')).to.equal('""');
                expect(opOoxA1Grammar.formatScalar(false)).to.equal('FALSE');
                expect(opOoxA1Grammar.formatScalar(true)).to.equal('TRUE');
                expect(opOoxA1Grammar.formatScalar(ErrorCode.NULL)).to.equal('#NULL!');
                expect(opOoxA1Grammar.formatScalar(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(opOoxA1Grammar.formatScalar(ErrorCode.VALUE)).to.equal('#VALUE!');
                expect(opOoxA1Grammar.formatScalar(ErrorCode.REF)).to.equal('#REF!');
                expect(opOoxA1Grammar.formatScalar(ErrorCode.NAME)).to.equal('#NAME?');
                expect(opOoxA1Grammar.formatScalar(ErrorCode.NUM)).to.equal('#NUM!');
                expect(opOoxA1Grammar.formatScalar(ErrorCode.NA)).to.equal('#N/A');
                expect(opOoxA1Grammar.formatScalar(ErrorCode.DATA)).to.equal('#GETTING_DATA');
                expect(opOoxA1Grammar.formatScalar(null)).to.equal('');
            });
            it('should convert values to localized text', function () {
                expect(uiOoxA1Grammar.formatScalar(42)).to.equal('42');
                expect(uiOoxA1Grammar.formatScalar(-12.5)).to.equal('-12,5');
                expect(uiOoxA1Grammar.formatScalar(1e300)).to.equal('1E+300');
                expect(uiOoxA1Grammar.formatScalar(1e-300)).to.equal('1E-300');
                expect(uiOoxA1Grammar.formatScalar(1e-310)).to.equal('0');
                expect(uiOoxA1Grammar.formatScalar(Number.POSITIVE_INFINITY)).to.equal('#ZAHL!');
                expect(uiOoxA1Grammar.formatScalar(Number.NaN)).to.equal('#ZAHL!');
                expect(uiOoxA1Grammar.formatScalar('abc')).to.equal('"abc"');
                expect(uiOoxA1Grammar.formatScalar('a"b"c')).to.equal('"a""b""c"');
                expect(uiOoxA1Grammar.formatScalar('')).to.equal('""');
                expect(uiOoxA1Grammar.formatScalar(ErrorCode.NULL)).to.equal('#NULL!');
                expect(uiOoxA1Grammar.formatScalar(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(uiOoxA1Grammar.formatScalar(ErrorCode.VALUE)).to.equal('#WERT!');
                expect(uiOoxA1Grammar.formatScalar(ErrorCode.REF)).to.equal('#BEZUG!');
                expect(uiOoxA1Grammar.formatScalar(ErrorCode.NAME)).to.equal('#NAME?');
                expect(uiOoxA1Grammar.formatScalar(ErrorCode.NUM)).to.equal('#ZAHL!');
                expect(uiOoxA1Grammar.formatScalar(ErrorCode.NA)).to.equal('#NV');
                expect(uiOoxA1Grammar.formatScalar(ErrorCode.DATA)).to.equal('#GETTING_DATA');
                expect(uiOoxA1Grammar.formatScalar(null)).to.equal('');
            });
            it('should convert values to English text', function () {
                expect(enOoxA1Grammar.formatScalar(42)).to.equal('42');
                expect(enOoxA1Grammar.formatScalar(-12.5)).to.equal('-12.5');
                expect(enOoxA1Grammar.formatScalar(1e300)).to.equal('1E+300');
                expect(enOoxA1Grammar.formatScalar(1e-300)).to.equal('1E-300');
                expect(enOoxA1Grammar.formatScalar(1e-310)).to.equal('0');
                expect(enOoxA1Grammar.formatScalar(Number.POSITIVE_INFINITY)).to.equal('#NUM!');
                expect(enOoxA1Grammar.formatScalar(Number.NaN)).to.equal('#NUM!');
                expect(enOoxA1Grammar.formatScalar('abc')).to.equal('"abc"');
                expect(enOoxA1Grammar.formatScalar('a"b"c')).to.equal('"a""b""c"');
                expect(enOoxA1Grammar.formatScalar('')).to.equal('""');
                expect(enOoxA1Grammar.formatScalar(ErrorCode.NULL)).to.equal('#NULL!');
                expect(enOoxA1Grammar.formatScalar(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(enOoxA1Grammar.formatScalar(ErrorCode.VALUE)).to.equal('#VALUE!');
                expect(enOoxA1Grammar.formatScalar(ErrorCode.REF)).to.equal('#REF!');
                expect(enOoxA1Grammar.formatScalar(ErrorCode.NAME)).to.equal('#NAME?');
                expect(enOoxA1Grammar.formatScalar(ErrorCode.NUM)).to.equal('#NUM!');
                expect(enOoxA1Grammar.formatScalar(ErrorCode.NA)).to.equal('#N/A');
                expect(enOoxA1Grammar.formatScalar(ErrorCode.DATA)).to.equal('#GETTING_DATA');
                expect(enOoxA1Grammar.formatScalar(null)).to.equal('');
            });
        });

        describe('method "formatGenericRange"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('formatGenericRange');
                expect(uiOoxA1Grammar).to.respondTo('formatGenericRange');
            });
            var s1 = sh(1), s2 = sh(3), s3 = sh(-1);
            it('should return the string representation of a cell address', function () {
                expect(opOoxA1Grammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
                expect(enOoxA1Grammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
                expect(enOdfA1Grammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
            });
            it('should return the string representation of a cell range address', function () {
                expect(opOoxA1Grammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
                expect(enOoxA1Grammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
                expect(opOdfA1Grammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
                expect(enOdfA1Grammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
            });
            it('should return the string representation of a cell address with sheet', function () {
                expect(opOoxA1Grammar.formatGenericRange(docModel, s1, null, 'B3')).to.equal('Sheet2!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, s1, null, 'B3')).to.equal('Sheet2!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, s1, null, 'B3')).to.equal('$Sheet2.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, s1, null, 'B3')).to.equal('$Sheet2!B3');
            });
            it('should return the string representation of a cell range address with sheet', function () {
                expect(opOoxA1Grammar.formatGenericRange(docModel, s1, null, 'B3', 'D5')).to.equal('Sheet2!B3:D5');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, s1, null, 'B3', 'D5')).to.equal('Sheet2!B3:D5');
                expect(opOdfA1Grammar.formatGenericRange(docModel, s1, null, 'B3', 'D5')).to.equal('$Sheet2.B3:D5');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, s1, null, 'B3', 'D5')).to.equal('$Sheet2!B3:D5');
            });
            it('should return the string representation of a reference with sheet range', function () {
                // cell address
                expect(opOoxA1Grammar.formatGenericRange(docModel, s1, s2, 'B3')).to.equal('\'Sheet2:Sheet 4\'!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, s1, s2, 'B3')).to.equal('\'Sheet2:Sheet 4\'!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, s1, s2, 'B3')).to.equal('$Sheet2.B3:$\'Sheet 4\'.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, s1, s2, 'B3')).to.equal('$Sheet2:$\'Sheet 4\'!B3');
                // cell range address
                expect(opOoxA1Grammar.formatGenericRange(docModel, s1, s2, 'B3', 'D5')).to.equal('\'Sheet2:Sheet 4\'!B3:D5');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, s1, s2, 'B3', 'D5')).to.equal('\'Sheet2:Sheet 4\'!B3:D5');
                expect(opOdfA1Grammar.formatGenericRange(docModel, s1, s2, 'B3', 'D5')).to.equal('$Sheet2.B3:$\'Sheet 4\'.D5');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, s1, s2, 'B3', 'D5')).to.equal('$Sheet2:$\'Sheet 4\'!B3:D5');
            });
            it('should return the string representation of a sheet reference error', function () {
                // cell address
                expect(opOoxA1Grammar.formatGenericRange(docModel, s3, null, 'B3')).to.equal('#REF!');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, s3, null, 'B3')).to.equal('#BEZUG!');
                expect(opOdfA1Grammar.formatGenericRange(docModel, s3, null, 'B3')).to.equal('#REF!');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, s3, null, 'B3')).to.equal('#BEZUG!');
                // cell range address
                expect(opOoxA1Grammar.formatGenericRange(docModel, s3, null, 'B3', 'D5')).to.equal('#REF!');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, s3, null, 'B3', 'D5')).to.equal('#BEZUG!');
                expect(opOdfA1Grammar.formatGenericRange(docModel, s3, null, 'B3', 'D5')).to.equal('#REF!');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, s3, null, 'B3', 'D5')).to.equal('#BEZUG!');
            });
            it('should handle simple/complex sheet names correctly', function () {
                // simple name
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('Sheet1'), null, 'B3')).to.equal('Sheet1!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('Sheet1'), null, 'B3')).to.equal('Sheet1!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1'), null, 'B3')).to.equal('$Sheet1.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1'), null, 'B3')).to.equal('$Sheet1!B3');
                // whitespace
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('Sheet 2'), null, 'B3')).to.equal('\'Sheet 2\'!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('Sheet 2'), null, 'B3')).to.equal('\'Sheet 2\'!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet 2'), null, 'B3')).to.equal('$\'Sheet 2\'.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet 2'), null, 'B3')).to.equal('$\'Sheet 2\'!B3');
                // leading digit
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('1Sheet'), null, 'B3')).to.equal('\'1Sheet\'!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('1Sheet'), null, 'B3')).to.equal('\'1Sheet\'!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('1Sheet'), null, 'B3')).to.equal('$\'1Sheet\'.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('1Sheet'), null, 'B3')).to.equal('$\'1Sheet\'!B3');
                // operator
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('Sheet+1'), null, 'B3')).to.equal('\'Sheet+1\'!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('Sheet+1'), null, 'B3')).to.equal('\'Sheet+1\'!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet+1'), null, 'B3')).to.equal('$\'Sheet+1\'.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet+1'), null, 'B3')).to.equal('$\'Sheet+1\'!B3');
                // simple sheet range
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2'), 'B3')).to.equal('Sheet1:Sheet2!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2'), 'B3')).to.equal('Sheet1:Sheet2!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2'), 'B3')).to.equal('$Sheet1.B3:$Sheet2.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2'), 'B3')).to.equal('$Sheet1:$Sheet2!B3');
                // complex first sheet
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('Sheet 1'), sh('Sheet2'), 'B3')).to.equal('\'Sheet 1:Sheet2\'!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('Sheet 1'), sh('Sheet2'), 'B3')).to.equal('\'Sheet 1:Sheet2\'!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet 1'), sh('Sheet2'), 'B3')).to.equal('$\'Sheet 1\'.B3:$Sheet2.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet 1'), sh('Sheet2'), 'B3')).to.equal('$\'Sheet 1\':$Sheet2!B3');
                // complex second sheet
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet 2'), 'B3')).to.equal('\'Sheet1:Sheet 2\'!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet 2'), 'B3')).to.equal('\'Sheet1:Sheet 2\'!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet 2'), 'B3')).to.equal('$Sheet1.B3:$\'Sheet 2\'.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet 2'), 'B3')).to.equal('$Sheet1:$\'Sheet 2\'!B3');
                // ODF: relative sheets
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1', false), null, 'B3')).to.equal('Sheet1.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1', false), null, 'B3')).to.equal('Sheet1!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet 2', false), null, 'B3')).to.equal('\'Sheet 2\'.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet 2', false), null, 'B3')).to.equal('\'Sheet 2\'!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet2'), 'B3')).to.equal('Sheet1.B3:$Sheet2.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet2'), 'B3')).to.equal('Sheet1:$Sheet2!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2', false), 'B3')).to.equal('$Sheet1.B3:Sheet2.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2', false), 'B3')).to.equal('$Sheet1:Sheet2!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet2', false), 'B3')).to.equal('Sheet1.B3:Sheet2.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet2', false), 'B3')).to.equal('Sheet1:Sheet2!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet 1', false), sh('Sheet2', false), 'B3')).to.equal('\'Sheet 1\'.B3:Sheet2.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet 1', false), sh('Sheet2', false), 'B3')).to.equal('\'Sheet 1\':Sheet2!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet 2', false), 'B3')).to.equal('Sheet1.B3:\'Sheet 2\'.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet 2', false), 'B3')).to.equal('Sheet1:\'Sheet 2\'!B3');
            });
            it('should treat booleans as complex sheet names', function () {
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('TRUE'), null, 'B3')).to.equal('\'TRUE\'!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('TRUE'), null, 'B3')).to.equal('TRUE!B3');
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('WAHR'), null, 'B3')).to.equal('WAHR!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('WAHR'), null, 'B3')).to.equal('\'WAHR\'!B3');
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('TRUE'), sh('WAHR'), 'B3')).to.equal('\'TRUE:WAHR\'!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('TRUE'), sh('WAHR'), 'B3')).to.equal('\'TRUE:WAHR\'!B3');
            });
            it('should treat cell addresses as complex sheet names', function () {
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('A1'), null, 'B3')).to.equal('\'A1\'!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('A1'), null, 'B3')).to.equal('\'A1\'!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('A1'), null, 'B3')).to.equal('$\'A1\'.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('A1'), null, 'B3')).to.equal('$\'A1\'!B3');
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('IW1'), null, 'B3')).to.equal('IW1!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('IW1'), null, 'B3')).to.equal('IW1!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('IW1'), null, 'B3')).to.equal('$IW1.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('IW1'), null, 'B3')).to.equal('$IW1!B3');
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('R1C1'), null, 'B3')).to.equal('\'R1C1\'!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('R1C1'), null, 'B3')).to.equal('\'R1C1\'!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('R1C1'), null, 'B3')).to.equal('$\'R1C1\'.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('R1C1'), null, 'B3')).to.equal('$\'R1C1\'!B3');
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh('Z1S1'), null, 'B3')).to.equal('Z1S1!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh('Z1S1'), null, 'B3')).to.equal('\'Z1S1\'!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh('Z1S1'), null, 'B3')).to.equal('$Z1S1.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh('Z1S1'), null, 'B3')).to.equal('$Z1S1!B3');
                expect(opOoxA1Grammar.formatGenericRange(docModel, sh(1), sh('A1'), 'B3')).to.equal('\'Sheet2:A1\'!B3');
                expect(uiOoxA1Grammar.formatGenericRange(docModel, sh(1), sh('A1'), 'B3')).to.equal('\'Sheet2:A1\'!B3');
                expect(opOdfA1Grammar.formatGenericRange(docModel, sh(1), sh('A1'), 'B3')).to.equal('$Sheet2.B3:$\'A1\'.B3');
                expect(uiOdfA1Grammar.formatGenericRange(docModel, sh(1), sh('A1'), 'B3')).to.equal('$Sheet2:$\'A1\'!B3');
            });
        });

        describe('method "formatReference"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('formatReference');
                expect(uiOoxA1Grammar).to.respondTo('formatReference');
            });
            var ref = a('C4'), c1 = cell('1.$2'), c2 = cell('$3.4'), s1 = sh(1), s2 = sh(3), s3 = sh(-1);
            it('should return the string representation of a cell address', function () {
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, c1, null)).to.equal('B$3');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, c1, null)).to.equal('B$3');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, c1, null)).to.equal('Z3S[-1]');
                expect(enOoxA1Grammar.formatReference(docModel, ref, null, null, c1, null)).to.equal('B$3');
                expect(enOoxRCGrammar.formatReference(docModel, ref, null, null, c1, null)).to.equal('R3C[-1]');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, c1, null)).to.equal('[.B$3]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, c1, null)).to.equal('B$3');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, c1, null)).to.equal('R3C[-1]');
                expect(enOdfA1Grammar.formatReference(docModel, ref, null, null, c1, null)).to.equal('B$3');
                expect(enOdfRCGrammar.formatReference(docModel, ref, null, null, c1, null)).to.equal('R3C[-1]');
            });
            it('should return the string representation of a cell range address', function () {
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, c1, c2)).to.equal('Z3S[-1]:Z[1]S4');
                expect(enOoxA1Grammar.formatReference(docModel, ref, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(enOoxRCGrammar.formatReference(docModel, ref, null, null, c1, c2)).to.equal('R3C[-1]:R[1]C4');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, c1, c2)).to.equal('[.B$3:.$D5]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, c1, c2)).to.equal('R3C[-1]:R[1]C4');
                expect(enOdfA1Grammar.formatReference(docModel, ref, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(enOdfRCGrammar.formatReference(docModel, ref, null, null, c1, c2)).to.equal('R3C[-1]:R[1]C4');
            });
            it('should return the string representation of a reference error', function () {
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, null, null)).to.equal('#REF!');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, null, null)).to.equal('#BEZUG!');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, null, null)).to.equal('#BEZUG!');
                expect(enOoxA1Grammar.formatReference(docModel, ref, null, null, null, null)).to.equal('#REF!');
                expect(enOoxRCGrammar.formatReference(docModel, ref, null, null, null, null)).to.equal('#REF!');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, null, null)).to.equal('[#REF!]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, null, null)).to.equal('#BEZUG!');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, null, null)).to.equal('#BEZUG!');
                expect(enOdfA1Grammar.formatReference(docModel, ref, null, null, null, null)).to.equal('#REF!');
                expect(enOdfRCGrammar.formatReference(docModel, ref, null, null, null, null)).to.equal('#REF!');
            });
            it('should return the string representation of a cell address with sheet', function () {
                expect(opOoxA1Grammar.formatReference(docModel, ref, s1, null, c1, null)).to.equal('Sheet2!B$3');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, s1, null, c1, null)).to.equal('Sheet2!B$3');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, s1, null, c1, null)).to.equal('Sheet2!Z3S[-1]');
                expect(opOdfA1Grammar.formatReference(docModel, ref, s1, null, c1, null)).to.equal('[$Sheet2.B$3]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, s1, null, c1, null)).to.equal('$Sheet2!B$3');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, s1, null, c1, null)).to.equal('$Sheet2!R3C[-1]');
            });
            it('should return the string representation of a cell range address with sheet', function () {
                expect(opOoxA1Grammar.formatReference(docModel, ref, s1, null, c1, c2)).to.equal('Sheet2!B$3:$D5');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, s1, null, c1, c2)).to.equal('Sheet2!B$3:$D5');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, s1, null, c1, c2)).to.equal('Sheet2!Z3S[-1]:Z[1]S4');
                expect(opOdfA1Grammar.formatReference(docModel, ref, s1, null, c1, c2)).to.equal('[$Sheet2.B$3:.$D5]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, s1, null, c1, c2)).to.equal('$Sheet2!B$3:$D5');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, s1, null, c1, c2)).to.equal('$Sheet2!R3C[-1]:R[1]C4');
            });
            it('should return the string representation of a reference error with sheet', function () {
                expect(opOoxA1Grammar.formatReference(docModel, ref, s1, null, null, null)).to.equal('Sheet2!#REF!');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, s1, null, null, null)).to.equal('Sheet2!#BEZUG!');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, s1, null, null, null)).to.equal('Sheet2!#BEZUG!');
                expect(opOdfA1Grammar.formatReference(docModel, ref, s1, null, null, null)).to.equal('[#REF!]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, s1, null, null, null)).to.equal('#BEZUG!');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, s1, null, null, null)).to.equal('#BEZUG!');
            });
            it('should return the string representation of a reference with sheet range', function () {
                // cell address
                expect(opOoxA1Grammar.formatReference(docModel, ref, s1, s2, c1, null)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, s1, s2, c1, null)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                expect(opOdfA1Grammar.formatReference(docModel, ref, s1, s2, c1, null)).to.equal('[$Sheet2.B$3:$\'Sheet 4\'.B$3]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, s1, s2, c1, null)).to.equal('$Sheet2:$\'Sheet 4\'!B$3');
                // cell range address
                expect(opOoxA1Grammar.formatReference(docModel, ref, s1, s2, c1, c2)).to.equal('\'Sheet2:Sheet 4\'!B$3:$D5');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, s1, s2, c1, c2)).to.equal('\'Sheet2:Sheet 4\'!B$3:$D5');
                expect(opOdfA1Grammar.formatReference(docModel, ref, s1, s2, c1, c2)).to.equal('[$Sheet2.B$3:$\'Sheet 4\'.$D5]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, s1, s2, c1, c2)).to.equal('$Sheet2:$\'Sheet 4\'!B$3:$D5');
                // reference error
                expect(opOoxA1Grammar.formatReference(docModel, ref, s1, s2, null, null)).to.equal('\'Sheet2:Sheet 4\'!#REF!');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, s1, s2, null, null)).to.equal('\'Sheet2:Sheet 4\'!#BEZUG!');
                expect(opOdfA1Grammar.formatReference(docModel, ref, s1, s2, null, null)).to.equal('[#REF!]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, s1, s2, null, null)).to.equal('#BEZUG!');
            });
            it('should return the string representation of a sheet reference error', function () {
                // cell address
                expect(opOoxA1Grammar.formatReference(docModel, ref, s3, null, c1, null)).to.equal('#REF!');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, s3, null, c1, null)).to.equal('#BEZUG!');
                expect(opOdfA1Grammar.formatReference(docModel, ref, s3, null, c1, null)).to.equal('[#REF!]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, s3, null, c1, null)).to.equal('#BEZUG!');
                // cell range address
                expect(opOoxA1Grammar.formatReference(docModel, ref, s3, null, c1, c2)).to.equal('#REF!');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, s3, null, c1, c2)).to.equal('#BEZUG!');
                expect(opOdfA1Grammar.formatReference(docModel, ref, s3, null, c1, c2)).to.equal('[#REF!]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, s3, null, c1, c2)).to.equal('#BEZUG!');
                // reference error
                expect(opOoxA1Grammar.formatReference(docModel, ref, s3, null, null, null)).to.equal('#REF!');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, s3, null, null, null)).to.equal('#BEZUG!');
                expect(opOdfA1Grammar.formatReference(docModel, ref, s3, null, null, null)).to.equal('[#REF!]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, s3, null, null, null)).to.equal('#BEZUG!');
            });
            it('should return string representation of a column range', function () {
                // relative->absolute
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('B:$D');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('B:$D');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('S[-1]:S4');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('[.B$1:.$D$65536]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('B:$D');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('C[-1]:C4');
                // absolute->relative
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('$B:D');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('$B:D');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('S2:S[1]');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('[.$B$1:.D$65536]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('$B:D');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('C2:C[1]');
            });
            it('should return string representation of a single column', function () {
                // absolute
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('$B:$B');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('$B:$B');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('S2');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('[.$B$1:.$B$65536]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('$B:$B');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('C2');
                // relative
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('B:B');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('B:B');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('S[-1]');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('[.B$1:.B$65536]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('B:B');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('C[-1]');
            });
            it('should return string representation of a column range with sheet', function () {
                // single sheet
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('Sheet2!$B:B');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('Sheet2!$B:B');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, sh(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('Sheet2!S2:S[-1]');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('[$Sheet2.$B$1:.B$65536]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('$Sheet2!$B:B');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, sh(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('$Sheet2!C2:C[-1]');
                // sheet range
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh(1), sh(3), cell('$1.$0'), cell('1.$65535'))).to.equal('\'Sheet2:Sheet 4\'!$B:B');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh(1), sh(3), cell('$1.$0'), cell('1.$65535'))).to.equal('\'Sheet2:Sheet 4\'!$B:B');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh(1), sh(3), cell('$1.$0'), cell('1.$65535'))).to.equal('[$Sheet2.$B$1:$\'Sheet 4\'.B$65536]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh(1), sh(3), cell('$1.$0'), cell('1.$65535'))).to.equal('$Sheet2:$\'Sheet 4\'!$B:B');
            });
            it('should not return string representation of a column range', function () {
                // missing absolute flags
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('$B$1:$D65536');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('$B$1:$D65536');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('[.$B$1:.$D65536]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('$B$1:$D65536');
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('$B1:$D$65536');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('$B1:$D$65536');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('[.$B1:.$D$65536]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('$B1:$D$65536');
                // wrong indexes
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('$B$2:$D$65536');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('$B$2:$D$65536');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('[.$B$2:.$D$65536]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('$B$2:$D$65536');
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('$B$1:$D$65535');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('$B$1:$D$65535');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('[.$B$1:.$D$65535]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('$B$1:$D$65535');
            });
            it('should return string representation of a row range', function () {
                // relative->absolute
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('2:$4');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('2:$4');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('Z[-2]:Z4');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('[.$A2:.$IV$4]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('2:$4');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('R[-2]:R4');
                // absolute->relative
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('$2:4');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('$2:4');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('Z2:Z');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('[.$A$2:.$IV4]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('$2:4');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('R2:R');
            });
            it('should return string representation of a single row', function () {
                // absolute
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('$2:$2');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('$2:$2');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('Z2');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('[.$A$2:.$IV$2]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('$2:$2');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('R2');
                // relative
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.1'))).to.equal('2:2');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.1'))).to.equal('2:2');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.1'))).to.equal('Z[-2]');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.1'))).to.equal('[.$A2:.$IV2]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.1'))).to.equal('2:2');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, cell('$0.1'), cell('$255.1'))).to.equal('R[-2]');
            });
            it('should return string representation of a row range with sheet', function () {
                // single sheet
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('Sheet2!$2:2');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('Sheet2!$2:2');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, sh(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('Sheet2!Z2:Z[-2]');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('[$Sheet2.$A$2:.$IV2]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('$Sheet2!$2:2');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, sh(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('$Sheet2!R2:R[-2]');
                // sheet range
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh(1), sh(3), cell('$0.$1'), cell('$255.1'))).to.equal('\'Sheet2:Sheet 4\'!$2:2');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh(1), sh(3), cell('$0.$1'), cell('$255.1'))).to.equal('\'Sheet2:Sheet 4\'!$2:2');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh(1), sh(3), cell('$0.$1'), cell('$255.1'))).to.equal('[$Sheet2.$A$2:$\'Sheet 4\'.$IV2]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh(1), sh(3), cell('$0.$1'), cell('$255.1'))).to.equal('$Sheet2:$\'Sheet 4\'!$2:2');
            });
            it('should not return string representation of a row range', function () {
                // missing absolute flags
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('$A$2:IV$4');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('$A$2:IV$4');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('[.$A$2:.IV$4]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('$A$2:IV$4');
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('A$2:$IV$4');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('A$2:$IV$4');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('[.A$2:.$IV$4]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('A$2:$IV$4');
                // wrong indexes
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('$B$2:$IV$4');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('$B$2:$IV$4');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('[.$B$2:.$IV$4]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('$B$2:$IV$4');
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('$A$2:$IU$4');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('$A$2:$IU$4');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('[.$A$2:.$IU$4]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('$A$2:$IU$4');
            });
            it('should return string representation of a sheet range', function () {
                expect(opOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('$1:$65536');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('$1:$65536');
                expect(uiOoxRCGrammar.formatReference(docModel, ref, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('Z1:Z65536');
                expect(opOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('[.$A$1:.$IV$65536]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('$1:$65536');
                expect(uiOdfRCGrammar.formatReference(docModel, ref, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('R1:R65536');
            });
            it('should handle simple/complex sheet names correctly', function () {
                var c1 = cell('0.0');
                // simple name
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('Sheet1'), null, c1, null)).to.equal('Sheet1!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('Sheet1'), null, c1, null)).to.equal('Sheet1!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1'), null, c1, null)).to.equal('[$Sheet1.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1'), null, c1, null)).to.equal('$Sheet1!A1');
                // whitespace
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('Sheet 2'), null, c1, null)).to.equal('\'Sheet 2\'!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('Sheet 2'), null, c1, null)).to.equal('\'Sheet 2\'!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet 2'), null, c1, null)).to.equal('[$\'Sheet 2\'.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet 2'), null, c1, null)).to.equal('$\'Sheet 2\'!A1');
                // leading digit
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('1Sheet'), null, c1, null)).to.equal('\'1Sheet\'!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('1Sheet'), null, c1, null)).to.equal('\'1Sheet\'!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('1Sheet'), null, c1, null)).to.equal('[$\'1Sheet\'.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('1Sheet'), null, c1, null)).to.equal('$\'1Sheet\'!A1');
                // operator
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('Sheet+1'), null, c1, null)).to.equal('\'Sheet+1\'!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('Sheet+1'), null, c1, null)).to.equal('\'Sheet+1\'!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet+1'), null, c1, null)).to.equal('[$\'Sheet+1\'.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet+1'), null, c1, null)).to.equal('$\'Sheet+1\'!A1');
                // simple sheet range
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('Sheet1'), sh('Sheet2'), c1, null)).to.equal('Sheet1:Sheet2!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('Sheet1'), sh('Sheet2'), c1, null)).to.equal('Sheet1:Sheet2!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1'), sh('Sheet2'), c1, null)).to.equal('[$Sheet1.A1:$Sheet2.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1'), sh('Sheet2'), c1, null)).to.equal('$Sheet1:$Sheet2!A1');
                // complex first sheet
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('Sheet 1'), sh('Sheet2'), c1, null)).to.equal('\'Sheet 1:Sheet2\'!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('Sheet 1'), sh('Sheet2'), c1, null)).to.equal('\'Sheet 1:Sheet2\'!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet 1'), sh('Sheet2'), c1, null)).to.equal('[$\'Sheet 1\'.A1:$Sheet2.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet 1'), sh('Sheet2'), c1, null)).to.equal('$\'Sheet 1\':$Sheet2!A1');
                // complex second sheet
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('Sheet1'), sh('Sheet 2'), c1, null)).to.equal('\'Sheet1:Sheet 2\'!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('Sheet1'), sh('Sheet 2'), c1, null)).to.equal('\'Sheet1:Sheet 2\'!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1'), sh('Sheet 2'), c1, null)).to.equal('[$Sheet1.A1:$\'Sheet 2\'.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1'), sh('Sheet 2'), c1, null)).to.equal('$Sheet1:$\'Sheet 2\'!A1');
                // ODF: relative sheets
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1', false), null, c1, null)).to.equal('[Sheet1.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1', false), null, c1, null)).to.equal('Sheet1!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet 2', false), null, c1, null)).to.equal('[\'Sheet 2\'.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet 2', false), null, c1, null)).to.equal('\'Sheet 2\'!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1', false), sh('Sheet2'), c1, null)).to.equal('[Sheet1.A1:$Sheet2.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1', false), sh('Sheet2'), c1, null)).to.equal('Sheet1:$Sheet2!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1'), sh('Sheet2', false), c1, null)).to.equal('[$Sheet1.A1:Sheet2.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1'), sh('Sheet2', false), c1, null)).to.equal('$Sheet1:Sheet2!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1', false), sh('Sheet2', false), c1, null)).to.equal('[Sheet1.A1:Sheet2.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1', false), sh('Sheet2', false), c1, null)).to.equal('Sheet1:Sheet2!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet 1', false), sh('Sheet2', false), c1, null)).to.equal('[\'Sheet 1\'.A1:Sheet2.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet 1', false), sh('Sheet2', false), c1, null)).to.equal('\'Sheet 1\':Sheet2!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1', false), sh('Sheet 2', false), c1, null)).to.equal('[Sheet1.A1:\'Sheet 2\'.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Sheet1', false), sh('Sheet 2', false), c1, null)).to.equal('Sheet1:\'Sheet 2\'!A1');
            });
            it('should treat booleans as complex sheet names', function () {
                var c1 = cell('0.0');
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('TRUE'), null, c1, null)).to.equal('\'TRUE\'!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('TRUE'), null, c1, null)).to.equal('TRUE!A1');
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('WAHR'), null, c1, null)).to.equal('WAHR!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('WAHR'), null, c1, null)).to.equal('\'WAHR\'!A1');
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('TRUE'), sh('WAHR'), c1, null)).to.equal('\'TRUE:WAHR\'!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('TRUE'), sh('WAHR'), c1, null)).to.equal('\'TRUE:WAHR\'!A1');
            });
            it('should treat cell addresses as complex sheet names', function () {
                var c1 = cell('0.0');
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('A1'), null, c1, null)).to.equal('\'A1\'!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('A1'), null, c1, null)).to.equal('\'A1\'!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('A1'), null, c1, null)).to.equal('[$\'A1\'.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('A1'), null, c1, null)).to.equal('$\'A1\'!A1');
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('IW1'), null, c1, null)).to.equal('IW1!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('IW1'), null, c1, null)).to.equal('IW1!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('IW1'), null, c1, null)).to.equal('[$IW1.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('IW1'), null, c1, null)).to.equal('$IW1!A1');
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('R1C1'), null, c1, null)).to.equal('\'R1C1\'!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('R1C1'), null, c1, null)).to.equal('\'R1C1\'!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('R1C1'), null, c1, null)).to.equal('[$\'R1C1\'.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('R1C1'), null, c1, null)).to.equal('$\'R1C1\'!A1');
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh('Z1S1'), null, c1, null)).to.equal('Z1S1!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh('Z1S1'), null, c1, null)).to.equal('\'Z1S1\'!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh('Z1S1'), null, c1, null)).to.equal('[$Z1S1.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh('Z1S1'), null, c1, null)).to.equal('$Z1S1!A1');
                expect(opOoxA1Grammar.formatReference(docModel, ref, sh(1), sh('A1'), c1, null)).to.equal('\'Sheet2:A1\'!A1');
                expect(uiOoxA1Grammar.formatReference(docModel, ref, sh(1), sh('A1'), c1, null)).to.equal('\'Sheet2:A1\'!A1');
                expect(opOdfA1Grammar.formatReference(docModel, ref, sh(1), sh('A1'), c1, null)).to.equal('[$Sheet2.A1:$\'A1\'.A1]');
                expect(uiOdfA1Grammar.formatReference(docModel, ref, sh(1), sh('A1'), c1, null)).to.equal('$Sheet2:$\'A1\'!A1');
            });
        });

        describe('method "formatName"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('formatName');
                expect(uiOoxA1Grammar).to.respondTo('formatName');
            });
            var s1 = sh(1), s2 = sh(3), s3 = sh(-1);
            it('should return the string representation', function () {
                expect(opOoxA1Grammar.formatName(docModel, null, null, 'name')).to.equal('name');
                expect(uiOoxA1Grammar.formatName(docModel, null, null, 'name')).to.equal('name');
                expect(opOoxA1Grammar.formatName(docModel, null, s1, 'Name')).to.equal('Sheet2!Name');
                expect(uiOoxA1Grammar.formatName(docModel, null, s1, 'Name')).to.equal('Sheet2!Name');
                expect(opOoxA1Grammar.formatName(docModel, null, s2, 'NAME')).to.equal('\'Sheet 4\'!NAME');
                expect(uiOoxA1Grammar.formatName(docModel, null, s2, 'NAME')).to.equal('\'Sheet 4\'!NAME');
                expect(opOoxA1Grammar.formatName(docModel, null, s3, 'name')).to.equal('[0]!name');
                expect(uiOoxA1Grammar.formatName(docModel, null, s3, 'name')).to.equal('[0]!name');
            });
            it('should return the string representation for explicitly global names', function () {
                expect(opOoxA1Grammar.formatName(docModel, 0, null, 'name')).to.equal('[0]!name');
                expect(uiOoxA1Grammar.formatName(docModel, 0, null, 'name')).to.equal('[0]!name');
                expect(opOoxA1Grammar.formatName(docModel, 0, s1, 'Name')).to.equal('[0]Sheet2!Name');
                expect(uiOoxA1Grammar.formatName(docModel, 0, s1, 'Name')).to.equal('[0]Sheet2!Name');
                expect(opOoxA1Grammar.formatName(docModel, 0, s2, 'NAME')).to.equal('\'[0]Sheet 4\'!NAME');
                expect(uiOoxA1Grammar.formatName(docModel, 0, s2, 'NAME')).to.equal('\'[0]Sheet 4\'!NAME');
                expect(opOoxA1Grammar.formatName(docModel, 0, s3, 'name')).to.equal('[0]!name');
                expect(uiOoxA1Grammar.formatName(docModel, 0, s3, 'name')).to.equal('[0]!name');
            });
            it('should return the string representation for external names', function () {
                expect(opOoxA1Grammar.formatName(docModel, 1, null, 'name')).to.equal('[1]!name');
                expect(uiOoxA1Grammar.formatName(docModel, 1, null, 'name')).to.equal('[1]!name');
                expect(opOoxA1Grammar.formatName(docModel, 1, s1, 'Name')).to.equal('[1]!Name');
                expect(uiOoxA1Grammar.formatName(docModel, 1, s1, 'Name')).to.equal('[1]!Name');
                expect(opOoxA1Grammar.formatName(docModel, 1, sh('Sheet1'), 'Name')).to.equal('[1]Sheet1!Name');
                expect(uiOoxA1Grammar.formatName(docModel, 1, sh('Sheet1'), 'Name')).to.equal('[1]Sheet1!Name');
                expect(opOoxA1Grammar.formatName(docModel, 1, sh('Sheet1'), 'Name')).to.equal('[1]Sheet1!Name');
                expect(uiOoxA1Grammar.formatName(docModel, 1, sh('Sheet1'), 'Name')).to.equal('[1]Sheet1!Name');
            });
            it('should handle simple/complex sheet names correctly', function () {
                expect(opOoxA1Grammar.formatName(docModel, null, sh('Sheet1'), 'name')).to.equal('Sheet1!name');
                expect(uiOoxA1Grammar.formatName(docModel, null, sh('Sheet1'), 'name')).to.equal('Sheet1!name');
                expect(opOoxA1Grammar.formatName(docModel, null, sh('Sheet 2'), 'name')).to.equal('\'Sheet 2\'!name');
                expect(uiOoxA1Grammar.formatName(docModel, null, sh('Sheet 2'), 'name')).to.equal('\'Sheet 2\'!name');
                expect(opOoxA1Grammar.formatName(docModel, null, sh('1Sheet'), 'name')).to.equal('\'1Sheet\'!name');
                expect(uiOoxA1Grammar.formatName(docModel, null, sh('1Sheet'), 'name')).to.equal('\'1Sheet\'!name');
                expect(opOoxA1Grammar.formatName(docModel, null, sh('Sheet+1'), 'name')).to.equal('\'Sheet+1\'!name');
                expect(uiOoxA1Grammar.formatName(docModel, null, sh('Sheet+1'), 'name')).to.equal('\'Sheet+1\'!name');
            });
            it('should treat booleans as complex sheet names', function () {
                expect(opOoxA1Grammar.formatName(docModel, null, sh('TRUE'), 'name')).to.equal('\'TRUE\'!name');
                expect(uiOoxA1Grammar.formatName(docModel, null, sh('TRUE'), 'name')).to.equal('TRUE!name');
                expect(opOoxA1Grammar.formatName(docModel, null, sh('WAHR'), 'name')).to.equal('WAHR!name');
                expect(uiOoxA1Grammar.formatName(docModel, null, sh('WAHR'), 'name')).to.equal('\'WAHR\'!name');
            });
            it('should treat cell addresses as complex sheet names', function () {
                expect(opOoxA1Grammar.formatName(docModel, null, sh('A1'), 'name')).to.equal('\'A1\'!name');
                expect(uiOoxA1Grammar.formatName(docModel, null, sh('A1'), 'name')).to.equal('\'A1\'!name');
                expect(opOoxA1Grammar.formatName(docModel, null, sh('IW1'), 'name')).to.equal('IW1!name');
                expect(uiOoxA1Grammar.formatName(docModel, null, sh('IW1'), 'name')).to.equal('IW1!name');
                expect(opOoxA1Grammar.formatName(docModel, null, sh('R1C1'), 'name')).to.equal('\'R1C1\'!name');
                expect(uiOoxA1Grammar.formatName(docModel, null, sh('R1C1'), 'name')).to.equal('\'R1C1\'!name');
                expect(opOoxA1Grammar.formatName(docModel, null, sh('Z1S1'), 'name')).to.equal('Z1S1!name');
                expect(uiOoxA1Grammar.formatName(docModel, null, sh('Z1S1'), 'name')).to.equal('\'Z1S1\'!name');
            });
        });

        describe('method "formatMacro"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('formatMacro');
                expect(uiOoxA1Grammar).to.respondTo('formatMacro');
            });
            var s1 = sh(1), s2 = sh(3), s3 = sh(-1);
            it('should return the string representation', function () {
                expect(opOoxA1Grammar.formatMacro(docModel, null, null, 'func')).to.equal('func');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, null, 'func')).to.equal('func');
                expect(opOoxA1Grammar.formatMacro(docModel, null, s1, 'Func')).to.equal('Sheet2!Func');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, s1, 'Func')).to.equal('Sheet2!Func');
                expect(opOoxA1Grammar.formatMacro(docModel, null, s2, 'FUNC')).to.equal('\'Sheet 4\'!FUNC');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, s2, 'FUNC')).to.equal('\'Sheet 4\'!FUNC');
                expect(opOoxA1Grammar.formatMacro(docModel, null, s3, 'func')).to.equal('[0]!func');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, s3, 'func')).to.equal('[0]!func');
            });
            it('should omit UDF prefix in UI grammar', function () {
                expect(opOoxA1Grammar.formatMacro(docModel, null, null, '_xludf.sum')).to.equal('_xludf.sum');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, null, '_xludf.sum')).to.equal('sum');
                expect(opOoxA1Grammar.formatMacro(docModel, null, s1, '_xludf.Sum')).to.equal('Sheet2!_xludf.Sum');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, s1, '_xludf.Sum')).to.equal('Sheet2!Sum');
                expect(opOoxA1Grammar.formatMacro(docModel, null, s2, '_xludf.SUM')).to.equal('\'Sheet 4\'!_xludf.SUM');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, s2, '_xludf.SUM')).to.equal('\'Sheet 4\'!SUM');
                expect(opOoxA1Grammar.formatMacro(docModel, null, s3, '_xludf.sum')).to.equal('[0]!_xludf.sum');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, s3, '_xludf.sum')).to.equal('[0]!sum');
            });
            it('should add UDF prefix in operations', function () {
                expect(opOoxA1Grammar.formatMacro(docModel, null, null, 'sum')).to.equal('_xludf.sum');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, null, 'sum')).to.equal('sum');
                expect(opOoxA1Grammar.formatMacro(docModel, null, s1, 'Sum')).to.equal('Sheet2!_xludf.Sum');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, s1, 'Sum')).to.equal('Sheet2!Sum');
                expect(opOoxA1Grammar.formatMacro(docModel, null, s2, 'SUM')).to.equal('\'Sheet 4\'!_xludf.SUM');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, s2, 'SUM')).to.equal('\'Sheet 4\'!SUM');
                expect(opOoxA1Grammar.formatMacro(docModel, null, s3, 'sum')).to.equal('[0]!_xludf.sum');
                expect(uiOoxA1Grammar.formatMacro(docModel, null, s3, 'sum')).to.equal('[0]!sum');
            });
        });

        describe('method "getBooleanName"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('getBooleanName');
                expect(uiOoxA1Grammar).to.respondTo('getBooleanName');
            });
            it('should return native boolean names', function () {
                expect(opOoxA1Grammar.getBooleanName(false)).to.equal('FALSE');
                expect(opOoxA1Grammar.getBooleanName(true)).to.equal('TRUE');
            });
            it('should return localized boolean names', function () {
                expect(uiOoxA1Grammar.getBooleanName(false)).to.equal('FALSCH');
                expect(uiOoxA1Grammar.getBooleanName(true)).to.equal('WAHR');
            });
            it('should return English boolean names', function () {
                expect(enOoxA1Grammar.getBooleanName(false)).to.equal('FALSE');
                expect(enOoxA1Grammar.getBooleanName(true)).to.equal('TRUE');
            });
        });

        describe('method "getBooleanValue"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('getBooleanValue');
                expect(uiOoxA1Grammar).to.respondTo('getBooleanValue');
            });
            it('should parse native boolean names', function () {
                expect(opOoxA1Grammar.getBooleanValue('false')).to.equal(false);
                expect(opOoxA1Grammar.getBooleanValue('False')).to.equal(false);
                expect(opOoxA1Grammar.getBooleanValue('FALSE')).to.equal(false);
                expect(opOoxA1Grammar.getBooleanValue('true')).to.equal(true);
                expect(opOoxA1Grammar.getBooleanValue('True')).to.equal(true);
                expect(opOoxA1Grammar.getBooleanValue('TRUE')).to.equal(true);
            });
            it('should parse localized boolean names', function () {
                expect(uiOoxA1Grammar.getBooleanValue('falsch')).to.equal(false);
                expect(uiOoxA1Grammar.getBooleanValue('Falsch')).to.equal(false);
                expect(uiOoxA1Grammar.getBooleanValue('FALSCH')).to.equal(false);
                expect(uiOoxA1Grammar.getBooleanValue('wahr')).to.equal(true);
                expect(uiOoxA1Grammar.getBooleanValue('Wahr')).to.equal(true);
                expect(uiOoxA1Grammar.getBooleanValue('WAHR')).to.equal(true);
            });
            it('should parse English boolean names', function () {
                expect(enOoxA1Grammar.getBooleanValue('false')).to.equal(false);
                expect(enOoxA1Grammar.getBooleanValue('False')).to.equal(false);
                expect(enOoxA1Grammar.getBooleanValue('FALSE')).to.equal(false);
                expect(enOoxA1Grammar.getBooleanValue('true')).to.equal(true);
                expect(enOoxA1Grammar.getBooleanValue('True')).to.equal(true);
                expect(enOoxA1Grammar.getBooleanValue('TRUE')).to.equal(true);
            });
            it('should return null for invalid names', function () {
                expect(opOoxA1Grammar.getBooleanValue('1')).to.equal(null);
                expect(opOoxA1Grammar.getBooleanValue('abc')).to.equal(null);
                expect(opOoxA1Grammar.getBooleanValue('FALSCH')).to.equal(null);
                expect(uiOoxA1Grammar.getBooleanValue('1')).to.equal(null);
                expect(uiOoxA1Grammar.getBooleanValue('abc')).to.equal(null);
                expect(uiOoxA1Grammar.getBooleanValue('FALSE')).to.equal(null);
                expect(enOoxA1Grammar.getBooleanValue('1')).to.equal(null);
                expect(enOoxA1Grammar.getBooleanValue('abc')).to.equal(null);
                expect(enOoxA1Grammar.getBooleanValue('FALSCH')).to.equal(null);
            });
        });

        describe('method "getErrorName"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('getErrorName');
                expect(uiOoxA1Grammar).to.respondTo('getErrorName');
            });
            it('should return native error names', function () {
                expect(opOoxA1Grammar.getErrorName(ErrorCode.NULL)).to.equal('#NULL!');
                expect(opOoxA1Grammar.getErrorName(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(opOoxA1Grammar.getErrorName(ErrorCode.VALUE)).to.equal('#VALUE!');
                expect(opOoxA1Grammar.getErrorName(ErrorCode.REF)).to.equal('#REF!');
                expect(opOoxA1Grammar.getErrorName(ErrorCode.NAME)).to.equal('#NAME?');
                expect(opOoxA1Grammar.getErrorName(ErrorCode.NUM)).to.equal('#NUM!');
                expect(opOoxA1Grammar.getErrorName(ErrorCode.NA)).to.equal('#N/A');
                expect(opOoxA1Grammar.getErrorName(ErrorCode.DATA)).to.equal('#GETTING_DATA');
            });
            it('should return localized error names', function () {
                expect(uiOoxA1Grammar.getErrorName(ErrorCode.NULL)).to.equal('#NULL!');
                expect(uiOoxA1Grammar.getErrorName(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(uiOoxA1Grammar.getErrorName(ErrorCode.VALUE)).to.equal('#WERT!');
                expect(uiOoxA1Grammar.getErrorName(ErrorCode.REF)).to.equal('#BEZUG!');
                expect(uiOoxA1Grammar.getErrorName(ErrorCode.NAME)).to.equal('#NAME?');
                expect(uiOoxA1Grammar.getErrorName(ErrorCode.NUM)).to.equal('#ZAHL!');
                expect(uiOoxA1Grammar.getErrorName(ErrorCode.NA)).to.equal('#NV');
                expect(uiOoxA1Grammar.getErrorName(ErrorCode.DATA)).to.equal('#GETTING_DATA');
            });
            it('should return English error names', function () {
                expect(enOoxA1Grammar.getErrorName(ErrorCode.NULL)).to.equal('#NULL!');
                expect(enOoxA1Grammar.getErrorName(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(enOoxA1Grammar.getErrorName(ErrorCode.VALUE)).to.equal('#VALUE!');
                expect(enOoxA1Grammar.getErrorName(ErrorCode.REF)).to.equal('#REF!');
                expect(enOoxA1Grammar.getErrorName(ErrorCode.NAME)).to.equal('#NAME?');
                expect(enOoxA1Grammar.getErrorName(ErrorCode.NUM)).to.equal('#NUM!');
                expect(enOoxA1Grammar.getErrorName(ErrorCode.NA)).to.equal('#N/A');
                expect(enOoxA1Grammar.getErrorName(ErrorCode.DATA)).to.equal('#GETTING_DATA');
            });
        });

        describe('method "getErrorCode"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('getErrorCode');
                expect(uiOoxA1Grammar).to.respondTo('getErrorCode');
            });
            it('should parse native error names', function () {
                expect(opOoxA1Grammar.getErrorCode('#NULL!')).to.equal(ErrorCode.NULL);
                expect(opOoxA1Grammar.getErrorCode('#null!')).to.equal(ErrorCode.NULL);
                expect(opOoxA1Grammar.getErrorCode('#DIV/0!')).to.equal(ErrorCode.DIV0);
                expect(opOoxA1Grammar.getErrorCode('#VALUE!')).to.equal(ErrorCode.VALUE);
                expect(opOoxA1Grammar.getErrorCode('#REF!')).to.equal(ErrorCode.REF);
                expect(opOoxA1Grammar.getErrorCode('#NAME?')).to.equal(ErrorCode.NAME);
                expect(opOoxA1Grammar.getErrorCode('#NUM!')).to.equal(ErrorCode.NUM);
                expect(opOoxA1Grammar.getErrorCode('#N/A')).to.equal(ErrorCode.NA);
                expect(opOoxA1Grammar.getErrorCode('#GETTING_DATA')).to.equal(ErrorCode.DATA);
            });
            it('should parse localized error names', function () {
                expect(uiOoxA1Grammar.getErrorCode('#NULL!')).to.equal(ErrorCode.NULL);
                expect(uiOoxA1Grammar.getErrorCode('#DIV/0!')).to.equal(ErrorCode.DIV0);
                expect(uiOoxA1Grammar.getErrorCode('#WERT!')).to.equal(ErrorCode.VALUE);
                expect(uiOoxA1Grammar.getErrorCode('#wert!')).to.equal(ErrorCode.VALUE);
                expect(uiOoxA1Grammar.getErrorCode('#BEZUG!')).to.equal(ErrorCode.REF);
                expect(uiOoxA1Grammar.getErrorCode('#NAME?')).to.equal(ErrorCode.NAME);
                expect(uiOoxA1Grammar.getErrorCode('#ZAHL!')).to.equal(ErrorCode.NUM);
                expect(uiOoxA1Grammar.getErrorCode('#NV')).to.equal(ErrorCode.NA);
                expect(uiOoxA1Grammar.getErrorCode('#GETTING_DATA')).to.equal(ErrorCode.DATA);
            });
            it('should parse English error names', function () {
                expect(enOoxA1Grammar.getErrorCode('#NULL!')).to.equal(ErrorCode.NULL);
                expect(enOoxA1Grammar.getErrorCode('#null!')).to.equal(ErrorCode.NULL);
                expect(enOoxA1Grammar.getErrorCode('#DIV/0!')).to.equal(ErrorCode.DIV0);
                expect(enOoxA1Grammar.getErrorCode('#VALUE!')).to.equal(ErrorCode.VALUE);
                expect(enOoxA1Grammar.getErrorCode('#REF!')).to.equal(ErrorCode.REF);
                expect(enOoxA1Grammar.getErrorCode('#NAME?')).to.equal(ErrorCode.NAME);
                expect(enOoxA1Grammar.getErrorCode('#NUM!')).to.equal(ErrorCode.NUM);
                expect(enOoxA1Grammar.getErrorCode('#N/A')).to.equal(ErrorCode.NA);
                expect(enOoxA1Grammar.getErrorCode('#GETTING_DATA')).to.equal(ErrorCode.DATA);
            });
            it('should return null for invalid names', function () {
                expect(opOoxA1Grammar.getErrorCode('abc')).to.equal(null);
                expect(opOoxA1Grammar.getErrorCode('#NULL')).to.equal(null);
                expect(opOoxA1Grammar.getErrorCode('NULL')).to.equal(null);
                expect(opOoxA1Grammar.getErrorCode('#WERT!')).to.equal(null);
                expect(uiOoxA1Grammar.getErrorCode('abc')).to.equal(null);
                expect(uiOoxA1Grammar.getErrorCode('#NULL')).to.equal(null);
                expect(uiOoxA1Grammar.getErrorCode('NULL')).to.equal(null);
                expect(uiOoxA1Grammar.getErrorCode('#VALUE!')).to.equal(null);
                expect(enOoxA1Grammar.getErrorCode('abc')).to.equal(null);
                expect(enOoxA1Grammar.getErrorCode('#NULL')).to.equal(null);
                expect(enOoxA1Grammar.getErrorCode('NULL')).to.equal(null);
                expect(enOoxA1Grammar.getErrorCode('#WERT!')).to.equal(null);
            });
        });

        describe('method "getOperatorName"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('getOperatorName');
                expect(uiOoxA1Grammar).to.respondTo('getOperatorName');
            });
            it('should return native operator names', function () {
                expect(opOoxA1Grammar.getOperatorName('add')).to.equal('+');
                expect(opOoxA1Grammar.getOperatorName('le')).to.equal('<=');
                expect(opOoxA1Grammar.getOperatorName('list')).to.equal(',');
                expect(opOoxA1Grammar.getOperatorName('isect')).to.equal(' ');
                expect(opOdfA1Grammar.getOperatorName('add')).to.equal('+');
                expect(opOdfA1Grammar.getOperatorName('le')).to.equal('<=');
                expect(opOdfA1Grammar.getOperatorName('list')).to.equal('~');
                expect(opOdfA1Grammar.getOperatorName('isect')).to.equal('!');
            });
            it('should return localized operator names', function () {
                expect(uiOoxA1Grammar.getOperatorName('add')).to.equal('+');
                expect(uiOoxA1Grammar.getOperatorName('le')).to.equal('<=');
                expect(uiOoxA1Grammar.getOperatorName('list')).to.equal(';');
                expect(uiOoxA1Grammar.getOperatorName('isect')).to.equal(' ');
                expect(uiOdfA1Grammar.getOperatorName('add')).to.equal('+');
                expect(uiOdfA1Grammar.getOperatorName('le')).to.equal('<=');
                expect(uiOdfA1Grammar.getOperatorName('list')).to.equal(';');
                expect(uiOdfA1Grammar.getOperatorName('isect')).to.equal(' ');
            });
            it('should return English operator names', function () {
                expect(enOoxA1Grammar.getOperatorName('add')).to.equal('+');
                expect(enOoxA1Grammar.getOperatorName('le')).to.equal('<=');
                expect(enOoxA1Grammar.getOperatorName('list')).to.equal(',');
                expect(enOoxA1Grammar.getOperatorName('isect')).to.equal(' ');
                expect(enOdfA1Grammar.getOperatorName('add')).to.equal('+');
                expect(enOdfA1Grammar.getOperatorName('le')).to.equal('<=');
                expect(enOdfA1Grammar.getOperatorName('list')).to.equal(',');
                expect(enOdfA1Grammar.getOperatorName('isect')).to.equal(' ');
            });
            it('should return null for invalid keys', function () {
                expect(opOoxA1Grammar.getOperatorName('abc')).to.equal(null);
                expect(opOoxA1Grammar.getOperatorName('ADD')).to.equal(null);
                expect(opOoxA1Grammar.getOperatorName('+')).to.equal(null);
            });
        });

        describe('method "getOperatorKey"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('getOperatorKey');
                expect(uiOoxA1Grammar).to.respondTo('getOperatorKey');
            });
            it('should parse native operator names', function () {
                expect(opOoxA1Grammar.getOperatorKey('+')).to.equal('add');
                expect(opOoxA1Grammar.getOperatorKey('<=')).to.equal('le');
                expect(opOoxA1Grammar.getOperatorKey(',')).to.equal('list');
                expect(opOoxA1Grammar.getOperatorKey(' ')).to.equal('isect');
                expect(opOdfA1Grammar.getOperatorKey('+')).to.equal('add');
                expect(opOdfA1Grammar.getOperatorKey('<=')).to.equal('le');
                expect(opOdfA1Grammar.getOperatorKey('~')).to.equal('list');
                expect(opOdfA1Grammar.getOperatorKey('!')).to.equal('isect');
            });
            it('should parse localized operator names', function () {
                expect(uiOoxA1Grammar.getOperatorKey('+')).to.equal('add');
                expect(uiOoxA1Grammar.getOperatorKey('<=')).to.equal('le');
                expect(uiOoxA1Grammar.getOperatorKey(';')).to.equal('list');
                expect(uiOoxA1Grammar.getOperatorKey(' ')).to.equal('isect');
                expect(uiOdfA1Grammar.getOperatorKey('+')).to.equal('add');
                expect(uiOdfA1Grammar.getOperatorKey('<=')).to.equal('le');
                expect(uiOdfA1Grammar.getOperatorKey(';')).to.equal('list');
                expect(uiOdfA1Grammar.getOperatorKey(' ')).to.equal('isect');
            });
            it('should parse English operator names', function () {
                expect(enOoxA1Grammar.getOperatorKey('+')).to.equal('add');
                expect(enOoxA1Grammar.getOperatorKey('<=')).to.equal('le');
                expect(enOoxA1Grammar.getOperatorKey(',')).to.equal('list');
                expect(enOoxA1Grammar.getOperatorKey(' ')).to.equal('isect');
                expect(enOdfA1Grammar.getOperatorKey('+')).to.equal('add');
                expect(enOdfA1Grammar.getOperatorKey('<=')).to.equal('le');
                expect(enOdfA1Grammar.getOperatorKey(',')).to.equal('list');
                expect(enOdfA1Grammar.getOperatorKey(' ')).to.equal('isect');
            });
            it('should return null for invalid names', function () {
                expect(opOoxA1Grammar.getOperatorKey('abc')).to.equal(null);
                expect(opOoxA1Grammar.getOperatorKey('add')).to.equal(null);
                expect(opOoxA1Grammar.getOperatorKey(';')).to.equal(null);
                expect(uiOoxA1Grammar.getOperatorKey('abc')).to.equal(null);
                expect(uiOoxA1Grammar.getOperatorKey('add')).to.equal(null);
                expect(uiOoxA1Grammar.getOperatorKey(',')).to.equal(null);
                expect(enOoxA1Grammar.getOperatorKey('abc')).to.equal(null);
                expect(enOoxA1Grammar.getOperatorKey('add')).to.equal(null);
                expect(enOoxA1Grammar.getOperatorKey(';')).to.equal(null);
            });
        });

        describe('method "getFunctionName"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('getFunctionName');
                expect(uiOoxA1Grammar).to.respondTo('getFunctionName');
            });
            it('should return native function names', function () {
                expect(opOoxA1Grammar.getFunctionName('ABS')).to.equal('ABS');
                expect(opOoxA1Grammar.getFunctionName('SUM')).to.equal('SUM');
                expect(opOoxA1Grammar.getFunctionName('AGGREGATE')).to.equal('_xlfn.AGGREGATE');
                expect(opOdfA1Grammar.getFunctionName('ABS')).to.equal('ABS');
                expect(opOdfA1Grammar.getFunctionName('SUM')).to.equal('SUM');
                expect(opOdfA1Grammar.getFunctionName('AGGREGATE')).to.equal('COM.MICROSOFT.AGGREGATE');
            });
            it('should return localized function names', function () {
                expect(uiOoxA1Grammar.getFunctionName('ABS')).to.equal('ABS');
                expect(uiOoxA1Grammar.getFunctionName('SUM')).to.equal('SUMME');
                expect(uiOoxA1Grammar.getFunctionName('AGGREGATE')).to.equal('AGGREGAT');
                expect(uiOdfA1Grammar.getFunctionName('ABS')).to.equal('ABS');
                expect(uiOdfA1Grammar.getFunctionName('SUM')).to.equal('SUMME');
                expect(uiOdfA1Grammar.getFunctionName('AGGREGATE')).to.equal('AGGREGAT');
            });
            it('should return English function names', function () {
                expect(enOoxA1Grammar.getFunctionName('ABS')).to.equal('ABS');
                expect(enOoxA1Grammar.getFunctionName('SUM')).to.equal('SUM');
                expect(enOoxA1Grammar.getFunctionName('AGGREGATE')).to.equal('AGGREGATE');
                expect(enOdfA1Grammar.getFunctionName('ABS')).to.equal('ABS');
                expect(enOdfA1Grammar.getFunctionName('SUM')).to.equal('SUM');
                expect(enOdfA1Grammar.getFunctionName('AGGREGATE')).to.equal('AGGREGATE');
            });
            it('should return null for invalid keys', function () {
                expect(opOoxA1Grammar.getFunctionName('123')).to.equal(null);
                expect(opOoxA1Grammar.getFunctionName('abc')).to.equal(null);
                expect(opOoxA1Grammar.getFunctionName('abs')).to.equal(null);
                expect(opOoxA1Grammar.getFunctionName('ISLEAPYEAR')).to.equal(null);
                expect(uiOoxA1Grammar.getFunctionName('123')).to.equal(null);
                expect(uiOoxA1Grammar.getFunctionName('abc')).to.equal(null);
                expect(uiOoxA1Grammar.getFunctionName('abs')).to.equal(null);
                expect(uiOoxA1Grammar.getFunctionName('ISLEAPYEAR')).to.equal(null);
                expect(enOoxA1Grammar.getFunctionName('123')).to.equal(null);
                expect(enOoxA1Grammar.getFunctionName('abc')).to.equal(null);
                expect(enOoxA1Grammar.getFunctionName('abs')).to.equal(null);
                expect(enOoxA1Grammar.getFunctionName('ISLEAPYEAR')).to.equal(null);
            });
        });

        describe('method "getFunctionKey"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('getFunctionKey');
                expect(uiOoxA1Grammar).to.respondTo('getFunctionKey');
            });
            it('should parse native function names', function () {
                expect(opOoxA1Grammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(opOoxA1Grammar.getFunctionKey('abs')).to.equal('ABS');
                expect(opOoxA1Grammar.getFunctionKey('SUM')).to.equal('SUM');
                expect(opOoxA1Grammar.getFunctionKey('sum')).to.equal('SUM');
                expect(opOoxA1Grammar.getFunctionKey('_xlfn.AGGREGATE')).to.equal('AGGREGATE');
                expect(opOdfA1Grammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(opOdfA1Grammar.getFunctionKey('abs')).to.equal('ABS');
                expect(opOdfA1Grammar.getFunctionKey('SUM')).to.equal('SUM');
                expect(opOdfA1Grammar.getFunctionKey('sum')).to.equal('SUM');
                expect(opOdfA1Grammar.getFunctionKey('COM.MICROSOFT.AGGREGATE')).to.equal('AGGREGATE');
            });
            it('should parse localized function names', function () {
                expect(uiOoxA1Grammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(uiOoxA1Grammar.getFunctionKey('abs')).to.equal('ABS');
                expect(uiOoxA1Grammar.getFunctionKey('SUMME')).to.equal('SUM');
                expect(uiOoxA1Grammar.getFunctionKey('summe')).to.equal('SUM');
                expect(uiOoxA1Grammar.getFunctionKey('AGGREGAT')).to.equal('AGGREGATE');
                expect(uiOdfA1Grammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(uiOdfA1Grammar.getFunctionKey('abs')).to.equal('ABS');
                expect(uiOdfA1Grammar.getFunctionKey('SUMME')).to.equal('SUM');
                expect(uiOdfA1Grammar.getFunctionKey('summe')).to.equal('SUM');
                expect(uiOdfA1Grammar.getFunctionKey('AGGREGAT')).to.equal('AGGREGATE');
            });
            it('should parse English function names', function () {
                expect(enOoxA1Grammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(enOoxA1Grammar.getFunctionKey('abs')).to.equal('ABS');
                expect(enOoxA1Grammar.getFunctionKey('SUM')).to.equal('SUM');
                expect(enOoxA1Grammar.getFunctionKey('sum')).to.equal('SUM');
                expect(enOoxA1Grammar.getFunctionKey('AGGREGATE')).to.equal('AGGREGATE');
                expect(enOdfA1Grammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(enOdfA1Grammar.getFunctionKey('abs')).to.equal('ABS');
                expect(enOdfA1Grammar.getFunctionKey('SUM')).to.equal('SUM');
                expect(enOdfA1Grammar.getFunctionKey('sum')).to.equal('SUM');
                expect(enOdfA1Grammar.getFunctionKey('AGGREGATE')).to.equal('AGGREGATE');
            });
            it('should return null for invalid names', function () {
                expect(opOoxA1Grammar.getFunctionKey('123')).to.equal(null);
                expect(opOoxA1Grammar.getFunctionKey('abc')).to.equal(null);
                expect(opOoxA1Grammar.getFunctionKey('SUMME')).to.equal(null);
                expect(opOoxA1Grammar.getFunctionKey('ISLEAPYEAR')).to.equal(null);
                expect(opOoxA1Grammar.getFunctionKey('FORMULATEXT')).to.equal(null);
                expect(opOoxA1Grammar.getFunctionKey('FORMELTEXT')).to.equal(null);
                expect(uiOoxA1Grammar.getFunctionKey('123')).to.equal(null);
                expect(uiOoxA1Grammar.getFunctionKey('abc')).to.equal(null);
                expect(uiOoxA1Grammar.getFunctionKey('SUM')).to.equal(null);
                expect(uiOoxA1Grammar.getFunctionKey('ISTSCHALTJAHR')).to.equal(null);
            });
        });

        describe('method "getTableRegionName"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('getTableRegionName');
                expect(uiOoxA1Grammar).to.respondTo('getTableRegionName');
            });
            it('should return native region names', function () {
                expect(opOoxA1Grammar.getTableRegionName('ALL')).to.equal('#All');
                expect(opOoxA1Grammar.getTableRegionName('HEADERS')).to.equal('#Headers');
                expect(opOoxA1Grammar.getTableRegionName('DATA')).to.equal('#Data');
                expect(opOoxA1Grammar.getTableRegionName('TOTALS')).to.equal('#Totals');
                expect(opOoxA1Grammar.getTableRegionName('ROW')).to.equal('#This Row');
                expect(opOdfA1Grammar.getTableRegionName('ALL')).to.equal('#All');
                expect(opOdfA1Grammar.getTableRegionName('HEADERS')).to.equal('#Headers');
                expect(opOdfA1Grammar.getTableRegionName('DATA')).to.equal('#Data');
                expect(opOdfA1Grammar.getTableRegionName('TOTALS')).to.equal('#Totals');
                expect(opOdfA1Grammar.getTableRegionName('ROW')).to.equal('#This Row');
            });
            it('should return localized region names', function () {
                expect(uiOoxA1Grammar.getTableRegionName('ALL')).to.equal('#Alle');
                expect(uiOoxA1Grammar.getTableRegionName('HEADERS')).to.equal('#Kopfzeilen');
                expect(uiOoxA1Grammar.getTableRegionName('DATA')).to.equal('#Daten');
                expect(uiOoxA1Grammar.getTableRegionName('TOTALS')).to.equal('#Ergebnisse');
                expect(uiOoxA1Grammar.getTableRegionName('ROW')).to.equal('#Diese Zeile');
                expect(uiOdfA1Grammar.getTableRegionName('ALL')).to.equal('#Alle');
                expect(uiOdfA1Grammar.getTableRegionName('HEADERS')).to.equal('#Kopfzeilen');
                expect(uiOdfA1Grammar.getTableRegionName('DATA')).to.equal('#Daten');
                expect(uiOdfA1Grammar.getTableRegionName('TOTALS')).to.equal('#Ergebnisse');
                expect(uiOdfA1Grammar.getTableRegionName('ROW')).to.equal('#Diese Zeile');
            });
            it('should return English region names', function () {
                expect(enOoxA1Grammar.getTableRegionName('ALL')).to.equal('#All');
                expect(enOoxA1Grammar.getTableRegionName('HEADERS')).to.equal('#Headers');
                expect(enOoxA1Grammar.getTableRegionName('DATA')).to.equal('#Data');
                expect(enOoxA1Grammar.getTableRegionName('TOTALS')).to.equal('#Totals');
                expect(enOoxA1Grammar.getTableRegionName('ROW')).to.equal('#This Row');
                expect(enOdfA1Grammar.getTableRegionName('ALL')).to.equal('#All');
                expect(enOdfA1Grammar.getTableRegionName('HEADERS')).to.equal('#Headers');
                expect(enOdfA1Grammar.getTableRegionName('DATA')).to.equal('#Data');
                expect(enOdfA1Grammar.getTableRegionName('TOTALS')).to.equal('#Totals');
                expect(enOdfA1Grammar.getTableRegionName('ROW')).to.equal('#This Row');
            });
            it('should return null for invalid keys', function () {
                expect(opOoxA1Grammar.getTableRegionName('all')).to.equal(null);
                expect(opOoxA1Grammar.getTableRegionName('#All')).to.equal(null);
                expect(opOoxA1Grammar.getTableRegionName(' ')).to.equal(null);
            });
        });

        describe('method "getTableRegionKey"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('getTableRegionKey');
                expect(uiOoxA1Grammar).to.respondTo('getTableRegionKey');
            });
            it('should parse native region names', function () {
                expect(opOoxA1Grammar.getTableRegionKey('#all')).to.equal('ALL');
                expect(opOoxA1Grammar.getTableRegionKey('#All')).to.equal('ALL');
                expect(opOoxA1Grammar.getTableRegionKey('#ALL')).to.equal('ALL');
                expect(opOoxA1Grammar.getTableRegionKey('#Headers')).to.equal('HEADERS');
                expect(opOoxA1Grammar.getTableRegionKey('#Data')).to.equal('DATA');
                expect(opOoxA1Grammar.getTableRegionKey('#Totals')).to.equal('TOTALS');
                expect(opOoxA1Grammar.getTableRegionKey('#This Row')).to.equal('ROW');
                expect(opOdfA1Grammar.getTableRegionKey('#All')).to.equal('ALL');
                expect(opOdfA1Grammar.getTableRegionKey('#Headers')).to.equal('HEADERS');
                expect(opOdfA1Grammar.getTableRegionKey('#Data')).to.equal('DATA');
                expect(opOdfA1Grammar.getTableRegionKey('#Totals')).to.equal('TOTALS');
                expect(opOdfA1Grammar.getTableRegionKey('#This Row')).to.equal('ROW');
            });
            it('should parse localized region names', function () {
                expect(uiOoxA1Grammar.getTableRegionKey('#alle')).to.equal('ALL');
                expect(uiOoxA1Grammar.getTableRegionKey('#Alle')).to.equal('ALL');
                expect(uiOoxA1Grammar.getTableRegionKey('#ALLE')).to.equal('ALL');
                expect(uiOoxA1Grammar.getTableRegionKey('#Kopfzeilen')).to.equal('HEADERS');
                expect(uiOoxA1Grammar.getTableRegionKey('#Daten')).to.equal('DATA');
                expect(uiOoxA1Grammar.getTableRegionKey('#Ergebnisse')).to.equal('TOTALS');
                expect(uiOoxA1Grammar.getTableRegionKey('#Diese Zeile')).to.equal('ROW');
                expect(uiOdfA1Grammar.getTableRegionKey('#Alle')).to.equal('ALL');
                expect(uiOdfA1Grammar.getTableRegionKey('#Kopfzeilen')).to.equal('HEADERS');
                expect(uiOdfA1Grammar.getTableRegionKey('#Daten')).to.equal('DATA');
                expect(uiOdfA1Grammar.getTableRegionKey('#Ergebnisse')).to.equal('TOTALS');
                expect(uiOdfA1Grammar.getTableRegionKey('#Diese Zeile')).to.equal('ROW');
            });
            it('should parse English region names', function () {
                expect(enOoxA1Grammar.getTableRegionKey('#all')).to.equal('ALL');
                expect(enOoxA1Grammar.getTableRegionKey('#All')).to.equal('ALL');
                expect(enOoxA1Grammar.getTableRegionKey('#ALL')).to.equal('ALL');
                expect(enOoxA1Grammar.getTableRegionKey('#Headers')).to.equal('HEADERS');
                expect(enOoxA1Grammar.getTableRegionKey('#Data')).to.equal('DATA');
                expect(enOoxA1Grammar.getTableRegionKey('#Totals')).to.equal('TOTALS');
                expect(enOoxA1Grammar.getTableRegionKey('#This Row')).to.equal('ROW');
                expect(enOdfA1Grammar.getTableRegionKey('#All')).to.equal('ALL');
                expect(enOdfA1Grammar.getTableRegionKey('#Headers')).to.equal('HEADERS');
                expect(enOdfA1Grammar.getTableRegionKey('#Data')).to.equal('DATA');
                expect(enOdfA1Grammar.getTableRegionKey('#Totals')).to.equal('TOTALS');
                expect(enOdfA1Grammar.getTableRegionKey('#This Row')).to.equal('ROW');
            });
            it('should return null for invalid names', function () {
                expect(opOoxA1Grammar.getTableRegionKey('All')).to.equal(null);
                expect(opOoxA1Grammar.getTableRegionKey('#Alle')).to.equal(null);
                expect(opOoxA1Grammar.getTableRegionKey('#ThisRow')).to.equal(null);
                expect(uiOoxA1Grammar.getTableRegionKey('Alle')).to.equal(null);
                expect(uiOoxA1Grammar.getTableRegionKey('#All')).to.equal(null);
                expect(uiOoxA1Grammar.getTableRegionKey('#DieseZeile')).to.equal(null);
                expect(enOoxA1Grammar.getTableRegionKey('All')).to.equal(null);
                expect(enOoxA1Grammar.getTableRegionKey('#Alle')).to.equal(null);
                expect(enOoxA1Grammar.getTableRegionKey('#ThisRow')).to.equal(null);
            });
        });

        describe('method "isSimpleTableColumn"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('isSimpleTableColumn');
                expect(uiOoxA1Grammar).to.respondTo('isSimpleTableColumn');
            });
            it('should return true for simple table columns', function () {
                expect(opOoxA1Grammar.isSimpleTableColumn('Col1')).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleTableColumn('Col1')).to.equal(true);
                expect(opOoxA1Grammar.isSimpleTableColumn('Col\xA11')).to.equal(true);
                expect(uiOoxA1Grammar.isSimpleTableColumn('Col\xA11')).to.equal(true);
            });
            it('should return false for complex table columns', function () {
                expect(opOoxA1Grammar.isSimpleTableColumn('Col 1')).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleTableColumn('Col 1')).to.equal(false);
                expect(opOoxA1Grammar.isSimpleTableColumn('Col!1')).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleTableColumn('Col!1')).to.equal(false);
                expect(opOoxA1Grammar.isSimpleTableColumn('Col_1')).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleTableColumn('Col_1')).to.equal(false);
            });
            it('should return result for comma character according to grammar', function () {
                expect(opOoxA1Grammar.isSimpleTableColumn('Col,1')).to.equal(false);
                expect(uiOoxA1Grammar.isSimpleTableColumn('Col,1')).to.equal(true);
                expect(enOoxA1Grammar.isSimpleTableColumn('Col,1')).to.equal(false);
            });
        });

        describe('method "encodeTableColumn"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('encodeTableColumn');
                expect(uiOoxA1Grammar).to.respondTo('encodeTableColumn');
            });
            it('should return encoded table columns', function () {
                expect(opOoxA1Grammar.encodeTableColumn('Col1')).to.equal('Col1');
                expect(uiOoxA1Grammar.encodeTableColumn('Col1')).to.equal('Col1');
                expect(opOoxA1Grammar.encodeTableColumn(' !:*+{')).to.equal(' !:*+{');
                expect(uiOoxA1Grammar.encodeTableColumn(' !:*+{')).to.equal(' !:*+{');
                expect(opOoxA1Grammar.encodeTableColumn('[]\'#@')).to.equal('\'[\']\'\'\'#@');
                expect(uiOoxA1Grammar.encodeTableColumn('[]\'#@')).to.equal('\'[\']\'\'\'#\'@');
            });
        });

        describe('method "decodeTableColumn"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('decodeTableColumn');
                expect(uiOoxA1Grammar).to.respondTo('decodeTableColumn');
            });
            it('should return encoded table columns', function () {
                expect(opOoxA1Grammar.decodeTableColumn('Col\'\'\'#\'1')).to.equal('Col\'#1');
                expect(uiOoxA1Grammar.decodeTableColumn('Col\'\'\'#\'1')).to.equal('Col\'#1');
            });
        });

        describe('method "isSpaceIntersection"', function () {
            it('should exist', function () {
                expect(opOoxA1Grammar).to.respondTo('isSpaceIntersection');
                expect(uiOoxA1Grammar).to.respondTo('isSpaceIntersection');
            });
            it('should return whether intersection operator is space', function () {
                expect(opOoxA1Grammar.isSpaceIntersection()).to.equal(true);
                expect(opOdfA1Grammar.isSpaceIntersection()).to.equal(false);
                expect(uiOoxA1Grammar.isSpaceIntersection()).to.equal(true);
                expect(uiOdfA1Grammar.isSpaceIntersection()).to.equal(true);
            });
        });
    });

    // ========================================================================
});
