/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/interpret/formulainterpreter'
], function (AppHelper, SheetHelper, FormulaUtils, FormulaInterpreter) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;

    // class Interpreter ======================================================

    describe('Spreadsheet class FormulaInterpreter', function () {

        it('should exist', function () {
            expect(FormulaInterpreter).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // initialize test document
        var interpreter = null;
        AppHelper.createSpreadsheetApp('ooxml').done(function (app) {
            interpreter = new FormulaInterpreter(app.getModel());
        });

        // public methods -----------------------------------------------------

        describe('constructor', function () {
            it('should create an instance', function () {
                expect(interpreter).to.be.an.instanceof(FormulaInterpreter);
            });
        });

        describe('method "equalScalars"', function () {
            it('should exist', function () {
                expect(interpreter).to.respondTo('equalScalars');
            });
            it('should compare scalars', function () {
                expect(interpreter.equalScalars(0, 0)).to.equal(true);
                expect(interpreter.equalScalars(0, 42)).to.equal(false);
                expect(interpreter.equalScalars('', '')).to.equal(true);
                expect(interpreter.equalScalars('', 'abc')).to.equal(false);
                expect(interpreter.equalScalars('abc', 'ABC')).to.equal(true);
                expect(interpreter.equalScalars(false, false)).to.equal(true);
                expect(interpreter.equalScalars(false, true)).to.equal(false);
                expect(interpreter.equalScalars(ErrorCode.NULL, ErrorCode.NULL)).to.equal(true);
                expect(interpreter.equalScalars(ErrorCode.NULL, ErrorCode.VALUE)).to.equal(false);
                expect(interpreter.equalScalars(0, '')).to.equal(false);
                expect(interpreter.equalScalars(0, false)).to.equal(false);
                expect(interpreter.equalScalars('', false)).to.equal(false);
            });
        });

        describe('method "compareScalars"', function () {
            it('should exist', function () {
                expect(interpreter).to.respondTo('compareScalars');
            });
            it('should compare scalars', function () {
                expect(interpreter.compareScalars(0, 0)).to.equal(0);
                expect(interpreter.compareScalars(0, 42)).to.equal(-1);
                expect(interpreter.compareScalars(42, 0)).to.equal(1);
                expect(interpreter.compareScalars('', '')).to.equal(0);
                expect(interpreter.compareScalars('', 'abc')).to.equal(-1);
                expect(interpreter.compareScalars('abc', '')).to.equal(1);
                expect(interpreter.compareScalars('abc', 'ABC')).to.equal(0);
                expect(interpreter.compareScalars(false, false)).to.equal(0);
                expect(interpreter.compareScalars(false, true)).to.equal(-1);
                expect(interpreter.compareScalars(true, false)).to.equal(1);
                expect(interpreter.compareScalars(ErrorCode.NULL, ErrorCode.NULL)).to.equal(0);
                expect(interpreter.compareScalars(ErrorCode.NULL, ErrorCode.VALUE)).to.equal(-1);
                expect(interpreter.compareScalars(ErrorCode.VALUE, ErrorCode.NULL)).to.equal(1);
                expect(interpreter.compareScalars(0, '')).to.equal(-1);
                expect(interpreter.compareScalars(0, false)).to.equal(-1);
                expect(interpreter.compareScalars('', false)).to.equal(-1);
            });
        });

        describe('method "createWarnResult"', function () {
            it('should exist', function () {
                expect(interpreter).to.respondTo('createWarnResult');
            });
            it('should create a warning result', function () {
                expect(interpreter.createWarnResult('val', ErrorCode.NULL)).to.deep.equal({ type: 'warn', code: 'NULL', value: ErrorCode.NULL });
                expect(interpreter.createWarnResult('ref', FormulaUtils.CIRCULAR_ERROR)).to.deep.equal({ type: 'warn', code: 'circular', value: 0 });
                expect(interpreter.createWarnResult('any', FormulaUtils.UNSUPPORTED_ERROR)).to.deep.equal({ type: 'warn', code: 'unsupported', value: ErrorCode.NA });
            });
            it('should create a matrix result for matrix context', function () {
                expect(interpreter.createWarnResult('mat', ErrorCode.NULL)).to.have.a.property('value').that.stringifiesTo('{#NULL}');
                expect(interpreter.createWarnResult('mat', FormulaUtils.CIRCULAR_ERROR)).to.have.a.property('value').that.stringifiesTo('{0}');
                expect(interpreter.createWarnResult('mat', FormulaUtils.UNSUPPORTED_ERROR)).to.have.a.property('value').that.stringifiesTo('{#NA}');
            });
        });

        describe('method "createErrorResult"', function () {
            it('should exist', function () {
                expect(interpreter).to.respondTo('createErrorResult');
            });
            it('should create a warning result', function () {
                expect(interpreter.createErrorResult('val', 'missing')).to.deep.equal({ type: 'error', code: 'missing', value: ErrorCode.NA });
            });
            it('should create a matrix result for matrix context', function () {
                expect(interpreter.createErrorResult('mat', 'missing')).to.have.a.property('value').that.stringifiesTo('{#NA}');
            });
        });

        describe('method "interpretTokens"', function () {
            it('should exist', function () {
                expect(interpreter).to.respondTo('interpretTokens');
            });
        });
    });

    // ========================================================================
});
