/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/model/operationcontext',
    'io.ox/office/spreadsheet/model/colrowcollection'
], function (AppHelper, SheetHelper, Utils, OperationContext, ColRowCollection) {

    'use strict';

    // convenience shortcuts
    var i = SheetHelper.i;

    // class ColRowCollection =================================================

    describe('Spreadsheet class ColRowCollection', function () {

        it('should exist', function () {
            expect(ColRowCollection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a0', attrs: {}, default: true },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1', attrs: { column: { width: 9 } } }
        ];

        // initialize test document
        var docModel = null, sheetModel = null, colCollection = null, colFactor = 1;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            sheetModel = docModel.getSheetModel(0);
            colCollection = sheetModel.getColCollection();
            colFactor = Utils.convertLength(docModel.getDefaultDigitWidth(), 'px', 'mm') * 100;
        });

        // creates a new operation context
        function op(json) {
            return new OperationContext(docModel, json);
        }

        function getEntries(collection) {
            return collection._getCloneData().entryModels;
        }

        function getColOffset(index) {
            return index * Math.ceil(9 * colFactor);
        }

        // private methods ----------------------------------------------------

        describe('method "_getCloneData"', function () {
            it('should exist', function () {
                expect(colCollection).to.respondTo('_getCloneData');
            });
            it('should return the internal data of the collection', function () {
                var cloneData = colCollection._getCloneData();
                expect(cloneData).to.be.an('object');
                expect(cloneData).to.have.a.property('entryModels').that.is.an('array').and.is.empty;
            });
        });

        // operation implementations ------------------------------------------

        describe('method "applyChangeOperation"', function () {
            it('should exist', function () {
                expect(colCollection).to.respondTo('applyChangeOperation');
            });
            it('should insert a new collection entry with an auto-style', function () {
                colCollection.applyChangeOperation(op({ start: 2, end: 6, s: 'a1' }));
                var entries = getEntries(colCollection);
                expect(entries).to.have.length(1);
                var colModel1 = entries[0];
                expect(colModel1).to.have.a.property('interval').that.deep.equals(i('C:G'));
                expect(colModel1).to.have.a.property('style').that.equals('a1');
                expect(colModel1).to.have.a.property('size').that.is.a('number');
                expect(colModel1).to.have.a.property('sizeHmm', getColOffset(1));
                expect(colModel1).to.have.a.property('offset').that.is.a('number');
                expect(colModel1).to.have.a.property('offsetHmm', getColOffset(2));
                expect(colModel1).to.respondTo('getExplicitEntryAttributes');
                expect(colModel1.getExplicitEntryAttributes()).to.deep.equal({});
            });
            it('should shorten collection entry at the beginning', function () {
                colCollection.applyChangeOperation(op({ start: 1, end: 2, s: 'a2' }));
                var entries = getEntries(colCollection);
                expect(entries).to.have.length(2);
                var colModel1 = entries[0];
                expect(colModel1).to.have.a.property('interval').that.deep.equals(i('B:C'));
                expect(colModel1).to.have.a.property('style').that.equals('a2');
                var colModel2 = entries[1];
                expect(colModel2).to.have.a.property('interval').that.deep.equals(i('D:G'));
                expect(colModel2).to.have.a.property('style').that.equals('a1');
            });
            it('should shorten collection entry at the end', function () {
                colCollection.applyChangeOperation(op({ start: 6, end: 7, s: 'a2' }));
                var entries = getEntries(colCollection);
                expect(entries).to.have.length(3);
                var colModel1 = entries[1];
                expect(colModel1).to.have.a.property('interval').that.deep.equals(i('D:F'));
                expect(colModel1).to.have.a.property('style').that.equals('a1');
                var colModel2 = entries[2];
                expect(colModel2).to.have.a.property('interval').that.deep.equals(i('G:H'));
                expect(colModel2).to.have.a.property('style').that.equals('a2');
            });
            it('should split collection entry in the middle', function () {
                colCollection.applyChangeOperation(op({ start: 4, s: 'a2' }));
                var entries = getEntries(colCollection);
                expect(entries).to.have.length(5);
                var colModel1 = entries[1];
                expect(colModel1).to.have.a.property('interval').that.deep.equals(i('D:D'));
                expect(colModel1).to.have.a.property('style').that.equals('a1');
                var colModel2 = entries[2];
                expect(colModel2).to.have.a.property('interval').that.deep.equals(i('E:E'));
                expect(colModel2).to.have.a.property('style').that.equals('a2');
                var colModel3 = entries[3];
                expect(colModel3).to.have.a.property('interval').that.deep.equals(i('F:F'));
                expect(colModel3).to.have.a.property('style').that.equals('a1');
            });
            it('should merge collection entries before', function () {
                colCollection.applyChangeOperation(op({ start: 0, s: 'a2' }));
                var entries = getEntries(colCollection);
                expect(entries).to.have.length(5);
                var colModel1 = entries[0];
                expect(colModel1).to.have.a.property('interval').that.deep.equals(i('A:C'));
                expect(colModel1).to.have.a.property('style').that.equals('a2');
            });
            it('should merge collection entries in the middle', function () {
                colCollection.applyChangeOperation(op({ start: 3, s: 'a2' }));
                var entries = getEntries(colCollection);
                expect(entries).to.have.length(3);
                var colModel1 = entries[0];
                expect(colModel1).to.have.a.property('interval').that.deep.equals(i('A:E'));
                expect(colModel1).to.have.a.property('style').that.equals('a2');
            });
            it('should merge collection entries after', function () {
                colCollection.applyChangeOperation(op({ start: 8, s: 'a2' }));
                var entries = getEntries(colCollection);
                expect(entries).to.have.length(3);
                var colModel1 = entries[2];
                expect(colModel1).to.have.a.property('interval').that.deep.equals(i('G:I'));
                expect(colModel1).to.have.a.property('style').that.equals('a2');
            });
            it('should remove collection entries with default format', function () {
                colCollection.applyChangeOperation(op({ start: 0, end: 1, s: '' }));
                var entries = getEntries(colCollection);
                expect(entries).to.have.length(3);
                var colModel1 = entries[0];
                expect(colModel1).to.have.a.property('interval').that.deep.equals(i('C:E'));
                expect(colModel1).to.have.a.property('style').that.equals('a2');
            });
            it('should insert a new collection entry with attributes', function () {
                colCollection.applyChangeOperation(op({ start: 12, end: 16, attrs: { column: { visible: false } } }));
                var entries = getEntries(colCollection);
                expect(entries).to.have.length(4);
                var colModel1 = entries[3];
                expect(colModel1).to.have.a.property('interval').that.deep.equals(i('M:Q'));
                expect(colModel1).to.have.a.property('style').that.equals('');
                expect(colModel1.getExplicitEntryAttributes()).to.deep.equal({ visible: false });
                expect(colModel1).to.have.a.property('sizeHmm', 0);
                expect(colModel1).to.have.a.property('offsetHmm', getColOffset(12));
            });
            it('should merge attributes of existing collection entries', function () {
                colCollection.applyChangeOperation(op({ start: 15, end: 18, attrs: { column: { width: 42 } } }));
                var entries = getEntries(colCollection);
                expect(entries).to.have.length(6);
                var colModel1 = entries[3];
                expect(colModel1).to.have.a.property('interval').that.deep.equals(i('M:O'));
                expect(colModel1.getExplicitEntryAttributes()).to.deep.equal({ visible: false });
                expect(colModel1).to.have.a.property('offsetHmm', getColOffset(12));
                var colModel2 = entries[4];
                expect(colModel2).to.have.a.property('interval').that.deep.equals(i('P:Q'));
                expect(colModel2.getExplicitEntryAttributes()).to.deep.equal({ width: 42, visible: false });
                expect(colModel1).to.have.a.property('offsetHmm', getColOffset(12));
                var colModel3 = entries[5];
                expect(colModel3).to.have.a.property('interval').that.deep.equals(i('R:S'));
                expect(colModel3.getExplicitEntryAttributes()).to.deep.equal({ width: 42 });
                expect(colModel1).to.have.a.property('offsetHmm', getColOffset(12));
            });
            it('should remove collection entries with default format', function () {
                colCollection.applyChangeOperation(op({ start: 14, attrs: { column: { visible: null } } }));
                var entries = getEntries(colCollection);
                expect(entries).to.have.length(6);
                var colModel1 = entries[3];
                expect(colModel1).to.have.a.property('interval').that.deep.equals(i('M:N'));
                expect(colModel1.getExplicitEntryAttributes()).to.deep.equal({ visible: false });
                expect(colModel1).to.have.a.property('offsetHmm', getColOffset(12));
                var colModel2 = entries[4];
                expect(colModel2).to.have.a.property('interval').that.deep.equals(i('P:Q'));
                expect(colModel2).to.have.a.property('offsetHmm', getColOffset(13));
                var colModel3 = entries[5];
                expect(colModel3).to.have.a.property('interval').that.deep.equals(i('R:S'));
                expect(colModel3).to.have.a.property('offsetHmm', getColOffset(13));
            });
        });

        // public methods -----------------------------------------------------
    });

    // ========================================================================
});
